/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.Socket;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 *
 * @author brownrigg
 * @version $Id: WMSHttpServletRequest.java 3361 2007-10-23 18:38:52Z rick $
 */
public class WMSHttpServletRequest {
    
    /** Creates a new instance of WMSHttpServletRequest */
    public WMSHttpServletRequest(Socket s) throws IOException {
        
        BufferedReader inp = new BufferedReader(
                new InputStreamReader(s.getInputStream()));
        
        // a generic error message used in any exceptions to be thrown...
        String errMsg = "Invalid HTTP request";
        
        // the first line is special...        
        String line = inp.readLine();
        StringTokenizer parser = new StringTokenizer(line);

        if (parser.countTokens() < 3) 
            throw new IOException(errMsg);
        method = parser.nextToken();
        url = parser.nextToken();
        String protocol = parser.nextToken();
        
        if (!method.equalsIgnoreCase("GET") && !method.equalsIgnoreCase("POST"))
            throw new IOException(errMsg);

        // break apart request and the query-string...
        int i = url.indexOf('?');
        if (i >= 0 && (i+1) != url.length()) {
            queryString = HttpUtil.decodeEscapes(url.substring(i+1));
            url = url.substring(0, i);
        }
    
        // Gather up the headers. For now, we'll ignore the body...        
        while (true) {
            line = inp.readLine();
            if (line == null || line.length() == 0) break;
            headers.add(line);
        }
    }
    
    public String getHeader(String hdrName) {
        if (hdrName == null || hdrName.length() == 0) 
            return null;
        
        for (String hdr : headers) {
            if (hdr.startsWith(hdrName)) {
                String val = hdr.substring(hdrName.length());
                val = val.substring(val.indexOf(':')+1);
                return val;
            }
        }
        return null;
    }
    
    public String getParameter(String name) {
        if (queryString == null || name == null || name.length() == 0)
            return null;      

        if (parameters == null) 
            extractParameters();
        
        // recall that the spec states parameter names shall be case insensitive...
        return parameters.get(name.toUpperCase());
    }
    
    public String[] getParameterValues(String name) {
        String param = getParameter(name);
        if (param == null)
            return null;
        if (param.equals(""))
            return new String[1];
        
        StringTokenizer parser = new StringTokenizer(param, ",");
        int numValues = parser.countTokens();
        String[] values = new String[numValues];
        int i = 0;
        while (parser.hasMoreTokens())
            values[i++] = parser.nextToken();
        return values;
    }

    public String getMethod() {
        return method;
    }
    
    public Enumeration getParameterNames() {
        if (parameters == null) 
            extractParameters();
        return new WrappedIterator(parameters.keySet());
    }
        
    public String getQueryString() {
        return queryString;
    }
    
    /***** These might be useful to implement, but we won't until we know for a fact...
    public Enumeration getHeaders() {
        return headers.elements();
    }
    
    public Enumeration getHeaderNames() {
        return null;
    }
    
    public Enumeration getParameterValues() {}
    *****************/
    
    /**
     * Convenience method to pull out all query parameters.
     */
    private void extractParameters() {
        parameters = new HashMap<String, String>();
        
        StringTokenizer parser = new StringTokenizer(queryString, "&");
        while (parser.hasMoreTokens()) {
            String param = parser.nextToken();
            int i = param.indexOf('=');
            if (i < 0) 
                continue;  // silently ignore bogus stuff here (may be revealed as error later)
            
            String value = param.substring(i+1);           
            param = param.substring(0,i).toUpperCase(); // spec says case-insensitive
            parameters.put(param, value);
        }                
    }
    
    
    //
    // A utility class to bend an Iterator into an Enumeration.
    //
    private class WrappedIterator implements Enumeration<String> {
        private Iterator<String> iterator;
        public WrappedIterator (Set<String> s) {
            iterator = s.iterator();
        }
        
        public boolean hasMoreElements() {
            return iterator.hasNext();
        }

        public String nextElement() {
            return iterator.next();
        }        
    }
    
    // Hide this...  
    private WMSHttpServletRequest() {
    }
    
    private String method = "null";
    private String url = "null";
    private String queryString = null;
    private Vector<String> headers = new Vector<String>();
    private HashMap<String, String> parameters = null;
}
