/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPServerInterface.cpp

	Contains:	Implementation of RTPServerInterface class. 
					
	$Log: RTPServerInterface.cpp,v $
	Revision 1.3  1999/02/26 04:56:40  cl
	Fixed total vs. current sessions bug
	
	Revision 1.2  1999/02/19 23:08:26  ds
	Created
	

*/

#include "RTPServerInterface.h"
#include "RTSPServerInterface.h"
#include "RTSPRequestInterface.h"
#include "RTSPMessages.h"
#include "RTSPProtocol.h"
#include "OS.h"

unsigned int	RTPServerInterface::sTotalSessions = 0;
UInt32			RTPServerInterface::sCurrentBandwidthInBits = 0;
UInt32			RTPServerInterface::sAvgBandwidthInBits = 0;
UInt32			RTPServerInterface::sPacketsPerSecond = 0;
UInt64			RTPServerInterface::sTotalBytes = 0;
unsigned int	RTPServerInterface::sPeriodicBytes = 0;
unsigned int	RTPServerInterface::sPeriodicPackets = 0;
UInt32			RTPServerInterface::sIsRefusingConnections = false;
		
OSRefTable*		RTPServerInterface::sRTPMap = NULL;
StrPtrLen 		RTPServerInterface::sSDPSuffix(".sdp", 4);
char*			RTPServerInterface::sCNameBase = "QTSS";
RTPSocketPool*	RTPServerInterface::sSocketPool = NULL;
IncomingRTCPDataTask*	RTPServerInterface::sRTCPDataTask = NULL;

UInt32			RTPServerInterface::sServerState = RTPServerInterface::kStartingUpState;

RTSPProtocol::RTSPStatusCode RTPServerInterface::IsOkToAddNewSession(RTSPRequestInterface* inRTSPRequest)
{
	//we may want to deny this connection for a couple of different reasons
	//if the server is refusing new connections
	if (RTPServerInterface::GetServerState() == kRefusingConnectionsState)
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientNotEnoughBandwidth,
												RTSPMessages::kRefusingConnections);

	//if the max connection limit has been hit
	SInt32 maxConns = RTSPServerInterface::GetRTSPPrefs()->GetMaxConnections();
	if ((maxConns > -1) && (RTPServerInterface::GetCurrentSessionCount() >= (UInt32)maxConns))
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientNotEnoughBandwidth,
										RTSPMessages::kTooManyClients);

	//if the max bandwidth limit has been hit
	SInt32 maxKBits = RTSPServerInterface::GetRTSPPrefs()->GetMaxKBitsBandwidth();
	if ( (maxKBits > -1) && (sCurrentBandwidthInBits >= ((UInt32)maxKBits*1024)) )
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientNotEnoughBandwidth,
												RTSPMessages::kTooMuchThruput);

	//if the server is too loaded down (CPU too high, whatever)
	// --INSERT WORKING CODE HERE--
	
	return RTSPProtocol::kSuccessOK;													
}

void	RTPServerInterface::SendDescribeResponse(RTSPRequestInterface* inRequest, iovec* describeData, UInt32 inNumVectors,
													UInt32 inTotalLength)
{
	//write content type header
	static StrPtrLen sContentType("application/sdp", 15);
	inRequest->AppendHeader(RTSPProtocol::kContentTypeHeader, &sContentType);
	
	//write content base header
	Assert(inRequest->GetQTSSParameter(qtssURLParam) != NULL);
	Assert(inRequest->GetQTSSParameter(qtssURLParam)->Len > 0);
	inRequest->AppendContentBaseHeader(inRequest->GetQTSSParameter(qtssAbsoluteURLParam));
	
	//write content size header
	inRequest->AppendContentLength(inTotalLength);
	
	//I believe the only error that can happen is if the client has disconnected.
	//if that's the case, just ignore it, hopefully the calling module will detect
	//this and return control back to the server ASAP 
	(void)inRequest->SendHeader();

	QTSS_ErrorCode err = inRequest->Send(describeData, inNumVectors, inTotalLength);
	AssertV((err == QTSS_NoErr) || (err == QTSS_RequestFailed), (int)err);
}

UInt32 RTPServerInterface::GetServerState()
{
	//check to see if a fatal error has occurred here.
	if (RTSPServerInterface::HasFatalErrorOccurred())
		sServerState = kFatalErrorState;
	return sServerState;
}

void RTPServerInterface::SetServerState(UInt32 serverState)
{
	//check to see if a fatal error has occurred here.
	if (RTSPServerInterface::HasFatalErrorOccurred())
		sServerState = kFatalErrorState;
	if (sServerState != kFatalErrorState)
		sServerState = serverState;
}

UInt32 RTPServerInterface::GetACName(char* ioCNameBuffer)
{
	//clear out the whole buffer
	::memset(ioCNameBuffer, 0, kMaxCNameLen);
	
	//cName identifier
	ioCNameBuffer[0] = 1;
	
	//Unique cname is constructed from the base name and the current time
	::sprintf(&ioCNameBuffer[1], " %s%qd", sCNameBase, OS::Milliseconds() / 1000);
	UInt32 cNameLen = ::strlen(ioCNameBuffer);
	//2nd byte of CName should be length
	ioCNameBuffer[1] = cNameLen - 2;//don't count indicator or length byte

	//pad length to a 4 byte boundary
	UInt32 padLength = cNameLen % 4;
	if (padLength > 0)
		cNameLen += 4 - padLength;
	
	Assert((cNameLen % 4) == 0);
	return cNameLen;
}

UDPSocketPair*	RTPSocketPool::ConstructUDPSocketPair()
{
	//construct a pair of UDP sockets, the lower one for RTP data (outgoing only, no demuxer
	//necessary), and one for RTCP data (incoming, so definitely need a demuxer).
	//These are nonblocking sockets that DON'T receive events (we are going to poll for data)
	return new ('usce')
		UDPSocketPair(	new ('usck') UDPSocket(Socket::kNonBlocking, RTPServerInterface::sRTCPDataTask),
						new ('usck') UDPSocket(Socket::kNonBlocking | UDPSocket::kWantsDemuxer, RTPServerInterface::sRTCPDataTask));
}

void RTPSocketPool::DestructUDPSocketPair(UDPSocketPair* inPair)
{
	delete inPair->GetSocketA();
	delete inPair->GetSocketB();
	delete inPair;
}

SInt64 IncomingRTCPDataTask::Run()
{
	const UInt32 kMaxRTCPPacketSize = 2048;
	char thePacketBuffer[kMaxRTCPPacketSize];
	StrPtrLen thePacket(thePacketBuffer, 0);
	
	//This task goes through all the UDPSockets in the RTPSocketPool, checking to see
	//if they have data. If they do, it demuxes the packets and sends the packet onto
	//the proper RTP session.
	(void)this->GetEvents();
	
	//Must be done atomically wrt the socket pool.
	OSMutexLocker locker(RTPServerInterface::sSocketPool->GetMutex());
	for (OSQueueIter iter(RTPServerInterface::sSocketPool->GetSocketQueue());
			!iter.IsDone(); iter.Next())
	{
		UInt32 theRemoteAddr = 0;
		UInt16 theRemotePort = 0;

		UDPSocketPair* thePair = (UDPSocketPair*)iter.GetCurrent()->GetEnclosingObject();
		Assert(thePair != NULL);
		
		for (UInt32 x = 0; x < 2; x++)
		{
			UDPSocket* theSocket = NULL;
			if (x == 0)
				theSocket = thePair->GetSocketA();
			else
				theSocket = thePair->GetSocketB();
				
			UDPDemuxer* theDemuxer = theSocket->GetDemuxer();
			if (theDemuxer != NULL)
				theDemuxer->GetMutex()->Lock();
			
			//get all the outstanding packets for this socket
			while (true)
			{
				theSocket->RecvFrom(&theRemoteAddr, &theRemotePort, thePacket.Ptr,	
								kMaxRTCPPacketSize, &thePacket.Len);
				if (thePacket.Len == 0)
					break;//no more packets on this socket!
					
				//if this socket has a demuxer, find the target UDPDemuxerTask
				if (theDemuxer != NULL)
				{
					UDPDemuxerTask* theTask = theDemuxer->GetTask(theRemoteAddr, theRemotePort);
					if (theTask != NULL)
						theTask->ProcessPacket(&thePacket);
				}
			}
			if (theDemuxer != NULL)
				theDemuxer->GetMutex()->Unlock();
		}
	}
	return kRTCPPollIntervalInMilSecs;
}
