/************************************************************************/
/* Module : db.c						        */
/* Purpose: database module				                */
/* By     : Keith R. Davis					        */
/* Date   : 12/11/95					                */
/* Notes  : Copyright(c) 1996-98 Mutiny Bay Software			*/
/*          Copyright(c) 1994, Regents of the University of California  */
/************************************************************************/

#include <Xm/Xm.h>			/* motif lib header		*/
#include <Xm/PrimitiveP.h>              /* motif primitive widget head. */
#include <sys/signal.h>                 /* system headers               */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <string.h>			/* string header		*/
#include <stdio.h>                      /* stdio header                 */
#include <pwd.h>                        /* passwd header                */

#include "textBuf.h"                    /* text widget headers          */
#include "textSel.h"
#include "text.h"
#include "textDisp.h"
#include "textP.h"
#include "window.h"
#include "highlight.h"
#include "highlightData.h"

#include "xutil.h"                      /* X utils header               */
#include "msgbox.h"                     /* message box header           */
#include "llist.h"                      /* linked list header           */
#include "mquel.h"                      /* mquel module header          */
#include "callback.h"                   /* callback header              */
#include "db.h"				/* db data header		*/
#include "util.h"                       /* utility function header      */

/* database globals , this needs to be eliminated !! */	
	
PGconn *db = NULL;		        /* db ptr		        */
char dbname[DB_NAME_SZ];		/* db name 	         	*/
char host[HOST_NAME_SZ];      	        /* host name 	                */
char port[PORT_NAME_SZ];		/* port 		        */
char username[UNAME_NAME_SZ];		/* user name 		        */
char *passwd = NULL;		        /* password  		        */
char spool_file[MAX_PATH_LEN];          /* spool file                   */         
char printer[PRINT_NAME_SZ];            /* printer                      */
int connected = 0;                      /* connect flag                 */
int user_prompt = 0;                    /* user & pwd prompt flag       */
int qy_cancelled = 0;                   /* query cancel flag            */
int qy_running = 0;                     /* query running flag           */

/************************************************************************/
/* Checks if user has selected to cancel the currently running query    */
/************************************************************************/
static int check_cancelled(void)
{
  Display *dpy = XtDisplay(AppWidgetsPtr->shell);
  Window win = XtWindow(AppWidgetsPtr->btn_exe);
  XEvent event;

  /* Update the display */
  XFlush(dpy);
  XmUpdateDisplay(AppWidgetsPtr->shell);

  /* Check if the cancel button was pressed */
  while(XCheckMaskEvent(dpy,
			ButtonPressMask | ButtonReleaseMask | KeyPressMask,
			&event)){
    if(event.xany.window == win)
      XtDispatchEvent(&event);
    else
      XBell(dpy, 50);
  }
  return qy_cancelled;
}

/************************************************************************/
/* Function: DB_Connect                                                 */
/* Purpose : establishes connection to postgres database                */
/* Params  :                                                            */
/* Returns : 1 on SUCCESS / 0 on FAILURE                                */
/* Notes   :                                                            */
/************************************************************************/

int DB_Connect(void)
{
  char db_msg[ERR_MSG_SIZE];
  char       *connect_string;
  
  XUTIL_BeginWait(AppWidgetsPtr->mainwindow);

  if(connected)
    PQfinish(db);
  
  if(user_prompt)
    db = PQsetdbLogin(host, port, NULL, NULL, dbname, username, 
		      (passwd != NULL) ? passwd : "");  
  else
    db = PQsetdbLogin(host, port, NULL, NULL, dbname, NULL, NULL);

  if (PQstatus(db) == CONNECTION_BAD) {

    sprintf(db_msg,"Connection to database '%s' failed.\nERROR: %s", 
	    dbname, PQerrorMessage(db));
    MSGBOX_Error("Connect", db_msg, AppWidgetsPtr->mainwindow);
    sprintf(db_msg,"Not connected.");
    XtVaSetValues(AppWidgetsPtr->message,
		  XtVaTypedArg, XmNlabelString, XmRString,
		  db_msg, strlen(db_msg)+1, NULL);
    connected = 0;
    XUTIL_EndWait(AppWidgetsPtr->mainwindow);
    return 0;
  }    
  sprintf(db_msg,"Connected to database '%s'.", dbname); 
  connected = 1;
  XtVaSetValues(AppWidgetsPtr->message,
		  XtVaTypedArg, XmNlabelString, XmRString,
		  db_msg, strlen(db_msg)+1, NULL);

  XUTIL_EndWait(AppWidgetsPtr->mainwindow);

  return 1;
}

/************************************************************************/
/* Function: DB_Close                                                   */
/* Purpose : kills connection to postgres database                      */
/* Params  :                                                            */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_Close(void)
{
  /* disconnect from database */
  if(connected)
    PQfinish(db);
}

/************************************************************************/
/* Function: DB_SubmitQy                                                */
/* Purpose : submits SQL window contents to postgres database           */
/* Params  :                                                            */
/* Returns : 1 on SUCCESS / 0 on FAILURE                                */
/* Notes   :                                                            */
/************************************************************************/

int DB_SubmitQy(void)
{
  textBuffer *buffer;
  char *qy_ptr;
  int buf_pos;
  int qy_pos;
  int qy_len;
  int in_quote = 0;
  int selected = 0;
  char db_msg[ERR_MSG_SIZE];
  char qy_buffer[QY_BUFFER_SIZE];
  PGresult* results;
  PGnotify* notify;

  if (!connected) {
    sprintf(db_msg,"No Connection to database.");
    MSGBOX_Error("Query", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  buffer = TextDGetBuffer(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD);

  if(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD->buffer->primary.selected){
    qy_ptr = (char*)doubleTrim(BufGetSelectionText(buffer));
    selected = 1;
  }
  else
    qy_ptr = 
      (char*)doubleTrim(BufGetAll(buffer));
  
  if((qy_len = strlen(qy_ptr)) == 0){
    XtFree((char*)qy_ptr);
    return 0;
  }
 
  qy_pos = 0;

  while(qy_pos < qy_len){
    buf_pos = 0;

    MQUEL_SetToolbar(0);

    while(qy_pos < qy_len){
      if((qy_ptr[qy_pos] == '\'') && !in_quote)
	in_quote = 1;
      else if((qy_ptr[qy_pos] == '\'') && in_quote)
	in_quote = 0;

      if(!in_quote && (qy_ptr[qy_pos] == ';'))
	 break;

      qy_buffer[buf_pos++] = qy_ptr[qy_pos++];
      if(strlen(qy_buffer) > (QY_BUFFER_SIZE - 2)){
    	sprintf(db_msg,"Query length exceeds max. query size, truncated at %d\n", 
		QY_BUFFER_SIZE);
    	DB_WriteToResult(db_msg, 0, NULL);
    	break;
      }
    }

    qy_pos++;
    qy_buffer[buf_pos++] = ';';
    qy_buffer[buf_pos] = '\0';

    // Submit the query
    if (!PQsendQuery(db, (char*)doubleTrim(qy_buffer))){
      DB_WriteToResult(PQerrorMessage(db), 0, NULL);
      DB_WriteToResult("\n", 0, NULL);
      TextDMakeInsertPosVisible(((TextWidget)AppWidgetsPtr->resultwindow)->text.textD);
      XtFree((char*)qy_ptr);
      MQUEL_SetToolbar(1);
      return 0;
    }

    qy_running = 1;
    qy_cancelled = 0;

    while(1) {
      PQconsumeInput(db);

      if(!PQisBusy(db))
	break;
      else
      	if(check_cancelled()){
	  PQrequestCancel(db);
	  break;
	}
    }

    results = PQgetResult(db);

    switch (PQresultStatus(results)) {
    case PGRES_TUPLES_OK:
      DB_PrintTuples(qy_buffer,
		     results,
		     spool_file,
		     XmToggleButtonGetState(AppWidgetsPtr->tglspool),  
		     XmToggleButtonGetState(AppWidgetsPtr->tglcol),
		     XmToggleButtonGetState(AppWidgetsPtr->tglecho),
		     GetColSeparator()
		     );
      PQclear(results);
      break;
    case PGRES_EMPTY_QUERY:
      /* do nothing */
      break;
    case PGRES_COMMAND_OK:
      sprintf(db_msg,"%s\n\n",PQcmdStatus(results));
      DB_WriteToResult(db_msg, 0, NULL);
      break;
    case PGRES_COPY_OUT:
      DB_HandleCopyOut(results);
      PQclear(results);
      break;
    case PGRES_COPY_IN:
      DB_HandleCopyIn(results);
      PQclear(results);
      break;
    case PGRES_NONFATAL_ERROR:
    case PGRES_FATAL_ERROR:
    case PGRES_BAD_RESPONSE:
      qy_cancelled = 1;
      DB_WriteToResult(PQerrorMessage(db), 0, NULL);
      /* reset the connection -  ???*/
      DB_Connect();
    } 
   
    /* check for asynchronous returns */
    notify = PQnotifies(db);
    if (notify) {
      sprintf(db_msg,"ASYNC NOTIFY of '%s' from backend pid '%d' received", 
	      notify->relname, notify->be_pid);
      DB_WriteToResult(db_msg, 0, NULL);
      free(notify);
    }
    if(qy_cancelled){
      qy_cancelled = 0;
      break;
    }
  }
  XtFree((char*)qy_ptr);
  MQUEL_SetToolbar(1);
  qy_running = 0;
  return 1;
}

/************************************************************************/
/* Function: DB_CancelQuery                                             */
/* Purpose : cancels a executing query on the current connection        */
/* Params  : none                                                       */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_CancelQy(void)
{
  if(qy_running)
    qy_cancelled = 1;
}

/************************************************************************/
/* Function: DB_OpenFile                                                */
/* Purpose : opens a file into the SQL window                           */
/* Params  : file : string w/ filename                                  */
/* Returns : 1 on SUCCESS / 0 on FAILURE                                */
/* Notes   :                                                            */
/************************************************************************/

int DB_OpenFile(char *file)
{
  FILE *sqlf;
  struct stat statbuf;
  int fileLen, readLen;
  char db_msg[ERR_MSG_SIZE];
  char *dataString;
  LLISTbuffer *tmp_buffer;
  int curr_mod;

  if((sqlf = fopen(file, "rw")) == NULL){
    sprintf(db_msg, "Error opening: %s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  if((tmp_buffer = LLIST_Insert(0, file, buffer_tail)) == NULL){
    sprintf(db_msg, "Not enough memory to open file :\n%s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  XUTIL_BeginWait(AppWidgetsPtr->mainwindow);

  /* Get the length of the file and the protection mode */
  if (fstat(fileno(sqlf), &statbuf) != 0) {
    sprintf(db_msg, "Error opening: %s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  fileLen = statbuf.st_size;
  /*window->fileMode = statbuf.st_mode;*/
  
  /* Allocate space for the whole contents of the file (unfortunately) */
  dataString = (char *)malloc(fileLen+1);  /* +1 = space for null */
  if (dataString == NULL) {
    sprintf(db_msg, "Not enough memory to open file :\n%s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  /* read data from the file */
  readLen = fread(dataString, sizeof(char), fileLen, sqlf);
  if (ferror(sqlf)) {
    sprintf(db_msg, "Error Reading :\n%s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    free(dataString);
    return 0;
  }

  /* NULL terminate the data */
  dataString[readLen] = 0;

  /* 
  ** write the file to the tmp buffer
  ** we also need a hack to preserve the current buffer's
  ** modification flag.
  */
  curr_mod = buffer_curr->modified;
  BufSetAll(tmp_buffer->buffer, dataString);
  buffer_curr->modified = curr_mod;

  /* Release the memory that holds fileString */
  free(dataString);

  XUTIL_EndWait(AppWidgetsPtr->mainwindow);

  if(fclose(sqlf) != 0){
    sprintf(db_msg, "Error closing file:\n%s", file);
    MSGBOX_Error("Open File", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }
  
  /* make the temp buffer the current buffer */
  buffer_curr = DB_SetBuffer(tmp_buffer);
  
  return 1;
}

/************************************************************************/
/* Function: DB_SaveFile                                                */
/* Purpose : saves a buffer into its source file                        */
/* Params  : buffer : ptr to buffer to save                             */
/* Returns : 1 on SUCCESS / 0 on FAILURE                                */
/* Notes   :                                                            */
/************************************************************************/

int DB_SaveFile(LLISTbuffer *buffer)
{
  FILE *sqlf;
  char *data;
  char db_msg[ERR_MSG_SIZE];

  if((sqlf = fopen(buffer->filename, "w+")) == NULL){
    sprintf(db_msg, "Error opening: %s", buffer->filename);
    MSGBOX_Error("Save", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }

  data = BufGetAll(buffer->buffer);

  XUTIL_BeginWait(AppWidgetsPtr->mainwindow);

  fputs(data, sqlf);
  fflush(sqlf);

  XUTIL_EndWait(AppWidgetsPtr->mainwindow);

  XtFree((char *)data);

  if(fclose(sqlf) != 0){
    sprintf(db_msg, "Error closing file: %s", buffer->filename);
    MSGBOX_Error("Save", db_msg, AppWidgetsPtr->mainwindow);
    return 0;
  }
  buffer->modified = 0;
  DB_SetTitle(buffer->filename);
  sprintf(db_msg, "Saved file: %s", buffer->filename);
  XtVaSetValues(AppWidgetsPtr->message,
		XtVaTypedArg, XmNlabelString, XmRString,
		db_msg, strlen(db_msg)+1, NULL);

  return 1;
}

/************************************************************************/
/* Notes   : The following three functions were lifted from the psql    */
/*           source code and were slightly modified to work with mpsql. */
/************************************************************************/

static void do_field(PQprintOpt *po, PGresult *res,
	 const int i, const int j, char *buf, const int fs_len,
	 char *fields[],
	 const int nFields, char *fieldNames[],
	 unsigned char fieldNotNum[], int fieldMax[],
	 const int fieldMaxLen, FILE *fout)
{
  char *pval, *p, *o;
  int plen;
  int skipit;
  int Spool;
  char dataString[DATA_SIZE];
  
  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  plen = PQgetlength(res, i, j);
  pval = PQgetvalue(res, i, j);
  
  if (plen < 1 || !pval || !*pval)
    {
      if (po->align || po->expanded)
	skipit = 1;
      else
	{
	  skipit = 0;
	  goto efield;
	}
    }
  else
    skipit = 0;
  
  if (!skipit)
    {
      for (p = pval, o = buf; *p; *(o++) = *(p++))
	{
	  if ((fs_len == 1 && (*p == *(po->fieldSep))) || *p == '\\' || *p == '\n')
	    *(o++) = '\\';
	  if (po->align && (*pval == 'E' || *pval == 'e' ||
			    !((*p >= '0' && *p <= '9') ||
			      *p == '.' ||
			      *p == 'E' ||
			      *p == 'e' ||
			      *p == ' ' ||
			      *p == '-')))
	    fieldNotNum[j] = 1;
	}
      *o = '\0';
      if (!po->expanded && (po->align || po->html3))
	{
	  int n = strlen(buf);
	  
	  if (n > fieldMax[j])
	    fieldMax[j] = n;
	  if (!(fields[i * nFields + j] = (char *) malloc(n + 1)))
	    {
	      perror("malloc");
	      exit(1);
	    }
	  strcpy(fields[i * nFields + j], buf);
	}
      else
	{
	  if (po->expanded)
	    {
	      if (po->html3){
		sprintf(dataString,
			"<tr><td align=left><b>%s</b></td>"
			"<td align=%s>%s</td></tr>\n",
			fieldNames[j],
			fieldNotNum[j] ? "left" : "right",
			buf);
		DB_WriteToResult(dataString, Spool, fout);
	      }
	      else
		{
		  if (po->align)
		    sprintf(dataString,
			    "%-*s%s %s\n",
			    fieldMaxLen - fs_len, fieldNames[j], po->fieldSep,
			    buf);
		  else
		    sprintf(dataString, "%s%s%s\n", fieldNames[j], po->fieldSep, buf);
		  DB_WriteToResult(dataString, Spool, fout);
		}
	    }
	  else
	    {
	      if (!po->html3)
		{
		  DB_WriteToResult(buf, Spool, fout);
		efield:
		  if ((j + 1) < nFields)
		    DB_WriteToResult(po->fieldSep, Spool, fout);
		  else
		    DB_WriteToResult("\n", Spool, fout);
		}
	    }
	}
    }
}

static char * do_header(FILE *fout, PQprintOpt *po, const int nFields, int fieldMax[],
		  char *fieldNames[], unsigned char fieldNotNum[],
		  const int fs_len, PGresult *res)
{
  int Spool;
  int j;
  char *border = NULL;
  char dataString[DATA_SIZE];

  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  if (po->html3)
    DB_WriteToResult("<tr>", Spool, fout);
  else
    {
      int j;
      int tot = 0;
      int n = 0;
      char *p = NULL;
      
      for (; n < nFields; n++)
	tot += fieldMax[n] + fs_len + (po->standard ? 2 : 0);
      if (po->standard)
	tot += fs_len * 2 + 2;
      border = (char*)malloc(tot + 1);
      if (!border)
	{
	  perror("malloc");
	  exit(1);
	}
      p = border;
      if (po->standard)
	{
	  char	   *fs = po->fieldSep;
	  
	  while (*fs++)
	    *p++ = ' ';
	}
      for (j = 0; j < nFields; j++)
	{
	  int len;
	  
	  for (len = fieldMax[j] + (po->standard ? 2 : 0); len--; *p++ = '-');
	  if (po->standard || (j + 1) < nFields)
	    {
	      char	   *fs = po->fieldSep;
	      
	      while (*fs++)
		*p++ = ' ';
	    } 
	}
      *p = '\0';
      if (po->standard){
	sprintf(dataString, "%s\n", border);
	DB_WriteToResult(dataString, Spool, fout);
      }
    }
  if (po->standard){
    sprintf(dataString, "%s", po->fieldSep);
    DB_WriteToResult(dataString, Spool, fout);
  }
  for (j = 0; j < nFields; j++)
    {
      char *s = PQfname(res, j);
      
      if (po->html3)
	{
	  sprintf(dataString, "<th align=%s>%s</th>",
		  fieldNotNum[j] ? "left" : "right", fieldNames[j]);
	  DB_WriteToResult(dataString, Spool, fout);
	}
      else
	{
	  int			n = strlen(s);
	  
	  if (n > fieldMax[j])
	    fieldMax[j] = n;
	  if (po->standard)
	    sprintf(dataString,
		    fieldNotNum[j] ? " %-*s " : " %*s ",
		    fieldMax[j], s);
	  else
	    sprintf(dataString, fieldNotNum[j] ? "%-*s" : "%*s", fieldMax[j], s);
	  if (po->standard || (j + 1) < nFields)
	    strcat(dataString, po->fieldSep);
	  DB_WriteToResult(dataString, Spool, fout);
	}
    }
  if (po->html3)
    DB_WriteToResult("</tr>\n", Spool, fout);
  else{
    sprintf(dataString, "\n%s\n", border);
    DB_WriteToResult(dataString, Spool, fout);
  }
  return border;
}


static void output_row(FILE *fout, PQprintOpt *po, const int nFields, char *fields[],
		   unsigned char fieldNotNum[], int fieldMax[], char *border,
		   const int row_index)
{
  int Spool;
  int field_index;
  char dataString[DATA_SIZE];

  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  if (po->html3)
    DB_WriteToResult("</tr>", Spool, fout);
  else if (po->standard)
    DB_WriteToResult(po->fieldSep, Spool, fout);
  for (field_index = 0; field_index < nFields; field_index++)
    {
      char *p = fields[row_index * nFields + field_index];
      
      if (po->html3)
	sprintf(dataString, "<td align=%s>%s</td>",
		fieldNotNum[field_index] ? "left" : "right", p ? p : "");
      else
	{
	  sprintf(dataString,
		  fieldNotNum[field_index] ?
		  (po->standard ? " %-*s " : "%-*s") :
		  (po->standard ? " %*s " : "%*s"),
		  fieldMax[field_index],
		  p ? p : "");
	  if (po->standard || field_index + 1 < nFields)
	    strcat(dataString, po->fieldSep);
	}
      DB_WriteToResult(dataString, Spool, fout);
      if (p)
	free(p);
    }
  if (po->html3)
    DB_WriteToResult("</tr>", Spool, fout);
  else if (po->standard){
    sprintf(dataString, "\n%s", border);
    DB_WriteToResult(dataString, Spool, fout);
  }
  DB_WriteToResult("\n", Spool, fout);
}

/************************************************************************/
/* Function: DB_FormatResults                                           */
/* Purpose : formats and sends query results to selected destination    */
/* Params  : fout : output dest file handle                             */
/*           res: query result struct                                   */
/*           po : ptr to PG Print Option Struct                         */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_FormatResults(FILE *fout, PGresult *res, PQprintOpt *po)
{
  int nFields;
  
  nFields = PQnfields(res);

  if(nFields > 0)
    {
      int i, j;
      int nTups;
      int *fieldMax = NULL;	/* in case we don't use them */
      unsigned char *fieldNotNum = NULL;
      char *border = NULL;
      char **fields = NULL;
      char **fieldNames;
      int  fieldMaxLen = 0;
      int  numFieldName;
      int  fs_len = strlen(po->fieldSep);
      int  total_line_length = 0;
      int  usePipe = 0;
      char *pagerenv;
      char buf[8192 * 2 + 1];
      char dataString[DATA_SIZE];
      int Spool;
      
      nTups = PQntuples(res);

      if (!(fieldNames = (char **) calloc(nFields, sizeof(char *))))
	{
	  perror("calloc");
	  exit(1);
	}
      if (!(fieldNotNum = (unsigned char *) calloc(nFields, 1)))
	{
	  perror("calloc");
	  exit(1);
	}
      if (!(fieldMax = (int *) calloc(nFields, sizeof(int))))
	{
	  perror("calloc");
	  exit(1);
	}
 
      for (j = 0; j < nFields; j++)
	{
	  int len;
	  char *s = PQfname(res, j);
	 
	    
	  fieldNames[j] = s;
	  len = s ? strlen(s) : 0;
	  fieldMax[j] = len;
	  len += fs_len;
	  if (len > fieldMaxLen)
	    fieldMaxLen = len;
	  total_line_length += len;
	}
      
      total_line_length += nFields * strlen(po->fieldSep) + 1;
      
      if (fout == NULL)
	Spool = 0;
      else
	Spool = 1;
      
      if (!po->expanded && (po->align || po->html3))
	{
	  if (!(fields = (char **) calloc(nFields * (nTups + 1), sizeof(char *))))
	    {
	      perror("calloc");
	      exit(1);
	    }
	}
      else if (po->header && !po->html3)
	{
	  if (po->expanded)
	    {
	      if (po->align)
		sprintf(dataString, "%-*s%s Value\n",
			fieldMaxLen - fs_len, "Field", po->fieldSep);
	      else
		sprintf(dataString, "%s%sValue\n", "Field", po->fieldSep);
	      DB_WriteToResult(dataString, Spool, fout);
	    }
	  else
	    {
	      int			len = 0;
	      
	      for (j = 0; j < nFields; j++)
		{
		  char	   *s = fieldNames[j];
		  strcat(dataString, s);
		  len += strlen(s) + fs_len;
		  if ((j + 1) < nFields)
		    strcat(dataString, po->fieldSep);
		}
	      strcat(dataString, "\n");
	      for (len -= fs_len; len--; strcat(dataString, "-"));
	      strcat(dataString, "\n");
	      DB_WriteToResult(dataString, Spool, fout);
	    }
	}
      if (po->expanded && po->html3)
	{
	  sprintf(dataString,
		  "<center><h2>"
		  "Query retrieved %d rows * %d fields"
		  "</h2></center>\n",
		  nTups, nFields);
	  DB_WriteToResult(dataString, Spool, fout);
	}
      for (i = 0; i < nTups; i++)
	{
	  if(check_cancelled())
	    break;
	  if (po->expanded)
	    {
	      if (po->html3)
		sprintf(dataString,
			"<table><caption align=high>%d</caption>\n", i);
	      else
		sprintf(dataString, "-- RECORD %d --\n", i);
	      DB_WriteToResult(dataString, Spool, fout);
	    }
	  for (j = 0; j < nFields; j++)
	    do_field(po, res, i, j, buf, fs_len, fields, nFields,
		     fieldNames, fieldNotNum,
		     fieldMax, fieldMaxLen, fout);
	  if (po->html3 && po->expanded)
	    DB_WriteToResult("</table>\n", Spool, fout);
	}
      if (!po->expanded && (po->align || po->html3))
	{
	  if (po->html3)
	    {
	      if (po->header) 
		{
		  sprintf(dataString,
		  	  "<table><caption align=high>"
		  	  "Retrieved %d rows * %d fields"
		  	  "</caption>\n",
		  	  nTups, nFields);
		  DB_WriteToResult(dataString, Spool, fout);
		}
	      else {
		sprintf(dataString, "<table>");
		DB_WriteToResult(dataString, Spool, fout);
	      }
	    }
	  if (po->header)
	    border = do_header(fout, po, nFields, fieldMax, fieldNames,
	    		       fieldNotNum, fs_len, res);
	  for (i = 0; i < nTups; i++){
	    if(check_cancelled())
	      break;
	    output_row(fout, po, nFields, fields,
	    	       fieldNotNum, fieldMax, border, i);
	    if(TextGetCursorPos(AppWidgetsPtr->resultwindow) > RSLT_BUFFER_SIZE)
	       DB_ShrinkBuffer(AppWidgetsPtr->resultwindow);
	  }
	  free(fields);
	  if (border)
	    free(border);
	}
      if (po->header && !po->html3){
	  sprintf(dataString, "(%d row%s)\n\n", PQntuples(res),(PQntuples(res) == 1) ? "" : "s");
	  DB_WriteToResult(dataString, Spool, fout);
      }
      free(fieldMax);
      free(fieldNotNum);
      free(fieldNames);
      if (usePipe)
	{
	  pclose(fout);
	  pqsignal(SIGPIPE, SIG_DFL);
	}
      if (po->html3 && !po->expanded)
	DB_WriteToResult("</table>\n", Spool, fout);
    }
}

/************************************************************************/
/* Function: DB_PrintTuples                                             */
/* Purpose : prints query results to selected destination               */
/* Params  : qy : string w/ qy                                          */
/*           res: query result struct                                   */
/*           fout: string w/ file to write to                           */
/*           Spool: spooling flag 0=FALSE / 1=TRUE                      */
/*           PrintAttNames: print attribute names 0=FALSE / 1=TRUE      */
/*           Echo: echo query 0=FALSE / 1=TRUE                          */
/*           sep: column sep char                                       */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_PrintTuples(char *qy, PGresult *res, char *fout, int Spool, int PrintAttNames, 
		    int Echo, char sep)
{
  FILE *spoolf = NULL;
  char colSep[2];
  char dataString[DATA_SIZE];
  PQprintOpt opt;

  memset(colSep, '\0', 2);
  colSep[0] = sep;

  opt.align = XmToggleButtonGetState(AppWidgetsPtr->align);
  opt.standard = XmToggleButtonGetState(AppWidgetsPtr->standard);
  opt.expanded = XmToggleButtonGetState(AppWidgetsPtr->expand);
  opt.header = PrintAttNames;
  opt.html3 = XmToggleButtonGetState(AppWidgetsPtr->html);
  opt.fieldSep = strdup(colSep);
  opt.pager = 0;

  if(Spool){
    if((spoolf = fopen(fout, "w+")) == NULL){
      sprintf(dataString, "Error opening/creating file: %s\n", fout);
      DB_WriteToResult(dataString, 0, NULL);
      return;
    }
  }

  if(Echo){
    sprintf(dataString, "QUERY: %s\n\n", 
	    (char *)doubleTrim(qy));
    DB_WriteToResult(dataString, Spool, spoolf);
  }

  DB_FormatResults(spoolf, res, &opt);
 
  if(Spool){
    if(fclose(spoolf) != 0){
      sprintf(dataString, "Error closing file: %s\n", fout);
      DB_WriteToResult(dataString, 0, NULL);
      return;
    }
    DB_WriteToResult("Query finished.\n", 0, NULL);  
  }

  TextDMakeInsertPosVisible(((TextWidget)AppWidgetsPtr->resultwindow)->text.textD);
}

/************************************************************************/
/* Function: DB_WriteToResult                                           */
/* Purpose : writes data to result window                               */
/* Params  : data : string w/ data to write                             */
/*           spool: spool flag 0=FALSE / 1=TRUE                         */
/*           file : spool file (FILE*)                                  */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_WriteToResult(char *data, int spool, FILE *file)
{
  textBuffer *buffer;
  int ins_pos;
  
  if(spool)
    fprintf(file, data);
  else{
    buffer = TextGetBuffer(AppWidgetsPtr->resultwindow);
    ins_pos = buffer->length;
    BufInsert(buffer, ins_pos, data);
    TextDSetInsertPosition(((TextWidget)AppWidgetsPtr->resultwindow)->text.textD, buffer->length);
    
    //TextDMakeInsertPosVisible(((TextWidget)AppWidgetsPtr->resultwindow)->text.textD);
  }
}

/************************************************************************/
/* Function: DB_WriteToSQL                                              */
/* Purpose : writes data to SQL window                                  */
/* Params  : data : string w/ data to write                             */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_WriteToSQL(char *data)
{
  textBuffer *buffer;
  int ins_pos;

  buffer = TextGetBuffer(AppWidgetsPtr->sqlwindow);
  ins_pos = buffer->length;
  BufInsert(buffer, ins_pos, data);
}

/************************************************************************/
/* Function: DB_WriteToBuffer                                           */
/* Purpose : writes data to text widget (buffer)                        */
/* Params  : data : string w/ data to write                             */
/*           w : text widget to write to (buffer)                       */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_WriteToBuffer(char *data, Widget w)
{
  textBuffer *buffer;
  int ins_pos;

  buffer = TextGetBuffer(w);
  ins_pos = buffer->length;
  BufInsert(buffer, ins_pos, data);
}

/************************************************************************/
/* Function: DB_OpenScratchBuffer                                       */
/* Purpose : opens a new scratch buffer                                 */
/* Params  :                                                            */
/* Returns : ptr to new buffer on SUCCESS / NULL on FAILURE             */
/* Notes   :                                                            */
/************************************************************************/

LLISTbuffer* DB_OpenScratchBuffer(void)
{
  char db_msg[ERR_MSG_SIZE];
  LLISTbuffer *tmp_buffer;

  if((tmp_buffer = LLIST_Insert(0, "scratch", buffer_tail)) == NULL){
    sprintf(db_msg, "Not enough memory to open buffer.");
    MSGBOX_Error("New", db_msg, AppWidgetsPtr->mainwindow);
    return NULL;
  }
  else
    return tmp_buffer;
}

/************************************************************************/
/* Function: DB_ClearDisplay                                            */
/* Purpose : clear the sql window display with an empty buffer          */
/* Params  : none                                                       */
/* Returns : ptr to new current buffer                                  */
/* Notes   : this is a lame hack to fix highlight sync problems         */
/************************************************************************/

static LLISTbuffer* DB_ClearDisplay()
{
  int curr_mod, new_mod;
  char *tmp;
  textBuffer *disp;

  /* save a copy of the current file's data */
  curr_mod = buffer_curr->modified;
  new_mod = empty_buffer.modified;
  buffer_curr->ins_pos = TextDGetInsertPosition(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD);

  /* stop highlighting */
  StopHighlighting(WinDataPtr);

  /* get a pointer to the display buffer */
  disp = ((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD->buffer;

  /* copy the contents to a tmp array */
  tmp = BufGetAll(disp);

  /* erase the display */
  BufSetAll(disp, "");

  /* set the display to the dummy buffer */
  WinDataPtr->buffer = empty_buffer.buffer;
  TextSetBuffer(AppWidgetsPtr->sqlwindow, empty_buffer.buffer);

  /* start highlighting */
  StartHighlighting(WinDataPtr, True);

  /* reset the display's original buffer */
  BufSetAll(buffer_curr->buffer, tmp);
  buffer_curr->modified = curr_mod;
  empty_buffer.modified = new_mod;

  /* free the tmp storage */
  XtFree((char *)tmp);

  return &empty_buffer;
}

/************************************************************************/
/* Function: DB_SetBuffer                                               */
/* Purpose : sets a select buffer to current SQL window buffer          */
/* Params  : buffer : ptr to buffer to display                          */
/* Returns : ptr to new current buffer                                  */
/* Notes   :                                                            */
/************************************************************************/

LLISTbuffer* DB_SetBuffer(LLISTbuffer *buffer)
{
  int curr_mod, new_mod;
  int curr_ins;
  char *tmp;
  textBuffer *disp;

  XUTIL_BeginWait(AppWidgetsPtr->mainwindow);

  buffer_curr = DB_ClearDisplay();

  /* save a copy of the current file's data */
  curr_mod = buffer_curr->modified;
  new_mod = buffer->modified;
  buffer_curr->ins_pos = TextDGetInsertPosition(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD);

  /* stop highlighting */
  StopHighlighting(WinDataPtr);

  /* set the display to the new buffer */
  WinDataPtr->buffer = buffer->buffer;
  TextSetBuffer(AppWidgetsPtr->sqlwindow, WinDataPtr->buffer);

  /* start highlighting */
  if(XmToggleButtonGetState(AppWidgetsPtr->highlight))
    StartHighlighting(WinDataPtr, True);
  
  /* set the cursor position */
  TextDSetInsertPosition(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD, buffer->ins_pos);
  TextDMakeInsertPosVisible(((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD);

  /* reset the display's original buffer */
  buffer_curr->modified = curr_mod;
  buffer->modified = new_mod;

  DB_SetTitle(buffer->filename);

  XUTIL_EndWait(AppWidgetsPtr->mainwindow);

  return buffer;
}

/************************************************************************/
/* Function: DB_SetTitle                                                */
/* Purpose : sets app's title window to passed string                   */
/* Params  : win_title : string w/ window title                         */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_SetTitle(char *win_title)
{
  char title[MAX_PATH_LEN]= "MPSQL - ";

  XtVaSetValues(AppWidgetsPtr->shell,
		XmNtitle, strcat(title, win_title),
		NULL);

}

/************************************************************************/
/* Function: DB_DeleteBuffer                                            */
/* Purpose : deletes a buffer                                           */
/* Params  : buffer : ptr to buffer to delete                           */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_DeleteBuffer(LLISTbuffer *buffer)
{
  if((LLIST_Count() - 1) == 0){
    DB_OpenScratchBuffer();
    buffer_curr = DB_SetBuffer(buffer_tail->prev);
  }
  else{
    if(buffer->prev == buffer_head)
      buffer_curr = DB_SetBuffer(buffer->next);
    else 
      buffer_curr = DB_SetBuffer(buffer->prev);
  }
  LLIST_Delete(buffer);
} 

/************************************************************************/
/* Function: DB_PrintBuffer                                             */
/* Purpose : prints contents of a buffer                                */
/* Params  : buffer : ptr to buffer to print                            */
/*           printer: ptr to printer command string to use              */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void DB_PrintBuffer(LLISTbuffer *buffer, char *printer)
{
  char tmp_file[MAX_PATH_LEN];
  char tmp_name[MAX_PATH_LEN];
  char print[MAX_PATH_LEN];
  FILE *sqlf;
  char *data;
  char db_msg[ERR_MSG_SIZE];

  sprintf(tmp_file, "%s", tmpnam(tmp_name));

  if((sqlf = fopen(tmp_file, "w+")) == NULL){
    sprintf(db_msg, "Error opening tmp file: %s", tmp_file);
    MSGBOX_Error("Print", db_msg, AppWidgetsPtr->mainwindow);
    return;
  }

  data = BufGetAll(buffer->buffer);

  fputs(data, sqlf);
  fflush(sqlf);

  if(fclose(sqlf) != 0){
    sprintf(db_msg, "Error closing tmp file: %s", tmp_file);
    MSGBOX_Error("Print", db_msg, AppWidgetsPtr->mainwindow);
    XtFree((char *)data); 
    return;
  }

  sprintf(print, "%s %s",
	  doubleTrim((char *)XmTextGetString(AppWidgetsPtr->printer)) ,tmp_file); 
  system(print);
  sprintf(print, "rm -f %s", tmp_file);
  system(print);
  XtFree((char *)data);
}

/************************************************************************/
/* Function: DB_ChkModBuffers                                           */
/* Purpose : checks the linked list for modified buffers                */
/* Params  :                                                            */
/* Returns : 0 all saved / 1 unsaved buffers                            */
/* Notes   :                                                            */
/************************************************************************/

int DB_ChkModBuffers(void)
{
  int i;
  LLISTbuffer *curr;

  for(i = 1; i <= LLIST_Count(); i++){
    curr = LLIST_Get(i);
    if(curr->modified == 1)
      return 1;
  }
  return 0;
}

/************************************************************************/
/* Function: DB_ShrinkBuffer                                            */
/* Purpose : removes text from top of specified text widget             */
/* Params  : w : text widget to remove text from                        */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_ShrinkBuffer(Widget w)
{
  int i;
  textBuffer *buffer;
  char *txt;

  buffer = TextDGetBuffer(((TextWidget)AppWidgetsPtr->resultwindow)->text.textD);

  txt = BufGetAll(buffer);

  while(TextGetCursorPos(AppWidgetsPtr->resultwindow) >= RSLT_BUFFER_SIZE){
    i = 0;
    while(txt[i] != '\n') 
      i++;
    BufReplace(buffer, 0, ++i, "");
  }
  XtFree((char*)txt);
}

/************************************************************************/
/* Function: DB_HandleCopyOut                                           */
/* Purpose : copies data out as specified by passed copy command        */
/* Params  : res: result set from copy command                          */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_HandleCopyOut(PGresult *res)
{
    int copydone = 1;
    char copybuf[COPYBUFSIZ];
    int ret;
    
    while (!copydone) {
	ret = PQgetline(db, copybuf, COPYBUFSIZ);
	if (copybuf[0] == '.' && copybuf[1] =='\0') {
	    copydone = 0;	/* don't print this... */
	} else {
	    fputs(copybuf, stdout);
	    switch (ret) {
	    case EOF:
		copydone = 0;
	    case 0:
		fputc('\n', stdout);
		break;
	    case 1:
		break;
	    }
	}
    }
    fflush(stdout);
    PQendcopy(db);
}

/************************************************************************/
/* Function: DB_HandleCopyIn                                            */
/* Purpose : copies data in as specified by passed copy command         */
/* Params  : res: result set from copy command                          */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_HandleCopyIn(PGresult *res)
{
    int copydone = 1;
    int firstload;
    int linedone;
    char copybuf[COPYBUFSIZ];
    char *s;
    int buflen;
    int c;
    
    fflush(stdin);
    if ((c = getc(stdin)) != '\n' && c != EOF) {
	(void) ungetc(c, stdin);
    }
    
    while (!copydone) {			/* for each input line ... */
	firstload = 0;
	linedone = 1;
	while (!linedone) {		/* for each buffer ... */
	    s = copybuf;
	    buflen = COPYBUFSIZ;
	    for (; buflen > 1 &&
		 !(linedone = (c = getc(stdin)) == '\n' || c == EOF);
		 --buflen) {
		*s++ = c;
	    }
	    if (c == EOF) {
		/* reading from stdin, but from a file */
		PQputline(db, ".");
		copydone = 0;
		break;
	    }
	    *s = '\0';
	    PQputline(db, copybuf);
	    if (firstload) {
		if (!strcmp(copybuf, ".")) {
		    copydone = 0;
		}
		firstload = 1;
	    }
	}
	PQputline(db, "\n");
    }
    PQendcopy(db);
}

/************************************************************************/
/* Function: DB_TrapStderr                                              */
/* Purpose : Traps stderr for redirection to application                */
/* Params  : none                                                       */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_TrapStderr(void)
{
  int errfd[2];
  (void) signal(SIGPIPE,SIG_IGN);

  if(pipe(errfd))
    return;

  dup2(errfd[1], fileno(stderr));
  close(errfd[1]);

  XtAppAddInput(app, errfd[0], (caddr_t) XtInputReadMask, 
		(XtInputCallbackProc) DB_RedirectStderr, NULL);
  return;
}

/************************************************************************/
/* Function: DB_RedirectStderr                                          */
/* Purpose : Callback for writing stderr to the result window           */
/* Params  : client_data : pointer to callback client data              */
/*           fd          : file handle pointer (stderr)                 */
/*           id          : XtAppAddInput process id pointer             */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DB_RedirectStderr(XtPointer client_data, int *fd, XtInputId *id)
{
  char buf[EXPLAINBUFSIZ];
  int nbytes;

  if((nbytes = read(*fd,buf,EXPLAINBUFSIZ - 1)) == (EOF))
    return;
  buf[nbytes] = '\0';
  if(nbytes)
    DB_WriteToResult(buf, 0, NULL);
  return;
}
