# -*- coding: utf-8 -*-
## PloneArticle
## Copyright (C)2006 Ingeniweb

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING. If not, write to the
## Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
"""PloneArticle security tests

Use this file as a skeleton for your own tests
"""
__version__ = "$Revision: 1.5 $"
# $Source: /cvsroot/ingeniweb/PloneArticle/tests/testCMFEditions.py,v $
# $Id: testCMFEditions.py,v 1.5 2006/09/22 01:29:10 encolpe Exp $
__docformat__ = 'restructuredtext'

from types import TupleType, ListType
from copy import deepcopy
from common import *
from Products.CMFCore.utils import getToolByName

if pa_config.HAS_PLONE_2_1:
    FILE_PORTAL_TYPE = 'File'
    IMAGE_PORTAL_TYPE = 'Image'
    LINK_PORTAL_TYPE = 'Link'
else:
    FILE_PORTAL_TYPE = 'ATFile'
    IMAGE_PORTAL_TYPE = 'ATImage'
    LINK_PORTAL_TYPE = 'ATLink'

ARTICLE_ATTRIBUTES = (
    '__ordered_attachment_refs__',
    '__ordered_image_refs__',
    '__ordered_link_refs__',
    '_locked_by',
    '_locked_date',
    '_unlocked_by',
    '_unlocked_date',
    '_objects',
)

# Fake upload object
class TestFile:
    __allow_access_to_unprotected_subobjects__ = 1
    filename = 'afile.fil'
    def seek(*args): pass
    def tell(*args): return 0
    def read(*args): return 'file_contents'

tests = []

class TestCFMEditions(PloneArticleTestCase):

    def createEmptyArticle(self, container, content_id = 'article'):
        # return empty article
        container.invokeFactory(type_name='PloneArticle', id=content_id)
        article = getattr(container, content_id)
        return article

    def createExternalFile(self, container=None, content_id='external_file'):
        container = container or self.member_folder
        container.invokeFactory(type_name=FILE_PORTAL_TYPE, id=content_id)
        return getattr(container, content_id)

    def createArticle(self, container):
        """Create content in self.article
        """

        # Add a file not in PloneArticle
        self.external_file = self.createExternalFile(container)

        # Add an image not in PloneArticle
        content_id = 'external_image'
        container.invokeFactory(type_name=IMAGE_PORTAL_TYPE, id=content_id)
        self.external_image = getattr(container, content_id)

        #create empty article
        self.article = self.createEmptyArticle(container)

        # Add files in article
        content_id = 'article_file1'
        self.article.invokeFactory(type_name=FILE_PORTAL_TYPE, id=content_id)
        self.article_file1 = getattr(self.article, content_id)
        content_id = 'article_file2'
        self.article.invokeFactory(type_name=FILE_PORTAL_TYPE, id=content_id)
        self.article_file2 = getattr(self.article, content_id)

        # Add images in article
        content_id = 'article_image1'
        self.article.invokeFactory(type_name=IMAGE_PORTAL_TYPE, id=content_id)
        self.article_image1 = getattr(self.article, content_id)
        content_id = 'article_image2'
        self.article.invokeFactory(type_name=IMAGE_PORTAL_TYPE, id=content_id)
        self.article_image2 = getattr(self.article, content_id)

        # Add links in article
        content_id = 'article_link1'
        self.article.invokeFactory(type_name=LINK_PORTAL_TYPE, id=content_id)
        self.article_link1 = getattr(self.article, content_id)
        content_id = 'article_link2'
        self.article.invokeFactory(type_name=LINK_PORTAL_TYPE, id=content_id)
        self.article_link2 = getattr(self.article, content_id)

        # Add files and images as references of article
        self.article.addAttachmentFromObject(self.external_file)
        self.article.addImageFromObject(self.external_image)
        self.article.addAttachmentFromObject(self.article_file1)
        self.article.addImageFromObject(self.article_image1)
        self.article.addLinkFromObject(self.article_link1)


    def testPloneArticleInterface(self):
        self.loginAsPortalMember()
        self.createArticle(self.member_folder)
        self.failUnless(IArticleAttachments.isImplementedBy(self.article))
        self.failUnless(IArticleImages.isImplementedBy(self.article))
        self.logout()


    def test_cmfedition_00_tool_CMFEditions_support(self,):
        """Testing CMFEditions configuration for PloneArticle
        """
        portal_article = getToolByName(self.portal, 'portal_article')

        #default is 0 bt we configure it to 1 in tests setup
        self.assertEqual(portal_article.hasCMFEditionsSupport(), 1)


        rep_tool = getToolByName(self.portal, 'portal_repository', None)
        mod_tool = getToolByName(self.portal, 'portal_modifier', None)

        self.assertEqual(rep_tool.getVersionableContentTypes(),
                         ['PloneArticle', 'PloneArticleMultiPage'])
        self.assertEqual(rep_tool.getPolicyMap(),
                         {
                             'PloneArticle': ['version_on_revert'],
                             'PloneArticleMultiPage': ['version_on_revert'],
                         })

        # We have to check three things on modifier:
        # 1. Does it exist
        # 2. Does it implement TalesMoifier
        # 3. Does the condition check every PloneArticle portal_type
        modifier = mod_tool.get('PloneArticleV3Modifier')
        from Products.CMFEditions.Modifiers import ConditionalTalesModifier
        from Products.PloneArticle.CMFEditionsModifier.PloneArticleV3CMFEModifier import PloneArticleV3Modifier
        self.failUnless(isinstance(modifier, ConditionalTalesModifier))
        self.failUnless(isinstance(modifier.getModifier(), PloneArticleV3Modifier))
        self.failUnless(modifier.isEnabled())
        self.failUnless(modifier.getTalesCondition() == "python: hasattr(object, 'portal_type') and object.portal_type in ('PloneArticle', 'PloneArticleMultiPage')")

        # Testing setattr misfunctions
        portal_article.updateCMFEditionsSupport(value=0)
        self.assertEqual(portal_article.hasCMFEditionsSupport(), 0)

        portal_article.updateCMFEditionsSupport(value=1)
        self.assertEqual(portal_article.hasCMFEditionsSupport(), 1)


    def test_cmfedition_00_is_article_versionable(self):
        """Versionable article content types

        We verify that article is configured as versionned document.
        It cannot be versionned 'on edit' in PloneArticle < version 4.0.
        The option 'version_on_revert' seems to be reasonable here.
        """
        portal_repository = getToolByName(self.portal, 'portal_repository', None)
        versionable_types = portal_repository.getVersionableContentTypes()
        # Are PloneArticle content types versionable ?
        self.failIf('PloneArticle' not in versionable_types)
        self.failIf('PloneArticleMultiPage' not in versionable_types)

        policy_map = portal_repository.getPolicyMap()
        # Are they correctly configured ?
        self.failUnless(policy_map['PloneArticle'] == ['version_on_revert'])
        self.failUnless(policy_map['PloneArticleMultiPage'] == ['version_on_revert'])

    def test_cmfedition_article_01_adding_a_version(self):
        """Adding a version on a PloneArticle

        We verify that the current article is not altered by the versionning.

        What's the different between 'applyVersionControl' and 'save'?
        1. They use different permissions
        2. '_recursiveSave' autoapply parameter is always equal to True with
           'applyVersionControl'. 'save' use self.autoapply => (on_edit ?)
        We would only test 'applyVersionControl' in other test because on_edit
        not occurs when we save attachments, images or links.
        """
        self.loginAsPortalMember()
        self.createArticle(self.member_folder)

        portal_repository = getToolByName(self.portal, 'portal_repository', None)
        portal_archivist = getToolByName(self.portal, 'portal_archivist', None)

        article_content_v0 = deepcopy(self.article.__dict__)

        portal_repository.applyVersionControl(self.article, comment='save no 1')
        # Is current version modified by the versionning API ? It does not.
        article_content_v1 = deepcopy(self.article.__dict__)

        portal_repository.save(self.article, comment='save no 2')
        # Is current version modified by the versionning API ? It does not.
        article_content_v2 = deepcopy(self.article.__dict__)

        for key in ARTICLE_ATTRIBUTES:
            #print key
            value_v0 = article_content_v0[key]
            if hasattr(value_v0, 'getItems'):
                value_v0 = value_v0.getItems()
            #PrettyPrinter().pprint(value_v0)

            value_v1 = article_content_v1[key]
            if hasattr(value_v1, 'getItems'):
                value_v1 = value_v1.getItems()
            #PrettyPrinter().pprint(value_v1)

            value_v2 = article_content_v2[key]
            if hasattr(value_v2, 'getItems'):
                value_v2 = value_v2.getItems()
            #PrettyPrinter().pprint(value_v2)

            self.failUnless(value_v0 == value_v1)
            self.failUnless(value_v0 == value_v2)


        # Now verify that two versions are stored
        history = portal_repository.getHistory(self.article, countPurged=False)
        self.failUnless(len(history) == 2)
 
        # Now we check that saved version are correct

        # Version number: history give us a reversed list
        self.failUnless(history[0].version_id == 1)
        self.failUnless(history[1].version_id == 0)

        # Version referenced object
        self.failUnless(history[0].object.UID() == self.article.UID())
        self.failUnless(history[1].object.UID() == self.article.UID())

        self.logout()


    def test_cmfedition_article_02_retrieve_a_version(self):
        """Retrieve a version on a PloneArticle

        We add and remove internal data and compare them with stored version
        """
        self.loginAsPortalMember()
        self.createArticle(self.member_folder)

        portal_repository = getToolByName(self.portal, 'portal_repository', None)
        portal_archivist = getToolByName(self.portal, 'portal_archivist', None)

        article_content_v0 = deepcopy(self.article.__dict__)
        refs_v0 = (
            article_content_v0['__ordered_attachment_refs__'].getItems(),
            article_content_v0['__ordered_image_refs__'].getItems(),
            article_content_v0['__ordered_link_refs__'].getItems(),
        )

        portal_repository.applyVersionControl(self.article, comment='save n°1: Nothing but default')

        # For the first version we add contents
        self.article.addAttachmentFromObject(self.article_file2)
        self.article.addImageFromObject(self.article_image2)
        self.article.addLinkFromObject(self.article_link2)

        article_content_v1 = deepcopy(self.article.__dict__)
        refs_v1 = (
            article_content_v1['__ordered_attachment_refs__'].getItems(),
            article_content_v1['__ordered_image_refs__'].getItems(),
            article_content_v1['__ordered_link_refs__'].getItems(),
        )

        # We verify that modifications are saved
        for index in range(0,3,1):
            self.failIf(refs_v0[index] == refs_v1[index])

        portal_repository.applyVersionControl(self.article, comment='save n°2: Add contents')

        # We retrieve the v0 and compare it
        clone_content_v0 = deepcopy(portal_repository.retrieve(self.article, 0).object.__dict__)
        for key in ARTICLE_ATTRIBUTES:
            #print key
            value_v0 = article_content_v0[key]
            if hasattr(value_v0, 'getItems'):
                value_v0 = value_v0.getItems()
            #PrettyPrinter().pprint(value_v0)

            clone_v0 = clone_content_v0[key]
            if hasattr(clone_v0, 'getItems'):
                clone_v0 = clone_v0.getItems()
            #PrettyPrinter().pprint(clone_v0)

            self.failUnless(value_v0 == clone_v0)

        # For the second version we remove contents
        # We would verify against v0 and v1
        attachment_uids = [e.UID() for e in self.article.getAttachments()]
        self.article.removeAttachments(UIDs=attachment_uids[:2])
        image_uids = [e.UID() for e in self.article.getImages()]
        self.article.removeImages(UIDs=[image_uids[0], image_uids[2]])
        link_uids = [e.UID() for e in self.article.getLinks()]
        self.article.removeLink(UID=link_uids[0])

        article_content_v2 = deepcopy(self.article.__dict__)
        refs_v2 = (
            article_content_v2['__ordered_attachment_refs__'].getItems(),
            article_content_v2['__ordered_image_refs__'].getItems(),
            article_content_v2['__ordered_link_refs__'].getItems(),
        )

        # We verify that modifications are saved
        for index in range(0,3,1):
            self.failIf(refs_v1[index] == refs_v2[index])

        portal_repository.applyVersionControl(self.article, comment='save n°3: Remove some new and old contents')

        # We retrieve the v0 and compare it
        clone_content_v0 = deepcopy(portal_repository.retrieve(self.article, 0).object.__dict__)
        for key in ARTICLE_ATTRIBUTES:
            #print key
            value_v0 = article_content_v0[key]
            if hasattr(value_v0, 'getItems'):
                value_v0 = value_v0.getItems()
            #PrettyPrinter().pprint(value_v0)

            clone_v0 = clone_content_v0[key]
            if hasattr(clone_v0, 'getItems'):
                clone_v0 = clone_v0.getItems()
            #PrettyPrinter().pprint(clone_v0)

            self.failUnless(value_v0 == clone_v0)

        # We retrieve the v1 and compare it
        clone_content_v1 = deepcopy(portal_repository.retrieve(self.article, 1).object.__dict__)
        for key in ARTICLE_ATTRIBUTES:
            #print key
            value_v1 = article_content_v1[key]
            if hasattr(value_v1, 'getItems'):
                value_v1 = value_v1.getItems()
            #PrettyPrinter().pprint(value_v1)

            clone_v1 = clone_content_v1[key]
            if hasattr(clone_v1, 'getItems'):
                clone_v1 = clone_v1.getItems()
            #PrettyPrinter().pprint(clone_v1)

            self.failUnless(value_v1 == clone_v1)


    def test_cmfedition_article_03_revert_version_v2_v1_v0(self):
        """Revert a version on a PloneArticle v2 to v1 then to v0

        We add and remove internal data and compare them with reverted version
        """
        self.loginAsPortalMember()
        self.createArticle(self.member_folder)

        portal_repository = getToolByName(self.portal, 'portal_repository', None)
        portal_archivist = getToolByName(self.portal, 'portal_archivist', None)

        article_content_v0 = deepcopy(self.article.__dict__)
        refs_v0 = (
            article_content_v0['__ordered_attachment_refs__'].getItems(),
            article_content_v0['__ordered_image_refs__'].getItems(),
            article_content_v0['__ordered_link_refs__'].getItems(),
        )

        portal_repository.applyVersionControl(self.article, comment='save n°1: Nothing but default')

        # For the first version we add contents
        self.article.addAttachmentFromObject(self.article_file2)
        self.article.addImageFromObject(self.article_image2)
        self.article.addLinkFromObject(self.article_link2)

        article_content_v1 = deepcopy(self.article.__dict__)
        refs_v1 = (
            article_content_v1['__ordered_attachment_refs__'].getItems(),
            article_content_v1['__ordered_image_refs__'].getItems(),
            article_content_v1['__ordered_link_refs__'].getItems(),
        )

        # We verify that modifications are saved
        for index in range(0,3,1):
            self.failIf(refs_v0[index] == refs_v1[index])

        portal_repository.applyVersionControl(self.article, comment='save n°2: Add contents')

        # For the second version we remove contents
        # We would verify against v0 and v1
        attachment_uids = [e.UID() for e in self.article.getAttachments()]
        self.article.removeAttachments(UIDs=attachment_uids[:2])
        image_uids = [e.UID() for e in self.article.getImages()]
        self.article.removeImages(UIDs=[image_uids[0], image_uids[2]])
        link_uids = [e.UID() for e in self.article.getLinks()]
        self.article.removeLink(UID=link_uids[0])

        article_content_v2 = deepcopy(self.article.__dict__)
        refs_v2 = (
            article_content_v2['__ordered_attachment_refs__'].getItems(),
            article_content_v2['__ordered_image_refs__'].getItems(),
            article_content_v2['__ordered_link_refs__'].getItems(),
        )

        # We verify that modifications are saved
        for index in range(0,3,1):
            self.failIf(refs_v1[index] == refs_v2[index])

        portal_repository.applyVersionControl(self.article, comment='save n°3: Remove some new and old contents')

        # We revert the v1 and compare it
        portal_repository.revert(self.article, 1)
        clone_content_v1 = deepcopy(self.article.__dict__)

        for key in ARTICLE_ATTRIBUTES:
            print
            print key
            value_v1 = article_content_v1[key]
            if hasattr(value_v1, 'getItems'):
                value_v1 = value_v1.getItems()
            if isinstance(value_v1, (TupleType, ListType)):
                value_v1 = list(value_v1)
                value_v1.sort()
            print 'v1:'
            PrettyPrinter().pprint(value_v1)

            clone_v1 = clone_content_v1[key]
            if hasattr(clone_v1, 'getItems'):
                clone_v1 = clone_v1.getItems()
            if isinstance(clone_v1, (TupleType, ListType)):
                clone_v1 = list(clone_v1)
                clone_v1.sort()
            print 'clone v1:'
            PrettyPrinter().pprint(clone_v1)

            self.failUnless(value_v1 == clone_v1)

        # We revert the v0 and compare it
        portal_repository.revert(self.article, 0)
        clone_content_v0 = deepcopy(self.article.__dict__)

        for key in ARTICLE_ATTRIBUTES:
            print
            print key
            value_v0 = article_content_v0[key]
            if hasattr(value_v0, 'getItems'):
                value_v0 = value_v0.getItems()
            if isinstance(value_v0, (TupleType, ListType)):
                value_v0 = list(value_v0)
                value_v0.sort()
            print 'v0:'
            PrettyPrinter().pprint(value_v0)

            clone_v0 = clone_content_v0[key]
            if hasattr(clone_v0, 'getItems'):
                clone_v0 = clone_v0.getItems()
            if isinstance(clone_v0, (TupleType, ListType)):
                clone_v0 = list(clone_v0)
                clone_v0.sort()
            print 'clone v0:'
            PrettyPrinter().pprint(clone_v0)

            self.failUnless(value_v0 == clone_v0)


if pa_config.HAS_CMFEDITIONS:
    tests.append(TestCFMEditions)

if __name__ == '__main__':
    framework()
else:
    # While framework.py provides its own test_suite()
    # method the testrunner utility does not.
    import unittest
    def test_suite():
        suite = unittest.TestSuite()
        for test in tests:
            suite.addTest(unittest.makeSuite(test))
        return suite
