/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <swidget/SEditor.h>
#include <swidget/SEditorIF.h>

/**
 * All editors are like this one.
 */
SEditor::SEditor(void)
{
  cleared = false;
  lineBreak = "\n";
  editorIF = 0;
  vimode = false;
  undoIndex = 0;
  setup();
  
}

SEditor::SEditor (const SString& lb)
{
  cleared = false;
  lineBreak = lb;
  editorIF = 0;
  vimode = false;
  undoIndex = 0;
  setup();
}

void
SEditor::setup()
{
  inputMethods.append (SS_DEFAULT_IM);
  inputMethods.append ("unicode");
  inputMethods.append (SS_KINPUT2_IM);
  inputMethods.append ("Hungarian");
  inputMethods.append ("Kana");
  inputMethods.append ("Hangul");
  fonts.append (SS_DEFAULT_FONT);
  fonts.append ("Bitmap");
  fonts.append ("TrueType");
}

/**
 * Copy editor and lose undo's
 */
SEditor::SEditor (const SEditor& in)
{
  editorIF = in.editorIF;
  vimode = in.vimode;
  inputMethods = in.inputMethods;
  fonts = in.fonts;
  undoIndex = 0;
  lineBreak = in.lineBreak;
}

/**
 * Copy editor and lose undo's
 */
SEditor
SEditor::operator=(const SEditor& in)
{
  editorIF = in.editorIF;
  vimode = in.vimode;
  inputMethods = in.inputMethods;
  fonts = in.fonts;
  undoIndex = 0;
  lineBreak = in.lineBreak;
  return *this;
}


SEditor::~SEditor ()
{
}

void
SEditor::setInterface(SEditorIF* _editorIF)
{
 editorIF = _editorIF; 
}

/**
 * called when a key has been pressed.
 */
void
SEditor::keyPressed (SWindowListener::SKey key, const SString& s,
            bool ctrl, bool shift, bool meta)
{
  if (editorIF == 0) return;
  if (key==SWindowListener::Key_Meta_R || key == SWindowListener::Key_Meta_L
   || key == SWindowListener::Key_Alt_R || key==SWindowListener::Key_Alt_L)
  {
    editorIF->deselectText();
    editorIF->startSelect();
    return;
  }
  /* Whoops - we did not receive keyrelease */
  if (editorIF->isSelecting() && !meta)
  {
    editorIF->endSelect();
  }
  SString string = s;
  STextIndex b = editorIF->getCaret();
  STextIndex a;
  STextIndex ba;
  SUndo::SType std;
  cleared = false;
  switch (key)
  {
  case SWindowListener::Key_Escape:
   // vi
    if (vimode)
    {
      editorIF->setEditable(!editorIF->isEditable());
    }
    else
    {
      editorIF->focusOut();
    }
    return;
  case SWindowListener::Key_Home:
    editorIF->caretGoTo(STextIndex(0,0));
    break;
  case SWindowListener::Key_End:
    /* I live this hack */
    editorIF->caretGoTo(STextIndex(100000000,0));
    break;
  case SWindowListener::Key_Prior:
    editorIF->pageUp();
    return;
  case SWindowListener::Key_Next:
    editorIF->pageDown();
    return;
  case SWindowListener::Key_Up:
    editorIF->caretUp();
    return;
  case SWindowListener::Key_Down:
    editorIF->caretDown();
    return;
  case SWindowListener::Key_Left:
  // Ifound this toggle very annoying.
#if 0
    if (editorIF->getDirection () == '>')
    {
       editorIF->setDirection ('<');
       return;
    }
#endif
    editorIF->caretLeft();
    return;
  case SWindowListener::Key_Right:
  // Ifound this toggle very annoying.
#if 0
    if (editorIF->getDirection () == '<')
    {
       editorIF->setDirection ('>');
       return;
    }
#endif
    editorIF->caretRight();
    return;
  case SWindowListener::Key_BackSpace:
    /* switch - delete */
    std = SUndo::SS_EraseSelect;
    ba = editorIF->getSelectedIndex(true);
    string = editorIF->eraseSelectedText();
    if (string.size() == 0)
    {
      if (editorIF->getDirection () == '<')
      {
         std = SUndo::SS_EraseOne;
         string = editorIF->erase();
      }
      else
      {
         std = SUndo::SS_BackSpaceOne;
         string = editorIF->backspace();
      }
      /* not even used */
      ba = editorIF->getCaret();
    }
    if (string.size() != 0)
    {
      a = editorIF->getCaret();
      add (SUndo(std, editorIF->getDirection (), string, b, a, ba));
    }
    return;
  case SWindowListener::Key_Delete:
    /* switch - backspace */
    std = SUndo::SS_EraseSelect;
    ba = editorIF->getSelectedIndex(true);
    string = editorIF->eraseSelectedText();
    if (string.size() == 0)
    {
      if (editorIF->getDirection () == '<')
      {
         std = SUndo::SS_BackSpaceOne;
         string = editorIF->backspace();
      }
      else
      {
         std = SUndo::SS_EraseOne;
         string = editorIF->erase();
      }
      /* not even used */
      ba = editorIF->getCaret();
    }
    if (string.size() != 0)
    {
      a = editorIF->getCaret();
      add (SUndo(std, editorIF->getDirection (), string, b, a, ba));
    }
    return;
  case SWindowListener::Key_Enter:
    if (editorIF->getDirection () == '<')
    {
      b = editorIF->getIndexBeforeLineBreak();
      if (cleared) return;
      editorIF->caretGoTo(b);
    }
    ba = editorIF->insertDirtyText(lineBreak);
    if (cleared) return;
    a = editorIF->getCaret();
    add (SUndo(SUndo::SS_Insert, editorIF->getDirection (), lineBreak, b, a, ba));
    break;
  case SWindowListener::Key_Return:
    if (editorIF->getDirection () == '<')
    {
      b = editorIF->getIndexBeforeLineBreak();
      if (cleared) return;
      editorIF->caretGoTo(b);
    }
    ba = editorIF->insertDirtyText(lineBreak);
    if (cleared) return;
    a = editorIF->getCaret();
    add (SUndo(SUndo::SS_Insert, editorIF->getDirection (), lineBreak, b, a, ba));
    break;
  case SWindowListener::Key_F1:
    fKey (0, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F2:
    fKey (1, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F3:
    fKey (2, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F4:
    fKey (3, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F5:
    fKey (4, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F6:
    fKey (5, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F7:
    fKey (6, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F8:
    fKey (7, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F9:
    fKey (8, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F10:
    fKey (9, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F11:
    fKey (10, ctrl|shift|meta);
    break;
  case SWindowListener::Key_F12:
    fKey (11, ctrl|shift|meta);
    break;
  case SWindowListener::Key_u:
  case SWindowListener::Key_U:
    if (ctrl|meta)
    {
      undo ();
      break;
    }
  case SWindowListener::Key_d:
  case SWindowListener::Key_D:
    if (ctrl|meta)
    {
      if (editorIF->getDirection () == '<')
      {
        editorIF->setDirection ('>');
      }
      else
      {
        editorIF->setDirection ('<');
      }
      break;
    }
  case SWindowListener::Key_r:
  case SWindowListener::Key_R:
    if (ctrl|meta)
    {
      redo ();
      break;
    }
  case SWindowListener::Key_f:
  case SWindowListener::Key_F:
    if (ctrl|meta)
    {
      editorIF->pageDown ();
      break;
    }
  case SWindowListener::Key_v:
  case SWindowListener::Key_V:
    if (ctrl|meta)
    {
      pasteText (b);
      return;
    }
  case SWindowListener::Key_x:
  case SWindowListener::Key_X:
    if (ctrl|meta)
    {
      /* switch - delete */
      std = SUndo::SS_EraseSelect;
      ba = editorIF->getSelectedIndex(true);
      string = editorIF->eraseSelectedText();
      if (cleared) return;
      if (string.size() != 0)
      {
        a = editorIF->getCaret();
        add (SUndo(std, editorIF->getDirection (), string, b, a, ba));
      }
      break;
    }
  case SWindowListener::Key_b:
  case SWindowListener::Key_B:
    if (ctrl|meta)
    {
      editorIF->pageUp ();
      break;
    }
  case SWindowListener::Key_p:
  case SWindowListener::Key_P:
  case SWindowListener::Key_k:
  case SWindowListener::Key_K:
    if (ctrl|meta)
    {
      editorIF->caretUp ();
      break;
    }
  case SWindowListener::Key_n:
  case SWindowListener::Key_N:
  case SWindowListener::Key_j:
  case SWindowListener::Key_J:
    if (ctrl||meta)
    {
      editorIF->caretDown ();
      break;
    }
  case SWindowListener::Key_h:
  case SWindowListener::Key_H:
    if (ctrl|meta)
    {
      editorIF->caretLeft ();
      break;
    }
  case SWindowListener::Key_l:
  case SWindowListener::Key_L:
    if (ctrl||meta)
    {
      editorIF->caretRight ();
      break;
    }
  case SWindowListener::Key_m:
  case SWindowListener::Key_M:
    if (ctrl)
    {
      editorIF->endSelect();
      editorIF->caretGoTo(STextIndex(b.line, 0));
      ba = editorIF->getIndexAfterLineBreak ();
      string =  editorIF->eraseText(ba);
      if (cleared) return;
      if (string.size() != 0)
      {
        a = editorIF->getCaret();
        add (SUndo(SUndo::SS_Erase, editorIF->getDirection (), 
             string, b, a, ba));
      }
      break;
    }
  case SWindowListener::Key_Send:
  default:
    if (vimode && key != SWindowListener::Key_Send &&
        s == ":" && !editorIF->isEditable())
    {
      /* push the focus to the command window. */
      editorIF->focusOut();
      break;
    }
    if (s.size())
    {
      ba =  editorIF->insertDirtyText(s);
      a = editorIF->getCaret();
      add (SUndo(SUndo::SS_Insert, editorIF->getDirection (), s, b, a, ba));
    }
  }
}

/**
 * called when a key was released.
 */
void
SEditor::keyReleased (SWindowListener::SKey key, const SString& s,
            bool ctrl, bool shift, bool meta)
{
  if (editorIF == 0) return;
  if (key==SWindowListener::Key_Meta_R || key == SWindowListener::Key_Meta_L
   || key == SWindowListener::Key_Alt_R || key==SWindowListener::Key_Alt_L)
  {
    editorIF->endSelect();
    return;
  }
}

/**
 * called when a mouse button was pressed.
 */
void
SEditor::buttonPressed (int button, const STextIndex& index)
{
  if (editorIF == 0) return;
  editorIF->deselectText();
  if (button == 1)
  {
    pasteText(editorIF->getCaret());
    return;
  }
  editorIF->caretGoTo (index);
  editorIF->startSelect();
}

/**
 * paste a text ftom clip
 */
void
SEditor::pasteText (const STextIndex& index)
{
  if (editorIF == 0) return;
  editorIF->deselectText();
  editorIF->caretGoTo(index);
  STextIndex b = index;
  STextIndex ba = editorIF->insertText();
  SString s = editorIF->getText (b, ba);
  STextIndex a = editorIF->getCaret();
  add (SUndo(SUndo::SS_Insert, editorIF->getDirection (), s, b, a, ba));
}

/**
 * Set clean text.
 */
void
SEditor::insertText (const SString& text)
{
  if (editorIF == 0) return;
  editorIF->deselectText();
  STextIndex b = editorIF->getCaret();
  STextIndex ba = editorIF->insertText(text);
  SString s = editorIF->getText (b, ba);
  STextIndex a = editorIF->getCaret();
  add (SUndo(SUndo::SS_Insert, editorIF->getDirection (), s, b, a, ba));
}

/**
 * called when a mouse button was released.
 */
void
SEditor::buttonReleased (int button, const STextIndex& index)
{
  if (editorIF == 0) return;
  if (button == 1)
  {
    return;
  }
  editorIF->selectText (index);
  editorIF->endSelect ();
}

/**
 * called when a mouse button was dragged.
 */
void
SEditor::buttonDragged (int button, const STextIndex& index)
{
  if (button == 1)
  {
    return;
  }
  if (editorIF == 0) return;
  editorIF->selectText (index);
}

/**
 * called when a mouse button was nulti-clicked.
 * It can come only if button was already released.
 */
void
SEditor::multiClicked (int button, const STextIndex& index, int count)
{
  if (editorIF == 0) return;
  if (button == 1)
  {
    return;
  }
  editorIF->selectText (index);
  editorIF->endSelect ();
  if (count == 2)
  {
    editorIF->selectWord ();
  }
  else if (count > 2)
  {
    editorIF->selectLine ();
  }
}

/**
 * This sets a set of input methods that can be activated with F1-F12.
 */
void
SEditor::setInputMethods (const SStringVector& in)
{
  inputMethods = in;
}

/**
 * set editor mode.
 */
void
SEditor::setVI (bool vi)
{
  vimode = vi;
}
/**
 * This sets a set of fonts that can be activated with <ctrl>F1-F12.
 */
void
SEditor::setFonts (const SStringVector& in)
{
  fonts = in;
}

/**
 * An FKEY was pressed.
 * @param num
 * <ul>
 * <li> F1 - 0 </li>
 * <li> F2 - 1 </li>
 * <li> Fn - n-1 </li>
 * </ul>
 */
void
SEditor::fKey(int num, bool ctrl)
{
  if (ctrl)
  {
    if (fonts.size() >= (unsigned int) num+1)
    {
      editorIF->setFont(fonts[num]);
    }
  }
  else
  {
    if (inputMethods.size() >= (unsigned int) num+1)
    {
       editorIF->setInputMethod(inputMethods[num]);
    }
  }
}
/**
 * Try to undo
 * @return false in case it can not be undone.
 */
bool
SEditor::undo()
{
  //fprintf (stderr, "undo\n");
  if (editorIF == 0) return false;
  /* clear state should be visible */
  if (editorIF->clearState()) return true;
  if (undoIndex == 0)
  {
    //fprintf (stderr, "SEditor::nothing to undo - undoindex\n");
    return false;
  }
  if (!editorIF->isEditable())
  {
     return false;
  }
  if (editorIF->isSelecting())
  {
    // fprintf (stderr, "SEditor::can not undo while selecting.\n");
     return false;
  }
  undoIndex--;
  SUndo u = undoBuffer[undoIndex];
  //fprintf (stderr, "Undo:%d.\n", u.type);
  STextIndex in;
  switch (u.type)
  {
  case SUndo::SS_Insert:
    editorIF->caretGoTo(u.beforeAfter);
    editorIF->eraseText (u.before);
    editorIF->setDirection(u.caret);
    break;
  case SUndo::SS_Erase:
  case SUndo::SS_EraseOne:
  case SUndo::SS_BackSpaceOne:
    editorIF->caretGoTo(u.after);
    in = editorIF->insertText(u.string);
    editorIF->caretGoTo(u.before);
    break;
  case SUndo::SS_EraseSelect:
    editorIF->caretGoTo(u.after);
    STextIndex in = editorIF->insertText(u.string);
    editorIF->caretGoTo(in);
    editorIF->startSelect();

    /* this is the sleection spot */
    editorIF->selectText(u.after);
    editorIF->endSelect();
    editorIF->caretGoTo(u.after);
    break;
  }
  return true;
}

/**
 * Try to redo
 * @return false in case it can not be undone.
 */
bool
SEditor::redo()
{
  //fprintf (stderr, "redo\n");
  if (editorIF == 0) return false;
  editorIF->clearState();
  if (undoIndex == undoBuffer.size())
  {
    //fprintf (stderr, "SEditor::redo ends here.\n");
    return false;
  }
  if (!editorIF->isEditable()) return false;
  if (editorIF->isSelecting())
  {
    //fprintf (stderr, "SEditor::can not redo while selecting.\n");
    return false;
  }

  SUndo u = undoBuffer[undoIndex];
  undoIndex++;
  //fprintf (stderr, "redo:%d.\n", u.type);
  STextIndex in;
  switch (u.type)
  {
  case SUndo::SS_Insert:
    editorIF->caretGoTo(u.before);
    in = editorIF->insertText(u.string);
    break;
  case SUndo::SS_Erase:
  case SUndo::SS_EraseSelect:
    editorIF->caretGoTo(u.after);
    editorIF->eraseText(u.beforeAfter);
    break;
  case SUndo::SS_BackSpaceOne:
    editorIF->caretGoTo(u.before);
    editorIF->backspace();
    break;
  case SUndo::SS_EraseOne:
    editorIF->caretGoTo(u.before);
    editorIF->erase();
    break;
  }
  return true;
}

/**
 * add an item to undo buffer
 * @param u is the new item.
 * TODO: clear buffer after some size.
 */
void
SEditor::add (const SUndo& u)
{
  if (undoBuffer.size() != undoIndex)
  {
    if (undoIndex==0)
    {
      undoBuffer.clear();
    }
    else 
    {
      undoBuffer.truncate(undoIndex);
    }
    undoIndex = undoBuffer.size();
  }
  undoBuffer.append (u);
  undoIndex++;
}

/**
 * clear the undo buffer.
 */
void
SEditor::clearUndo()
{
  undoBuffer.clear();
  undoIndex = 0;
}

void
SEditor::clear ()
{
  cleared = true;
  vimode = false;
  undoIndex = 0;
  undoBuffer.clear();
}

void
SEditor::setLineBreak (const SString& _lineBreak)
{
  lineBreak = _lineBreak;
}

SString
SEditor::getLineBreak () const
{
  return SString (lineBreak);
}
