/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "stoolkit/STextData.h"

#include "stoolkit/SUniMap.h"
#include "stoolkit/SEncoder.h"
#include "stoolkit/SBinHashtable.h"
#include "stoolkit/SUtil.h"
#include "stoolkit/SCluster.h"

SBinHashtable<SGlyphShared*> glyphCache;

/*------------------------------------------------------------------------------
 *                     STextDataEvent
 *------------------------------------------------------------------------------
 */
/**
 * @param attr is tru if no text was added or removed. only attribute changed.
 */
STextDataEvent::STextDataEvent (const STextIndex& _start, bool attr)
{
  start = _start;
  valid = false;
  attribute = attr;
}
/**
 * Create a stoolkit STextDataEvent
 */
STextDataEvent::STextDataEvent (void)
{
  clear ();
}

/**
 * Set the remaining.
 */
void
STextDataEvent::setRemaining (const STextIndex& _remaining)
{
  remaining = _remaining;
  valid = true;
}

/**
 * Nothing to destroy. Just in case
 */
STextDataEvent::~STextDataEvent ()
{
}

/**
 * Clear all fields.
 */
void
STextDataEvent::clear ()
{
  start = STextIndex (0,0);
  remaining =  STextIndex (0,0);
  valid = false;
  attribute = true;
}

/**
 * Add a new event to this in a sane way.
 */
void
STextDataEvent::add (const STextDataEvent& event)
{
  if (!event.valid)
  {
    return;
  }
  if (!valid)
  {
    start = event.start;
    remaining = event.remaining;
    if (!event.attribute) attribute = false;
  }
  else if (event.remaining < remaining)
  {
    remaining = event.remaining;
  }
  valid = true;
}

/*------------------------------------------------------------------------------
 *                     SGlyph
 *------------------------------------------------------------------------------
 */
/**
 * I would like to protect this to this package.
 * Not meant to be used in other packages 
 */
SGlyph::SGlyph (char type, SGlyphShared* _shared)
{
  selected = false;
  underlined = false;
  useComposition = true;
  lr = false;
  shared = _shared;
  if (shared->type ==0 && type!=0) shared->type = type;
  currentShape = shared->shaped ? 0 : -1; /* first or not shaped */
}

/**
 * This is the definition of one glyph. It may be composed of
 * several characters.
 * @param decomp is the character array in order. It ALWAYS contains chars.
 * @param comp is the composition character. It might be 0. In this
 *  case decomp needs to be used.
 * @param shaped is used only if shared is not chached. It tells if
 *  the glyph is shaped.
 * YOU SOULD ALSO SET useComposition aftwerwards.
 *
 * ONLY PRECOMPOSED OR SINGLE CHARS WILL BE CACHED!
 * comp is nonzero for precomposed chars.
 */
SGlyph::SGlyph (char type, const SV_UCS4 &decomp, SS_UCS4 comp,
  SS_UCS4 mirror, bool shaped, unsigned int cluster)
{
  selected = false;
  underlined = false;
  useComposition = true;
  lr = false;
  SS_UCS4 precomp = comp;
  /* There is no such thing as a one character composition */
  if (decomp.size()==1) precomp = decomp[0];

  SString skey = (precomp==0) ? SString() : SString((long)precomp);
  if (skey.size()==0 && decomp.size() > 1)
  {
    skey = SString ((char*) decomp.array(), sizeof (SS_UCS4) * decomp.size());
  }

  SGlyphShared* shr = 0;
  if ((shr=glyphCache.get (skey))==0)
  {
    shr = new  SGlyphShared();
    shr->lineend = decomp[0] == LS || decomp[0] == PS 
      || decomp[0] == CR || decomp[0] == LF;
    shr->tab = decomp.size()==1 && decomp[0] == TAB;
    shr->type = type;
    shr->shaped = shaped;
    shr->cluster = cluster;
    shr->mirror = mirror;

    /* 
     *  We save only 
     *  1. chars with precomposition 
     *  2. single chars
     */
    shr->mcomposition = precomp;

    /* Memory is expensive, dont waste on single chars */
    SV_UCS4 svs;
    if (decomp.size() > 1)
    {
      svs = decomp;
    }
    shr->ucs4v = svs;
    if (skey.size())
    {
      glyphCache.put (skey, shr);
    }
  }
  shared = shr; 
  if (shared->type ==0 && type!=0) shared->type = type;
  currentShape = shared->shaped ? 0 : -1; /* first or not shaped */
}

/*------------------------------------------------------------------------------
 *                    SGlyph Work on characters 
 *------------------------------------------------------------------------------
 */
/**
 * Get the composition of the glyph. If this is a shaped
 * Glyph this is the ORIGINAL composition - if any. 
 */ 
const SS_UCS4
SGlyph::getChar() const
{
  return shared->mcomposition;
}

/*
 * Get the mirrored version of the glyph.
 */

const SS_UCS4
SGlyph::getMirroredChar() const
{
  return shared->mirror;
}

/**
 * Get the current shape of a shaped glyph.
 * If glyph is not shaping return getChar.
 */
const SS_UCS4
SGlyph::getShapedChar() const
{
  if (shared->shaped && currentShape >=0)
  {
    SS_UCS4 curr = shared->ucs4v.array()[(unsigned int)currentShape];
    if (curr) return curr;
    /* some fallback. if we are lucky... */
  }
  return shared->mcomposition;
}

/**
 * Get the first character of this composition or decomposition
 */
const SS_UCS4
SGlyph::getFirstChar () const
{
  SS_UCS4 g = getChar();
  if (g==0 && size() > 0) 
  {
     g = getDecompArray()[0];
  }
  return g;
}

/*------------------------------------------------------------------------------
 *                    SGlyph Work on decompositions 
 *------------------------------------------------------------------------------
 */

/**
 * Get the precomposed characters _or_ the composition characters.
 * If precomposed characters we input, return them, if decomposed
 * were input return them. It returns the unicode representation 
 * of the given cluster.
 */
SV_UCS4
SGlyph::getChars() const
{
  SV_UCS4 ret;
  if ((useComposition && getChar() != 0) || size()==0)
  {
    ret.append (getChar());
    return SV_UCS4(ret);
  }
  /* TODO: shaping on clusters */
  if (shared->cluster!=0)
  {
    unsigned int sz = shared->ucs4v.size();
    for (unsigned int i=(unsigned int)shared->cluster; i<sz; i++)
    {
      ret.append (shared->ucs4v[i]);
    }
    return SV_UCS4(ret);
  }
  const SS_UCS4* buff = getDecompArray();
  unsigned int sz = size();
  for (unsigned int i=0; i<sz; i++)
  {
    ret.append (buff[i]);
  }
  return SV_UCS4(ret);
}

/**
 * @return the size of the decomposition buffer
 */
unsigned int
SGlyph::size() const
{
  if (shared->shaped)
  {
    /* For shaped we have an offset */
    return shared->ucs4v.size()-4;
  }
  if (shared->cluster!=0)
  {
    return (unsigned int) shared->cluster;
  }
  return shared->ucs4v.size();
}


/**
 * Return the decomposition array of the glyph.
 * If the glyph is a shaped glyph it still returns the
 * correct decomposition - if any.
 */
const SS_UCS4*
SGlyph::getDecompArray() const
{
  if (shared->shaped)
  {
    /* For shaped we have an offset */
    return &shared->ucs4v.array()[4];
  }
  return shared->ucs4v.array();
}

/**
 * Return the shape array of the glyph.
 * If the glyph is not shaped returned 0
 * the array has a size of foru and it contains the
 * isolated, initial, medial, final forms
 * If a form iz 0 it is not defined.
 */
const SS_UCS4*
SGlyph::getShapeArray() const
{
  if (shared->shaped)
  {
    /* For shaped we have an offset */
    return shared->ucs4v.array();
  }
  return 0;
}

/**
 * return the decomposed character at a certain place.
 */
SS_UCS4
SGlyph::operator[] (unsigned int index) const
{
  if (shared->shaped)
  {
    /* For shaped we have an offset */
    return shared->ucs4v.array()[4 + index];
  }
  return shared->ucs4v.array()[index];
}


/*------------------------------------------------------------------------------
 *                    SGlyph::is... 
 *------------------------------------------------------------------------------
 */

/*
 * check if the character has to be mirrored in RTL
 */

bool
SGlyph::isMirrorable() const
{
  return (shared->mirror != 0);
}

/**
 * check if this character is special - lineend or tab
 * that requires special rendering
 */
bool
SGlyph::isSpecial() const
{
  return (shared->lineend || shared->tab || getFirstChar() < 0x20);
}

/*
 * Get character type.
 */
unsigned int
SGlyph::getType()  const
{
  return (unsigned int) ((unsigned char)shared->type);
}

/**
 * Check if this is a white space.
 */
bool
SGlyph::isWhiteSpace() const
{
  if (isSpecial ()) return true;
  switch (getFirstChar())
  {
  case 0x20:
  case 0x3000:
    return true;
  }
  return false;
}

/**
 * Check if this is a transparent character.
 */
bool
SGlyph::isTransparent() const
{
  // From Miikka-Markus Alhonen:
  // "T = Mn + Cf - ZWNJ - ZWJ" (ArabicShaping-4.txt of UCD 3.1.1)
  // This means that every character belonging to character classes 
  // Mn (04) or Cf (0E) except for ZWNJ U+200C and ZWJ U+200D - and 
  // nothing else - is transparent.
  // So, even all the characters in Combining Diacritical Marks 
  // U+0300 - U+036F, Hebrew vowels U+0591 - U+05BD, Syriac vowels 
  // U+0730 - U+074A etc. are transparent characters, not just the 
  // Arabic tashkeel
  //return (gcategory.encode (u4) == 0x4 && ...); // Mark, Non-Spacing
  if (shared->type == 0x4 || shared->type == 0x6 
     || shared->type == 0xe)
  {
     SS_UCS4 u4 = getFirstChar();;
     return  (u4 != 0x200C && u4 != 0x200D);
  }
  return false;
}


bool
SGlyph::isLineEnd() const
{
  return shared->lineend;
}

bool
SGlyph::isTab() const
{
  return shared->tab;
}

bool
SGlyph::isCluster() const
{
  return shared->cluster != 0;
}

/**
 * We use private area to define our own ligatures.
 */
bool
SGlyph::isYuditLigature() const
{
  if (size()==0) return 0;
  /* We might above ligature for shaping */
  SS_UCS4 ch = getShapedChar();
  return isLigature (ch);
/*
  if (ch >= 0x80000000 && ch < 0xA0000000)
  {
    return true;
  }
  return false;
*/
}

/**
 * We use private area to define our own precomposed characters.
 */
bool
SGlyph::isYuditComposition() const
{
  if (size()==0) return 0;
  /* We might above ligature for shaping */
  SS_UCS4 ch = getShapedChar();
  if (ch>= 0xA0000000)
  {
    return true;
  }
  return false;
}

/**
 * Construct a key from characters inside the glyph
 * @return the key. the key does not contain attributes, 
 * it only has glyph info.
 */
SString
SGlyph::charKey() const
{
 
  if (size() == 0)
  {
    SS_UCS4 chr = getChar();
    return SString ((char*) &chr, sizeof (SS_UCS4));
  }
  return SString ((char*) shared->ucs4v.array(), 
         shared->ucs4v.size() * sizeof (SS_UCS4));
}

/*------------------------------------------------------------------------------
 *                    SGlyph::comparison
 *------------------------------------------------------------------------------
 */
bool
SGlyph::operator == (const SGlyph& g2) const
{
  if (g2.size() != size()) return false;
  if (getChar() != 0 && getChar() == g2.getChar()) return true;
  if (size() == 0)
  {
     return (getChar() == g2.getChar());
  }

  /* percfect match */
  for (unsigned int i=0; i<shared->ucs4v.size(); i++)
  {
     if (shared->ucs4v[i] != g2.shared->ucs4v[i]) return false;
  }
  return true;
}

bool
SGlyph::operator != (const SGlyph& g2) const
{
  if (g2.size() != size()) return true;
  if (getChar() != 0 && getChar() != g2.getChar()) return true;
  if (size() == 0)
  {
     return (getChar() != g2.getChar());
  }

  for (unsigned int i=0; i<shared->ucs4v.size(); i++)
  {
     if (shared->ucs4v[i] != g2.shared->ucs4v[i]) return true;
  }
  return false;
}

bool
SGlyph::operator == (const SType& _type) const
{
  /* take care of CRLF - ugly but it works */
  return (SS_UCS4) _type == getFirstChar();
}

bool
SGlyph::operator != (const SType& _type) const
{
  /* take care of CRLF - ugly but it works */
  return (SS_UCS4) _type != getFirstChar();
}

/*------------------------------------------------------------------------------
 *  SGlyph: Copy constructor, assign operator, SObject clone, destructor
 *------------------------------------------------------------------------------
 */
/**
 * Create a glyph from another glyph
 * @param glyph is the other glyph.
 */
SGlyph::SGlyph (const SGlyph& glyph)
{
  lr = glyph.lr;
  underlined = glyph.underlined;
  selected = glyph.selected;
  useComposition = glyph.useComposition;
  currentShape = glyph.currentShape;
  shared = glyph.shared;
}

/**
 * Nothing to desctruct now.
 */
SGlyph::~SGlyph ()
{
}

/**
 * Assign a glyph.
 * @param glyph is the other glyph.
 */
SGlyph
SGlyph::operator=(const SGlyph& glyph)
{
  lr = glyph.lr;
  underlined = glyph.underlined;
  selected = glyph.selected;
  useComposition = glyph.useComposition;

  shared = glyph.shared;
  return *this;
}

/**
 * All objects are to define this.
 */
SObject*
SGlyph::clone() const
{
  return new SGlyph (*this);
}

/*------------------------------------------------------------------------------
 *                           STextData
 *------------------------------------------------------------------------------
 */

/**
 * Create and empty STextData 
 */
STextData::STextData (void)
{
}

/**
 * Create a text data from utf8.
 * @param utf8 is the input text
 */
STextData::STextData (const SString& utf8)
{
  insert (utf8);
}

/**
 * clear the data and assign new data
 */
STextData
STextData::operator = (const STextData & data)
{
  clear ();
  insert (data);
  return *this;
}

/**
 * Glyph Lines are pointers
 */
STextData::~STextData ()
{
  for (unsigned int i=0; i<lines.size(); i++)
  {
     delete lines[i];
  }
}

/**
 * Get the contained text
 */
SString
STextData::getText () const
{
  if (size()==0) return SString();
  return getText (STextIndex (0, 0), STextIndex (size(), size(size()))); 
}

/**
 * Get the text between the cursor and here
 */
SString
STextData::getText (const STextIndex& index) const
{
  return getText (textIndex, index); 
}

/**
 * Extract a bi-di marked utf-8 text.
 * @param begin is the starting index
 * @param end is the ending index
 */
SString
STextData::getText (const STextIndex& begin, const STextIndex& end) const
{
  SString ret;
  if (size() == 0) return SString(ret);

  STextIndex start;
  STextIndex stop;
  if (end > begin)
  {
    start = begin;
    stop = end;
  }
  else
  {
    start = end;
    stop = begin;
  }
  if (stop.line > size())
  {
     stop.line = size(); stop.index = size(size()-1);
  }
  for (unsigned int i=start.line; i<=stop.line && i<size(); i++)
  {
     unsigned int b = (i==start.line) ? start.index : 0;
     unsigned int e = (i==stop.line)? stop.index : size (i);
     ret.append (getText (i, b, e));
  }
  return SString(ret);
}

/**
 * Extract a bi-di marked utf-8 text.
 * TODO: implements a better bidi.
 * @param beginLine is the starting line
 * @param beginIndex is the starting position
 * @param endIndex is the ending position
 */
SString
STextData::getText (unsigned int beginLine, unsigned int beginIndex,
   unsigned int endIndex) const
{
  SV_UCS4 ucs;

  for (unsigned int i=beginIndex; i<endIndex; i++)
  {
     if (glyphAt(STextIndex (beginLine, i)).lr)
     {
          ucs.append (glyphAt(STextIndex (beginLine, i)).getChars());
     }
     else
     {
        unsigned int start = i;
        while (i<endIndex && !glyphAt(STextIndex (beginLine, i)).lr)
        {
           i++;
        }
        unsigned int end = i;
        i--; /* We will increment it later in the external loop */
        ucs.append (SGlyph::RLO);
        while (end>start)
        {
          --end;
          ucs.append (glyphAt(STextIndex (beginLine, end)).getChars());
        }
        ucs.append (SGlyph::PDF);
     }
  }
  SEncoder ic;
  return (SString(ic.encode (ucs)));
}

/**
 * find string in text data from current position.
 * return an index to the end of that data.
 * move the current position to the beginning of the data.
 * it does not search across lines.
 */
STextIndex
STextData::find (const SString& string)
{
  STextData d (string);

  /**
   * go through all lines.
   */
  for (unsigned int i=textIndex.line; i<size(); i++)
  {
     unsigned int begin = (i==textIndex.line) ? textIndex.index : 0;
     unsigned int end = size (i);

     SGlyphLine* line = lines[i];
     /**
      * go through all glyphs
      */
     for (unsigned int j=begin; j+d.size(0)<=end; j++)
     {
        /* hack. We dont disclose the SGlyphLIne= */
        bool found = true;
        for (unsigned int k=0; k<d.size(0); k++)
        {
          const SGlyph* g = line->peek(j+k);
          const SGlyph& g2 = d.glyphAt (STextIndex(0, k));
          if (g2 != *g)
          {
            found = false;
          }
        }
        if (found)
        {
          move (STextIndex(i, j));
          return STextIndex (i, j+d.size(0));
        }
     }
  }
  return STextIndex (0,0);
}

/**
 * Get a glyph at a certain position
 * @param index is the glyph index
 * @return a reference to the glyph.
 */
const SGlyph&
STextData::glyphAt (const STextIndex& index) const
{
  return (*lines[index.line])[index.index];
}

/**
 * move the insertion point.
 * @param line is the line number.
 * @param index is the line index.
 */
void
STextData::move (const STextIndex& index)
{
  STextIndex ndx (index);
  if (ndx.line > size()) ndx.line = size();
  if (ndx.line > 0 && !isProperLine (ndx.line-1))
  {
     ndx.line--;
     ndx.index = size(ndx.line);
  }
  if (ndx.line == size())
  {
    ndx.index = 0;
  }
  else if (ndx.index >= size(ndx.line))
  {
    ndx.index = size(ndx.line);
    if (isProperLine (ndx.line) && ndx.index == size(ndx.line))
    {
      ndx.index = ndx.index-1;
    }
  }
  textIndex = ndx;
}

/**
 * set the text data
 */
void
STextData::setText (const SString& string)
{
  SEncoder ic;
  SV_UCS4 ucs4 = ic.decode (string, false);
  setText (ucs4);
}

/**
 * set the text data
 */
void
STextData::setText (const SV_UCS4& ucs4)
{
  clear ();
  /* split into lines */
  unsigned int start = 0;
  unsigned int end = 0;
  do
  {
    SGlyphLine* gl = new SGlyphLine();
    start = end;

    /* split the lines and do stoolkit composition */
    end = split (ucs4, gl, start);
    /* put bidi into OUR visual order */
    bidiToVisual (gl);
    reShapeSegment (gl, 0, 0);
    if (gl->size()) 
    {
      lines.append (gl);
    }
    else
    {
      delete gl;
    }
  } while (start != end);
  STextDataEvent nevt (STextIndex(0,0));
  event.add (nevt);
}

/**
 * insert one signle glyph and move the cursor to the right
 * @param glyph is the glyph to insert
 * @return an index to end, text between current cursor and end
 * has been inserted.
 */
void
STextData::insert (const SGlyph& glyph, STextIndex* ret)
{
  STextIndex oldIndex = textIndex;

  if (size() <= textIndex.line)
  {
     lines.append (new SGlyphLine());
  }

  SGlyphLine* line = lines[textIndex.line];
  line->insert (textIndex.index, glyph);
  if (ret->line == textIndex.line)
  {
    ret->index++;
  }
  textIndex.index = textIndex.index+1;

  STextDataEvent nevt (oldIndex);
  /* there are two cases. we opened a new line or not */
  if (glyph.isLineEnd())
  {
    /* cut line in two */
    if (line->size() > textIndex.index)
    {
      SGlyphLine *newLine = new SGlyphLine (*line);
      newLine->remove (0, textIndex.index);
      line->truncate (textIndex.index);
      lines.insert (textIndex.line+1, newLine);

      textIndex.index = newLine->size()-1;
      textIndex.line = textIndex.line + 1;
      setMaxLimits (&nevt, textIndex);
      textIndex.index = 0;
      ret->line++;
      if (ret->line== textIndex.line) ret->index=0;
    }
    else /* new line */
    {
      textIndex.index = 0;
      textIndex.line = textIndex.line + 1;
      setMaxLimits (&nevt, textIndex);
      ret->line++;
      ret->index=0;
      if (ret->line== textIndex.line) ret->index=0;
    }
  }
  else
  {
    setMaxLimits (&nevt, textIndex);
  }
  event.add (nevt);
}

/**
 * Insert a text from data
 * @param data is the STextData
 */
void
STextData::insert (const STextData& data)
{
  STextIndex ind;
  for (unsigned int i=0; i<data.size(); i++)
  {
    for (unsigned int j=0; j<data.size(i); j++)
    {
      insert (data.glyphAt (STextIndex (i, j)), &ind);
    }
  }
}

/**
 * Insert a text from utf8. Do fuzzy algoritm to determine
 * direction and all kinds of fuzzy properties, like composition.
 * The resulting text is added to the textbuffer in screen-order.
 * FIXME: 
 * - Implement a bidi algoritm. The current version treats everything
 *   as strong left-to-right character unless it is enclosed in  RLO-PDF.
 * - Get a table of all composing characters and break down the glyphs so
 *   that one glyph contains the base character and the compositng marks.
 * @param utf8 an utf8 encoded text.
 * @return an index to end, text between current cursor and end
 * has been inserted. 
 */
STextIndex
STextData::insert (const SString& utf8)
{
  SEncoder ic;
  SV_UCS4 ucs4 = ic.decode (utf8, false);
  /* split into lines */
  unsigned int start = 0;
  unsigned int end = 0;
  STextIndex old=textIndex;
  unsigned int i;
  do
  {
    SGlyphLine gl;
    start = end;

    /* split the lines and do stoolkit composition */
    end = split (ucs4, &gl, start);
    /* put bidi into OUR visual order */
    bidiToVisual (&gl);
    reShapeSegment (&gl);
    for (i=0; i<gl.size(); i++)
    {
      insert (gl[i], &old);
    }
  } while (start != end);
  /* shape the begining and ending position */
  STextIndex vold = getTextIndex (old, -1); 
  reShape (vold);
  /* before */
  STextIndex op = getTextIndex (vold, -1);
  if (vold != op) reShape (op);

  /* after */
  STextIndex oa = getTextIndex (vold, 1);
  if (vold != oa) reShape (oa);

  reShape (textIndex);
  /* before */
  STextIndex tp = getTextIndex (textIndex, -1);
  if (textIndex != tp) reShape (tp);

  /* before */
  STextIndex ta = getTextIndex (textIndex, 1);
  if (textIndex != ta) reShape (ta);

  return STextIndex (old);
}

/**
 * remove glyphs between current and line and size.
 * current can be less or greater.
 * @param index is the ending high end is non-exclusive
 */
void
STextData::remove (const STextIndex& index)
{
  if (size() == 0) return;

  STextIndex newIndex = reorder (index);
  unsigned int i;
  bool join = false;
  for (i=textIndex.line; i<=newIndex.line && i<size(); i++)
  {
     unsigned int begin = (i==textIndex.line) ? textIndex.index : 0;
     unsigned int end = (i==newIndex.line)? newIndex.index : size (i);

     SGlyphLine* line = lines[i];
     if (begin == 0 && end == size(i))
     {
        if (isProperLine(i)) join = true;
        line->clear();
     }
     else for (unsigned int j=begin; j<end; j++)
     {
        /* hack. We dont disclose the SGlyphLIne= */
        if (j+1==end)
        {
          SGlyph g = glyphAt (STextIndex(i, begin));
          if (g.isLineEnd()) join = true;
        }
        line->remove (begin);
     }
  }
  unsigned int lr = 0;

  /* Remove zeroes */
  for (i=textIndex.line+1; i<=newIndex.line && i<size() + lr; i++)
  {
     SGlyphLine* line = lines[i-lr];
     if (line->size() == 0)
     {
       lines.remove (i-lr);
       delete line;
       lr++;
     }
  }

  /* now we have everything collapsed. */

  STextDataEvent nevt (textIndex);
  setMaxLimits (&nevt, textIndex);
  event.add (nevt);

  /* a joining line was created */
  if (join)
  {
    /* wanted, but impossible */
    if (textIndex.line+1 == size())
    {
      SGlyphLine* line = lines[textIndex.line];
      if (line->size()==0)
      {
        lines.remove (textIndex.line);
        delete line;
      }
      return;
    }
    SGlyphLine* gl = lines[textIndex.line+1];
    lines.remove (textIndex.line+1);

    STextIndex sv = textIndex;
    STextIndex dm = textIndex;
    for (i=0; i<gl->size(); i++)
    {
      insert ((*gl)[i], &dm);
    }
    delete gl;
    move (sv);
  }
  else
  { /* we may not removed this */
    SGlyphLine* line = lines[textIndex.line];
    if (line->size()==0)
    {
      lines.remove (textIndex.line);
      delete line;
    }
  }
  reShape (textIndex);
  /* before */
  STextIndex tp = getTextIndex (textIndex, -1);
  if (textIndex != tp) reShape (tp);

  /* before */
  STextIndex ta = getTextIndex (textIndex, 1);
  if (textIndex != ta) reShape (ta);
}

/**
 * select a region of text and move cursor to it.
 * @param index is the new index. high end is non-exclusive
 * @param is is true if select, unselect otherwise. 
 */
void
STextData::select (const STextIndex& index, bool is)
{
  if (size() == 0) return;

  STextIndex newIndex = reorder (index);
  bool reordered = (newIndex != index);


  if (newIndex.line > size())
  {
     newIndex.line = size(); newIndex.index = size(size()-1);
  }
  for (unsigned int i=textIndex.line; i<=newIndex.line && i<size(); i++)
  {
     unsigned int begin = (i==textIndex.line) ? textIndex.index : 0;
     unsigned int end = (i==newIndex.line)? newIndex.index : size (i);

     SGlyphLine* line = lines[i];
     for (unsigned int j=begin; j<end; j++)
     {
        /* hack. We dont disclose the SGlyphLIne= */
        SGlyph* g = (SGlyph*) line->peek(j);
        g->selected = is;
     }
  }
  if (reordered)
  {
    STextDataEvent nevt (textIndex, true);
    setMaxLimits (&nevt, newIndex);
    event.add (nevt);
  }
  else
  {
    STextDataEvent nevt (textIndex, true);
    setMaxLimits (&nevt, newIndex);
    event.add (nevt);
    textIndex =  newIndex;
  }
}

bool
STextData::setLineType (const SString& str)
{
  if (str==SS_LB_UNIX) return setLineType (SGlyph::ELF);
  if (str==SS_LB_MAC) return setLineType (SGlyph::ECR);
  if (str==SS_LB_DOS) return setLineType (SGlyph::ECRLF);
  if (str==SS_LB_LS) return setLineType (SGlyph::ELS);
  if (str==SS_LB_PS) return setLineType (SGlyph::EPS);
  return false;
}

/**
 * Set the line type for the whole data structure.
 * @param type will be forced after each newline. 
 * @return true if anything changed.
 */
bool
STextData::setLineType (SGlyph::SLineEnd type)
{
  SV_UCS4 v;
  switch (type)
  {
  case SGlyph::ELF:
    v.append ((SS_UCS2) SGlyph::LF);
    return setLineType (SGlyph(0x0d, v, (SS_UCS4) SGlyph::LF, 0, false, 0));
  case SGlyph::ECR:
    v.append ((SS_UCS2) SGlyph::CR);
    return setLineType (SGlyph(0x0d, v, (SS_UCS4) SGlyph::CR, 0, false, 0));
  case SGlyph::ECRLF:
    v.append ((SS_UCS2) SGlyph::CR);
    v.append ((SS_UCS2) SGlyph::LF);
    return setLineType (SGlyph(0x0d, v, 0, 0, false, 0));
  case SGlyph::ELS:
    v.append ((SS_UCS2) SGlyph::LS);
    return setLineType (SGlyph(0x0b, v, (SS_UCS4) SGlyph::LS, 0, false, 0));
  case SGlyph::EPS:
    v.append ((SS_UCS2) SGlyph::PS);
    return setLineType (SGlyph(0x0c, v, (SS_UCS4) SGlyph::PS, 0, false, 0));
  }
  return false;
}
/**
 * Set the line type for the whole data structure.
 * @param type will be forced after each newline. 
 * @return true if anything changed.
 */
bool
STextData::setLineType (const SGlyph& _glyph)
{
  bool retvle = false;
  SGlyph glyph = _glyph;
  glyph.lr = true;
  for (unsigned int i=0; i<lines.size(); i++)
  {
    SGlyphLine* gl = lines[i];
    if (gl==0 || gl->size()==0) continue;
    const SGlyph& g = *gl->peek(gl->size()-1);
    if (!g.isLineEnd())
    {
      continue;
    }
    if (glyph == g)
    {
       continue;
    }
    gl->replace (gl->size()-1, glyph);
    retvle = true;
  }
  return retvle;
}

/**
 * select a region of text and move cursor to it.
 * @param index is the new index - high end is non-exclusive
 * @param is is true if underline, un-underline otherwise.
 */
void
STextData::underline (const STextIndex& index, bool is)
{
  if (size() == 0) return;

  STextIndex newIndex = reorder (index);
  bool reordered = (newIndex != index);

  for (unsigned int i=textIndex.line; i<=newIndex.line && i<size(); i++)
  {
     unsigned int begin = (i==textIndex.line) ? textIndex.index : 0;
     unsigned int end = (i==newIndex.line)? newIndex.index : size (i);

     SGlyphLine* line = lines[i];
     for (unsigned int j=begin; j<end; j++)
     {
        /* hack. We dont disclose the SGlyphLIne= */
        SGlyph* g = (SGlyph*) line->peek(j);
        g->underlined = is;
     }
  }
  if (reordered)
  {
    STextDataEvent nevt (textIndex, true);
    setMaxLimits (&nevt, newIndex);
    event.add (nevt);
  }
  else
  {
    STextDataEvent nevt (textIndex, true);
    setMaxLimits (&nevt, newIndex);
    event.add (nevt);
    textIndex =  newIndex;
  }
}

/**
 * Clear the text
 */
void
STextData::clear ()
{
  for (unsigned int i=0; i<lines.size(); i++)
  {
     delete lines[i];
  }
  lines.clear();
  textIndex = STextIndex (0, 0);
  STextDataEvent nevt (STextIndex (0, 0));
  nevt.remaining = STextIndex (0, 0);
  nevt.valid = true;
  event.add (nevt);
}

void
STextData::fireEvent ()
{
  if (!event.valid || listener == 0) return ;
  listener->textChanged (this, event); 
  event.clear();
}

void
STextData::clearEvent ()
{
  event.clear();
}

/**
 * return the number of lines in text
 */
unsigned int
STextData::size() const
{
  return lines.size();
}

/**
 * return the number of lines in line
 */
unsigned int
STextData::size(unsigned int line) const
{
  if (line >= size()) return 0;
  const SGlyphLine* l = lines[line];
  return l->size();
}

/**
 * Get the text index relative to offset.
 * @param offset is character offset to textIndex.
 */
STextIndex
STextData::getTextIndex(int charOffset) const
{
  return getTextIndex (textIndex, charOffset);
}
/**
 * Get the text index relative to offset.
 * @param offset is character offset to textIndex.
 */
STextIndex
STextData::getTextIndex(const STextIndex& base, int charOffset) const
{
  STextIndex ret =  base;
  if (charOffset > 0)
  {
    for (int i=0; i<charOffset; i++)
    {
      if (ret == STextIndex (size(), 0)) break;
      if (size(ret.line) <= ret.index+1)
      {
        if (!isProperLine (ret.line))
        {
          if (size(ret.line) == ret.index+1)
          {
             ret.index = ret.index + 1;
          }
          break;
        }
        ret.index = 0;
        ret.line = ret.line + 1;
        continue;
      }
      ret.index = ret.index + 1;
      continue;
    }
  }
  else if (charOffset < 0)
  {
    for (int i=0; i>charOffset; i--)
    {
      if (ret == STextIndex (0, 0)) break;
      if (ret.index > 0)
      {
        ret.index = ret.index - 1;
        continue;
      }
      ret.line = ret.line-1;
      /*
      * For proper lines,
      * The beginning of the line is the same location as the end. 
      */
      ret.index = size(ret.line);
      if (isProperLine (ret.line))
      {
        ret.index = ret.index-1;
      }
    }
     
  }
  return STextIndex (ret);
}

/**
 * Check if line ends with newline glyph
 */
bool
STextData::isProperLine (unsigned int line) const
{
  if (line >= size()) return false;
  if (size(line)==0) return false;
  const SGlyph g = glyphAt(STextIndex(line, size(line)-1));
  return g.isLineEnd();
}

/**
 * add data listener.
 * TODO: now it only sets it.
 */
void
STextData::addTextDataListener (STextDataListener* _listener)
{
  listener = _listener;
}

/**
 * Set the maximum limits.
 * @param evt -> remainingLine and remainingPosition will be set
 * @param index is the one that needs to be converted to remaining index
 */
void
STextData::setMaxLimits (STextDataEvent* evt, const STextIndex& index)
{
  evt->valid = true;
  unsigned int lsize = size();
  if (index.line >= lsize)
  {
     evt->setRemaining (STextIndex (0,0));
     return;
  }
  unsigned int rem  = lsize - index.line - 1;
  unsigned int cindex = size(index.line);
  evt->setRemaining (STextIndex (rem, (index.index>=cindex) ? 0 : cindex - index.index));
}

/**
 * Convert remainin-line to maxline
 */
STextIndex
STextData::getMaxTextIndex (const STextDataEvent& evt) const
{
  if (!evt.valid) return STextIndex (0,0);
  unsigned int lsize = size();
  if (lsize <= evt.remaining.line)
  {
     return STextIndex (lsize,0);
  }
  unsigned int line = lsize - evt.remaining.line -1;
  if (size(line) < evt.remaining.index)
  {
     return STextIndex (line, size(line));
  }
  return STextIndex (line, size(line)+1-evt.remaining.index);
}

/**
 * Put the stuff in order, so that it will always be in increasing order.
 * @param end is the desired end
 */
STextIndex
STextData::reorder (const STextIndex& index)
{
  if (index > textIndex)
  {
     return STextIndex (index);
  }
  STextIndex tmp = textIndex;
  textIndex = index;
  return STextIndex (tmp);
}  

/**
 * return true if character is a target for select.
 */
bool
STextData::isWhiteSpace (const STextIndex& index) const
{
  return glyphAt (index).isWhiteSpace();
}

/**
 * Re-shape current glyph.
 * The full line is taken!
 * @param index is the index around which we re-shape.
 */
void
STextData::reShapeOne(const STextIndex& index)
{
  /* current */
  if (index.line < size() && index.index < size (index.line))
  {
    SGlyphLine *line = lines[index.line];; 

    /* full line before=0, after=0*/
    const SGlyph* before = 0;
    const SGlyph* after = 0;
    /* This is the insertion point! */
    int cbefore = getNonTransparentLeft (line, index.index);
    int cafter = getNonTransparentRight (line, index.index);
    if (cbefore >= 0) before = line->peek((unsigned int)cbefore);
    if (cafter >= 0) after = line->peek((unsigned int)cafter);
    SGlyph* g = (SGlyph*) line->peek(index.index);
    char shape = getShape (*g, before, after);

    // Hack for speed. This is a constant in reality ...
    if (g->currentShape != shape)
    {
       g->currentShape = shape;
       /* this glyph changed shape */
       STextDataEvent nevt (index, true);
       setMaxLimits (&nevt, index);
       event.add (nevt);
    }
  }
}

/**
 * Re-shape current glyph.
 * The full line is taken!
 * @param index is the index around which we re-shape.
 */
void
STextData::reShape(const STextIndex& index)
{
  /* current */
  if (index.line < size() && index.index < size (index.line))
  {
    SGlyphLine *line = lines[index.line];; 

    /* full line before=0, after=0*/
    const SGlyph* before = 0;
    const SGlyph* after = 0;
    int cbefore = getNonTransparentLeft (line, index.index);
    int cafter = getNonTransparentRight (line, index.index);
    if (cbefore >= 0) before = line->peek((unsigned int)cbefore);
    if (cafter >= 0) after = line->peek((unsigned int)cafter);
    SGlyph* g = (SGlyph*) line->peek(index.index);
    char shape = getShape (*g, before, after);

    // Hack for speed. This is a constant in reality ...
    if (g->currentShape != shape)
    {
       g->currentShape = shape;
       /* this glyph changed shape */
       STextDataEvent nevt (index, true);
       setMaxLimits (&nevt, index);
       event.add (nevt);
    }
    /* do it after and before to skip spaces */
    if (cbefore>=0)
    {
       STextIndex ib = index;
       ib.index = (unsigned int) cbefore;
       reShapeOne (ib);
    }
    if (cafter>=0)
    {
       STextIndex ia = index;
       ia.index = (unsigned int) cafter;
       reShapeOne (ia);
    }
  }
}

/**
 * Get the shape at the current position
 * Please note that it works in visual order!
 * @return 
 * <ul> 
 *  <li> -1  no shape </li>
 *  <li> 0   isolated </li>
 *  <li> 1   initial (space after-rl)</li>
 *  <li> 2   medial </li>
 *  <li> 3   final (space before-rl)</li>
 * </ul>
 * @param gbefore is the glyph before this line, transparent chars skipped
 * @param gafter is the glyph before this line, transparent chars skipped
 */
char
STextData::getShape(const SGlyph& glyph, 
    const SGlyph* gbefore,
    const SGlyph* gafter)
{
  static SS_UCS4 initials[4] = {0x0, 0x0, 0x0, 0x0};
  static SS_UCS4 dualjoining[4] = {1, 1, 1, 1};

  /* is it a shapeable one ? */
  const SS_UCS4* now = glyph.getShapeArray();
  if (now == 0)
  {
    return -1;
  }
  const SS_UCS4* before = initials;
  const SS_UCS4* after = initials;
  if (gbefore) 
  {
    before=gbefore->getShapeArray();
    if (before==0) before=initials;
    /* tatweel and ZWJ are dual joining */
    if (gbefore->getChar() == 0x0640 ||
	gbefore->getChar() == 0x200d) before = dualjoining;
  }
  if (gafter) 
  {
    after=gafter->getShapeArray();
    if (after==0) after=initials;
    /* tatweel and ZWJ are dual joining */
    if (gafter->getChar() == 0x0640 ||
	gafter->getChar() == 0x200d) after = dualjoining;
  }

  /* Make it all rl for simplicity */
  if (!glyph.lr)
  {
    const SS_UCS4* tmp = before;
    before = after;
    after = tmp;
  }
#define USE_MY_OLD_HACK 1
#if USE_MY_OLD_HACK
  if ((before[(unsigned int)SGlyph::INITIAL])
     && now[(unsigned int)SGlyph::MEDIAL]
     && (after[(unsigned int)SGlyph::FINAL]))
  {
    return (char)SGlyph::MEDIAL;
  }
  if (after[(unsigned int)SGlyph::FINAL]
    && now[(unsigned int)SGlyph::INITIAL])
  {
    return (char)SGlyph::INITIAL;
  }
  if (before[(unsigned int)SGlyph::INITIAL]
     && now[(unsigned int)SGlyph::FINAL])
  {
    return (char)SGlyph::FINAL;
  }
  if (now[(unsigned int)SGlyph::ISOLATED])
  {
    return (char)SGlyph::ISOLATED;
  }
#else /* USE_MY_OLD_HACK */

  /**
   * Trying to implement Miikka-Markus Alhonen's stuff.
     1) combine the transparent characters with their base characters, 
        so that they won't affect the following step in any way
     2) join the letters to each other
     3) form the ligatures taking into consideration the contextual 
        joining forms of the first and the last letter of the ligature 
        according to the following rules:
     -If the first letter of the ligature is initial and the last medial, the
      outcome should be considered as initial; so to speak, only an 
      initial ligature could be used in this case, not an isolated one for 
      instance.
     -first = initial, last = final, outcome = isolated
     -first = medial, last = medial, outcome = medial
     -first = medial, last = final, outcome = final
     -first = isolated, last = final, outcome = isolated (e.g. U+FDF2)
     -first = isolated, last = isolated, outcome = isolated (e.g. U+FDF6)
    */
  if ((before[(unsigned int)SGlyph::INITIAL])
     && now[(unsigned int)SGlyph::INITIAL]
     && (after[(unsigned int)SGlyph::MEDIAL]))
  {
    return (char)SGlyph::INITIAL;
  }
  //   -first = initial, last = final, outcome = isolated
  if ((before[(unsigned int)SGlyph::INITIAL])
     && now[(unsigned int)SGlyph::ISOLATED]
     && (after[(unsigned int)SGlyph::FINAL]))
  {
    return (char)SGlyph::ISOLATED;
  }
   //-first = medial, last = medial, outcome = medial
  if ((before[(unsigned int)SGlyph::MEDIAL])
     && now[(unsigned int)SGlyph::MEDIAL]
     && (after[(unsigned int)SGlyph::MEDIAL]))
  {
    return (char)SGlyph::MEDIAL;
  }
  //   -first = medial, last = final, outcome = final
  if ((before[(unsigned int)SGlyph::MEDIAL])
     && now[(unsigned int)SGlyph::FINAL]
     && (after[(unsigned int)SGlyph::FINAL]))
  {
    return (char)SGlyph::FINAL;
  }
  //   -first = isolated, last = final, outcome = isolated (e.g. U+FDF2)
  if ((before[(unsigned int)SGlyph::ISOLATED])
     && now[(unsigned int)SGlyph::ISOLATED]
     && (after[(unsigned int)SGlyph::FINAL]))
  {
    return (char)SGlyph::ISOLATED;
  }
  //   -first = isolated, last = isolated, outcome = isolated (e.g. U+FDF6)
  if ((before[(unsigned int)SGlyph::ISOLATED])
     && now[(unsigned int)SGlyph::ISOLATED]
     && (after[(unsigned int)SGlyph::ISOLATED]))
  {
    return (char)SGlyph::ISOLATED;
  }
#endif /* USE_MY_OLD_HACK */
  /* fallback */
  return -1;
}

/**
 * Split a text into lines
 * @param ucs4 is a text that can contain newline chars.
 * @param from is the starting index.
 * @parem gl is the return glyphline.
 * @return ending index, that is equal to starting index 
 * if ther is no more data.
 */
unsigned int
split (const SV_UCS4& ucs4, SGlyphLine* gl, unsigned int from)
{
  SUniMap composer ("precompose");
  SUniMap category ("category");
  SUniMap shaper ("shape");
  SUniMap mirroring ("mirroring");

  gl->clear();
  unsigned int i=from;
  SS_UCS4 composition;
  SS_UCS4 mirror;
  bool    useComposed;
  bool    isShaped;
  bool    isLigature;
  //gl->ensure (ucs4.size());
  SGlyphShared* shr = 0;
  while (i<ucs4.size())
  {
    char chartype = 0;
    unsigned int n =0;
    SV_UCS4 ret;
    composition = 0;
    mirror = 0;
    useComposed = false;
    isShaped = false;
    isLigature = false;
    unsigned int clusterIndex = 0;
    SV_UCS4 du4;
    shr = 0;
    switch (ucs4[i])
    {
    case SGlyph::CR:
      du4.append (ucs4[i++]);
      if (i<ucs4.size() && ucs4[i] == SGlyph::LF)
      {
        du4.append (ucs4[i++]);
      }
      break;
    case SGlyph::LF:
      du4.append (ucs4[i++]);
      if (i<ucs4.size() && ucs4[i] == SGlyph::CR)
      {
        du4.append (ucs4[i++]);
      }
      break;
    case SGlyph::PS:
    case SGlyph::LS:
    case SGlyph::RLO:
    case SGlyph::LRO:
    case SGlyph::PDF:
      du4.append (ucs4[i++]);
      break ;
    default:
      /* shaping stand alone characters is allowed. */
      if (category.isOK())
      {
        /* max 0x1e */
        chartype = (char) category.encode (ucs4[i]);
      }
      if (mirroring.isOK())
      {
	 mirror = mirroring.encode (ucs4[i]);
      }
      
      if (shaper.isOK())
      {
        /* encode shapes if possible */
        unsigned int n = shaper.lift (ucs4, i, false, &ret);
        /* the composition comes at the end  - if any */
        if (n>=i+1 && ret.size()==4)
        {
           composition =  ucs4[i];
           useComposed = true;
           /* The four variants for shaping */
           du4.append (ret[0]);
           du4.append (ret[1]);
           du4.append (ret[2]);
           du4.append (ret[3]);
           if (n>i+1) // composed
           {
             useComposed = false;
             composition = 0;
             SV_UCS4 retc;
             /* good to have - for cacheing ...*/
             unsigned int nc = composer.lift (ucs4, i, false, &retc);
             if (nc > i+1 && retc.size() == 1)
             {
                composition = retc[0];
             }
             /* append the extra bits at the end. */
             for (unsigned int j=i; j<n; j++)
             {
               du4.append (ucs4[j]);
             }
           }
           isShaped = true;
           i = n;
           break;
        }
        ret.clear();
      }
      /* check for clusters */
      if ((n = getCluster (ucs4, i, &ret)) > i
          && ret.size() > 0 && ret[0] < 0x80000000)
      {
         unsigned int j;
         for (j=0; j<ret.size(); j++)
         {
           if (ret[j] > 0x80000000 && ret[j] < 0xA0000000)
           {
              composition = ret[j];
              /* Save this unicode ligature. Fonts have unicode order.  */
              putLigatureCluster (composition, du4.array(), du4.size());
              putLigatureUnicode (composition, &ucs4.array()[i], n-i);
              break;
           }
           du4.append (ret[j]);
         }
         clusterIndex = j;
         while (i<n)
         {
          du4.append (ucs4[i++]);
         }
         ret.clear ();
      }
      else if (composer.isOK () && category.isOK())
      {
        ret.clear();
        n = composer.lift (ucs4, i, false, &ret);
        if (ret.size()==1)
        {
           composition = ret[0];
        }
        if (n==i) /* it was not a composition */
        {
          /* quick get */
          shr = glyphCache.get (SString((long)ucs4[i]));
          if (shr != 0)
          {
            /* removed dependency on composition - glyphcache fixed */
            /*
             * Glyphcache can have precomposed or single chars only. 
             * But these single chars have have some comps...
             */
            if (i+1 >= ucs4.size ())
               // Mark, Non-Spacing, Mark-Enclosing
            {
              /* so next one is non-composing */
              i++;
              useComposed = true;
              break;
            }
            SS_UCS2 u2=category.encode (ucs4[i+1]);
            if (u2 != 0x4 && u2 != 0x6 ) 
            {
              i++;
              useComposed = true;
              break;
            }
            shr = 0;
            du4.append (ucs4[i++]);
            /* check if next is composing, support this level only */
            while (i<ucs4.size())
            {
              u2 = category.encode (ucs4[i]);
              if (u2 == 0x4 ||  u2 == 0x6) // Mark, Non-Spacing, Mark-Enclosing
              {
                 du4.append (ucs4[i++]);
              }
              else
              {
                 break;
              }
            }
            break;
          }
          ret.clear();
          /* check if it can be broken down into a composition */
          unsigned int n = composer.lift (ucs4, i, true, &ret);
          if (ucs4[i] != 0 && n == i+1 
              && ret.size() > 0 && ret.size() <= SD_MAX_COMPOSE)
          {
            useComposed = true;
            n = i+1;
            
            composition = ucs4[i++];
            du4.append (ret);
          }
          else
          {
            du4.append (ucs4[i++]);
            /* check if next is composing, support this level only */
            while (i<ucs4.size())
            {
               SS_UCS2 u2 = category.encode (ucs4[i]);
               if (u2 == 0x4 || u2 == 0x6) // Mark, Non-Spacing
               {
                  du4.append (ucs4[i++]);
               }
               else
               {
                  break;
               }
            }
          }
        }
        else
        {
          while (i<n)
          {
            du4.append (ucs4[i++]);
          }
        }
      }
      else
      {
        du4.append (ucs4[i++]);
      }
      break;
    }
    /* we got this shared glyph. It can only be precomposed or single  */
    if (shr)
    {
      SGlyph g(chartype, shr);
      g.useComposition = (useComposed || g.size()==0);
      gl->append (g);
      if (g.isLineEnd())
      {
        /* line can not start with pdf. */
        if (i<ucs4.size() && ucs4[i]==SGlyph::PDF) i++;
        break;
      }
    }
    else
    {
      SGlyph g (chartype, du4, composition, mirror, isShaped, clusterIndex);
      g.useComposition = (useComposed || du4.size()==1);
      gl->append (g);
      if (g.isLineEnd())
      {
        /* line can not start with pdf. */
        if (i<ucs4.size() && ucs4[i]==SGlyph::PDF) i++;
        break;
      }
    }
  }
  return i;
}

/**
 * put the text into visual order.
 * @param gl is a line of glyphs. 
 * TODO: make a bettter bidi.
 */
void
bidiToVisual (SGlyphLine* gl)
{
  SGlyphLine in = *gl;
  gl->clear();
  SBinVector<bool> stack;
  stack.append (true);
  bool lrnow = true;
  unsigned int rlends=0;

  /*
   * go through the input and construct gl.
   */
  for (unsigned int i=0; i<in.size(); i++)
  {
    if (in[i] == SGlyph::LRO)
    {
       stack.append (true);
       lrnow = true;
    }
    else if (in[i] == SGlyph::RLO)
    {
       stack.append (false);
       lrnow = false;
       for (rlends=i+1; rlends<in.size(); rlends++)
       {
          if (in[rlends] == SGlyph::LRO || in[rlends] == SGlyph::RLO
              || in[rlends] == SGlyph::PDF)
          {
             break;
          }
       }
    }
    else if (in[i] == SGlyph::PDF)
    {
       if (stack.size() > 1)
       {
         lrnow = stack[stack.size()-2];
         stack.truncate (stack.size()-1);
       }
       else
       {
         fprintf (stderr, "STextLine::insert unmatched STextType::PDF\n");
         lrnow = true;
       }
       if (!lrnow) 
       {
         for (rlends=i+1; rlends<in.size(); rlends++)
         {
           if (in[rlends] == SGlyph::LRO || in[rlends] == SGlyph::RLO
                || in[rlends] == SGlyph::PDF)
           {
              break;
           }
         }
       }
    }
    else
    {
       if (lrnow)
       {
         SGlyph g (in[i]);  
         g.lr = lrnow;
         gl->append (g);
       }
       else
       {
         rlends--;
         SGlyph g (in[rlends]);
         g.lr = lrnow;
         gl->append (g);
       }
    }
  }
}

/**
 * This one is supposed to shape a single segment.
 * This one is called with textIndex positioned at insertin point.
 */
void
STextData::reShapeSegment (SGlyphLine* gl)
{
  const SGlyph* before = 0;
  const SGlyph* after = 0;
  STextIndex index = textIndex;
  if (index.line < size())
  {
    SGlyphLine *line = lines[index.line];; 

    /* full line before=0, after=0*/
    int cbefore = getNonTransparentLeft (line, index.index+1);
    int cafter = getNonTransparentRight (line, index.index);
    if (cbefore >= 0) before = line->peek((unsigned int)cbefore);
    if (cafter >= 0) after = line->peek((unsigned int)cafter);
  }
  reShapeSegment (gl, before, after);
}


/**
 * try to shape a single segment.
 * return true if it is possible.
 * @param before is the visual glyph before this line.
 * @param after is the visual glyph after this line.
 * Todo: do glyphline-breaking of shaping is not possible, use
 * lineShaper to get glyph breakdown.
 */
bool
STextData::reShapeSegment (
      SGlyphLine* gl,
      const SGlyph* _before, 
      const SGlyph* _after)
{


  bool ok = true;
  unsigned int gsize = gl->size();
  //const SGlyph* before = (i==0) ? _before :  gl->peek(i-1);
  //const SGlyph* after = (i+1>=gsize) ? _after :  gl->peek(i+1);
  const SGlyph* before = _before;
  const SGlyph* after = _after;
  for (unsigned int i=0; i<gsize; )
  {
    SGlyph* g = (SGlyph*) gl->peek(i);
    if (g->getShapeArray()==0)
    {
       i++;
       continue;
    }
    int cbefore = getNonTransparentLeft (gl, i);
    int cafter = getNonTransparentRight (gl, i);
    before = (cbefore >= 0) ? gl->peek((unsigned int)cbefore) : _before;
    after = (cafter >= 0) ? gl->peek((unsigned int)cafter) : _after;
    char shape = getShape ((*gl)[i], before, after);
    g->currentShape = shape;
    /* removed recursive shaping */
    if (shape < 0) ok = false;
    i++;
  }
  return ok;
}

/**
 * TODO: use this method. :)
 * Get the first non-spacing glyph left, starting here and incuding this.
 * return the index or -1 if not found.
 */
int 
STextData::getNonTransparentLeft(SGlyphLine* gl, unsigned int index)
{
 if (index > gl->size()) return -1;
 for (int i=((int)index) - 1; i >= 0; i--)
 {
   const SGlyph& g = (*gl)[(unsigned int)i];
   if (!g.isTransparent()) return i; 
 }
 return -1;
}

/**
 * TODO: use this method. :)
 * Get the first non-spacing glyph left, starting here and including this.
 * return the index or -1 if not found.
 */
int 
STextData::getNonTransparentRight(SGlyphLine* gl, unsigned int index)
{
 for (int i=index +1; i < (int)gl->size(); i++)
 {
   const SGlyph& g = (*gl)[(unsigned int)i];
   if (!g.isTransparent()) return i; 
 }
 return -1;
}
