#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h> 
#include <errno.h>
extern int errno;
#include <unistd.h>
#include <signal.h>
#include <math.h>
#include <sys/stat.h>

#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>          /* For inet_ntoa. */
#include <sys/ioctl.h>
#include <fcntl.h>

#include "../include/string.h"

#include "ytypes.h"
#include "soundpaths.h"
#include "ysound.h"
#include "ymixer.h"
#include "ymode.h"
#include "yconnection.h"
#include "yhost.h"
#include "playstack.h"
#include "ynet.h"

#include "yiff.h"
#include "options.h"


void YNetPrintError(
	FILE *stream,
	YConnectionNumber con_num,
	u_int32_t chunk_length,
	u_int16_t major_op_code,
	u_int16_t minor_op_code,
	const char *mesg
);

int YNetSendToConnection(
	YConnectionNumber con_num, const char *buf, int len
);

int YNetSendAudioChangePreset(
	YConnectionNumber con_num,
	const char *audio_mode_name
);
int YNetSendAudioChangeValues(
	YConnectionNumber con_num,
	int sample_size,
	int channels,
	YDataLength sample_rate,
	int direction,
	int allow_fragmenting,
	int num_fragments,
	int fragment_size
);

int YNetParseAudioChange(YCNP_STD_INPUTS_PROTO);

int YNetSendServerStats(
	YConnectionNumber con_num,
	int protocol_version_major,
	int protocol_version_minor,
	Coefficient cycle_load,
	const char *vendor_name
);
int YNetParseServerStats(YCNP_STD_INPUTS_PROTO);

int YNetSendCycleChange(YConnectionNumber con_num, long cycle_us);
int YNetParseCycleChange(YCNP_STD_INPUTS_PROTO);

int YNetSendDisconnect(YConnectionNumber con_num, u_int16_t reason);
int YNetParseDisconnect(YCNP_STD_INPUTS_PROTO);

int YNetSendSetHost(
	YConnectionNumber con_num,
	u_int16_t minor_op_code,
	const YIPUnion *ip
);
int YNetParseSetHost(YCNP_STD_INPUTS_PROTO);

int YNetSendSetMixerDevice(
	YConnectionNumber con_num,
	int mixer_code,
	Coefficient value1, Coefficient value2
);
int YNetParseMixerDevice(YCNP_STD_INPUTS_PROTO);

int YNetSendSoundObjectPlay(
	YConnectionNumber con_num,
	YID yid,
	const char *path,
	YDataPosition pos,
	YVolumeStruct *volume,
	int sample_rate,
	int repeats
);
int YNetParseSoundObjectPlay(YCNP_STD_INPUTS_PROTO);

int YNetSendSoundObjectKill(YConnectionNumber con_num, YID yid);
int YNetParseSoundObjectKill(YCNP_STD_INPUTS_PROTO);

int YNetSendSoundObjectAttributes(
	YConnectionNumber con_num,
	const char *path,
	int format,
	int sample_size,
	int channels,
	int sample_rate
);
int YNetParseSoundObjectAttributes(YCNP_STD_INPUTS_PROTO);

int YNetSendShutdown(YConnectionNumber con_num, u_int16_t reason);
int YNetParseShutdown(YCNP_STD_INPUTS_PROTO);

int YNetSendSync(YConnectionNumber con_num, long us);
int YNetParseSync(YCNP_STD_INPUTS_PROTO);

int YNetSendAudioStats(
        YConnectionNumber con_num,
        Audio *audio
);
int YNetParseAudioStats(YCNP_STD_INPUTS_PROTO);

int YNetSendListAudioModes(
	YConnectionNumber con_num,
	const char *audio_mode_name,
	int sample_rate, int channels,
	int sample_size, int fragment_size_bytes,
	char direction, Boolean allow_fragmenting, int num_fragments
);
int YNetParseListAudioModes(YCNP_STD_INPUTS_PROTO);

int YNetSendPlaySoundObjectValues(
	YConnectionNumber con_num,
	PlayStack *ps_ptr
);
int YNetParsePlaySoundObjectValues(YCNP_STD_INPUTS_PROTO);

int YNetParse(
	YConnectionNumber con_num,
	const u_int8_t *buf,
	u_int32_t chunk_length,
        u_int16_t major_op_code,
        u_int16_t minor_op_code
);
int YNetRecv(YConnectionNumber con_num);;


#define MIN(a,b)	((a) < (b) ? (a) : (b))
#define MAX(a,b)	((a) > (b) ? (a) : (b))


extern int YAntiShift(int in);   /* In main.c */


/*
 *	Marks currently handled connection to be closed.
 */
static Boolean close_this_connection;


/*
 *	Procedure to print network error message.
 */
void YNetPrintError(
        FILE *stream,
        YConnectionNumber con_num,
        u_int32_t chunk_length,
        u_int16_t major_op_code,
        u_int16_t minor_op_code,
        const char *mesg
)
{
        if(stream == NULL)
            return;

        fprintf(
            stream,
            "Y server protocol error from connection: %i\n",
            con_num
        );
        fprintf(      
            stream,  
            "    Major OP Code: %i\n",
            major_op_code
        );
        fprintf(
            stream,
            "    Minor OP Code: %i\n",
            minor_op_code 
        );
        fprintf(
            stream,
            "    Segment Size: %i bytes\n",
            chunk_length
        );
        if(mesg != NULL)
            fprintf(
                stream,
                "    Remarks: %s\n",   
                mesg
	    );


	return;
}

/*
 *	Send buffer buf of length len to connection.
 *
 *	The connection's socket is checked to see if it's
 *	available for sending data. Returns -1 if it was not
 *	able to send the buffer or the number of bytes sent.
 */
int YNetSendToConnection(YConnectionNumber con_num, const char *buf, int len) 
{
	int s;
        struct timeval timeout;
        fd_set writefds;

        int bytes_sent;
        YConnection *con;


        /* Connection assumed valid. */
        con = yconnection[con_num];


	s = con->socket;

        timeout.tv_sec = 0;
        timeout.tv_usec = 0;
        FD_ZERO(&writefds);
        FD_SET(s, &writefds);
        select(s + 1, NULL, &writefds, NULL, &timeout);

        if(!FD_ISSET(s, &writefds))
            return(-1);

        bytes_sent = send(s, buf, len, 0);
        if(bytes_sent < 0)
            return(-1);
        else
            return(bytes_sent);
}


/*
 *	Send Audio mode change.
 */
int YNetSendAudioChangePreset(
	YConnectionNumber con_num,
	const char *audio_mode_name
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <...modename...>
         */
        const YDataLength this_seg_len = 8 + YAudioNameMax;
        YDataLength actual_seg_len;
        int name_len;
        char buf[this_seg_len];


        if(audio_mode_name == NULL)
            return(-1);

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);  
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YAudioChange);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YAudioChangePreset);

        name_len = MIN(strlen(audio_mode_name), YAudioNameMax);
        strncpy(
            &buf[8],
            audio_mode_name,
            name_len
        );
        /* Do not null terminate string. */

        /*   Since we have a string in this seg, we need to
         *   calculate the actual length.
         */
        actual_seg_len = 8 + name_len;

        /* Update segment header's chunk size! */
        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)actual_seg_len);

        return(YNetSendToConnection(con_num, buf, actual_seg_len));

}
/*
 *	Sends audio values change.
 */
int YNetSendAudioChangeValues(
        YConnectionNumber con_num,
        int sample_size,
        int channels,
        YDataLength sample_rate,
        int direction,
        int allow_fragmenting,
        int num_fragments,
        int fragment_size
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_samplesize> <u16_channels> <u32_samplerate>
         *   <u8_direction>
         *   <u8_allowfrags> <u32_numfrags> <u32_fragsize>
         */
        const YDataLength this_seg_len = 8 + 8 + 1 + 9;

        char buf[this_seg_len];


        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YAudioChange);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YAudioChangeValues);

        *(u_int16_t *)(&buf[8]) = htons((u_int16_t)sample_size);
        *(u_int16_t *)(&buf[10]) = htons((u_int16_t)channels);
        *(u_int32_t *)(&buf[12]) = htonl((u_int32_t)sample_rate);

        *(u_int8_t *)(&buf[16]) = (u_int8_t)direction;

        *(u_int8_t *)(&buf[17]) = (u_int8_t)allow_fragmenting;
        *(u_int32_t *)(&buf[18]) = htonl((u_int32_t)num_fragments);
        *(u_int32_t *)(&buf[22]) = htonl((u_int32_t)fragment_size);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles Audio mode change.
 */
int YNetParseAudioChange(YCNP_STD_INPUTS_PROTO)
{
	int i;
	YConnection **yc_ptr;
	YMode *ymode_ptr;
	Audio *audio_ptr;

	/* Change mode values or change to preset mode? */
        if(minor_op_code == YAudioChangeValues)
        {
            /* Parse set Audio values. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <u16_samplesize> <u16_channels> <u32_samplerate>
             *   <u8_direction>
             *   <u8_allowfrags> <u32_numfrags> <u32_fragsize>
             */
            const YDataLength this_seg_len = 8 + 8 + 1 + 9;
	    int sample_size;
            int channels;
	    YDataLength sample_rate;
	    int direction;
	    int allow_fragmenting;
	    int num_fragments;
	    int fragment_size_bytes;


            if(chunk_length < this_seg_len)
                return(-1);

	    /* Get Audio values. */
            sample_size = ntohs(*(u_int16_t *)(&buf[8]));
            channels = ntohs(*(u_int16_t *)(&buf[10]));
            sample_rate = ntohl(*(u_int32_t *)(&buf[12]));

	    direction = (int)buf[16];

            allow_fragmenting = (int)buf[17];
            num_fragments = ntohl(*(u_int32_t *)(&buf[18]));
            fragment_size_bytes = ntohl(*(u_int32_t *)(&buf[22]));


	    /* Special case: If the sample rate is 0, this implies
	     * that the DSP device should be closed (shelled out)
	     * to let other applications open it.
	     */
	    if(sample_rate == 0)
	    {
                /* Shell out the recorder. */
                YSoundShellOut(recorder);

		/* Send back shell out response. */
                YNetSendAudioChangeValues(
                    con_num,
                    sample_size,
                    channels,
                    0,		/* Sample rate should be 0. */
                    direction,
                    allow_fragmenting,
                    num_fragments,
                    fragment_size_bytes
                );

		/* Return now, do not delete playstacks or modify Audio. */
		return(0);
	    }

            /* Delete all playstacks and notify their respective owners
             * about the deletion.
             */
            for(i = 0; i < total_playstacks; i++)
                YiffDestroyPlaystack(i);

            /* Shutdown the recorder. */
            YSoundShutdown(recorder);

            /* Free recorder structure. */
            free(recorder);
            recorder = NULL;


            /* Set new Audio values to option Audio values. */
	    audio_ptr = &option.audio;
            audio_ptr->sample_size = sample_size;
            audio_ptr->channels = channels;
            audio_ptr->sample_rate = sample_rate;

#ifdef OSS_BUFFRAG
            audio_ptr->allow_fragments = ((allow_fragmenting) ?
		True : False
	    );
            audio_ptr->num_fragments = num_fragments;
            audio_ptr->fragment_size = YAntiShift(fragment_size_bytes - 1);
#endif  /* OSS_BUFFRAG */

            audio_ptr->direction = direction;


            /* Allocate recorder structure. */
            recorder = (Recorder *)calloc(1, sizeof(Recorder));
            if(recorder == NULL)
                return(-1);
	    /* Initialize recorder. */
            if(YSoundInit(recorder, &option.audio))
	    {
		/* Failed to initialize. */

		/* Shutdown the recorder. */
		YSoundShutdown(recorder);

		/* Free recorder structure. */
		free(recorder);
		recorder = NULL;

		/* Send back notification of initialization failure. */
                YNetSendAudioChangeValues(
                    con_num,
                    0, 0, 0,
                    0, 0, 0, 0
                );

		return(-1);
	    }

            /*  Need to sync audio device right after
             *  initialization.
             */
            YSoundSync(recorder, 0);


            /* Notify all connections about Audio values change. */
            for(i = 0, yc_ptr = yconnection;
                i < total_yconnections;
                i++, yc_ptr++
	    )
            {
		if(*yc_ptr == NULL)
		    continue;

                if((*yc_ptr)->socket < 0)
                    continue;

		YNetSendAudioChangeValues(
		    i,
                    sample_size,
                    channels,
                    sample_rate,
                    direction,
                    allow_fragmenting,
                    num_fragments,
                    fragment_size_bytes
		);
            }
        }
        else
        {
            /* Parse change preset Audio mode. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <...modename...>
             */
            const YDataLength base_seg_len = 8;
            int name_len;
	    int mode_num;
	    int mode_name_different = 0;
	    char mode_name[YAudioNameMax];
	    Audio *audio_ptr;


            if(chunk_length < base_seg_len)
                return(-1);

            name_len = (YDataLength)chunk_length - (YDataLength)base_seg_len;
            if(name_len >= YAudioNameMax)
                name_len = YAudioNameMax - 1;

            if(name_len > 0)
            {
                strncpy(
                    mode_name,  
                    &buf[8],
                    name_len
                );
                mode_name[name_len] = '\0';

		/* Check if the given Audio mode name is different from
		 * the one specified in the Audio structure.
		 */
		if(recorder->audio.audio_mode_name != NULL)
		{
		    if(strcasecmp(
			recorder->audio.audio_mode_name,
			mode_name
		    ))
		        mode_name_different = 1;
		}
		else
		{
		    /* Last Audio mode name not recorded on Audio
		     * structure, so assume its different.
		     */
		    mode_name_different = 1;
		}
		/* Mode name should always be considered different if
		 * Audio is currently shelled out.
		 */
		if(recorder->audio.shelled_out)
		    mode_name_different = 1;


		/* Match preset Audio mode, skip if current Audio
		 * mode matches.
		 */
                mode_num = YModeMatch(mode_name);
                if(YModeIsAllocated(mode_num) &&
                   mode_name_different
		)
                {
		    ymode_ptr = ymode[mode_num];

		    /* Delete all playstacks and notify their respective
		     * owners about the deletion.
		     */
		    for(i = 0; i < total_playstacks; i++)
			YiffDestroyPlaystack(i);

		    /* Shutdown the recorder. */
		    YSoundShutdown(recorder);

		    /* Free recorder structure. */
		    free(recorder);
		    recorder = NULL;


                    /* Set option Audio values to new mode's values. */
		    audio_ptr = &option.audio;

		    /* Audio modes are always considered user set. */
                    audio_ptr->cycle_set = CYCLE_SET_USER;

                    audio_ptr->cycle.ms = ymode_ptr->cycle.ms;
                    audio_ptr->cycle.us = ymode_ptr->cycle.us;

		    /* Leave audio_ptr->compensated_cycle alone. */

                    audio_ptr->write_ahead.ms = ymode_ptr->write_ahead.ms;
                    audio_ptr->write_ahead.us = ymode_ptr->write_ahead.us;

                    audio_ptr->sample_size = ymode_ptr->sample_size;
                    audio_ptr->channels = ymode_ptr->channels;
                    audio_ptr->sample_rate = ymode_ptr->sample_rate;

#ifdef OSS_BUFFRAG
                    audio_ptr->allow_fragments = ymode_ptr->allow_fragments;
                    audio_ptr->num_fragments = ymode_ptr->num_fragments;
                    audio_ptr->fragment_size = ymode_ptr->fragment_size;
#endif  /* OSS_BUFFRAG */

                    audio_ptr->flip_stereo = ymode_ptr->flip_stereo;
                    audio_ptr->direction = ymode_ptr->direction;


                    /* Allocate recorder structure. */
                    recorder = (Recorder *)calloc(1, sizeof(Recorder));
                    if(recorder == NULL)
                        return(-1);
		    /* Initialize recorder. */
                    if(YSoundInit(recorder, audio_ptr))
		    {
                        /* Failed to initialize. */

                        /* Shutdown the recorder. */
                        YSoundShutdown(recorder);

                        /* Free recorder structure. */
                        free(recorder);
                        recorder = NULL;

			/* Send back notification of initialization
			 * failure.
			 */
                        YNetSendAudioChangePreset(con_num, "");

			return(-1);
		    }

                    /*  Need to sync audio device right after
		     *  initialization.
		     */
                    YSoundSync(recorder, 0);

		    /* Record name of preset Audio mode used to
		     * set this Audio.
		     */
		    free(recorder->audio.audio_mode_name);
		    recorder->audio.audio_mode_name = StringCopyAlloc(
			ymode_ptr->name
		    );

                    /* Notify all connections of Audio mode change. */
                    for(i = 0, yc_ptr = yconnection;
                        i < total_yconnections;
                        i++, yc_ptr++
                    )
                    {
                        if(*yc_ptr == NULL)
                            continue;

                        if((*yc_ptr)->socket < 0)
                            continue;

                        YNetSendAudioChangePreset(
                            i,
                            mode_name
                        );
                    }
		}
		/* Audio mode the same? */
		else if(!mode_name_different)
		{
		    /* Send back fake success response. */
		    YNetSendAudioChangePreset(con_num, mode_name);
		}
		else
		{
		    /* No such preset Audio mode. */

                    /* Notify connection of Audio mode change
		     * failure (send back empty Audio mode name).
		     */
                    YNetSendAudioChangePreset(con_num, "");
		}
            }
	    else
	    {
		/* No mode name specified. */

                /* Notify connection of Audio mode change
                 * failure (send back empty Audio mode name).
                 */
                YNetSendAudioChangePreset(con_num, "");
	    }
        }


	return(0);
}

/*
 *	Sends server stats.
 */
int YNetSendServerStats(
	YConnectionNumber con_num,
	int protocol_version_major,
	int protocol_version_minor,
	Coefficient cycle_load,
	const char *vendor_name
)
{
	/*   <u32_cs> <u16_majop> <u16_minop>
	 *   <u32_protocolvermaj> <u32_protocolvermin>
         *   <u16_load>
         *   <...vendorname...>
         */
        const YDataLength this_seg_len = 8 + 8 + 2 + YVendorNameMax;
        YDataLength actual_seg_len;
        int vendor_name_len;
        char buf[this_seg_len];


        if(vendor_name == NULL)
            return(-1);

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YServerStats);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YServerStatsSet);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)protocol_version_major);
        *(u_int32_t *)(&buf[12]) = htonl((u_int32_t)protocol_version_minor);

        *(u_int16_t *)(&buf[16]) = htons((u_int16_t)(
	    cycle_load * (Coefficient)((u_int16_t)-1)
	));
 
        vendor_name_len = MIN(strlen(vendor_name), YVendorNameMax);
        strncpy(
            &buf[18],
            vendor_name,
            vendor_name_len
        );
        /* Do not null terminate string. */

        /*   Since we have a string in this seg, we need to
         *   calculate the actual length.
         */
        actual_seg_len = 8 + 8 + 2 + vendor_name_len;

        /* Update segment header's chunk size! */
        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)actual_seg_len);

        return(YNetSendToConnection(con_num, buf, actual_seg_len));
}
/*
 *	Parses server stats.
 */
int YNetParseServerStats(YCNP_STD_INPUTS_PROTO)
{
        if(minor_op_code == YServerStatsGet)
        {
            /* Parse server stats get. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             */
            const YDataLength this_seg_len = 8;
        
            if(chunk_length < this_seg_len)
                return(-1);

	    YNetSendServerStats(
		con_num,
		YPROTOCOL_VERSION_MAJOR,
		YPROTOCOL_VERSION_MINOR,
		ystats.cycle_load,
		PROG_VENDOR_NAME
	    );
        }
        else if(minor_op_code == YServerStatsSet)
        {
            /* Parse server stats set. */
 
            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <u32_protocolvermaj> <u32_protocolvermin>
             *   <u16_load>
             *   <...vendorname...>
             */
            const YDataLength base_seg_len = 8 + 8 + 2;
        
            if(chunk_length < base_seg_len)
                return(-1);
        }


        return(0);
}

/*
 *	Sends cycle change.
 */
int YNetSendCycleChange(YConnectionNumber con_num, long cycle_us)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_cycleus>
         */
        const YDataLength this_seg_len = 8 + 4;

        char buf[this_seg_len];


        /* Linux requires this > than 100 us. */
	if(cycle_us < 100)
	    cycle_us = 100;

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YCycleChange);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)cycle_us);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles cycle change.
 */
int YNetParseCycleChange(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_cycleus>
         */
        const YDataLength this_seg_len = 8 + 4;
	long cycle_us;
	int i;
	YConnection **yc_ptr;


        if(chunk_length < this_seg_len)
            return(-1);

        cycle_us = ntohl(*(u_int32_t *)(&buf[8]));
	/* Most UNIXes requires this > than 100 us. */
	if(cycle_us < 100)
	    cycle_us = 100;

	/* Set new cycle. */
        if(recorder == NULL)
        {
	    /* Recorder not available, send error response. */
	    YNetSendCycleChange(con_num, 0);
	}
	else
	{
	    /* Set new cycle. */
            if(recorder->audio.cycle_set == CYCLE_SET_USER)
            {
                recorder->audio.cycle.ms = cycle_us / 1000;
                recorder->audio.cycle.us = cycle_us % 1000;

                recorder->audio.compensated_cycle.ms =
                    recorder->audio.cycle.ms;
                recorder->audio.compensated_cycle.us =
                    recorder->audio.cycle.us;

		recorder->audio.write_ahead.ms = (long)(cycle_us * 1.5) /
		    1000;
		recorder->audio.write_ahead.us = (long)(cycle_us * 1.5) %
		    1000;

                option.audio.write_ahead.ms =
		    recorder->audio.write_ahead.ms;
                option.audio.write_ahead.us =
		    recorder->audio.write_ahead.us;


                /* Notify all connections of cycle change. */
                for(i = 0, yc_ptr = yconnection;
                    i < total_yconnections;
                    i++, yc_ptr++
		)
		{
		    if(*yc_ptr == NULL)
			continue;

                    if((*yc_ptr)->socket < 0)
                        continue;

                    YNetSendCycleChange(i, cycle_us);
		}
            }
	    else
	    {
		/* Program or hardware set cycle. */
		cycle_us = (recorder->audio.cycle.ms * 1000) +
                    recorder->audio.cycle.us;

		YNetSendCycleChange(con_num, cycle_us);
	    }
	}


        return(0);
}


/*
 *	Sends disconnect.
 */
int YNetSendDisconnect(YConnectionNumber con_num, u_int16_t reason)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_reason>
         */
        const YDataLength this_seg_len = 8 + 2;

	char buf[this_seg_len];

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YDisconnect);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int16_t *)(&buf[8]) = htons((u_int16_t)reason);

        return(YNetSendToConnection(con_num, buf, this_seg_len));            
}
/*
 *	Handle disconnect.
 */
int YNetParseDisconnect(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_reason>
         */
        const YDataLength this_seg_len = 8 + 2;
	u_int16_t reason;


        if(chunk_length < this_seg_len)
            return(-1);

        reason = ntohs(*(u_int16_t *)(&buf[8]));

	/* Mark currently handled connection number to be closed. */
	close_this_connection = True;

        return(0);
}

/*
 *	Sends set host.
 */
int YNetSendSetHost(
	YConnectionNumber con_num,
	u_int16_t minor_op_code,
	const YIPUnion *ip
)
{
        /*   <u32_cs> <u16_majop> <u16_minop> 
         *   <u8_ipc1> <u8_ipc2> <u8_ipc3> <u8_ipc4>
         */
        const YDataLength this_seg_len = 8 + 4;

        char buf[this_seg_len];

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSetHost);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)minor_op_code);

        *(u_int8_t *)(&buf[8]) = ip->charaddr[0];
        *(u_int8_t *)(&buf[9]) = ip->charaddr[1];
        *(u_int8_t *)(&buf[10]) = ip->charaddr[2];
        *(u_int8_t *)(&buf[11]) = ip->charaddr[3];

	return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles set host.
 */
int YNetParseSetHost(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u8_ipc1> <u8_ipc2> <u8_ipc3> <u8_ipc4>
         */
        const YDataLength this_seg_len = 8 + 4;
	YIPUnion ip;
	int status;

        if(chunk_length < this_seg_len)
            return(-1);

        ip.charaddr[0] = (u_int8_t)buf[8];
        ip.charaddr[1] = (u_int8_t)buf[9];
        ip.charaddr[2] = (u_int8_t)buf[10];
        ip.charaddr[3] = (u_int8_t)buf[11];

	switch(minor_op_code)
	{
	  case YSetHostAdd:
	    status = YHostAllocate(&ip);
	    if(status < 0)
		fprintf(stderr,
		    "Cannot add YHost %i.%i.%i.%i.\n",
		    ip.charaddr[0],
                    ip.charaddr[1],
                    ip.charaddr[2],          
                    ip.charaddr[3]
		);
	    break;

	  case YSetHostRemove:
	    status = YHostDeleteByHost(&ip);
            if(status < 0)
                fprintf(stderr,
                    "Cannot remove YHost %i.%i.%i.%i.\n",
                    ip.charaddr[0],
                    ip.charaddr[1],
                    ip.charaddr[2],
                    ip.charaddr[3] 
                );
            break;
	}

	/* Notify connection of host add or remove. */
	YNetSendSetHost(
            con_num,
            minor_op_code,
            &ip
	);

        return(0);
}

/*
 *      Sends a mixer device set.
 */
int YNetSendSetMixerDevice(
        YConnectionNumber con_num,
        int mixer_code,
        Coefficient value1, Coefficient value2
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_mixercode> <u32_value1> <u32_value2>
         */
        const YDataLength this_seg_len = 8 + 10;

        char buf[this_seg_len];
        u_int32_t ival1, ival2;


        ival1 = (u_int32_t)rint(value1 * (Coefficient)((u_int32_t)-1));
        ival2 = (u_int32_t)rint(value2 * (Coefficient)((u_int32_t)-1));


        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YMixerChannel);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YMixerChannelSet);

        *(u_int16_t *)(&buf[8]) = htons((u_int16_t)mixer_code);

        *(u_int32_t *)(&buf[10]) = htonl((u_int32_t)ival1);
        *(u_int32_t *)(&buf[14]) = htonl((u_int32_t)ival2);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles set mixer device.
 */
int YNetParseMixerDevice(YCNP_STD_INPUTS_PROTO)
{
        if(minor_op_code == YMixerChannelSet)
        {
            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <u16_mixercode> <u32_value1> <u32_value2>
             */
            const YDataLength this_seg_len = 8 + 10;
	    int i, status, mixer_code;
	    Coefficient value[YMixerValues];
	    YConnection **yc_ptr;


            if(chunk_length < this_seg_len)
                return(-1);

            mixer_code = ntohs(*(u_int16_t *)(&buf[8]));

	    if(0 < YMixerValues)
                value[0] =
                    (Coefficient)ntohl(*(u_int32_t *)(&buf[10])) /
                    (Coefficient)((u_int32_t)-1);
	    if(1 < YMixerValues)
		value[1] =
                    (Coefficient)ntohl(*(u_int32_t *)(&buf[14])) /
                    (Coefficient)((u_int32_t)-1);

	    /* Set mixer device. */
	    status = -1;
	    if(recorder != NULL)
	    {
		status = YMixerSet(recorder, mixer_code, value);
	    }
	    if(status < 0)
	    {
                /* Error setting mixer values, notify error. */
                YNetSendSetMixerDevice(
                    con_num, mixer_code,
		    0.0, 0.0
                );
	    }
	    else
	    {
		/* Notify all connections about mixer channel change. */
                for(i = 0, yc_ptr = yconnection;
                    i < total_yconnections;
                    i++, yc_ptr++
                )
                {
                    if((*yc_ptr) == NULL)
                        continue;

                    if((*yc_ptr)->socket < 0)
                        continue;

		    YNetSendSetMixerDevice(
		        i, mixer_code,
			(0 < YMixerValues) ? value[0] : 0.0,
			(1 < YMixerValues) ? value[1] : 0.0
		    );
		}
	    }
	}
	else if(minor_op_code == YMixerChannelGet)
	{
            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <u16_mixercode>
             */
            const YDataLength this_seg_len = 8 + 2;
            int status, mixer_code;
	    Coefficient value[YMixerValues];


            if(chunk_length < this_seg_len)
                return(-1);

            mixer_code = ntohs(*(u_int16_t *)(&buf[8]));

	    /* Get mixer channel device values. */
	    status = -1;
	    if(recorder != NULL)
	    {
		status = YMixerGet(
                    recorder, mixer_code, value
		);
	    }
	    if(status < 0)
	    {
		/* Error getting values, respond with 0 channel
		 * values.
		 */
                YNetSendSetMixerDevice(con_num, mixer_code, 0.0, 0.0);
	    }
	    else
	    {
		YNetSendSetMixerDevice(
		    con_num, mixer_code,
		    (0 < YMixerValues) ? value[0] : 0.0,
		    (1 < YMixerValues) ? value[1] : 0.0
		);
	    }
	}

	return(0);
}


/*
 *      Sends sound play.
 */
int YNetSendSoundObjectPlay(
	YConnectionNumber con_num,
	YID yid,
	const char *path,
	YDataPosition pos,
	YVolumeStruct *volume,
	int sample_rate,
	int repeats
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_yid> <u32_pos> <u16_lvol> <u16_rvol>
	 *   <u32_samplerate> <u32_repeats>
         *   <...path...>
         */
        const YDataLength this_seg_len = 8 + 12 + 8 + YPathMax; 

        char buf[this_seg_len];
        YDataLength path_len, actual_seg_len;

	if(sample_rate < 0)
	    sample_rate = 0;
	if(repeats < 0)
	    repeats = 0;

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSoundObjectPlay);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)yid);   
        *(u_int32_t *)(&buf[12]) = htonl((u_int32_t)pos);  
        *(u_int16_t *)(&buf[16]) = htons((u_int16_t)volume->left);
        *(u_int16_t *)(&buf[18]) = htons((u_int16_t)volume->right);

	*(u_int32_t *)(&buf[20]) = htonl((u_int32_t)sample_rate);
        *(u_int32_t *)(&buf[24]) = htonl((u_int32_t)repeats);

        path_len = MIN(strlen(path), YPathMax);
        strncpy(
            &buf[28],
            path,
            path_len
        );
        /* Do not null terminate string. */

        /*   Since we have a string in this seg, we need to
         *   calculate the actual length.
         */
        actual_seg_len = 8 + 12 + 8 + path_len;

        /* Update segment header's chunk size! */
        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)actual_seg_len);

        return(YNetSendToConnection(con_num, buf, actual_seg_len));
}
/*
 *      Handles sound play.
 */
int YNetParseSoundObjectPlay(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_yid> <u32_pos> <u16_lvol> <u16_rvol>
         *   <u32_sample_rate> <u32_repeats>
         *   <...path...>
         */
/*	const YDataLength this_seg_len = 8 + 12 + 8 + YPathMax; */
	const YDataLength this_seg_len_base = 8 + 12 + 8;

	YID yid;
	YDataPosition pos;
	u_int16_t lvol, rvol;
	int sample_rate;
	int repeats;

	int path_len;
	char path[YPathMax];
	YVolumeStruct volume;
	char *strptr;


        if(chunk_length < this_seg_len_base)
            return(-1);

        yid = ntohl(*(u_int32_t *)(&buf[8]));
        pos = ntohl(*(u_int32_t *)(&buf[12]));
        lvol = ntohs(*(u_int16_t *)(&buf[16]));
        rvol = ntohs(*(u_int16_t *)(&buf[18]));
	sample_rate = ntohl(*(u_int32_t *)(&buf[20]));
	repeats = ntohl(*(u_int32_t *)(&buf[24]));

	path_len = (int)chunk_length - this_seg_len_base;
	if(path_len >= YPathMax)
	    path_len = YPathMax - 1;

	if(path_len > 0)
	{
	    strncpy(
		path,
		&buf[28],
		path_len
	    );
	    path[path_len] = '\0';
	}
	else
	{
	    /* If path is not specified, do not continue. */

            YNetSendSoundObjectPlay(
                con_num,
                YIDNULL,
                path,
                pos,
                &volume,
		sample_rate,
                repeats
            );
	    return(0);
	}


	/* Try to match correct complete path. */
	strptr = SoundPathCompletePath(path);
	if(strptr == NULL)
	{
	    YNetSendSoundObjectPlay(
                con_num,
                YIDNULL,
		path,
		pos,
		&volume,
		sample_rate,
		repeats
	    );
	    return(0);
	}

	/* Do not start playing if recorder is not initialized. */
	if(recorder == NULL)
	{
            YNetSendSoundObjectPlay(
                con_num,
                YIDNULL,
                path,
                pos,
                &volume,
		sample_rate,
                repeats
            );
            return(0);
	}

	/* Create playstack. */
	volume.left = lvol;
	volume.right = rvol;
        YiffCreatePlaystack( 
            strptr,		/* Path. */
            con_num,		/* Owner. */
            yid,		/* YID. */
	    pos,		/* Position. */
            &volume,
	    sample_rate,
	    repeats		/* 0 means repeat forever. */
	);

        /* YiffCreatePlaystack() will send response. */


	return(0);
}


/*
 *	Sends sound kill.
 */
int YNetSendSoundObjectKill(YConnectionNumber con_num, YID yid)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_yid>
         */
        const YDataLength this_seg_len = 8 + 4;

	char buf[this_seg_len];

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSoundObjectKill);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)yid);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles sound kill.
 */
int YNetParseSoundObjectKill(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_yid>
         */
        const YDataLength this_seg_len = 8 + 4;
	YID yid;
	int i;
	PlayStack **ptr;


        if(chunk_length < this_seg_len)
            return(-1);

	yid = ntohl(*(u_int32_t *)(&buf[8])); 


	/* Kill playstack. */
        for(i = 0, ptr = playstack;
            i < total_playstacks;
            i++, ptr++
	)
        {
            if(*ptr == NULL)
		continue;

            if((*ptr)->owner != con_num)
		continue;

	    if((*ptr)->yid == YIDNULL)
		continue;

	    if((*ptr)->yid != yid)
		continue;

	    /* Delete playstack and notify owner connection about it. */
	    YiffDestroyPlaystack(i);
	}


        return(0);
}

/*
 *	Sends sound object attributes.
 */
int YNetSendSoundObjectAttributes(
	YConnectionNumber con_num,
	const char *path,
	int format,
	int sample_size,
	int channels,   
	int sample_rate
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_format>
	 *   <u16_samplesize> <u16_channels> <u32_samplerate>
         *   <...path...>
         */
        const YDataLength this_seg_len = 8 + 2 + 8 + YPathMax;

        char buf[this_seg_len];
        YDataLength path_len, actual_seg_len;

        if(path == NULL)
	    return(-1);

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSoundObjectAttributes);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YSoundObjectAttributesSet);

        *(u_int16_t *)(&buf[8]) = htons((u_int16_t)format);

        *(u_int16_t *)(&buf[10]) = htons((u_int16_t)sample_size);
        *(u_int16_t *)(&buf[12]) = htons((u_int16_t)channels);
        *(u_int32_t *)(&buf[14]) = htonl((u_int32_t)sample_rate);

        path_len = MIN(strlen(path), YPathMax);
	if(path_len > 0)
            strncpy(
                &buf[18],
                path,
                path_len
            );
        /* Do not null terminate string. */

        /*   Since we have a string in this seg, we need to
         *   calculate the actual length.
         */
        actual_seg_len = 8 + 2 + 8 + path_len;

        /* Update segment header's chunk size! */
        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)actual_seg_len);

        return(YNetSendToConnection(con_num, buf, actual_seg_len));
}
/*
 *	Handles sound object attributes.
 */
int YNetParseSoundObjectAttributes(YCNP_STD_INPUTS_PROTO)
{
	if(minor_op_code == YSoundObjectAttributesGet)
	{
            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <...path...>
             */
            const YDataLength base_seg_len = 8;
/*	    const YDataLength this_seg_len = 8 + YPathMax; */

	    YDataLength path_len;
	    char path[YPathMax];

	    char *strptr;
	    AFWDataStruct af_data;


	    if(chunk_length < base_seg_len)
		return(-1);

	    path_len = (YDataLength)chunk_length -
		(YDataLength)base_seg_len;
	    if(path_len >= YPathMax)
		path_len = YPathMax - 1;

	    if(path_len > 0)
	    {
	        strncpy(
		    path,
		    (char *)(&buf[8]),
		    path_len
	        );
		path[path_len] = '\0';
	    }
	    else
	    {
		*path = '\0';
	    }


            /* Try to match correct complete path. */
            strptr = SoundPathCompletePath(path);
            if(strptr == NULL)
            {
                YNetSendSoundObjectAttributes(
                    con_num,
                    "",
		    SndObjTypeNone,
		    0, 0, 0
		);
                return(0);
            }


	    /* Open audio file.*/
	    if(AFWOpen(strptr, &af_data))
		*path = '\0';	/* Failure, set path to nothing. */

	    /* Send audio file data. */
	    YNetSendSoundObjectAttributes(
		con_num,
		path,		/* Send back original path. */
		af_data.sndobj_format,
		af_data.sample_size,
		af_data.channels,
		af_data.sample_rate
	    );

	    /* Close audio file. */
	    AFWClose(
		&af_data,
		((recorder == NULL) ? NULL : &recorder->audio)
	    );
	}
	else if(minor_op_code == YSoundObjectAttributesSet)
	{
	    fprintf(stderr,
 "YNetParseSoundObjectAttributes(): Setting of sound object attributes not allowed.\n"
	    );



	}

	return(0);
}

/*
 *	Sends shutdown.
 */
int YNetSendShutdown(YConnectionNumber con_num, u_int16_t reason)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_reason>
         */
        const YDataLength this_seg_len = 8 + 2;

        char buf[this_seg_len];

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YShutdown);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int16_t *)(&buf[8]) = htons((u_int16_t)reason);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles shutdown.
 */
int YNetParseShutdown(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u16_reason>
         */
        const YDataLength this_seg_len = 8 + 2;
        u_int16_t reason;


        if(chunk_length < this_seg_len)
            return(-1);

        reason = ntohs(*(u_int16_t *)(&buf[8]));


/* Check if connection has permission to shut server down? */

	/*   Set runlevel to 1 to shutdown gently. Comfermation
	 *   will be sent to all connections just before shutdown.
	 */
	runlevel = 1;


        return(0);
}

/*
 *	Sends sync.
 */
int YNetSendSync(YConnectionNumber con_num, long us)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_cycleaheadus>
         */
        const YDataLength this_seg_len = 8 + 4;

        char buf[this_seg_len];
	u_int32_t ca;

        if(us < 0)
            us = 0;
	ca = us;

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSync);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)0);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)ca);

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Handles sync.
 */
int YNetParseSync(YCNP_STD_INPUTS_PROTO)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_cycleaheadus>
         */
        const YDataLength this_seg_len = 8 + 4;

	long cahead_us;


        if(chunk_length < this_seg_len)
            return(-1);

        cahead_us = ntohl(*(u_int32_t *)(&buf[8]));

        if(recorder != NULL)
        {
            /* Recaliberate cycle if program calculates it. */
            if(recorder->audio.cycle_set == CYCLE_SET_PROGRAM)
                YSoundCaliberateCycle(recorder);

            /* Set new write ahead. */
            if(cahead_us > 0)
            {
                recorder->audio.write_ahead.ms = cahead_us / 1000;
                recorder->audio.write_ahead.us = cahead_us % 1000;
	    }

            /* Sync sound device. */
            YSoundSync(recorder, 0);
	}

	/* Send sync comfermation. */
	YNetSendSync(con_num, cahead_us);


        return(0);
}

/*
 *	Sends set audio stats.
 */
int YNetSendAudioStats(
	YConnectionNumber con_num,
	Audio *audio
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u8_cycleset>
	 *   <u32_cycleus> <u32_compensatedcycleus>
	 *   <u32_writeaheadus> <u32_cumulativelatencyus>
	 *   <u16_samplesize> <u16_channels> <u32_samplerate>
	 *   <u32_bytespersec>
	 *   <u8_allowfragments> <u32_numfragments> <u32_fragmentsize>
	 *   <u8_flipstereo>
	 *   <u8_direction>
         */
        const YDataLength this_seg_len = 8 + 1 + 8 + 8 + 8 + 4 + 9 + 1 + 1;

        char buf[this_seg_len];


	if(audio == NULL)
	{
	 /* Send blank values (implying recorder not available). */
	 memset(buf, 0, this_seg_len * sizeof(char));

         *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
         *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YAudioStats);
         *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YAudioStatsSet);
	}
	else
	{
	 *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
         *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YAudioStats);
         *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YAudioStatsSet);

         *(u_int8_t *)(&buf[8]) = (u_int8_t)audio->cycle_set;

         *(u_int32_t *)(&buf[9]) = htonl((u_int32_t)(
	    (audio->cycle.ms * 1000) + audio->cycle.us
	 ));
         *(u_int32_t *)(&buf[13]) = htonl((u_int32_t)(    
            (audio->compensated_cycle.ms * 1000) + audio->compensated_cycle.us
         ));

         *(u_int32_t *)(&buf[17]) = htonl((u_int32_t)(
            (audio->write_ahead.ms * 1000) + audio->write_ahead.us
         ));
         *(u_int32_t *)(&buf[21]) = htonl((u_int32_t)(
            (audio->cumulative_latency.ms * 1000) + audio->cumulative_latency.us
         ));

         *(u_int16_t *)(&buf[25]) = htons((u_int16_t)audio->sample_size);
         *(u_int16_t *)(&buf[27]) = htons((u_int16_t)audio->channels);
         *(u_int32_t *)(&buf[29]) = htonl((u_int32_t)audio->sample_rate);   

         *(u_int32_t *)(&buf[33]) = htonl((u_int32_t)audio->bytes_per_second);

#ifdef OSS_BUFFRAG
         *(u_int8_t *)(&buf[37]) = (u_int8_t)((audio->allow_fragments) ? 1 : 0);
         *(u_int32_t *)(&buf[38]) = htonl((u_int32_t)audio->num_fragments);
         *(u_int32_t *)(&buf[42]) = htonl((u_int32_t)(
	    1 << audio->fragment_size
	 ));
#endif	/* OSS_BUFFRAG */

         *(u_int8_t *)(&buf[46]) = (u_int8_t)((audio->flip_stereo) ? 1 : 0);

         *(u_int8_t *)(&buf[47]) = (u_int8_t)audio->direction;
	}

        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Parses audio stats.
 */
int YNetParseAudioStats(YCNP_STD_INPUTS_PROTO)
{
	if(minor_op_code == YAudioStatsGet)
	{
            /*   <u32_cs> <u16_majop> <u16_minop>
	     */
            const YDataLength this_seg_len = 8;

	    if(chunk_length < this_seg_len)
		return(-1);

	    if(recorder == NULL)
		YNetSendAudioStats(con_num, NULL);
	    else
	        YNetSendAudioStats(con_num, &recorder->audio);
	}
	else if(minor_op_code == YAudioStatsSet)
        {

	}


	return(0);
}

/*
 *	Send list audio modes, mode values.
 */
int YNetSendListAudioModes(
	YConnectionNumber con_num,
	const char *audio_mode_name,
	int sample_rate, int channels,
	int sample_size, int fragment_size_bytes,
	char direction, Boolean allow_fragmenting, int num_fragments
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_samplerate> <u32_channels>
         *   <u32_samplesize> <u32_fragmentsizebytes>
         *   <u8_direction> <u8_allowfragmenting> <u32_numfragments>
         *   <...audiomodename...>
         */
        const YDataLength this_seg_len = 8 + 8 + 8 + 6 + YAudioNameMax;

        char buf[this_seg_len];
        YDataLength name_len, actual_seg_len;


        if(audio_mode_name == NULL)
            return(-1);

        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YListAudioModes);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YListAudioModesSet);

        *(u_int32_t *)(&buf[8]) = htonl((u_int32_t)sample_rate);
        *(u_int32_t *)(&buf[12]) = htonl((u_int32_t)channels);
        *(u_int32_t *)(&buf[16]) = htonl((u_int32_t)sample_size);
        *(u_int32_t *)(&buf[20]) = htonl((u_int32_t)fragment_size_bytes);

	*(u_int8_t *)(&buf[24]) = (u_int8_t)direction;
	*(u_int8_t *)(&buf[25]) = (u_int8_t)((allow_fragmenting) ? 1 : 0);
	*(u_int32_t *)(&buf[26]) = htonl((u_int32_t)num_fragments);

        name_len = MIN(strlen(audio_mode_name), YAudioNameMax);
        strncpy(
            &buf[30],
            audio_mode_name,
            name_len
        );
        /* Do not null terminate string. */

        /*   Since we have a string in this seg, we need to
         *   calculate the actual length.
         */
        actual_seg_len = 8 + 8 + 8 + 6 + name_len;

        /* Update segment header's chunk size! */
        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)actual_seg_len);

        return(YNetSendToConnection(con_num, buf, actual_seg_len));
}
/*
 *	Parses list audio modes.
 */
int YNetParseListAudioModes(YCNP_STD_INPUTS_PROTO)
{
        if(minor_op_code == YListAudioModesGet)
        {
            /* Parse list audio modes get. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             */
            const YDataLength this_seg_len = 8;
	    int i;
	    YMode **ymode_ptr;

                
            if(chunk_length < this_seg_len)
                return(-1);

	    /* Send list of preset audio modes. */
	    for(i = 0, ymode_ptr = ymode;
                i < total_ymodes;
                i++, ymode_ptr++
	    )
	    {
		if(*ymode_ptr == NULL)
		    continue;

                YNetSendListAudioModes(
                    con_num,   
                    (*ymode_ptr)->name,
                    (*ymode_ptr)->sample_rate,
		    (*ymode_ptr)->channels,
		    (*ymode_ptr)->sample_size,
#ifdef OSS_BUFFRAG
		    (1 << ((*ymode_ptr)->fragment_size)),
#else
		    1024,
#endif
		    (*ymode_ptr)->direction,
#ifdef OSS_BUFFRAG
		    (*ymode_ptr)->allow_fragments,
		    (*ymode_ptr)->num_fragments
#else
		    False,
		    1
#endif
                );
	    }


	    /* End with an audio mode with no name (indicating end). */
            YNetSendListAudioModes(
                con_num,
                "",
                0, 0,
		0, 0,
		0, False, 0
	    );
	}
	else if(minor_op_code == YListAudioModesSet)
        {

	}         
        
        return(0);
}

/*
 *	Sends sound object values.
 */
int YNetSendPlaySoundObjectValues(
	YConnectionNumber con_num,
	PlayStack *ps_ptr
)
{
        /*   <u32_cs> <u16_majop> <u16_minop>
         *   <u32_flags> <u32_yid>
         *   <u32_pos> <u32_totalrepeats>
         *   <u16_volleft> <u16_volright> <u32_sample_rate>
         */
	const YDataLength this_seg_len = 8 + 8 + 8 + 8;

        char buf[this_seg_len];
	u_int32_t flags;
	u_int16_t volume_left, volume_right;


	if(ps_ptr == NULL)
	    return(-1);

	/* Set flags to match client side. */
	flags = (1 << 1) || (1 << 2) || (1 << 5) || (1 << 6) || (1 << 7);

	/* Calculate volume. */
	volume_left = ps_ptr->volume_left * (Coefficient)((u_int16_t)-1);
        volume_right = ps_ptr->volume_right * (Coefficient)((u_int16_t)-1);


        *(u_int32_t *)(&buf[0]) = htonl((u_int32_t)this_seg_len);
        *(u_int16_t *)(&buf[4]) = htons((u_int16_t)YSoundObjectPlayValues);
        *(u_int16_t *)(&buf[6]) = htons((u_int16_t)YSoundObjectPlayValuesSet);

	*(u_int32_t *)(&buf[8]) = htonl((u_int32_t)flags);
	*(u_int32_t *)(&buf[12]) = htonl((u_int32_t)ps_ptr->yid);

        *(u_int32_t *)(&buf[16]) = htonl((u_int32_t)ps_ptr->position);
        *(u_int32_t *)(&buf[20]) = htonl((u_int32_t)ps_ptr->total_repeats);

        *(u_int16_t *)(&buf[24]) = htons((u_int16_t)volume_left);
        *(u_int16_t *)(&buf[26]) = htons((u_int16_t)volume_right);

        *(u_int32_t *)(&buf[28]) = htonl((u_int32_t)ps_ptr->applied_sample_rate);


        return(YNetSendToConnection(con_num, buf, this_seg_len));
}
/*
 *	Parses play sound object values.
 */
int YNetParsePlaySoundObjectValues(YCNP_STD_INPUTS_PROTO)
{
        if(minor_op_code == YSoundObjectPlayValuesGet)
        {
            /* Parse list audio modes get. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             */
            const YDataLength this_seg_len = 8;

            if(chunk_length < this_seg_len)
                return(-1);
	}
        else if(minor_op_code == YSoundObjectPlayValuesSet)
        {
            /* Parse list audio modes get. */

            /*   <u32_cs> <u16_majop> <u16_minop>
             *   <u32_flags> <u32_yid>
             *   <u32_pos> <u32_totalrepeats>
             *   <u16_volleft> <u16_volright> <u32_sample_rate>
             */
            const YDataLength this_seg_len = 8 + 8 + 8 + 8;

	    int i;
	    u_int32_t flags;
	    YID yid;
	    u_int32_t position;
	    u_int32_t total_repeats;
	    u_int16_t volume_left, volume_right;
	    u_int32_t sample_rate;
	    PlayStack **ptr, *ps_ptr;


            if(chunk_length < this_seg_len)
                return(-1);

            flags = ntohl(*(u_int32_t *)(&buf[8]));
	    yid = ntohl(*(u_int32_t *)(&buf[12]));

            position = ntohl(*(u_int32_t *)(&buf[16]));
            total_repeats = ntohl(*(u_int32_t *)(&buf[20]));

	    volume_left = ntohs(*(u_int16_t *)(&buf[24]));
            volume_right = ntohs(*(u_int16_t *)(&buf[26]));

            sample_rate = ntohl(*(u_int32_t *)(&buf[28]));

	    /* Check if yid exists. */
            for(i = 0, ptr = playstack;
                i < total_playstacks;
                i++, ptr++
            )
            {
		ps_ptr = *ptr;
                if(ps_ptr == NULL)
                    continue;

                if(ps_ptr->owner != con_num)
                    continue;

                if(ps_ptr->yid == YIDNULL)
                    continue;
 
                if(ps_ptr->yid == yid)
		{
		    /* Set new values to play stack. */

		    /* Change position? */
		    if(flags & (1 << 2))
			ps_ptr->position = (YDataPosition)position;

		    /* Change total repeats? */
		    if(flags & (1 << 5))
			ps_ptr->total_repeats = (int)total_repeats;

                    /* Change volume? */
                    if(flags & (1 << 6))
		    {
                        ps_ptr->volume_left = (double)volume_left /
			    (double)((u_int16_t)-1);
			ps_ptr->volume_right = (double)volume_right /
                            (double)((u_int16_t)-1);
		    }

		    /* Change sample size? */
		    if(flags & (1 << 7))
			ps_ptr->applied_sample_rate = (int)sample_rate;

		    /* Send response of values set. */
		    YNetSendPlaySoundObjectValues(con_num, ps_ptr);
		}
	    }
        }

        return(0);
}


/*
 *      Parses the buf.
 */
int YNetParse(
        YConnectionNumber con_num,
	const u_int8_t *buf,
        u_int32_t chunk_length,
        u_int16_t major_op_code,
        u_int16_t minor_op_code
)
{
	int status;
        YConnection *con;      


	/* Connection assumed valid. */
	con = yconnection[con_num];


        switch(major_op_code)
        {
          case YAudioChange:
            status = YNetParseAudioChange(YCNP_STD_INPUTS);
            break;

          case YCycleChange:
            status = YNetParseCycleChange(YCNP_STD_INPUTS);
            break;
          
          case YDisconnect:
            status = YNetParseDisconnect(YCNP_STD_INPUTS);
            break;

          case YSetHost:
            status = YNetParseSetHost(YCNP_STD_INPUTS);
            break;

          case YListHosts:
/* Work on this later. */
            break;
 
          case YMixerChannel:
            status = YNetParseMixerDevice(YCNP_STD_INPUTS);
            break;
        
          case YListMixers:
/* Work on this later. */
            break; 
 
          case YSoundObjectPlay:
            status = YNetParseSoundObjectPlay(YCNP_STD_INPUTS);
            break;
 
          case YSoundObjectKill:
            status = YNetParseSoundObjectKill(YCNP_STD_INPUTS);
            break;

	  case YSoundObjectAttributes:
	    status = YNetParseSoundObjectAttributes(YCNP_STD_INPUTS);
	    break;
 
          case YShutdown:
            status = YNetParseShutdown(YCNP_STD_INPUTS);
            break;
 
          case YSync:
            status = YNetParseSync(YCNP_STD_INPUTS);
            break;

          case YAudioStats:
	    status = YNetParseAudioStats(YCNP_STD_INPUTS);
            break;

          case YServerStats:
            status = YNetParseServerStats(YCNP_STD_INPUTS);
            break;

	  case YListAudioModes:
            status = YNetParseListAudioModes(YCNP_STD_INPUTS);
            break;

	  case YSoundObjectPlayValues:
	    status = YNetParsePlaySoundObjectValues(YCNP_STD_INPUTS);
            break;


          default:
	    YNetPrintError(
		stderr,
		con_num,
                chunk_length,
                major_op_code,
                minor_op_code,	
		"Unsupported Major Op Code"
	    );
            break;
        }

	return(0);
}

/*
 *	Handles incoming data from connection con_num.
 */
int YNetRecv(YConnectionNumber con_num)
{
	YConnection *con;
        int i, n, s, status, loops;
        struct timeval timeout;
        fd_set readfds;

        u_int8_t *buf_ptr;  
        int bytes_read, segments_handled;

        u_int32_t chunk_length;
        u_int16_t major_op_code, minor_op_code;



	/* Reset close connection marker. */
	close_this_connection = False;


	/* Connection is assumed valid and connected. */
	con = yconnection[con_num];

	/* Get connection's socket number. */
	s = con->socket;


        if(con->buf == NULL)   
            return(0);

        if(con->buf_cont < 0)
            con->buf_cont = 0;


        if((con->buf_len - con->buf_cont) <= 0)
        {
            /* Contents buffer overflowed and thus we are unable to
	     * parse the current buffer any more. So reset the
	     * contents buffer and and disconnect connection since
	     * there is no way to seek the next chunk position.
             */
            con->buf_cont = 0;

            /* Disconnect and delete connection. */
            if(con->socket > -1)
            {
                close(con->socket);
                con->socket = -1;
            }
	    YiffCloseConnection(con_num);

            fprintf(
                stderr,
 "YNetRecv(): Connection %i: Contents overflowed buffer length %ld.\n",
                con_num,
                con->buf_len
            );

            return(0);
        }


        /* Do not block, check if there is any data to be read. */
        timeout.tv_sec = 0;
        timeout.tv_usec = 0;
        FD_ZERO(&readfds);
        FD_SET(s, &readfds);
        status = select(s + 1, &readfds, NULL, NULL, &timeout);
        if(status == -1)
            perror("select");

        if(!FD_ISSET(s, &readfds))
            return(0);


        buf_ptr = &(con->buf[con->buf_cont]);
        bytes_read = recv(
            s,
            buf_ptr,
            con->buf_len - con->buf_cont,
            0
        );
        if(bytes_read == 0)
        {
            /* Disconnect and delete connection. */
            if(con->socket > -1)
            {
                close(con->socket);
                con->socket = -1;
            }
            YiffCloseConnection(con_num);

            return(0);
        }
        else if(bytes_read <= 0)
        {
            /* Handle error. */
            switch(errno)
            {
              case EWOULDBLOCK:
                break;

              case EINTR:
                break;

              default:
                /* Disconnect and delete connection. */
                if(con->socket > -1)
                {
                    close(con->socket);
                    con->socket = -1;
                }
                YiffCloseConnection(con_num);
                break;   
            }
            return(0);
        } 

        /* Increment buffer contents. */
        con->buf_cont += bytes_read;
        if(con->buf_cont > con->buf_len)
            con->buf_cont = con->buf_len;

        segments_handled = 0;
	for(loops = 0; loops < 30; loops++)
        {
            /* Check if we got atleast the first 8 bytes, needed
             * for chunk size, major op code, and minor op code.
             */
            if(con->buf_cont < 8)
                break;

            /* All chunks start off with the same format, that being:
             *   <4 bytes, chunk length> <2 bytes, major op code>
             *   <2 bytes, minor op code>
             */
            chunk_length = (u_int32_t)MAX(
		(long)ntohl(*((u_int32_t *)(&con->buf[0]))), (long)0
	    );
            major_op_code = ntohs(*((u_int16_t *)(&con->buf[4])));
            minor_op_code = ntohs(*((u_int16_t *)(&con->buf[6])));


            /* Specified chunk length must be 8 or greater. */
            if(chunk_length < 8)
            {
                YNetPrintError(
                    stderr,
                    con_num,
                    chunk_length,
                    major_op_code,
                    minor_op_code,
 "Recieved a segment with header specified chunk length less than 8 bytes"
                ); 

                /* Disconnect and delete connection. */
                if(con->socket > -1)
                {
                    close(con->socket);
                    con->socket = -1;
                }
                YiffCloseConnection(con_num);

                return(0);
            }


            /* Did we get the entire chunk? Is the indicated chunk
             * length greater than the contents buffer length?
             */
            if((YDataLength)chunk_length > (YDataLength)con->buf_cont)
		break;
/*
{
 printf("Server: %i %i %i: Recieved incomplete.\n",
  chunk_length, major_op_code, minor_op_code
 );
                break;
}
else
 printf("Server: %i %i | %i %i: Recieved.\n",
  chunk_length, con->buf_cont, major_op_code, minor_op_code
 );
 */

            /* Parse this segement and put data into event structure. */
            YNetParse(
                con_num,
                con->buf,
                chunk_length,
                major_op_code,
                minor_op_code
            );
	    segments_handled++;

	    /*   Stop parsing and close this connection? This would
	     *   be said if the connection needed to be closed for some
	     *   reason determined in a higher call to YNetParse()
	     *   above.
	     */
	    if(close_this_connection)
	    {
                /* Disconnect and delete connection. */
                if(con->socket > -1)
                {
                    close(con->socket);
                    con->socket = -1;
                }
                YiffCloseConnection(con_num);

		break;
	    }

            /* Shift the buffer. */
            for(i = 0, n = chunk_length;
                n < con->buf_cont;
                i++, n++
            )
                con->buf[i] = con->buf[n];

            /* Decrease buffer contents. */
            con->buf_cont = (YDataLength)con->buf_cont -
		(YDataLength)chunk_length;
            if(con->buf_cont < 0)
                con->buf_cont = 0;
        }

        /*   Return number of segments handled which implies events
         *   handled.
         */
        return(segments_handled);
}
