/*
 * english.c
 *
 * English-Chinese Input method
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "../im/impi.h"

#ifdef ENGLISH
char	imname[] = "Ӣ";
char	imdscrpt[] = "Ӣ뷨";
#endif

#ifdef MGM
char	imname[] = "";
char	imdscrpt[] = "뷨";
#endif

#ifdef GBKPY
char	imname[] = "ȫƴ";
char	imdscrpt[] = "GBKȫƴ뷨";
#endif

#define MAXNENTRIES				50000
#define MAXNCANDS				256

typedef struct
{
	char	*key;
	int		len;
	char	*cand;
}	ENTRY;

static ENTRY	dict[MAXNENTRIES];
static int		nentries;

static char		candword[MAXNCANDS][64];
static int		ncand;
static int		pagestart[MAXNCANDS];
static int		pageno;

static int	LoadDict(char filename[]);
static int	SearchDict(char key[]);
static void	reset();
static void	gen_candstr();
int isshengmu(char *s);

int	IMInit()
{
#ifdef ENGLISH
	char	*filename = "/usr/local/yh/lib/english.mb";
#endif

#ifdef MGM
	char	*filename = "/usr/local/yh/lib/mgm.mb";
#endif

#ifdef GBKPY
	char	*filename = "/usr/local/yh/lib/gbkpy.mb";
#endif

	maxeclen = 8;
	if (LoadDict(filename) < 0)
		return	-1;
	return	0;
}

int	IMClearup()
{
	return	0;
}

int	IMSelect(select)
int	select;
{
	return	0;
}

int	IMFilter(ch)
int		ch;
{
	int	searchcand;

	searchcand = 0;
	result[0] = '\0';
	candstr[0] = '\0';

	if (ch == ' ')
	{
		if (ncand > 0)
		{
			reset();
			strcpy(result, candword[pagestart[pageno]]);
		}
		else
		{
			reset();
			result[0] = ' ';
			result[1] = '\0';
		}
	}
	else if ((ncand > 0) && isdigit(ch))
	{
		int	i;

		i = pagestart[pageno] + ch - '0';
		if (i <= ncand)
		{
			reset();
			strcpy(result, candword[i]);
		}
		else
		{
			reset();
			result[0] = ch;
			result[1] = '\0';
		}
	}
	else if ((ncand > 0) && ((ch == '=') || (ch == '+')))
	{
		if (pagestart[pageno + 1] < ncand)
			pageno++;
	}
	else if ((ncand > 0) && (ch == '-'))
	{
		if (pageno)
			pageno--;
	}
	else if (((ch >= 'a') && (ch <= 'z')) || (ch == '\''))
	{
		searchcand = 1;
		if (eclen < maxeclen)
		{
			extcode[eclen++] = ch;
			extcode[eclen] = '\0';
		}
	}
	else if ((ch == KEY_BACKSPACE) && (eclen > 0))
	{
		searchcand = 1;
		extcode[--eclen] = '\0';
	}
	else
		handle_punct(ch);

	if (searchcand)
	{
		int	i, len;

		ncand = SearchDict(extcode);

		pagestart[pageno = 0] = 0;
		i = pagestart[pageno];
		do
		{
			len = 0;
			while ((i < ncand) && (len < 48))
			{
				len += strlen(candword[i++]) + 3;
			}
			pagestart[++pageno] = i;
		} while (i < ncand);

		pageno = 0;
	}

	gen_candstr();
	return	0;
}

static void	reset()
{
	ncand = 0;
	eclen = 0;
	extcode[0] = '\0';
	candstr[0] = '\0';
	result[0] = '\0';
}

static void	gen_candstr()
{
	int		i, j;
	char	*p;

	if (ncand == 0)
	{
		candstr[0] = '\0';
		return;
	}

	p = candstr;
	for (i = pagestart[pageno]; i < pagestart[pageno + 1]; i++)
	{
		if (i >= ncand)
			break;
		p += sprintf(p, " %d:%s", i - pagestart[pageno], candword[i]);
	}
}

static int	LoadDict(char filename[])
{
	FILE	*pf;
	char	key[16], buf[64], cand[512]; 
	int		pos;

	if ((pf = fopen(filename, "r")) == NULL)
		return	-1;

	nentries = 0;
	pos = 0;
	key[0] = '\0';
	fscanf(pf, "%s", key);
	while (nentries < MAXNENTRIES)
	{
		buf[0] = '\0';
		fscanf(pf, "%s", buf);

		if ((unsigned char) buf[0] < 128)
		{
			dict[nentries].key = strdup(key);
			dict[nentries].len = strlen(key);
			if (pos != 0)
			{
				dict[nentries].cand = malloc(pos + 1);
				memcpy(dict[nentries].cand, cand, pos + 1);
				pos = 0;
			}
			else
				dict[nentries].cand = "\0";
			nentries++;

			if (buf[0] == '\0')
				break;
						
			strcpy(key, buf);
			continue;
		}
		else
		{
			strcpy(&cand[pos], buf);
			pos += strlen(buf) + 1;
			cand[pos] = '\0';
		}
	}

	fclose(pf);
	return	0;
}

typedef unsigned char	*puchar;

int	is_gb(puchar p)
{
	return	((*p >= 0xa1) && (*p <= 0xfe) && (p[1] >= 0xa1) && (p[1] <= 0xfe));
}

int	cmpfn(puchar pa, puchar pb)
{
	if (is_gb(pa))
	{
		if (is_gb(pb))
			return	*pa - *pb;
		else
			return	-1;	
	}
	else
	{
		if (is_gb(pb))
			return	1;
		else
			return	*pa - *pb;	
	}
}

static int	SearchDict(char key[])
{
	register int	i;
	int				len;
	int				ncand;

	if ((len = strlen(key)) == 0)
		return	0;

	ncand = 0;
	for (i = 0; i < nentries; i++)
	{
		int	matched = 0;

#ifdef GBKPY
		if (isshengmu(key))
		{
			if ((len <= dict[i].len) && 
				(memcmp(key, dict[i].key, len) == 0))
				matched = 1;
		}
		else
		{
			if (strcmp(key, dict[i].key) == 0)
				matched = 1;
		}
#else
		if ((len <= dict[i].len) && 
			(memcmp(key, dict[i].key, len) == 0))
			matched = 1;
#endif

		if (matched)
		{
			int		pos;
			char	*p;

			pos = 0;
			while (ncand < MAXNCANDS)
			{
				p = &dict[i].cand[pos];
				if (*p == '\0')
					break;
				strcpy(candword[ncand++], p);
				pos += strlen(p) + 1;
			}
		}
	}

	qsort(candword, ncand, 64, (int (*)(const void *, const void *)) cmpfn); 	
	return	ncand;
}

int isshengmu(char *s)
{
	static char	*shengmutab[] =
	{
		"ch", "sh", "zh", "c", "d", "f", "g", "h", "j", "k", "l", "m",
		"n", "p", "q", "r", "s", "t", "w", "x", "y", "z", NULL
	};
	int	rv = 0;
	int	i;

	for (i = 0; shengmutab[i] != NULL; i++)
		if (strcmp(s, shengmutab[i]) == 0)
		{
			rv = 1;
			break;
		}

	return	rv;
}
