/* $Id: map.cpp,v 1.3 2003/07/02 20:05:45 flaterco Exp $ */

/*  Map class.  */

/*****************************************************************************\

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

\*****************************************************************************/

#include "map.h"


Map::Map (QWidget * parent, const char *name):
  QWidget (parent, name, WNorthWestGravity),
  pen (Qt::red, 1),
  polyline (3),
  buffer (width (), height ())
{
    //  Track mouse movement with no button pressed.

    QWidget::setMouseTracking (TRUE);


    if ((qApp->argc () > 0) && !buffer.load (qApp->argv ()[1]))
        buffer.fill (colorGroup ().base ());
    setBackgroundMode (QWidget::PaletteBase);

#ifndef QT_NO_CURSOR
    setCursor (Qt::crossCursor);
#endif
}



void
Map::clearScreen ()
{
    buffer.fill (colorGroup ().base ());
    repaint (FALSE);
}



void
Map::setZoomLevel (int level)
{
    //  Negative number zooms out one level.

    if (level < 0)
    {
        if (zoom_level > 0) zoom_level += level;
    }


    //  Otherwise, go to the specified level (usually 0).

    else
    {
        if (zoom_level != level) zoom_level = level;
    }

    redrawMap ();
}



void
Map::setExtents (int zoom, double slat, double nlat, double wlon, double elon)
{
    //  Set the boundaries.  This doesn't necessarily have to be the entire 
    //  world.

    zoom_level = zoom;

    bounds[zoom].slat = slat;
    bounds[zoom].nlat = nlat;

    bounds[zoom].wlon = wlon;
    bounds[zoom].elon = elon;

    if (bounds[zoom].wlon >= bounds[zoom].elon)
        bounds[zoom].elon += 360.0;

    range = bounds[zoom].elon - bounds[zoom].wlon;

    center_longitude = bounds[zoom].wlon + range / 2.0;


    //  Arbitrarily dividing by 12.  This just happens to be the same as Dave
    //  Flater's 30 degree split in XTide (what a coincidence).

    if (!zoom) move_inc = range / 12.0;
}



int 
Map::getExtents (double *slat, double *nlat, double *wlon, double *elon,
    double *full_slat, double *full_nlat, double *full_wlon,
    double *full_elon)
{
    //  The original extents, not zoomed in.

    *full_slat = bounds[0].slat;
    *full_nlat = bounds[0].nlat;
    *full_wlon = bounds[0].wlon;
    *full_elon = bounds[0].elon;


    //  The current extents.

    *slat = bounds[zoom_level].slat;
    *nlat = bounds[zoom_level].nlat;
    *wlon = bounds[zoom_level].wlon;
    *elon = bounds[zoom_level].elon;

    return (zoom_level);
}



void
Map::drawLine (int x0, int y0, int x1, int y1)
{
    QPainter painter;

    painter.begin (&buffer);
    painter.setPen (Qt::black);
    polyline.setPoints (2, x0, y0, x1, y1);
    painter.drawPolyline (polyline);

    QRect r = polyline.boundingRect ();

    r = r.normalize ();
    r.setLeft (r.left () - 1);
    r.setTop (r.top () - 1);
    r.setRight (r.right () + 1);
    r.setBottom (r.bottom () + 1);

    bitBlt (this, r.x (), r.y (), &buffer, r.x (), r.y (), r.width (),
        r.height ());
}



void Map::setRedrawZeroLevel ()
{
    redraw_zero_level = 1;
}



void
Map::drawCircle (int x, int y, int h, int w, int r, QColor color)
{
    QPainter painter;
    QBrush b1;

    painter.begin (&buffer);

    b1.setStyle (SolidPattern);
    b1.setColor (color);
    painter.setPen (color);

    painter.setBrush (b1);
    painter.drawChord (x, y, h, w, 0, r * 16);

    bitBlt (this, x, y , &buffer, x, y, w, h);

    painter.end ();
}



void
Map::drawRectangle (int x, int y, int h, int w)
{
    QPainter painter;
    QBrush b1;

    painter.begin (&buffer);

    b1.setStyle (SolidPattern);
    b1.setColor (Qt::blue);
    painter.setPen (Qt::blue);

    painter.setBrush (b1);
    painter.drawRect (x, y, h, w);

    bitBlt (this, x, y , &buffer, x, y, w, h);

    painter.end ();
}



void
Map::mousePressEvent (QMouseEvent * e)
{
    if (e->button () == LeftButton)
    {
        if (zoom_level < MAX_ZOOM_LEVEL)
        {
            double lat, lon;

            untranslateCoordinates (&lat, &lon, e->x (), e->y ());

            double xsize = (bounds[zoom_level].elon -
                bounds[zoom_level].wlon) / 4.0;
            double ysize = (bounds[zoom_level].nlat -
                bounds[zoom_level].slat) / 4.0;


            //  If this is the last zoom let's be a bit more fine-grained.

            if (zoom_level == (MAX_ZOOM_LEVEL - 1))
            {
                xsize *= 2.0;
                ysize *= 2.0;
            }


            //  Increment the zoom level and compute the new bounds.

            zoom_level++;


            bounds[zoom_level].wlon = lon - xsize / 2.0;
            bounds[zoom_level].elon = lon + xsize / 2.0;

            if (bounds[zoom_level].wlon < -180.0)
                bounds[zoom_level].wlon += 360.0;

            if (bounds[zoom_level].elon > 180.0)
                bounds[zoom_level].elon -= 360;

            if (bounds[zoom_level].wlon >= bounds[zoom_level].elon)
                bounds[zoom_level].elon += 360.0;


            bounds[zoom_level].slat = lat - ysize / 2.0;
            bounds[zoom_level].nlat = lat + ysize / 2.0;

            if (bounds[zoom_level].slat < bounds[0].slat)
            {
                bounds[zoom_level].slat = bounds[0].slat;
                bounds[zoom_level].nlat = bounds[zoom_level].slat + ysize;
            }

            if (bounds[zoom_level].nlat > bounds[0].nlat)
            {
                bounds[zoom_level].nlat = bounds[0].nlat;
                bounds[zoom_level].slat = bounds[zoom_level].nlat - ysize;
            }

            redrawMap ();
        }
        emit leftMouseSignal (e);
    }
    else if (e->button () == MidButton)
    {
        //  Let the parent do something.

        emit midMouseSignal (e);
    }
    else
    {
        //  Let the parent do something.

        emit rightMouseSignal (e);
    }
}



void
Map::moveMap (double center_lat, double center_lon)
{
    double xsize = (bounds[zoom_level].elon - bounds[zoom_level].wlon);
    double ysize = (bounds[zoom_level].nlat - bounds[zoom_level].slat);


    bounds[zoom_level].wlon = center_lon - xsize /2.0;
    bounds[zoom_level].elon = center_lon + xsize / 2.0;

    if (bounds[zoom_level].wlon < -180.0)
    {
	bounds[zoom_level].wlon += 360.0;
	bounds[zoom_level].elon = bounds[zoom_level].wlon + xsize;
    }
    if (bounds[zoom_level].elon > 180.0)
    {
	bounds[zoom_level].elon -= 360.0;
	bounds[zoom_level].wlon = bounds[zoom_level].elon - xsize;
    }



    bounds[zoom_level].slat = center_lat - ysize / 2.0;
    bounds[zoom_level].nlat = center_lat + ysize / 2.0;

    if (bounds[zoom_level].nlat > bounds[0].nlat)
    {
	bounds[zoom_level].nlat = bounds[0].nlat;
	bounds[zoom_level].slat = bounds[zoom_level].nlat - ysize;
    }
    if (bounds[zoom_level].slat < bounds[0].slat)
    {
	bounds[zoom_level].slat = bounds[0].slat;
	bounds[zoom_level].nlat = bounds[zoom_level].slat + ysize;
    }

    redrawMap ();
}



//  Move in some direction.  You can move west or east when fully zoomed out,
//  but not north or south.

void
Map::moveMap (int direction)
{
    double xsize = (bounds[zoom_level].elon - bounds[zoom_level].wlon);
    double ysize = (bounds[zoom_level].nlat - bounds[zoom_level].slat);


    //  Notice the 5% overlap.

    switch (direction)
    {
        case MAP_LEFT:
            if (zoom_level)
            {
                bounds[zoom_level].wlon = bounds[zoom_level].wlon - xsize *
                    0.95;
                bounds[zoom_level].elon = bounds[zoom_level].elon - xsize *
                    0.95;

                if (bounds[zoom_level].wlon < -180.0)
                {
                    bounds[zoom_level].wlon += 360.0;
                    bounds[zoom_level].elon = bounds[zoom_level].wlon + xsize;
                }
            }
            else
            {
                center_longitude -= move_inc;
                if (center_longitude < -180.0)
                    center_longitude += 360.0;
                bounds[0].wlon = center_longitude - range / 2.0;
                bounds[0].elon = center_longitude + range / 2.0;

                if (bounds[0].wlon < -180.0)
                    bounds[0].wlon += 360.0;
                if (bounds[0].elon > 180.0)
                    bounds[0].elon -= 360.0;

                if (bounds[0].wlon >= bounds[0].elon)
                    bounds[0].elon += 360.0;

                redraw_zero_level = 1;
            }
            break;

        case MAP_UP:
            if (zoom_level)
            {
                bounds[zoom_level].slat = bounds[zoom_level].slat + ysize *
                    0.95;
                bounds[zoom_level].nlat = bounds[zoom_level].nlat + ysize *
                    0.95;

                if (bounds[zoom_level].nlat > bounds[0].nlat)
                {
                    bounds[zoom_level].nlat = bounds[0].nlat;
                    bounds[zoom_level].slat = bounds[zoom_level].nlat - ysize;
                }
            }
            break;

        case MAP_RIGHT:
            if (zoom_level)
            {
                bounds[zoom_level].wlon = bounds[zoom_level].wlon + xsize *
                    0.95;
                bounds[zoom_level].elon = bounds[zoom_level].elon + xsize *
                    0.95;

                if (bounds[zoom_level].elon > 180.0)
                {
                    bounds[zoom_level].elon -= 360.0;
                    bounds[zoom_level].wlon = bounds[zoom_level].elon - xsize;
                }
            }
            else
            {
                center_longitude += move_inc;
                if (center_longitude > 180.0)
                    center_longitude -= 360.0;
                bounds[0].wlon = center_longitude - range / 2.0;
                bounds[0].elon = center_longitude + range / 2.0;

                if (bounds[0].wlon < -180.0)
                    bounds[0].wlon += 360.0;
                if (bounds[0].elon > 180.0)
                    bounds[0].elon -= 360.0;

                if (bounds[0].wlon >= bounds[0].elon)
                    bounds[0].elon += 360.0;

                redraw_zero_level = 1;
            }
            break;

        case MAP_DOWN:
            if (zoom_level)
            {
                bounds[zoom_level].slat = bounds[zoom_level].slat - ysize *
                    0.95;
                bounds[zoom_level].nlat = bounds[zoom_level].nlat - ysize *
                    0.95;

                if (bounds[zoom_level].slat < bounds[0].slat)
                {
                    bounds[zoom_level].slat = bounds[0].slat;
                    bounds[zoom_level].nlat = bounds[zoom_level].slat + ysize;
                }
            }
            break;

        default:
            break;
    }

    redrawMap ();
}



void
Map::mouseReleaseEvent (QMouseEvent *)
{
    //  Nothing here yet.
}



void
Map::mouseMoveEvent (QMouseEvent * e)
{
    last_x = e->x ();
    last_y = e->y ();


    //  Let the parent do something.

    emit moveMouseSignal (e);
}



void
Map::resizeEvent (QResizeEvent * e)
{
    static int ow, oh;

    QWidget::resizeEvent (e);

    int w = width () > buffer.width ()? width () : buffer.width ();
    int h = height () > buffer.height ()? height () : buffer.height ();


    //  Getting resize events from somewhere even when it ain't.

    if (ow != w || oh != h) redraw_zero_level = 1;

    ow = w;
    oh = h;


    QPixmap tmp (buffer);

    buffer.resize (w, h);
    buffer.fill (colorGroup ().base ());
    bitBlt (&buffer, 0, 0, &tmp, 0, 0, tmp.width (), tmp.height ());

    savemap.resize (w, h);
    bitBlt (&savemap, 0, 0, &tmp, 0, 0, tmp.width (), tmp.height ());

    draw_width = w - BORDER * 2;
    draw_height = h - (BORDER * 2);
}



void
Map::paintEvent (QPaintEvent * e)
{
    //  This is the equivalent of the X expose event.

    QWidget::paintEvent (e);

    QArray < QRect > rects = e->region ().rects ();
    for (uint i = 0; i < rects.count (); i++)
    {
        QRect r = rects[(int) i];

        bitBlt (this, r.x (), r.y (), &buffer, r.x (), r.y (), r.width (),
            r.height ());
    }
}



//  Convert from position to pixels.  Returns 0 if outside of viewable area.

int
Map::translateCoordinates (double lat, double lon, int *x, int *y)
{
    double lng = lon;

    if (bounds[zoom_level].elon > 180.0 && lng < bounds[zoom_level].wlon)
        lng += 360.0;
    if (bounds[zoom_level].wlon < -180.0 && lng > bounds[zoom_level].elon)
        lng -= 360.0;

    if (lat < bounds[zoom_level].slat || lat > bounds[zoom_level].nlat ||
        lng < bounds[zoom_level].wlon || lng > bounds[zoom_level].elon)
        return (0);

    *x = BORDER + (int) (((lng - bounds[zoom_level].wlon) /
        (bounds[zoom_level].elon - bounds[zoom_level].wlon)) * draw_width);
    *y = BORDER + (int) (((bounds[zoom_level].nlat -
        lat) / (bounds[zoom_level].nlat - bounds[zoom_level].slat)) * 
        draw_height);

    return (1);
}



//  Convert from pixels to position.

void
Map::untranslateCoordinates (double *lat, double *lon, int x, int y)
{
    *lon = bounds[zoom_level].wlon + ((double) (x - BORDER) /
        (double) (draw_width - BORDER)) * (bounds[zoom_level].elon -
        bounds[zoom_level].wlon);
    *lat = bounds[zoom_level].nlat - ((double) (y - BORDER) /
        (double) (draw_height - BORDER)) * (bounds[zoom_level].nlat -
        bounds[zoom_level].slat);
}




//  Draw the map.

void
Map::redrawMap ()
{
    float *latray, *lonray;
    int *segray, coast;
    static char files[6][12] = { "wvsfull.dat", "wvs250k.dat", "wvs1.dat",
        "wvs3.dat", "wvs12.dat", "wvs43.dat"
    };

    int wvsrtv (char *, int, int, float **, float **, int **);


    //  Set the watch cursor.

    setCursor (waitCursor);


    clearScreen ();


    int startlat = (int) (bounds[zoom_level].slat - 1.0);
    int endlat = (int) (bounds[zoom_level].nlat + 1.0);
    int startlon = (int) (bounds[zoom_level].wlon - 1.0);
    int endlon = (int) (bounds[zoom_level].elon + 1.0);


    latray = (float *) NULL;
    lonray = (float *) NULL;
    segray = (int *) NULL;


    int lat, lon, segx[2], segy[2];


    //  Get the proper coastline data set based on the zoom level.

    switch (zoom_level)
    {
        case 0:
        default:
            coast = 5;
            break;

        case 1:
            coast = 4;
            break;

        case 2:
            coast = 2;
            break;

        case 3:
            coast = 2;
            break;

        case 4:
            coast = 1;
            break;

        case 5:
            coast = 1;
            break;
    }


    //  Read and plot the coastlines.

    if (redraw_zero_level || zoom_level)
    {
        for (lat = startlat; lat < endlat; lat++)
        {
            for (lon = startlon; lon < endlon; lon++)
            {
                int k, nseg, offset;

                offset = 0;

                nseg = wvsrtv (files[coast], lat, lon, &latray, &lonray,
                    &segray);

                if (nseg)
                {
                  //  Get rid of single point islands that were required for 
                  //  NAVO.  These were created during the decimation
                  //  process.

                    if (nseg > 2 || latray[0] != latray[1] ||
                        lonray[0] != lonray[1])
                    {
                        for (k = 0; k < nseg; k++)
                        {
                            int cnt, m;

                            m = 0;
                            for (cnt = 0; cnt < segray[k]; cnt++)
                            {
                                if (translateCoordinates (
                                        (double) latray[offset + cnt],
                                        (double) lonray[offset + cnt],
                                        &segx[m], &segy[m]))
                                {
                                    //  Check for the weird situation when
                                    //  west and east are at 0.0 and 360.0.

                                    if (m && segx[1] - segx[0] < 50 &&
                                        segx[0] - segx[1] < 50)
                                    {
                                        drawLine (segx[0], segy[0],
                                            segx[1], segy[1]);

                                        segx[0] = segx[1];
                                        segy[0] = segy[1];
                                    }
                                    m = 1;
                                }
                                else
                                {
                                    m = 0;
                                }
                            }
                            offset += segray[k];
                        }
                    }
                }
            }
        }

        wvsrtv ("clean", 0, 0, &latray, &lonray, &segray);


        //  Draw the grid lines.

        for (lon = startlon; lon < endlon; lon++)
        {
            if (!(lon % (int) move_inc) && lon > bounds[zoom_level].wlon &&
                lon < bounds[zoom_level].elon)
            {
                if (translateCoordinates (bounds[zoom_level].slat,
                        (double) lon, &segx[0], &segy[0]))
                {
                    translateCoordinates (bounds[zoom_level].nlat,
                        (double) lon, &segx[1], &segy[1]);

                    drawLine (segx[0], segy[0], segx[1], segy[1]);
                }
            }
        }


        for (lat = startlat; lat < endlat; lat++)
        {
            if (!(lat % (int) move_inc) && lat > bounds[0].slat &&
                lat < bounds[0].nlat)
            {
                if (translateCoordinates ((double) lat,
                        bounds[zoom_level].wlon, &segx[0], &segy[0]))
                {
                    translateCoordinates ((double) lat,
                        bounds[zoom_level].elon, &segx[1], &segy[1]);

                    drawLine (segx[0], segy[0], segx[1], segy[1]);
                }
            }
        }


        //  The first time through we want to save the map so we don't have to
        //  read the file and wait to draw the top zoom level.

        if (redraw_zero_level && !zoom_level)
        {
            redraw_zero_level = 0;
            bitBlt (&savemap, 0, 0, &buffer, 0, 0, buffer.width (),
                buffer.height ());
        }
    }
    else
    {
        //  Paint the original pixmap and save it in buffer for expose events

        bitBlt (&buffer, 0, 0, &savemap, 0, 0, savemap.width (),
            savemap.height ());
        bitBlt (this, 0, 0, &savemap, 0, 0, savemap.width (),
            savemap.height ());
    }


    //  Let the parent do something.

    emit redrawSignal ();


    //  Go back to the arrow cursor.

    setCursor (arrowCursor);
}
