// $Id: xxTextMode.cc,v 1.4 2003/01/17 18:33:00 flaterco Exp $
/*  xxTextMode  Plain old text mode, in a window.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "xtide.hh"

void
xxTextMode::dismiss () {
  delete this; // Is this safe?
}

void
xxTextMode::help() {
  Dstr helpstring;

  if (doneness == 2)
    helpstring = "\
XTide Text Mode Window\n\
\n\
Use the forward and backward buttons to scroll the tide predictions\n\
forward or backward in time.  You can resize the window vertically\n\
to see more predictions at once.\n\
\n\
The options menu supplies the following commands:\n\
\n\
  Save:  Use this to write the tide predictions to a text file.\n\
\n\
  Set Mark:  This is used to set the 'mark level' for tide predictions.\n\
  The prediction window will show the times when the tide level crosses\n\
  the mark.  Some subordinate stations don't have this option.\n\
\n\
  Convert ft<->m:  Convert feet to meters or vice-versa.\n\
\n\
  Set Time:  Go to a different point in time.  Major adjustments that\n\
  cannot be made with the forward and backward buttons can be made with\n\
  Set Time.\n\
\n\
  New Graph Window:  Open a window with a tide graph for the current\n\
  location and time.\n\
\n\
  New Text Window:  Duplicates the existing text window.\n\
\n\
  New Raw Mode Window:  Open a window with a text listing of unprocessed\n\
  numerical time stamps and tide levels for the current location and time.\n\
\n\
  New Medium Rare Mode Window:  Open a window with a text listing of\n\
  processed timestamps and unprocessed numerical tide levels for the\n\
  current location and time.\n\
\n\
  New Clock Window:  Open a window with a tide clock for the current\n\
  location.\n\
\n\
  New Location Chooser:  Open new globe and location list windows to allow\n\
  selecting a new location.";
  else {

  if (doneness)
    helpstring = "XTide Medium Rare Mode Window\n";
  else
    helpstring = "XTide Raw Mode Window\n";
  helpstring += "\n\
Use the forward and backward buttons to scroll the tide predictions\n\
forward or backward in time.  You can resize the window vertically\n\
to see more predictions at once.\n\
\n\
The options menu supplies the following commands:\n\
\n\
  Save:  Use this to send output to a text file.  In this mode you are\n\
  given the opportunity to generate output for a different time interval\n\
  than is displayed on the screen.\n\
\n\
  Convert ft<->m:  Convert feet to meters or vice-versa.\n\
\n\
  Set Time:  Go to a different point in time.  Major adjustments that\n\
  cannot be made with the forward and backward buttons can be made with\n\
  Set Time.\n\
\n\
  Set Step:  Change the time increment of the output.\n\
\n\
  New Graph Window:  Open a window with a tide graph for the current\n\
  location and time.\n\
\n\
  New Text Window:  Open a window with a text listing of tide predictions\n\
  for the current location and time.\n\
\n";

  if (doneness)
    helpstring += "\
  New Raw Mode Window:  Open a window with a text listing of unprocessed\n\
  numerical time stamps and tide levels for the current location and time.\n\
\n\
  New Medium Rare Mode Window:  Duplicates the existing medium rare mode\n\
  window.\n";
  else
    helpstring += "\
  New Raw Mode Window:  Duplicates the existing raw mode window.\n\
\n\
  New Medium Rare Mode Window:  Open a window with a text listing of\n\
  processed timestamps and unprocessed numerical tide levels for the\n\
  current location and time.\n";

  helpstring += "\n\
  New Clock Window:  Open a window with a tide clock for the current\n\
  location.\n\
\n\
  New Location Chooser:  Open new globe and location list windows to allow\n\
  selecting a new location.";
  }
  xtidecontext->root->newHelpBox (helpstring);
}

void xxTextMode::predictexact (Station::Direction dir, struct tideline *tl) {
  xtidecontext->predictText (station, dir, tl->tm, tl->line, tl->etype);
}

void
xxTextModeforwardCallback (Widget w, XtPointer client_data,
XtPointer call_data) {
  xxTextMode *txt = (xxTextMode *)client_data;
  Timestamp tm = txt->lastline->tm;
  Station::EventType avoid_etype = txt->lastline->etype;

  // Move the first one to the end
  struct xxTextMode::tideline *tl = txt->firstline;
  txt->firstline = tl->next;
  txt->lastline->next = tl;
  txt->lastline = tl;
  tl->next = NULL;

  // Change its contents

  if (txt->doneness == 2) {
    tl->tm = tm;
    txt->predictexact (Station::forward, tl);
    // Avoid repetition
    if (tl->etype == avoid_etype)
      txt->predictexact (Station::forward, tl);
  } else {
    tm += txt->station->step;
    tl->tm = tm;
    txt->xtidecontext->rareModes (txt->station, tm, tm, txt->station->step, tl->line, txt->doneness);
    tl->line -= tl->line.length() - 1;
  }

  txt->draw();
}

void
xxTextModebackwardCallback (Widget w, XtPointer client_data,
XtPointer call_data) {
  xxTextMode *txt = (xxTextMode *)client_data;
  Timestamp tm = txt->firstline->tm;
  Station::EventType avoid_etype = txt->firstline->etype;

  // Move the last one to the start
  assert (txt->firstline != txt->lastline);
  struct xxTextMode::tideline *tl = txt->lastline;
  struct xxTextMode::tideline *temp2 = txt->firstline;
  while (temp2->next != tl)
    temp2 = temp2->next;
  temp2->next = NULL;
  txt->lastline = temp2;
  tl->next = txt->firstline;
  txt->firstline = tl;

  // Change its contents

  if (txt->doneness == 2) {
    tl->tm = tm;
    txt->predictexact (Station::backward, tl);
    // Avoid repetition
    if (tl->etype == avoid_etype)
      txt->predictexact (Station::backward, tl);
  } else {
    tm -= txt->station->step;
    tl->tm = tm;
    txt->xtidecontext->rareModes (txt->station, tm, tm, txt->station->step, tl->line, txt->doneness);
    tl->line -= tl->line.length() - 1;
  }

  txt->draw();
}

void
xxTextModeResizeHandler (Widget w, XtPointer client_data,
    XEvent *event, Boolean *continue_dispatch) {
  xxTextMode *txt = (xxTextMode *)client_data;
  switch (event->type) {
  case ConfigureNotify:
    {
      XConfigureEvent *ev = (XConfigureEvent *)event;
      Dimension newheight = ev->height;
      if (newheight != txt->height) {
	txt->height = newheight;
	// Extremely kludgy guesstimate of how many lines we should have now.
	int t = (int)((double)(newheight - (txt->origheight -
	  txt->origlabelheight)) * (double)origtextmodelines /
	  (double)(txt->origlabelheight));
	if (t < 1)
	  txt->lines = 1;
	else
	  txt->lines = t;
	txt->redraw();
      }
    }
    break;
  default:
    ;
  }
}

void
xxRareModeSaveCallback (Dstr &filename, Timestamp start_tm, Timestamp end_tm,
void *in_ptr) {
  xxTextMode *txt = (xxTextMode *)in_ptr;
  FILE *fp;
  if ((fp = fopen (filename.aschar(), "w"))) {
    Dstr text;
    txt->xtidecontext->rareModes (txt->station,
      start_tm, end_tm, txt->station->step, text, txt->doneness);
    fprintf (fp, "%s", text.aschar());
    fclose (fp);
  } else {
    Dstr details (filename);
    details += ": ";
    details += strerror (errno);
    details += ".";
    barf (CANT_OPEN_FILE, details, 0);
  }
}

void
xxTextModeSaveCallback (Dstr &filename, void *in_ptr) {
  xxTextMode *txt = (xxTextMode *)in_ptr;
  FILE *fp;
  if ((fp = fopen (filename.aschar(), "w"))) {
    Dstr text;
    txt->xtidecontext->textMode (txt->station,
      txt->t, txt->lastline->tm, text);
    fprintf (fp, "%s", text.aschar());
    fclose (fp);
  } else {
    Dstr details (filename);
    details += ": ";
    details += strerror (errno);
    details += ".";
    barf (CANT_OPEN_FILE, details, 0);
  }
}

void
xxTextMode::save () {
  if (doneness == 2)
    (void) new xxFilename (xtidecontext, mypopup, xxTextModeSaveCallback, this, "tides.txt");
  else
    (void) new xxRareModeSavePrompts (xtidecontext, mypopup, xxRareModeSaveCallback, this, "tides.txt", firstline->tm, lastline->tm, station->timeZone);
}

xxTextMode::xxTextMode (xxTideContext *in_tidecontext,
xxContext *in_xxcontext, Station *in_station, int doneness_in): xxDrawable (in_tidecontext,
in_xxcontext, in_station, 2) {
  doneness = doneness_in;
  t = Timestamp((time_t)(time(NULL)));
  construct();
}

xxTextMode::xxTextMode (xxTideContext *in_tidecontext,
xxContext *in_xxcontext, Station *in_station, int doneness_in, Timestamp t_in):
xxDrawable (in_tidecontext, in_xxcontext, in_station, 2) {
  doneness = doneness_in;
  t = t_in;
  construct();
}

void xxTextMode::construct () {
  lines = origtextmodelines;
  Dstr title (station->name);
  switch (doneness) {
  case 0:
    title += " (Raw)";
    break;
  case 1:
    title += " (Medium Rare)";
    break;
  case 2:
    title += " (Text)";
    break;
  default:
    assert (0);
  }
  mypopup->setTitle (title);
  XtAddEventHandler (mypopup->manager, StructureNotifyMask, False,
		     xxTextModeResizeHandler, (XtPointer)this);

  {
    Arg labelargs[6] =  {
      {XtNbackground, (XtArgVal)mypopup->pixels[Colors::background]},
      {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]},
      {XtNleft, (XtArgVal)XawChainLeft},
      {XtNright, (XtArgVal)XawChainLeft},
      {XtNtop, (XtArgVal)XawChainTop},
      {XtNbottom, (XtArgVal)XawChainTop}
    };
    Widget labelwidget = XtCreateManagedWidget (station->name.aschar(),
      labelWidgetClass, container->manager, labelargs, 6);
    namelabel = new xxContext (mypopup, labelwidget);
  }
  {
    Arg labelargs[7] =  {
      {XtNbackground, (XtArgVal)mypopup->pixels[Colors::background]},
      {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]},
      {XtNfromVert, (XtArgVal)namelabel->manager},
      {XtNleft, (XtArgVal)XawChainLeft},
      {XtNright, (XtArgVal)XawChainRight},
      {XtNtop, (XtArgVal)XawChainTop},
      {XtNbottom, (XtArgVal)XawChainBottom}
    };
    Widget labelwidget = XtCreateManagedWidget ("",
      labelWidgetClass, container->manager, labelargs, 7);
    label = new xxContext (mypopup, labelwidget);
  }
  {
    Arg buttonargs[7] =  {
      {XtNbackground, (XtArgVal)mypopup->pixels[Colors::button]},
      {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]},
      {XtNfromVert, (XtArgVal)label->manager},
      {XtNleft, (XtArgVal)XawChainLeft},
      {XtNright, (XtArgVal)XawChainLeft},
      {XtNtop, (XtArgVal)XawChainBottom},
      {XtNbottom, (XtArgVal)XawChainBottom}
    };
    Widget buttonwidget = XtCreateManagedWidget ("Backward", repeaterWidgetClass,
      container->manager, buttonargs, 7);
    XtAddCallback (buttonwidget, XtNcallback, xxTextModebackwardCallback,
     (XtPointer)this);
    backwardbutton = new xxContext (mypopup, buttonwidget);
  }
  {
    Arg buttonargs[8] =  {
      {XtNbackground, (XtArgVal)mypopup->pixels[Colors::button]},
      {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]},
      {XtNfromVert, (XtArgVal)label->manager},
      {XtNfromHoriz, (XtArgVal)backwardbutton->manager},
      {XtNleft, (XtArgVal)XawChainLeft},
      {XtNright, (XtArgVal)XawChainLeft},
      {XtNtop, (XtArgVal)XawChainBottom},
      {XtNbottom, (XtArgVal)XawChainBottom}
    };
    Widget buttonwidget = XtCreateManagedWidget ("Forward", repeaterWidgetClass,
      container->manager, buttonargs, 8);
    XtAddCallback (buttonwidget, XtNcallback, xxTextModeforwardCallback,
     (XtPointer)this);
    forwardbutton = new xxContext (mypopup, buttonwidget);
  }

  addNormalButtons (label->manager, forwardbutton->manager);

  firstline = lastline = NULL;
  redraw();
  mypopup->realize();
  mypopup->widthNoSmaller();

  {
    Arg args[1] = {
      {XtNheight, (XtArgVal)(&origheight)}
    };
    XtGetValues (container->manager, args, 1);
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
    cerr << "Text window original height " << origheight << endl;
#endif
    height = origheight;
  }
  {
    Dimension internalheight;
    Arg args[2] = {
      {XtNheight, (XtArgVal)(&origlabelheight)},
      {XtNinternalHeight, (XtArgVal)(&internalheight)}
    };
    XtGetValues (label->manager, args, 2);
    origlabelheight -= internalheight * 2;
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
    cerr << "Text label original height " << origlabelheight << endl;
#endif
  }
}

void xxTextMode::redraw() {
  Timestamp tm = t;

  clearbuf();

  unsigned a;
  for (a=0; a<lines; a++) {
    struct tideline *tl = new tideline;
    tl->tm = tm;

    if (doneness == 2) {
      predictexact (Station::forward, tl);
      tm = tl->tm;
    } else {
      xtidecontext->rareModes (station, tm, tm, station->step, tl->line, doneness);
      tl->line -= tl->line.length() - 1;
      tm += station->step;
    }

    tl->next = NULL;
    if (!a) {
      firstline = lastline = tl;
    } else {
      lastline->next = tl;
      lastline = tl;
    }
  }

  // Was having major trouble getting the label widget to resize itself
  // reliably.  This seems to help.
  {
    int heightguess = height - (origheight - origlabelheight);
    if (heightguess < 1)
      heightguess = 1;
    Arg labelargs[1] =  {
      {XtNheight, (XtArgVal)heightguess}
    };
    XtSetValues (label->manager, labelargs, 1);
  }

  draw();
}

void xxTextMode::draw () {
  Dstr tidetext;
  struct tideline *tl = firstline;
  while (tl) {
    tidetext += tl->line;
    tl = tl->next;
    if (tl)
      tidetext += "\n";
  }
  {
    Arg labelargs[1] =  {
      {XtNlabel, (XtArgVal)tidetext.aschar()}
    };
    XtSetValues (label->manager, labelargs, 1);
  }

  // Keep start time updated.
  if (doneness == 2)
    t = firstline->tm - Interval(60);
  else
    t = firstline->tm;
}

void
xxTextMode::clearbuf() {
  while (firstline) {
    struct tideline *tl = firstline->next;
    delete firstline;
    firstline = tl;
  }
  firstline = lastline = NULL;
}

xxTextMode::~xxTextMode() {
  mypopup->unrealize();
  delete label;
  delete namelabel;
  delete backwardbutton;
  delete forwardbutton;
  clearbuf();
}

void xxTextMode::global_redraw() {
  xxDrawable::global_redraw();
  redraw();
  Arg buttonargs[2] =  {
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };
  if (backwardbutton)
    XtSetValues (backwardbutton->manager, buttonargs, 2);
  if (forwardbutton)
    XtSetValues (forwardbutton->manager, buttonargs, 2);
  Arg args[2] =  {
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::background]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };
  if (label)
    XtSetValues (label->manager, args, 2);
  if (namelabel)
    XtSetValues (namelabel->manager, args, 2);
}

int xxTextMode::is_rare() {
  if (doneness < 2)
    return 1;
  return 0;
}
