// 	$Id: TideContext.cc,v 1.2 1999/12/05 20:12:17 dave Exp $	
/*  TideContext  In lieu of global variables and functions.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

void version_string (Dstr &text_out) {
  text_out = "XTide ";
  text_out += VERSION;
  if (PATCHLEVEL) {
    text_out += ".";
    text_out += PATCHLEVEL;
  }
#ifdef VSTATUS
  text_out += " ";
  text_out += VSTATUS;
#endif
}

TideContext::TideContext (int in_argc, char **in_argv, Colors *in_colors,
Settings *in_settings) {

  // It's unfortunate that this gets done twice.
  cls = new CommandLineSettings(in_argc, in_argv);

  colors = in_colors;
  settings = in_settings;

  // Get HFILE_PATH from environment or /etc/xtide.conf and create
  // persistent HarmonicsPath
  Dstr hfile_path (getenv ("HFILE_PATH"));
  if (hfile_path.isNull()) {
    FILE *configfile;
    if ((configfile = fopen ("/etc/xtide.conf", "r"))) {
      hfile_path.getline (configfile);
      fclose (configfile);
    }
  }
  if (hfile_path.isNull())
    hfile_path = "harmonics";
  harmonicsPath = new HarmonicsPath (hfile_path);

  homedir = getenv ("HOME");
  if (homedir.isNull())
    barf (NOHOMEDIR);

  disabledisclaimerfile = homedir;
  disabledisclaimerfile += "/.disableXTidedisclaimer";

  // Get this only when needed.
  myStationIndex = NULL;
}

TideContext::~TideContext () {
  if (myStationIndex)
    delete myStationIndex;
  delete harmonicsPath;
  delete cls;
  delete colors;
  delete settings;
}

// THIS CODE IS DUPLICATED IN XXTIDECONTEXT!  The duplication is needed
// to embed XEvent polling in xxTideContext.
StationIndex *TideContext::stationIndex () {
  if (!myStationIndex) {
    myStationIndex = new StationIndex();
    for (unsigned i=0; i<harmonicsPath->length(); i++) {
      Dstr msg ("Indexing ");
      msg += (*harmonicsPath)[i];
      msg += "...";
      log (msg, LOG_NOTICE);
      HarmonicsFile h ((*harmonicsPath)[i], settings);
      StationRef *s;
      while ((s = h.getNextStationRef()))
        myStationIndex->add (s);
    }
    if (myStationIndex->length() == 0)
      barf (NO_HFILE_PATH);
    myStationIndex->qsort();
  }
  return myStationIndex;
}

StationRef *TideContext::fastload (const Dstr &name) {
  if (myStationIndex) {
    return (*myStationIndex)[name];
  } else {
    for (unsigned i=0; i<harmonicsPath->length(); i++) {
      HarmonicsFile h ((*harmonicsPath)[i], settings);
      StationRef *s = h.getStationRef (name);
      if (s) {
        s->fastload = 1;
        return s;
      }
    }
    return NULL;
  }
}

void 
TideContext::predictText (Station *station, Station::Direction dir,
Timestamp &tm, Dstr &line, Station::EventType &etype, Timestamp *t_out) {
  Timestamp ts;
  PredictionValue pv;
  Dstr etype_desc, time_print, pv_print;
  station->predictExactTideEvent (tm, dir, ts, etype, etype_desc, pv);
  if (t_out)
    (*t_out) = ts;
  ts.print (time_print, station->timeZone, settings);
  if (!isSunMoonEvent (etype))
    pv.print (pv_print);
  line = time_print;
  line += " ";
  line += pv_print;
  line += "  ";
  line += etype_desc;

  // This is replaced by code in text_boilerplate.
#if 0
  // When known, append the direction of currents.  (The offending attributes
  // should be null if it's not a current station.)
  switch (etype) {
  case Station::max:
    if (!(station->maxCurrentAngle.isNull())) {
      line += " ";
      line += station->maxCurrentAngle.ppqdeg(0);
    }
    break;
  case Station::min:
    if (!(station->minCurrentAngle.isNull())) {
      line += " ";
      line += station->minCurrentAngle.ppqdeg(0);
    }
    break;
  default:
    ;
  }
#endif
}

void TideContext::textMode (Station *station, Timestamp start_tm,
Dstr &text_out) {
  Timestamp end_tm = start_tm + Interval(defpredictinterval);
  textMode (station, start_tm, end_tm, text_out);
}

void TideContext::text_boilerplate (Station *station, Dstr &text_out,
int html_form) {
  text_out = (char *)NULL;
  if (html_form)
    text_out += "<H3>";
  text_out += station->name;
  if (html_form)
    text_out += "<BR>";
  text_out += '\n';
  if (station->coordinates.isNull())
    text_out += "Coordinates unknown\n";
  else {
    Dstr t;
    station->coordinates.print (t);
    text_out += t;
    text_out += '\n';
  }

  // When known, append the direction of currents.  (The offending attributes
  // should be null if it's not a current station.)
  if (!(station->maxCurrentAngle.isNull())) {
    if (html_form)
      text_out += "<BR>";
    text_out += "Flood direction ";
    text_out += station->maxCurrentAngle.ppqdeg(0);
    text_out += '\n';
  }
  if (!(station->minCurrentAngle.isNull())) {
    if (html_form)
      text_out += "<BR>";
    text_out += "Ebb direction ";
    text_out += station->minCurrentAngle.ppqdeg(0);
    text_out += '\n';
  }

  if (html_form)
    text_out += "</H3>";
  text_out += '\n';
}

void TideContext::textMode (Station *station, Timestamp start_tm,
Timestamp end_tm, Dstr &text_out) {
  Dstr line;
  Station::EventType etype;
  Timestamp tm;

  text_boilerplate (station, text_out);

  // Leave a safety margin since these timestamps will generally already
  // be known tide events.
  tm = start_tm - Interval(60);

  // These can come back out of order for subordinate stations.
  struct TideContext::tideline *list = new struct TideContext::tideline;
  list->next = NULL;
  predictText (station, Station::forward, tm, list->text, etype,
    &(list->t_out));
  struct TideContext::tideline *last = list;
  struct TideContext::tideline *tl;
  while (tm < end_tm + Interval(60)) {
    tl = new struct TideContext::tideline;
    last->next = tl;
    last = tl;
    tl->next = NULL;
    predictText (station, Station::forward, tm, tl->text, etype,
      &(tl->t_out));
  }

  // Since this list is _almost_ in order, a bubble sort will be optimal.
  int done = 0;
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  int itctr = 0, swapctr = 0;
#endif
  while (!done) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
    itctr++;
#endif
    last = list;
    tl = list->next;
    done = 1;
    while (tl) {
      if (last->t_out > tl->t_out) {
        // Swap
        done = 0;
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
        swapctr++;
#endif
        Dstr temptext = last->text;
        Timestamp temptout = last->t_out;
        last->text = tl->text;
        last->t_out = tl->t_out;
        tl->text = temptext;
        tl->t_out = temptout;
      }
      last = tl;
      tl = tl->next;
    }
  }
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  cerr << "TideContext::textMode bubble sort took " << itctr <<
    " iterations and " << swapctr << " swaps" << endl;
#endif

  // Yay, we're done.
  while (list) {
    text_out += list->text;
    text_out += '\n';
    last = list;
    list = list->next;
    delete last;
  }
}

void TideContext::calendarMode (Station *station, Timestamp start_tm,
Timestamp end_tm, Dstr &text_out, int html_form) {
  text_boilerplate (station, text_out, html_form);
  Calendar cal (station, start_tm, end_tm);
  Dstr temp;
  cal.print (temp, html_form);
  text_out += temp;
}

void TideContext::bannerMode (Station *station, Timestamp start_tm,
Timestamp end_tm, Dstr &text_out) {
  text_boilerplate (station, text_out);
  Banner *banner = bannerFactory (station, settings->tw, start_tm, end_tm);
  Dstr temp;
  banner->drawTides (station, start_tm);
  banner->writeAsText (temp);
  text_out += temp;
  delete banner;
}

void TideContext::statsMode (Station *station, Timestamp start_tm,
Timestamp end_tm, Dstr &text_out) {
  text_boilerplate (station, text_out);
  PredictionValue maxl = station->maxLevel();
  PredictionValue minl = station->minLevel();
  PredictionValue meanl = (maxl + minl) / 2.0;
  Dstr temp;
  text_out += "Mathematical upper bound: ";
  maxl.print (temp);
  text_out += temp;
  text_out += "\nMathematical lower bound: ";
  minl.print (temp);
  text_out += temp;
  text_out += "\n";
  if (station->isCurrent)
    text_out += "Permanent current: ";
  else
    text_out += "Mean Sea Level: ";
  meanl.print (temp);
  text_out += temp;
  text_out += "\n\n";

  int first = 1;
  Timestamp tm = start_tm, maxt, mint;
  while (tm <= end_tm) {
    PredictionValue pv_out;
    Station::EventType etype;
    Dstr etype_desc;
    Timestamp t_out;
    station->predictExactTideEvent (tm, Station::forward, t_out,
      etype, etype_desc, pv_out);
    if (tm > end_tm)
      break;
    if (!isSunMoonEvent (etype)) {
      if (first || (pv_out < minl)) {
	mint = t_out;
	minl = pv_out;
      }
      if (first || (pv_out > maxl)) {
	maxt = t_out;
	maxl = pv_out;
      }
      first = 0;
    }
  }

  Dstr timezone = station->timeZone;
  text_out += "Searched interval from ";
  start_tm.print (temp, timezone, settings);
  text_out += temp;
  text_out += " to ";
  end_tm.print (temp, timezone, settings);
  text_out += temp;
  text_out += "\n";
  if (!first) {
    text_out += "Max was ";
    maxl.print (temp);
    text_out += temp;
    text_out += " at ";
    maxt.print (temp, timezone, settings);
    text_out += temp;
    text_out += "\n";
    text_out += "Min was ";
    minl.print (temp);
    text_out += temp;
    text_out += " at ";
    mint.print (temp, timezone, settings);
    text_out += temp;
    text_out += "\n";
  } else {
    text_out += "Found no tide events.\n";
  }
}

void
TideContext::rawMode (Station *station, Timestamp start_tm, Timestamp end_tm,
		      Interval step, Dstr &text_out) {
  text_out = (char *)NULL;
  assert (step > Interval(0));
  assert (start_tm <= end_tm);
  for (Timestamp t = start_tm; t <= end_tm; t += step) {
    text_out += t.timet();
    text_out += " ";
    text_out += station->predictApproximate(t).val();
    text_out += "\n";
  }
}

void TideContext::listModePlain (Dstr &text_out) {
  text_out = "Location list generated ";
  Timestamp now (time(NULL));
  Dstr tempnow, tz;
  tz = "UTC0";
  now.print (tempnow, tz, settings);
  text_out += tempnow;
  text_out += "\n\n";
  int namewidth = (int)(settings->tw) - 45;
  if (namewidth < 10)
    namewidth = 10;
  StationIndex *si = stationIndex();
  char fmt[80];
  sprintf (fmt, "%%-%d.%ds %%-20.20s %%-23.23s\n", namewidth, namewidth);
  char *buf = (char *) malloc (settings->tw + 30);
  for (unsigned i=0; i<si->length(); i++) {
    Dstr fname, c;
    (*si)[i]->shortHarmonicsFileName (fname);
    (*si)[i]->coordinates.print (c, 1);
    sprintf (buf, fmt, (*si)[i]->name.aschar(), fname.aschar(), c.aschar());
    text_out += buf;
  }
  free (buf);
}

void
TideContext::startLocListHTML (Dstr &d) {
  d += "<P><TABLE>\n<TR><TH>Location</TH><TH>File</TH>\n\
<TH>Coordinates</TH></TR>";
}

void
TideContext::endLocListHTML (Dstr &d) {
  d += "</TABLE></P>\n";
}

void
TideContext::listLocationHTML (Dstr &d, StationRef *sr) {
  assert (sr);
  d += "<TR><TD>";
  d += sr->name;
  d += "</TD><TD>";
  {
    Dstr fname;
    sr->shortHarmonicsFileName (fname);
    d += fname;
  }
  d += "</TD><TD>";
  Dstr tempc;
  sr->coordinates.print (tempc);
  d += tempc;
  d += "</TD></TR>\n";
}

void TideContext::listModeHTML (Dstr &text_out) {
  text_out = "<P> Location list generated ";
  Timestamp now (time(NULL));
  Dstr tempnow, tz;
  tz = "UTC0";
  now.print (tempnow, tz, settings);
  text_out += tempnow;
  text_out += "</P>\n\n";
  StationIndex *si = stationIndex();
  startLocListHTML (text_out);
  for (unsigned i=0; i<si->length(); i++) {
    listLocationHTML (text_out, (*si)[i]);
    if ((i % maxhtmltablen == 0) && (i != 0)) {
      endLocListHTML (text_out);
      startLocListHTML (text_out);
    }
  }
  endLocListHTML (text_out);
}
