/*
 *  Xtend is Copyright (C) 1998-1999 David M. Shaw <dshaw@jabberwocky.com>
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <limits.h>
#include <stdlib.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include "x10.h"
#include "util.h"
#include "exec.h"
#include "parse.h"
#include "version.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_UNAME
#include <sys/utsname.h>
#endif

static char RCSID[] = "@(#) $Id: xtend.c,v 2.9.2.13 1999/02/28 21:10:56 dshaw Exp $";
static char RCSID2[] = "@(#) $Silliness: Why are you snooping with ident? $";

extern int errno;
int verbose = 0, special = 0;
char parsefunction (char *byte);
void show_version (void);
void parseaddress (char byte);
void copyleft (void);
int setup (void);
void daemonize(void);

char *commands[] =
{
  "All Units Off",
  "All Lights On",
  "On",
  "Off",
  "Dim",
  "Bright",
  "All Lights Off",
  "Extended Code",
  "Hail Request",
  "Hail Acknowledge",
  "Pre-set Dim",
  "Pre-set Dim",
  "Extended Data Transfer",
  "Status On",
  "Status Off",
  "Status Request"
};

/* oh, that wacky X10 counting scheme. */
/* these are both house and unit codes */
static unsigned char units[] =
{6, 14, 2, 10, 1, 9, 5, 13, 7, 15, 3, 11, 0, 8, 4, 12};

static unsigned char david_check_this_later[] =
{12, 4, 2, 10, 14, 6, 8, 5, 3, 11, 15, 7, 1, 9};

/* As per Greg Gulik */
static unsigned char dims[] =
{1, 9, 5, 13, 3, 11, 7, 15, 2, 10, 6, 14, 4, 12, 8, 16};

int statusfile;
unsigned char addresstable[16][16];
char *commandtable[NUM_COMMANDS][16][16];
char *envtable[16][16];
unsigned short okaytowipe = 0;
char status = 0;
char *port = NULL, *file = NULL, *shell = NULL, *lockdir = DEFAULT_LOCKDIR,
  rcfilename[PATH_MAX] =
{'\0'};

int
main (int argc, char *argv[])
{
  int arg = 0, x10;
  char length = (int) RCSID2;
  char buf[MAXBUF];
  int background=1;

  while ((arg = getopt (argc, argv, "vVd:f:bgh")) != -1)
    switch (arg)
      {
      case 'f':
	strncpy (rcfilename, optarg, PATH_MAX);
	rcfilename[PATH_MAX - 1] = '\0';
	break;

      case 'v':
	verbose = 1;
	break;

      case 'b':
	background=0;
	break;

      case 'd':
	verbose = atoi (optarg);
	break;

      default:
      case 'h':
	fprintf (stderr, "Usage: %s [-v] [-V] [-f command_file] [-d debug_level] [-b] [-g] [-h]\n", argv[0]);
	fprintf (stderr, "\t-v\tverbose mode (same as -d 1)\n");
	fprintf (stderr, "\t-V\tprint version and default configuration information\n");
	fprintf (stderr, "\t-f\tspecify command file (instead of /etc/xtend.conf)\n");
	fprintf (stderr, "\t-d\tspecify debug level (implies -b)\n");
	fprintf (stderr, "\t-b\tdon't run in the background\n");
	fprintf (stderr, "\t-g\tprint the GNU copyleft information\n");
	fprintf (stderr, "\t-h\tprint this note\n");
	exit (0);

      case 'V':
	show_version ();
	exit (0);
	break;

      case 'g':
	copyleft ();
	exit (0);
	break;
      }

  if (verbose > 4)
    show_version ();

  parseinit ();

  /* we must daemonize before we setup as the lockfile contains our PID,
     and our PID is dependent on whether we daemonize */

  if(background && verbose==0)
    daemonize();

  x10 = setup ();

  /* ..and now we loop forever and ever and ever.... */

  for (;;)
    {
      if (special == 0)
	{
	  if (waitforpoll (x10))
	    {
	      length = getmessage (x10, buf);
	      handlemessage (buf, length);
	    }
	  else
	    {
	      close (x10);

	      if (special == 1)
		{
		  reload ();
		  x10 = setup ();
		}

	      if (special == 2)
		{
		  shutdown ();
		  exit (0);
		}
	    }
	}
    }
}

char
parsefunction (char *byte)
{
  char house, offset = 0;
  int index;
  x10code command;

  house = (byte[0] >> 4) & 0x0f;
  command = byte[0] & 0x0f;

  if(command==PREDIM2)
    {
      command=PREDIM1;
      house+=16;
    }

  /* Different decoding for pre-set dims and real houses */
  if(command==PREDIM1)
    house=dims[(int)house];
  else
    for (index = 0; index < 16; index++)
      {
	if (units[index] == house)
	  {
	    house = index;
	    break;
	  }
      }

  okaytowipe |= (0x01 << house);

  if (verbose > 1)
    {
      if(command==PREDIM1)
	printf("Pre-set dim, level %d",house+1);
      else
	printf ("House %c, %s", 65 + house, commands[(int) command]);

      if (command == DIM || command == BRIGHT)
	{
	  double level = ((double) byte[1] / 210) * 100;
	  printf (" %s%f%% (0x%X)", (level < 0 ? ">" : ""), (level < 0 ? 100 : level), byte[1]);
	}

      printf ("\n");
    }

  if (command == DIM || command == BRIGHT)
    offset = 1;

  docommands (house, command);

  return offset;
}

void
parseaddress (char byte)
{
  char house = 0, unit = 0, hi, lo;
  int index;

  hi = (byte >> 4) & 0x0f;
  lo = byte & 0x0f;

  for (index = 0; index < 16; index++)
    {
      if (units[index] == hi)
	house = index;
      if (units[index] == lo)
	unit = index;
    }

  if (verbose > 1)
    printf ("Address %c%d\n", 65 + house, unit + 1);

  if (okaytowipe)
    wipeaddresstable (house);

  /* mark the address as, er, addressed */
  addresstable[(int) house][(int) unit] |= ADDRESSBIT;
}

void
copyleft (void)
{
  printf ("\n Xtend %s, Copyright (C) 1998-1999 David M. Shaw <dshaw@jabberwocky.com>\n\n", VERSION);
  printf ("    This program is free software; you can redistribute it and/or modify\n");
  printf ("    it under the terms of the GNU General Public License as published by\n");
  printf ("    the Free Software Foundation; either version 2 of the License, or\n");
  printf ("    (at your option) any later version.\n\n");
  printf ("    This program is distributed in the hope that it will be useful,\n");
  printf ("    but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  printf ("    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
  printf ("    GNU General Public License for more details.\n\n");
  printf ("    You should have received a copy of the GNU General Public License\n");
  printf ("    along with this program; if not, write to the Free Software\n");
  printf ("    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n");
}

int
setup (void)
{
  int x10, x, fdstats;
  off_t err = (off_t) RCSID;	/* pointless hack to prevent gcc from
				   squawking */
  struct termios attribs;

  statusfile = -1;
  parsefile ();

  if (file == NULL && port == NULL)
    {
      fprintf (stderr,"Cannot run if both \"Port\" and \"File\" are undefined!\n");
      exit (-1);
    }

  if (file != NULL && port != NULL)
    {
      fprintf (stderr,"Cannot run if both \"Port\" and \"File\" are defined!\n");
      exit (-1);
    }

  if (verbose)
    showparse ();

  if (file == NULL)
    {
      /* serial port, so we need a lock */

      lockfile (port, lockdir, DO_LOCK);

      x10 = open (port, O_RDWR | O_NOCTTY);
    }
  else
    {
      /* heyu file */
      x10 = open (file, O_RDONLY | O_NOCTTY);
    }

  if ((fdstats = fcntl (x10, F_GETFD, 0)) < 0)
    {
      fprintf (stderr, "Couldn't fcntl x10: %s\n", strerror (errno));
      exit (-1);
    }

  fdstats |= FD_CLOEXEC;

  if (fcntl (x10, F_SETFD, fdstats) < 0)
    {
      fprintf (stderr, "Couldn't set close-on-exec for x10: %s\n", strerror (errno));
      exit (-1);
    }

  if (x10 == -1)
    {
      fprintf (stderr, "%s: %s", port == NULL ? file : port, strerror (errno));
      exit (-1);
    }

  if (verbose > 8)
    printf ("%s opened as fd %d\n", port == NULL ? file : port, x10);

/* if we're using a heyu-style file, we need to seek to the end of it */

  if (port == NULL)
    {
      err = lseek (x10, 0, SEEK_END);
      if (err == -1)
	{
	  fprintf (stderr, "main:lseek: %s\n", strerror (errno));
	  exit (-1);
	}
    }
  else
    {
/* if not, we need to set up the serial port to be cm11a happy.
   that means 4800 baud, 8 bits, 1 stop bit, and no parity */

      if (tcgetattr (x10, &attribs) < 0)
	{
	  fprintf (stderr, "main:tcgetattr: %s\n", strerror (errno));
	  exit (-1);
	}

      attribs.c_iflag = IGNBRK | IGNPAR;
      attribs.c_oflag = 0;
      attribs.c_lflag = 0;
      attribs.c_cflag = CLOCAL | B4800 | CS8 | CREAD;

      for (x = 0; x < NCCS; x++)
	attribs.c_cc[x] = 0;

      if (tcsetattr (x10, TCSANOW, &attribs) < 0)
	{
	  fprintf (stderr, "main:tcsetattr: %s\n", strerror (errno));
	  exit (-1);
	}
    }

  return x10;
}

void
show_version (void)
{
#ifdef HAVE_UNAME
  struct utsname uts;

  uname (&uts);
#endif

  printf ("Xtend %s, Copyright (C) 1998-1999 David M. Shaw <dshaw@jabberwocky.com>\n", VERSION);

#ifdef HAVE_UNAME
  printf ("System=%s %s %s\n", uts.sysname, uts.release, uts.machine);
#else
  printf ("System is unknown\n");
#endif
  printf ("Default Lockdir=%s\n", DEFAULT_LOCKDIR);
  printf ("Default Shell=%s\n", DEFAULT_SHELL);
  printf ("Default RCfile=$HOME/%s\n", DEFAULT_RCFILE);
}

void
daemonize(void)
{
  pid_t child;

  child=vfork();
  if(child==-1)
    {
      printf("Couldn't fork to daemonize: %s\n",strerror(errno));
      exit(-1);
    }

  if(child!=0)
    {
      /* I'm the parent. */
      _exit(0);
    }

  /* I'm the child. */

  setsid();
}
