/*
 *  Xtend is Copyright (C) 1998-1999 David M. Shaw <dshaw@jabberwocky.com>
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include "parse.h"
#include "util.h"
#include "x10.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

static char RCSID[] = "@(#) $Id: parse.c,v 2.7.2.10 1999/02/28 02:44:31 dshaw Exp $";

/* there are a few more X10 commands, like "all-on" and status requests,
   but not much hardware actually generates them...  */

char *commandcodes[] =
{
  "all-off",
  "lights-on",
  "on",
  "off",
  "dim",
  "bright",
  "lights-off",
  "",
  "hailreq",
  "hailack",
  "preset",
  "",
  "",
  "statuson",
  "statusoff",
  "statusrequest"
};

extern int errno;
extern char *port, *file, *shell, rcfilename[PATH_MAX], *lockdir, status;
extern unsigned char addresstable[16][16];
extern int verbose, special, statusfile;
extern char *commands[];
char *commandtable[NUM_COMMANDS][16][16];
char *statusfilename;
extern char *envtable[16][16];

void
showparse (void)
{
  int x, y, z = (int) RCSID;

  if (port == NULL)
    printf ("%s=%s\n", FILE_CMD, file);
  else
    {
      printf ("%s=%s\n", PORT_CMD, port);
      printf ("%s=%s\n", LOCK_CMD, lockdir);
    }

  printf ("Shell=%s\n", shell);
  printf ("RCfile=%s\n", rcfilename);

  printf ("Appliances: ");
  for (x = 0; x < 16; x++)
    for (y = 0; y < 16; y++)
      if (addresstable[x][y] & APPLBIT)
	printf ("%c%d ", x + 65, y + 1);

  printf ("\n");

  for (x = 0; x < NUM_COMMANDS; x++)
    {
      for (y = 0; y < 16; y++)
	{
	  for (z = 0; z < 16; z++)
	    {
	      if (commandtable[x][y][z] != NULL)
		if (x == 0 || x == 1)
		  {
		    printf ("X10 command=%s, house=%c, shell command=\n\t%s\n",
			    commands[x], y + 65, commandtable[x][y][z]);
		  }
		else
		  {
		    printf ("X10 command=%s, house=%c, unit=%d, shell command=\n\t%s\n",commands[x], y + 65, z + 1, commandtable[x][y][z]);
		  }
	    }
	}
    }
}

void
parseinit (void)
{
  int x, y, z;

  for (x = 0; x < NUM_COMMANDS; x++)
    for (y = 0; y < 16; y++)
      {
	envtable[x][y] = NULL;
	for (z = 0; z < 16; z++)
	  commandtable[x][y][z] = NULL;
      }
}

void
parsefile (void)
{
  FILE *rcfile;
  char line[RCLINEMAX];
  int dummy, lineno;

  /* we should start off nice and clean, X10 wise.. */
  for (dummy = 0; dummy < 16; dummy++)
    for (lineno = 0; lineno < 16; lineno++)
      addresstable[dummy][lineno] = 0;

  lineno = 0;
  status = 0;

  if (rcfilename[0] == '\0')
  {
      strcpy (rcfilename, DEFAULT_RCFILE);
    }

  shell = getenv ("SHELL");
  if (shell == NULL)
    shell = DEFAULT_SHELL;

  rcfile = fopen (rcfilename, "r");
  if (rcfile == NULL)
    {
      fprintf (stderr, "Can't open config file \"%s\": %s\n", rcfilename, strerror (errno));
      exit (-1);
    }

  while (fgets (line, RCLINEMAX, rcfile) != NULL)
    {
      lineno++;
      if ((strlen (line) == RCLINEMAX - 1) && (line[RCLINEMAX - 2] != '\n'))
	{
	  fprintf (stderr, "error:\t%s ....\n\tLine is too long! (%d character limit) - line %d ignored\n", line, RCLINEMAX - 2, lineno);
	  /* now eat the rest of the line so we can keep
	     parsing */
	  dummy = 'x';
	  while (dummy != '\n' && dummy != EOF)
	    dummy = fgetc (rcfile);
	}
      else
	{
	  /* chop the \n off the end */
	  line[strlen (line) - 1] = '\0';
	  if (parseline (line))
	    fprintf (stderr, " - line %d ignored\n", lineno);
	}
    }

  fclose (rcfile);

  /* now register the handlers for HUP, INT, and TERM.. */

  setsignals ();
}

int
parseline (char *line)
{
  char *token = NULL;
  char housecode;
  int unitcode;
  x10code commandcode;
  char *portfile;
  char *x10_command;
  char *shell_command;
  char line2[RCLINEMAX];
  int index = 0, index2 = 0, protect = 0, args = 0, fdstats;

  while (line[index] != '\0')
    {
      if (line[index] == '\\' && !protect)
	protect = 1;
      else
	{
	  if (!protect && line[index] == COMMENT_CHAR)
	    {
	      line[index] = '\0';
	      line[index + 1] = '\0';
	    }

	  protect = 0;
	  line2[index2++] = line[index];
	}
      index++;
    }

  line2[index2] = '\0';

  if (verbose > 5)
    printf ("\nLine: %s\n", line2);

  /* yeehah.  Now we have a line that can be tokenized */

  token = strtok (line2, TOKEN_SEP);
  if (token == NULL)
    {
      /* this is a commented out line */
      return 0;
    }

  if (strcasecmp (token, APPL_CMD) == 0)
    {
      while ((portfile = strtok (NULL, TOKEN_SEP)) != NULL)
	{
	  args = parseunitcode (line, portfile, &housecode, &unitcode);
	  if (args == 1)
	    {
	      fprintf (stderr, "error:\t%s\n\tCannot mark a whole house (%c) as an appliance!", line, housecode + 65);
	      args = 0;
	    }

	  if (args == 0)
	    {
	      fprintf (stderr, " - portion of line\n\tstarting with \"%s\" ignored.\n", portfile);
	      return 0;
	    }

	  if (verbose > 2)
	    printf ("marking %c%d as an appliance\n", housecode + 65, unitcode + 1);

	  addresstable[(int) housecode][(int) unitcode] |= APPLBIT;
	  if (statusfile != -1)
	    updatestatusfile (housecode, unitcode);
	}

      return 0;
    }

  if (strcasecmp (token, STATUSFILE_CMD) == 0)
    {
      portfile = strtok (NULL, TOKEN_SEP);
      if (portfile == NULL)
	{
	  fprintf (stderr, "error:\t%s\n\tno filename found on a %s line",
		   line, STATUSFILE_CMD);
	  return 1;
	}

      if (strlen (portfile) > PATH_MAX)
	{
	  fprintf (stderr, "%s is WAY too big! (>%d characters)\n", STATUSFILE_CMD, PATH_MAX);
	  exit (-1);
	}

      if (verbose)
	printf ("Statusfile=%s\n", portfile);

      statusfile = open (portfile, O_CREAT | O_TRUNC | O_WRONLY, 0644);
      if (statusfile == -1)
	{
	  fprintf (stderr, "Couldn't open statusfile %s: %s\n", portfile, strerror (errno));
	  exit (-1);
	}

      if ((fdstats = fcntl (statusfile, F_GETFD, 0)) < 0)
	{
	  fprintf (stderr, "Couldn't fcntl statusfile: %s\n", strerror (errno));
	  exit (-1);
	}

      fdstats |= FD_CLOEXEC;

      if (fcntl (statusfile, F_SETFD, fdstats) < 0)
	{
	  fprintf (stderr, "Couldn't set close-on-exec for statusfile: %s\n", strerror (errno));
	  exit (-1);
	}

      if ((write (statusfile, &addresstable[0][0], 256) == -1) ||
	  (write (statusfile, &addresstable[0][0], 256) == -1))
	{
	  fprintf (stderr, "Couldn't clear statusfile %s: %s\n", portfile, strerror (errno));
	  exit (-1);
	}

      return 0;
    }

  if (strcasecmp (token, LOCK_CMD) == 0)
    {
      portfile = strtok (NULL, TOKEN_SEP);
      if (portfile == NULL)
	{
	  fprintf (stderr, "error:\t%s\n\tno directory found on a %s line",
		   line, LOCK_CMD);
	  return 1;
	}

      /* this isn't likely to ever happen as PATH_MAX is a heck of
         a lot larger than RCLINEMAX */

      if (strlen (portfile) > PATH_MAX - 27)
	{
	  fprintf (stderr, "%s is WAY too big! (>%d characters)\n", LOCK_CMD, PATH_MAX - 26);
	  exit (-1);
	}

      lockdir = malloc (strlen (portfile) + 1);

      if (lockdir == NULL)
	{
	  fprintf (stderr, "Couldn't allocate memory for lockdir %s !\n", portfile);
	  exit (-1);
	}

      strcpy (lockdir, portfile);
      status |= 0x02;
      return 0;
    }

  if (strcasecmp (token, SHELL_CMD) == 0)
    {
      portfile = strtok (NULL, TOKEN_SEP);
      if (portfile == NULL)
	{
	  fprintf (stderr, "error:\t%s\n\tno shell found on a %s line",
		   line, SHELL_CMD);
	  return 1;
	}

      shell = malloc (strlen (portfile) + 1);

      if (shell == NULL)
	{
	  fprintf (stderr, "Couldn't allocate memory for shell %s !\n", portfile);
	  exit (-1);
	}

      strcpy (shell, portfile);
      status |= 0x01;
      return 0;
    }

  if (strcasecmp (token, PORT_CMD) == 0 || strcasecmp (token, FILE_CMD) == 0)
    {
      portfile = strtok (NULL, TOKEN_SEP);
      if (portfile == NULL)
	{
	  fprintf (stderr, "error:\t%s\n\tno port or file found on a %s or %s line",
		   line, PORT_CMD, FILE_CMD);
	  return 1;
	}

      port = malloc (strlen (portfile) + 1);

      if (port == NULL)
	{
	  fprintf (stderr, "Couldn't allocate memory for %s %s !\n", port == NULL ? FILE_CMD : PORT_CMD, portfile);
	  exit (-1);
	}

      strcpy (port, portfile);
      if (strcasecmp (token, FILE_CMD) == 0)
	{
	  file = port;
	  port = NULL;
	}
      return 0;
    }

  /* okay, so we know it's an address.  Is it a full address or just a
     house code? */

  if ((args = parseunitcode (line, token, &housecode, &unitcode)) == 0)
    return 1;

  x10_command = strtok (NULL, TOKEN_SEP);

  if (x10_command == NULL)
    {
      fprintf (stderr, "error:\t%s\n\tNo command on this line", line);
      return 1;
    }

  for (commandcode = 0; commandcode < NUM_COMMANDS; commandcode++)
    {
      if (verbose > 5)
	printf ("comparing %s to %s...\n", x10_command, commandcodes[commandcode]);

      if (strcasecmp (x10_command, commandcodes[commandcode]) == 0)
	{
	  if (verbose > 5)
	    printf ("matched %s\n", x10_command);
	  break;
	}
    }

  /* all-off and lights-on */
  if (commandcode == ALLOFF || commandcode == LIGHTSON)
    unitcode = 0;

  if (commandcode == NUM_COMMANDS)
    {
      fprintf (stderr, "error:\t%s\n\t\"%s\" isn't any X10 command that I know of",
	       line, x10_command);
      return 1;
    }

  if (args == 2 && (commandcode == ALLOFF || commandcode == LIGHTSON))
    {
      fprintf (stderr, "error:\t%s\n\tNo unit code is needed for a '%s' command", line, commandcodes[commandcode]);
      return 1;
    }

  if (args == 1 && commandcode != ALLOFF && commandcode != LIGHTSON)
    {
      fprintf (stderr, "error:\t%s\n\tA unit code is needed for a '%s' command", line, commandcodes[commandcode]);
      return 1;
    }

  /* this isn't the same strtok as before, as the shell command can
     have spaces */

  shell_command = strtok (NULL, "");

  if (shell_command == NULL)
    {
      fprintf (stderr, "error:\t%s\n\tNo shell command on this line", line);
      return 1;
    }

  if (verbose > 4)
    printf ("the shell command is %s\n", shell_command);

  if (commandtable[commandcode][(int) housecode][(int) unitcode] != NULL)
    {
      fprintf (stderr, "error:\t%s\n\tCannot redefine %c%d %s", line, housecode + 65, unitcode + 1, x10_command);
      return 1;
    }

  commandtable[commandcode][(int) housecode][(int) unitcode] =
    malloc (strlen (shell_command) + 1);

  if (commandtable[commandcode][(int) housecode][(int) unitcode] == NULL)
    {
      fprintf (stderr, "Couldn't allocate memory for %c%d %s !\n", housecode + 65, unitcode + 1, shell_command);
      exit (-1);
    }

  strcpy (commandtable[commandcode][(int) housecode][(int) unitcode], shell_command);

  return 0;
}

void
reload (void)
{
  int x, y, z;

  if (verbose > 0)
    printf ("Caught signal - Reloading %s\n", rcfilename);

  if (file == NULL)
    lockfile (port, lockdir, DO_UNLOCK);

  for (x = 0; x < NUM_COMMANDS; x++)
    for (y = 0; y < 16; y++)
      for (z = 0; z < 16; z++)
	{
	  free (commandtable[x][y][z]);
	  commandtable[x][y][z] = NULL;
	}
  free (port);
  port = NULL;
  free (file);
  file = NULL;

  if (status & 0x01)
    free (shell);
  shell = NULL;

  if (status & 0x02)
    free (lockdir);
  lockdir = DEFAULT_LOCKDIR;

  special = 0;

  return;
}

int
parseunitcode (char *line, char *token, char *housecode, int *unitcode)
{
  int args;

  *housecode = 'Z';
  *unitcode = 99;

  args = sscanf (token, "%c%d", housecode, unitcode);

  if (verbose > 5)
    printf ("parsing %s into %c%d, and got %d args\n", token, *housecode, *unitcode, args);

  *housecode = toupper (*housecode);
  if (!isalpha ((int)*housecode) || *housecode < 'A' || *housecode > 'P')
    {
      fprintf (stderr, "error:\t%s\n\thouse code %c is out of range (A-P)",
	       line, *housecode);
      return 0;
    }

  /* ASCII to X10... */
  *housecode -= 65;

  if (args == 2)
    {
      /* it's a full address */
      if (verbose > 6)
	printf ("it's a full address\n");

      *unitcode -= 1;

      if (*unitcode < 0 || *unitcode > 15)
	{
	  fprintf (stderr, "error:\t%s\n\tunit code %d is out of range (1-16)",
		   line, *unitcode + 1);
	  return 0;
	}
    }
  else
    *unitcode = 0;

  if (verbose > 4)
    printf ("the house is %c, the unit is %d\n", *housecode + 65, *unitcode + 1);

  return args;
}
