/*******************************************************************
 * WPA-PSK Function implementations for supplicant
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 * 
 * File: psk.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: psk.c,v 1.6 2006/01/12 17:54:02 chessing Exp $
 * $Date: 2006/01/12 17:54:02 $
 * $Log: psk.c,v $
 * Revision 1.6  2006/01/12 17:54:02  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.5  2005/11/18 19:14:47  chessing
 * Fixes to the OpenSSL detection settings in the autoconf file.  Added some checks for SHA changes in OpenSSL 0.9.8
 *
 * Revision 1.4  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.3  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/hmac.h>
#include <openssl/err.h>
#include <openssl/sha.h>
#include <assert.h>
#include <string.h>

// For OpenSSL 0.9.8 we need to explicitly include sha.h
#ifndef SHA_DIGEST_LENGTH
#include <openssl/sha.h>
#endif

// If we still don't have SHA_DIGEST_LENGTH defined, then define it ourselves.
#ifndef SHA_DIGEST_LENGTH
#define SHA_DIGEST_LENGTH 20
#endif

#include "profile.h"
#include "xsup_debug.h"

/*********************************************************************
 *
 * This code taken from 802.11i-D3.0, section F.8.2 reference implementation.
 *
 *********************************************************************/
void psk_wpa_pbkdf2_f(char *password, unsigned char *ssid, int ssidlength,
		      int iterations, int count, unsigned char *output)
{
  unsigned char digest[36], digest1[SHA_DIGEST_LENGTH];
  int i, j, k;

  for (i=0; i < strlen(password); i++)
    {
      assert((password[i] >= 32) && (password[i] <= 126));
    }

  /* U1 = PRF(P, S || int(i)) */
  memcpy(digest, ssid, ssidlength);
  digest[ssidlength] = (unsigned char)((count>>24) & 0xff);
  digest[ssidlength+1] = (unsigned char)((count>>16) & 0xff);
  digest[ssidlength+2] = (unsigned char)((count>>8) & 0xff);
  digest[ssidlength+3] = (unsigned char)(count & 0xff);

  // OpenSSL takes the parameters in a different order than what is
  // defined in F.8.2, so even though it looks wrong, this is correct. ;)
  HMAC(EVP_sha1(), password, strlen(password), digest, ssidlength+4, 
       digest1, (u_int *) &k);

  /* output = U1 */
  memcpy(output, digest1, SHA_DIGEST_LENGTH);

  for (i = 1; i < iterations; i++)
    {
      /* Un = PRF(P, Un-1) */
      HMAC(EVP_sha1(), password, strlen(password), digest1, SHA_DIGEST_LENGTH, 
	   digest, (u_int *) &k);
      memcpy(digest1, digest, k);

      /* output = output xor Un */
      for (j = 0; j < SHA_DIGEST_LENGTH; j++)
	{
	  output[j] ^= digest[j];
	}
    }
}

int psk_wpa_pbkdf2(char *password, unsigned char *ssid, int ssidlength,
		   unsigned char *output)
{
  if ((strlen(password) > 63) || (ssidlength > 32))
    {
      debug_printf(DEBUG_NORMAL, "Invalid WPA-PSK password!\n");
      return FALSE;
    }

  psk_wpa_pbkdf2_f(password, ssid, ssidlength, 4096, 1, output);
  psk_wpa_pbkdf2_f(password, ssid, ssidlength, 4096, 2, 
		   &output[SHA_DIGEST_LENGTH]);
  return TRUE;
}
