/*******************************************************************
 * Handler for profile related things.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: profile.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: profile.c,v 1.35 2006/02/23 22:26:53 chessing Exp $
 * $Date: 2006/02/23 22:26:53 $
 * $Log: profile.c,v $
 * Revision 1.35  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.34  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.33  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.32  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.31  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <string.h>

#include "snmp.h"
#include "profile.h"
#include "xsup_err.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "eapol.h"
#include "xsupconfig.h"

/*******************************************
 *
 * Initalize the default values for the structure.  In general, state machine
 * and user configuration variables won't need to be set here.  We should
 * set up variables that are in the root of the structure.
 *
 *******************************************/
int init_interface_struct(struct interface_data *work, char *intname)
{
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};

  if (!work)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  if (!intname)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface name passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  work->intName = (char *)malloc(strlen(intname)+1);
  if (work->intName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory in %s at line %d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  bzero(work->intName, strlen(intname)+1);
  strncpy(work->intName, intname, strlen(intname));

  bzero(work->source_mac, 6);
  bzero(work->dest_mac, 6);

  work->keyingMaterial = NULL;

  work->tick = FALSE;

  work->rsn_ie = NULL;
  work->wpa_ie = NULL;

  work->flags = WAS_DOWN;

  // The default MAC specified by the IEEE 802.1x spec.
  memcpy(&work->dest_mac[0], &dot1x_default_dest, 6);
	
  work->cur_essid = NULL;

  eapol_init(work);

  work->tempPassword = NULL;

  work->send_size = 0;
  work->recv_size = 0;

  return XENONE;
}

/**********************************************
 *
 * Clean out a structure.  Clear out everything but return a pointer to "next"
 * the pointer to next should be used in a successive call.
 *
 **********************************************/
void destroy_interface_struct(struct interface_data *intdata)
{
  if (!intdata) return;

  if (intdata->intName) free(intdata->intName);

  if (intdata->cur_essid) free(intdata->cur_essid);
  if (intdata->keyingMaterial) free(intdata->keyingMaterial);
}


void profile_dump_flags(struct interface_data *intdata)
{
  debug_printf(DEBUG_INT, "Flags in interface_data :\n");

  if (TEST_FLAG(intdata->flags, IS_WIRELESS)) 
    debug_printf(DEBUG_INT, "IS_WIRELESS\n");

  if (TEST_FLAG(intdata->flags, WAS_DOWN))
    debug_printf(DEBUG_INT, "WAS_DOWN\n");

  if (TEST_FLAG(intdata->flags, SCANNING))
    debug_printf(DEBUG_INT, "SCANNING\n");

  if (TEST_FLAG(intdata->flags, ALLMULTI))
    debug_printf(DEBUG_INT, "ALLMULTI\n");

  if (TEST_FLAG(intdata->flags, ROAMED))
    debug_printf(DEBUG_INT, "ROAMED\n");

  if (TEST_FLAG(intdata->flags, ONEDOWN))
    debug_printf(DEBUG_INT, "ONEDOWN\n");

  if (TEST_FLAG(intdata->flags, TERM_ON_FAIL))
    debug_printf(DEBUG_INT, "DONT_USE_TEMP\n");

  if (TEST_FLAG(intdata->flags, DONT_KEY))
    debug_printf(DEBUG_INT, "DONT_KEY\n");

  if (TEST_FLAG(intdata->flags, CLEAR_IPC))
    debug_printf(DEBUG_INT, "CLEAR_IPC\n");

  if (TEST_FLAG(intdata->flags, PASV_SCANNING))
    debug_printf(DEBUG_INT, "PASV_SCANNING\n");
}

/****************************************
 *
 * Get configuration information out of memory, and 
 *
 ****************************************/
char config_build(struct interface_data *ctx, char *network_name)
{
  struct config_network *result;
  struct config_data *config_info;

  config_info = config_get_config_info();

  if (config_info != NULL)
    {
      debug_printf(DEBUG_CONFIG, "Working from config file %s.\n",config_info->config_fname);

      // We were passed in a "network name".  First, look through the config
      // to see if it matches any friendly names.
      result = config_find_network(config_info->networks, network_name);

      if (result != NULL) 
	{
	  config_set_network_config(result);
	  return TRUE;
	}

      // If we are wireless, we don't want to switch to a default network,
      // we want to either sit tight, or scan for a network.
      if (TEST_FLAG(ctx->flags, IS_WIRELESS)) 
	{
	  config_set_network_config(NULL);
	  return FALSE;
	}

      // This is not good.  We don't have any configuration information
      // for the requested network.  So, we need to return the default
      // information, and a warning.
      debug_printf(DEBUG_NORMAL, "No configuration information for network \"%s\" found.  Using default.\n", network_name);

      result = config_find_network(config_info->networks, 
				   config_info->globals->default_net);

      if (result != NULL) 
	{
	  config_set_network_config(result);
	  return TRUE;
	}

    } else {
      debug_printf(DEBUG_CONFIG, "config_info == NULL!  No config to update!\n");
    }
  return FALSE;
}

/************************************
 *
 * Set statemachine/config related variables for this interface.
 *
 ************************************/
int config_set_globals(struct interface_data *myint)
{
  struct config_data *config_info;

  config_info = config_get_config_info();

  if (config_info == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No configuration information is available!\n");
      return -1;
    }

  if (myint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No interface information is available!\n");
      return -1;
    }

  // Start by seeing if we need to set any global values.
  if ((CONFIG_GLOBALS_AUTH_PER & config_info->globals->flags) ==
      CONFIG_GLOBALS_AUTH_PER)
    {
      myint->statemachine->authPeriod = config_info->globals->auth_period;
    }

  if ((CONFIG_GLOBALS_HELD_PER & config_info->globals->flags) ==
      CONFIG_GLOBALS_HELD_PER)
    {
      myint->statemachine->heldPeriod = config_info->globals->held_period;
    }

  if ((CONFIG_GLOBALS_MAX_STARTS & config_info->globals->flags) ==
      CONFIG_GLOBALS_MAX_STARTS)
    {
      myint->statemachine->maxStart = config_info->globals->max_starts;
    }
 
  return 0;
}
