// file      : xsde/cxx/serializer/serializer-header.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/serializer/serializer-header.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

namespace CXX
{
  namespace Serializer
  {
    namespace
    {
      struct Enumeration: Traversal::Enumeration,
                          protected virtual Context
      {
        Enumeration (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          String const& name (ename (e));
          SemanticGraph::Type& base (e.inherits ().base ());

          os << "class " << name << ": public virtual " << fq_name (base)
             << "{"
             << "public:" << endl
             << "// Serializer callbacks. Override them in your " <<
            "implementation." << endl
             << "//" << endl
             << endl;

          // pre
          //
          String const& arg (arg_type (e));

          if (arg == arg_type (base))
          {
            os << "// virtual void" << endl;

            if (arg == L"void")
              os << "// pre ();" << endl;
            else
              os << "// pre (" << arg << ") = 0;" << endl;

            os << endl;
          }
          else
          {
            os << "virtual void" << endl;

            if (arg == L"void")
              os << "pre ();";
            else
              os << "pre (" << arg << ") = 0;";

            os << endl;
          }

          // post
          //
          os << "// virtual void" << endl
             << "// post ();" << endl
             << endl;

          os << "};";
        }
      };

      //
      //
      struct List: Traversal::List, protected virtual Context
      {
        List (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& l)
        {
          String const& name (ename (l));
          SemanticGraph::Type& t (l.argumented ().type ());

          String item (unclash (name, "item"));
          String item_next (unclash (name, "item_next"));

          os << "class " << name << ": public virtual " << simple_base
             << "{"
             << "public:" << endl
             << "// Serializer callbacks. Override them in your " <<
            "implementation." << endl
             << "//" << endl
             << endl;

          // pre
          //
          String const& arg (arg_type (l));

          os << "virtual void" << endl;

          if (arg == L"void")
            os << "pre ();";
          else
            os << "pre (" << arg << ") = 0;";

          os << endl;

          // item
          //
          os << "virtual bool" << endl
             << item_next << " ();"
             << endl;

          String const& ret (ret_type (t));

          os << "virtual " << ret << endl
             << item << " ()" << (ret != L"void" ? " = 0" : "") <<  ";"
             << endl;

          // post
          //
          os << "// virtual void" << endl
             << "// post ();" << endl
             << endl;

          //
          //
          os << "// Serializer construction API." << endl
             << "//" << endl;

          // item_serializer
          //
          os << "void" << endl
             << unclash (name, "item_serializer") << " (" <<
            fq_name (t) << "&);"
             << endl;

          // serializers
          //
          os << "void" << endl
             << "serializers (" << fq_name (t) << "& /* item */);"
             << endl;

          // c-tor
          //
          os << "// Constructor." << endl
             << "//" << endl
             << name << " ();"
             << endl;

          //
          //
          os << "// Implementation." << endl
             << "//" << endl
             << "public:" << endl;

          os << "virtual void" << endl
             << "_serialize_content ();"
             << endl;

          os << "protected:" << endl
             << fq_name (t) << "* _xsde_" << item << "_;"
             << "};";
        }
      };

      struct Union: Traversal::Union, protected virtual Context
      {
        Union (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& u)
        {
          String const& name (ename (u));

          os << "class " << name << ": public virtual " << simple_base
             << "{"
             << "public:" << endl
             << "// Serializer callbacks. Override them in your " <<
            "implementation." << endl
             << "//" << endl
             << endl;

          // pre
          //
          String const& arg (arg_type (u));

          os << "virtual void" << endl;

          if (arg == L"void")
            os << "pre ();";
          else
            os << "pre (" << arg << ") = 0;";

          os << endl;

          // serialize_content
          //
          os << "virtual void" << endl
             << "_serialize_content () = 0;"
             << endl;

          // post
          //
          os << "// virtual void" << endl
             << "// post ();" << endl
             << endl;

          os << "};";
        }
      };

      //
      //
      struct ParticleTag: Traversal::Particle,
                          protected virtual Context
      {
        ParticleTag (Context& c)
            : Context (c), first_ (true)
        {
        }

        virtual Void
        traverse (Type& p)
        {
          if (first_)
            first_ = false;
          else
            os << "," << endl;

          os << etag (p);
        }

      private:
        Boolean first_;
      };

      struct CompositorCallback: Traversal::All,
                                 Traversal::Choice,
                                 Traversal::Sequence,
                                 protected virtual Context
      {
        CompositorCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::All& a)
        {
          // For the all compositor, maxOccurs=1 and minOccurs={0,1}.
          //
          if (a.contained_compositor ().min () == 0)
          {
            os << "virtual bool" << endl
               << epresent (a) << " ();"
               << endl;
          }

          Traversal::All::traverse (a);
        }

        virtual Void
        traverse (SemanticGraph::Choice& c)
        {
          if (c.contains_begin () != c.contains_end ())
          {
            UnsignedLong min;
            UnsignedLong max;

            if (c.contained_compositor_p ())
            {
              min = c.contained_compositor ().min ();
              max = c.contained_compositor ().max ();
            }
            else
            {
              min = c.contained_particle ().min ();
              max = c.contained_particle ().max ();
            }

            if (min == 0 && max == 1)
            {
              os << "virtual bool" << endl
                 << epresent (c) << " ();"
                 << endl;
            }
            else if (max != 1)
            {
              os << "virtual bool" << endl
                 << enext (c) << " ()" << (min != 0 ? " = 0" : "") << ";"
                 << endl;
            }

            //
            //
            os << "enum " << earm_tag (c)
               << "{";

            {
              ParticleTag particle (*this);
              Traversal::ContainsParticle contain_particle (particle);
              Traversal::Choice::contains (c, contain_particle);
            }

            os << "};";

            os << "virtual " << earm_tag (c) << endl
               << earm (c) << " ()" << (min != 0 ? " = 0" : "") << ";"
               << endl;

            Traversal::Choice::traverse (c);
          }
        }

        virtual Void
        traverse (SemanticGraph::Sequence& s)
        {
          UnsignedLong min;
          UnsignedLong max;

          if (s.contained_compositor_p ())
          {
            min = s.contained_compositor ().min ();
            max = s.contained_compositor ().max ();
          }
          else
          {
            min = s.contained_particle ().min ();
            max = s.contained_particle ().max ();
          }

          if (min == 0 && max == 1)
          {
            os << "virtual bool" << endl
               << epresent (s) << " ();"
               << endl;
          }
          else if (max != 1)
          {
            os << "virtual bool" << endl
               << enext (s) << " ()" << (min != 0 ? " = 0" : "") <<  ";"
               << endl;
          }

          Traversal::Sequence::traverse (s);
        }
      };

      struct ParticleCallback: Traversal::Element,
                               Traversal::Any,
                               protected virtual Context
      {
        ParticleCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          UnsignedLong min (e.contained_particle ().min ());
          UnsignedLong max (e.contained_particle ().max ());

          if (min == 0 && max == 1)
          {
            os << "virtual bool" << endl
               << epresent (e) << " ();"
               << endl;
          }
          else if (max != 1)
          {
            os << "virtual bool" << endl
               << enext (e) << " ()" << (min != 0 ? " = 0" : "") << ";"
               << endl;
          }

          String const& ret (ret_type (e.type ()));

          // Make it non-pure-virtual only if the return type is void.
          //
          os << "virtual " << ret << endl
             << ename (e) << " ()" << (ret != L"void" ? " = 0;" : ";")
             << endl;
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          UnsignedLong min (a.contained_particle ().min ());
          UnsignedLong max (a.contained_particle ().max ());

          if (min == 0 && max == 1)
          {
            os << "virtual bool" << endl
               << epresent (a) << " ();"
               << endl;
          }
          else if (max != 1)
          {
            os << "virtual bool" << endl
               << enext (a) << " ()" << (min != 0 ? " = 0" : "") << ";"
               << endl;
          }

          if (stl)
          {
            os << "virtual void" << endl
               << ename (a) << " (::std::string& ns, ::std::string& name)" <<
              (min != 0 ? " = 0;" : ";")
               << endl;
          }
          else
          {
            os << "virtual void" << endl
               << ename (a) << " (const char*& ns, const char*& name, " <<
              "bool& free)" << (min != 0 ? " = 0;" : ";")
               << endl;
          }

          os << "virtual void" << endl
             << eserialize (a) << " ()" << (min != 0 ? " = 0;" : ";")
             << endl;

        }
      };

      struct AttributeCallback: Traversal::Attribute,
                                Traversal::AnyAttribute,
                                protected virtual Context
      {
        AttributeCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          if (a.optional ())
          {
            os << "virtual bool" << endl
               << epresent (a) << " ();"
               << endl;
          }

          String const& ret (ret_type (a.type ()));

          // Make it non-pure-virtual only if the return type is void.
          //
          os << "virtual " << ret << endl
             << ename (a) << " ()" << (ret != L"void" ? " = 0;" : ";")
             << endl;
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          os << "virtual bool" << endl
             << enext (a) << " ();"
             << endl;

          if (stl)
          {
            os << "virtual void" << endl
               << ename (a) << " (::std::string& ns, ::std::string& name);"
               << endl;
          }
          else
          {
            os << "virtual void" << endl
               << ename (a) << " (const char*& ns, const char*& name, " <<
              "bool& free);"
               << endl;
          }

          os << "virtual void" << endl
             << eserialize (a) << " ();"
             << endl;
        }
      };


      //
      //
      struct ParticleAccessor: Traversal::Element,
                               protected virtual Context
      {
        ParticleAccessor (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          os << "void" << endl
             << eserializer (e) << " (" << fq_name (e.type ()) << "&);"
             << endl;
        }
      };

      struct AttributeAccessor: Traversal::Attribute,
                                protected virtual Context
      {
        AttributeAccessor (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          os << "void" << endl
             << eserializer (a) << " (" << fq_name (a.type ()) << "&);"
             << endl;
        }
      };


      //
      //
      struct ParticleMember: Traversal::Element,
                               protected virtual Context
      {
        ParticleMember (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          os << fq_name (e.type ()) << "* " << emember (e) << ";";
        }
      };

      struct AttributeMember: Traversal::Attribute,
                                protected virtual Context
      {
        AttributeMember (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          os << fq_name (a.type ()) << "* " << emember (a) << ";";
        }
      };


      //
      //
      struct Complex : Traversal::Complex,
                       protected virtual Context
      {
        Complex (Context& c)
            : Context (c),
              compositor_callback_ (c),
              particle_callback_ (c),
              attribute_callback_ (c),
              particle_accessor_ (c),
              attribute_accessor_ (c),
              particle_member_ (c),
              attribute_member_ (c)
        {
          // Callback.
          //
          contains_compositor_callback_ >> compositor_callback_;
          compositor_callback_ >> contains_particle_callback_;
          contains_particle_callback_ >> compositor_callback_;
          contains_particle_callback_ >> particle_callback_;

          names_attribute_callback_ >> attribute_callback_;

          // Accessor.
          //
          contains_compositor_accessor_ >> compositor_accessor_;
          compositor_accessor_ >> contains_particle_accessor_;
          contains_particle_accessor_ >> compositor_accessor_;
          contains_particle_accessor_ >> particle_accessor_;

          names_attribute_accessor_ >> attribute_accessor_;

          // Member.
          //
          contains_compositor_member_ >> compositor_member_;
          compositor_member_ >> contains_particle_member_;
          contains_particle_member_ >> compositor_member_;
          contains_particle_member_ >> particle_member_;

          names_attribute_member_ >> attribute_member_;
        }

        virtual Void
        traverse (Type& c)
        {
          String const& name (ename (c));

          // In case of an inheritance-by-restriction, we don't need to
          // generate serializer callbacks, etc. since they are the same
          // as in the base. We only need the serialization/validation code.
          //
          Boolean restriction (restriction_p (c));

          Boolean he (has<Traversal::Element> (c));
          Boolean ha (has<Traversal::Attribute> (c));

          Boolean hae (has_particle<Traversal::Any> (c));
          Boolean haa (has<Traversal::AnyAttribute> (c));

          Boolean hra (false); // Has required attribute.
          if (ha)
          {
            RequiredAttributeTest test (hra);
            Traversal::Names names_test (test);
            names (c, names_test);
          }

          //
          //
          os << "class " << name << ": public virtual ";

          if (c.inherits_p ())
            os << fq_name (c.inherits ().base ());
          else
            os << complex_base;

          os << "{"
             << "public:" << endl
             << "// Serializer callbacks. Override them in your " <<
            "implementation." << endl
             << "//" << endl
             << endl;

          // pre
          //
          String const& arg (arg_type (c));

          Boolean same (c.inherits_p () &&
                        arg == arg_type (c.inherits ().base ()));

          if (same)
          {
            os << "// virtual void" << endl;

            if (arg == L"void")
              os << "// pre ();" << endl;
            else
              os << "// pre (" << arg << ") = 0;" << endl;

            os << endl;
          }
          else
          {
            os << "virtual void" << endl;

            if (arg == L"void")
              os << "pre ();";
            else
              os << "pre (" << arg << ") = 0;";

            os << endl;
          }

          // Member callbacks.
          //
          if (!restriction && (ha || he || hae || haa))
          {
            if (ha || haa)
            {
              os << "// Attributes." << endl
                 << "//" << endl;

              names (c, names_attribute_callback_);
            }

            if (he || hae)
            {
              os << "// Elements." << endl
                 << "//" << endl;

              contains_compositor (c, contains_compositor_callback_);
            }
          }

          // post
          //
          os << "// virtual void" << endl
             << "// post ();" << endl
             << endl;

          //
          //
          if (!restriction && (he || ha))
          {
            os << "// Serializer construction API." << endl
               << "//" << endl;

            os << "void" << endl
               << "serializers (";

            {
              SerializerParamDecl decl (*this, false);
              decl.traverse (c);
            }

            os << ");"
               << endl;

            if (ha)
            {
              os << "// Individual attribute serializers." << endl
                 << "//" << endl;

              names (c, names_attribute_accessor_);
            }

            if (he)
            {
              os << "// Individual element serializers." << endl
                 << "//" << endl;

              contains_compositor (c, contains_compositor_accessor_);
            }
          }

          // Default c-tor.
          //
          if (!restriction && (he || ha))
          {
            os << "// Constructor." << endl
               << "//" << endl
               << name << " ();"
               << endl;
          }

          // Implementation.
          //
          if (he || ha || hae || haa)
          {
            os << "// Implementation." << endl
               << "//" << endl
               << "public:" << endl;

            if (ha || haa)
            {
              os << "virtual void" << endl
                 << "_serialize_attributes ();"
                 << endl;
            }

            if (he || hae)
            {
              os << "virtual void" << endl
                 << "_serialize_content ();"
                 << endl;
            }
          }

          if (!restriction && (he || ha))
          {
            os << "protected:" << endl;

            if (ha)
              names (c, names_attribute_member_);

            if (he)
              contains_compositor (c, contains_compositor_member_);
          }

          os << "};";
        }

      private:
        //
        //
        CompositorCallback compositor_callback_;
        ParticleCallback particle_callback_;
        Traversal::ContainsCompositor contains_compositor_callback_;
        Traversal::ContainsParticle contains_particle_callback_;

        AttributeCallback attribute_callback_;
        Traversal::Names names_attribute_callback_;

        //
        //
        Traversal::Compositor compositor_accessor_;
        ParticleAccessor particle_accessor_;
        Traversal::ContainsCompositor contains_compositor_accessor_;
        Traversal::ContainsParticle contains_particle_accessor_;

        AttributeAccessor attribute_accessor_;
        Traversal::Names names_attribute_accessor_;

        //
        //
        Traversal::Compositor compositor_member_;
        ParticleMember particle_member_;
        Traversal::ContainsCompositor contains_compositor_member_;
        Traversal::ContainsParticle contains_particle_member_;

        AttributeMember attribute_member_;
        Traversal::Names names_attribute_member_;
      };

      struct FundType : Context,
                        Traversal::AnyType,
                        Traversal::AnySimpleType,

                        Traversal::Fundamental::Byte,
                        Traversal::Fundamental::UnsignedByte,
                        Traversal::Fundamental::Short,
                        Traversal::Fundamental::UnsignedShort,
                        Traversal::Fundamental::Int,
                        Traversal::Fundamental::UnsignedInt,
                        Traversal::Fundamental::Long,
                        Traversal::Fundamental::UnsignedLong,
                        Traversal::Fundamental::Integer,
                        Traversal::Fundamental::NonPositiveInteger,
                        Traversal::Fundamental::NonNegativeInteger,
                        Traversal::Fundamental::PositiveInteger,
                        Traversal::Fundamental::NegativeInteger,

                        Traversal::Fundamental::Boolean,
                        Traversal::Fundamental::Float,
                        Traversal::Fundamental::Double,
                        Traversal::Fundamental::Decimal,

                        Traversal::Fundamental::String,
                        Traversal::Fundamental::NormalizedString,
                        Traversal::Fundamental::Token,
                        Traversal::Fundamental::Name,
                        Traversal::Fundamental::NameToken,
                        Traversal::Fundamental::NameTokens,
                        Traversal::Fundamental::NCName,
                        Traversal::Fundamental::Language,

                        Traversal::Fundamental::QName,

                        Traversal::Fundamental::Id,
                        Traversal::Fundamental::IdRef,
                        Traversal::Fundamental::IdRefs,

                        Traversal::Fundamental::AnyURI,

                        Traversal::Fundamental::Base64Binary,
                        Traversal::Fundamental::HexBinary,

                        Traversal::Fundamental::Date,
                        Traversal::Fundamental::DateTime,
                        Traversal::Fundamental::Duration,
                        Traversal::Fundamental::Day,
                        Traversal::Fundamental::Month,
                        Traversal::Fundamental::MonthDay,
                        Traversal::Fundamental::Year,
                        Traversal::Fundamental::YearMonth,
                        Traversal::Fundamental::Time,

                        Traversal::Fundamental::Entity,
                        Traversal::Fundamental::Entities
      {
        FundType (Context& c)
            : Context (c), xs_ns_ (xs_ns_name ())
        {
          impl_ns_ = "::xsde::cxx::serializer::";
          impl_ns_ += (validation ? L"validating" : L"non_validating");

          if (options.value<CLI::no_stl> ())
          {
            qname_type_ = L"const " + xs_ns_ + L"::qname*";
            string_type_ = L"const char*";
            string_seq_type_ = L"const " + xs_ns_ + L"::string_sequence*";
          }
          else
          {
            qname_type_ = xs_ns_ + L"::qname";
            string_type_ = L"::std::string";
            string_seq_type_ = xs_ns_ + L"::string_sequence";
          }

          buffer_type_ = L"const " + xs_ns_ + L"::buffer*";

          if (options.value<CLI::no_long_long> ())
          {
            long_type_ = L"long";
            unsigned_long_type_ = L"unsigned long";
          }
          else
          {
            long_type_ = L"long long";
            unsigned_long_type_ = L"unsigned long long";
          }
        }

        // anyType & anySimpleType.
        //
        virtual Void
        traverse (SemanticGraph::AnyType& t)
        {
          gen_typedef (t, "void", "any_type_sskel", "any_type_simpl");
        }

        virtual Void
        traverse (SemanticGraph::AnySimpleType& t)
        {
          gen_typedef (t, "void",
                       "any_simple_type_sskel", "any_simple_type_simpl");
        }

        // Boolean.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Boolean& t)
        {
          gen_typedef (t, "bool", "boolean_sskel", "boolean_simpl");
        }

        // Integral types.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Byte& t)
        {
          gen_typedef (t, "signed char", "byte_sskel", "byte_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedByte& t)
        {
          gen_typedef (t, "unsigned char",
                       "unsigned_byte_sskel", "unsigned_byte_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Short& t)
        {
          gen_typedef (t, "short", "short_sskel", "short_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedShort& t)
        {
          gen_typedef (t, "unsigned short",
                       "unsigned_short_sskel", "unsigned_short_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Int& t)
        {
          gen_typedef (t, "int", "int_sskel", "int_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedInt& t)
        {
          gen_typedef (t, "unsigned int",
                       "unsigned_int_sskel", "unsigned_int_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Long& t)
        {
          gen_typedef (t, long_type_, "long_sskel", "long_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedLong& t)
        {
          gen_typedef (t, unsigned_long_type_,
                       "unsigned_long_sskel", "unsigned_long_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Integer& t)
        {
          gen_typedef (t, "long", "integer_sskel", "integer_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NegativeInteger& t)
        {
          gen_typedef (t, "long",
                       "negative_integer_sskel", "negative_integer_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NonPositiveInteger& t)
        {
          gen_typedef (t, "long",
                       "non_positive_integer_sskel",
                       "non_positive_integer_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::PositiveInteger& t)
        {
          gen_typedef (t, "unsigned long",
                       "positive_integer_sskel", "positive_integer_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NonNegativeInteger& t)
        {
          gen_typedef (t, "unsigned long",
                       "non_negative_integer_sskel",
                       "non_negative_integer_simpl");
        }

        // Floats.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Float& t)
        {
          gen_typedef (t, "float", "float_sskel", "float_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Double& t)
        {
          gen_typedef (t, "double", "double_sskel", "double_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Decimal& t)
        {
          gen_typedef (t, "double", "decimal_sskel", "decimal_simpl");
        }

        // Strings.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::String& t)
        {
          gen_typedef (t, string_type_, "string_sskel", "string_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NormalizedString& t)
        {
          gen_typedef (t, string_type_,
                       "normalized_string_sskel", "normalized_string_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Token& t)
        {
          gen_typedef (t, string_type_, "token_sskel", "token_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NameToken& t)
        {
          nmtoken_ = gen_typedef (t, string_type_,
                                  "nmtoken_sskel", "nmtoken_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NameTokens& t)
        {
          // NMTOKENS uses NMTOKEN implementation to serialize individual
	  // items. As a result, we don't generate NMTOKENS if we didn't
	  // generate NMTOKEN. Here we assume NMTOKEN is handled before
	  // NMTOKENS.
          //
          if(nmtoken_)
            gen_typedef (t, string_seq_type_,
                         "nmtokens_sskel", "nmtokens_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Name& t)
        {
          gen_typedef (t, string_type_, "name_sskel", "name_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NCName& t)
        {
          gen_typedef (t, string_type_, "ncname_sskel", "ncname_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Language& t)
        {
          gen_typedef (t, string_type_, "language_sskel", "language_simpl");
        }

        // Qualified name.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::QName& t)
        {
          gen_typedef (t, qname_type_, "qname_sskel", "qname_simpl");
        }

        // ID/IDREF.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Id& t)
        {
          gen_typedef (t, string_type_, "id_sskel", "id_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::IdRef& t)
        {
          idref_ = gen_typedef (t, string_type_, "idref_sskel", "idref_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::IdRefs& t)
        {
          // IDREFS uses IDREF implementation to serialize individual items.
          // As a result, we don't generate IDREFS if we didn't generate
          // IDREF. Here we assume IDREF is handled before IDREFS.
          //
          if (idref_)
            gen_typedef (t, string_seq_type_, "idrefs_sskel", "idrefs_simpl");
        }

        // URI.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::AnyURI& t)
        {
          gen_typedef (t, string_type_, "uri_sskel", "uri_simpl");
        }

        // Binary.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Base64Binary& t)
        {
          gen_typedef (t, buffer_type_,
                       "base64_binary_sskel", "base64_binary_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::HexBinary& t)
        {
          gen_typedef (t, buffer_type_,
                       "hex_binary_sskel", "hex_binary_simpl");
        }


        // Date/time.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Date& t)
        {
          gen_typedef (t, xs_ns_ + L"::date", "date_sskel", "date_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::DateTime& t)
        {
          gen_typedef (t, xs_ns_ + L"::date_time",
                       "date_time_sskel", "date_time_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Duration& t)
        {
          gen_typedef (t, xs_ns_ + L"::duration",
                       "duration_sskel", "duration_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Day& t)
        {
          gen_typedef (t, xs_ns_ + L"::gday", "gday_sskel", "gday_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Month& t)
        {
          gen_typedef (t, xs_ns_ + L"::gmonth",
                       "gmonth_sskel", "gmonth_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::MonthDay& t)
        {
          gen_typedef (t, xs_ns_ + L"::gmonth_day",
                       "gmonth_day_sskel", "gmonth_day_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Year& t)
        {
          gen_typedef (t, xs_ns_ + L"::gyear", "gyear_sskel", "gyear_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::YearMonth& t)
        {
          gen_typedef (t, xs_ns_ + L"::gyear_month",
                       "gyear_month_sskel", "gyear_month_simpl");
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Time& t)
        {
          gen_typedef (t, xs_ns_ + L"::time", "time_sskel", "time_simpl");
        }

        // Entity.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Entity& t)
        {
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Entities& t)
        {
        }

      private:
        Boolean
        gen_typedef (SemanticGraph::Type& t,
                     String const& type,
                     String const& sskel,
                     String const& simpl)
        {
          if (ret_type (t) == type)
          {
            os << "typedef " << impl_ns_ << "::" << sskel << " " <<
              ename (t) << ";";

            String const& simpl_name (t.context ().get<String> ("impl"));

            os << "typedef " << impl_ns_ << "::" << simpl << " " <<
              simpl_name << ";"
               << endl;

            return true;
          }

          return false;
        }

        String xs_ns_;
        String impl_ns_;
        String qname_type_;
        String string_type_;
        String buffer_type_;
        String string_seq_type_;
        String long_type_;
        String unsigned_long_type_;

        Boolean idref_;
        Boolean nmtoken_;
      };

      struct FundNamespace : Namespace,
                             protected virtual Context
      {
        FundNamespace (Context& c)
            : Context (c), Namespace (c)
        {
        }

        void
        traverse (Type& ns)
        {
          pre (ns);

          String impl ("::xsde::cxx::serializer::");
          impl += (validation ? L"validating" : L"non_validating");

          String const c (char_type);

          os << "// Built-in XML Schema types mapping." << endl
             << "//" << endl
             << "using ::xsde::cxx::string_sequence;"
             << "using ::xsde::cxx::qname;"
             << "using ::xsde::cxx::buffer;"
             << "using ::xsde::cxx::time_zone;"
             << "using ::xsde::cxx::gday;"
             << "using ::xsde::cxx::gmonth;"
             << "using ::xsde::cxx::gyear;"
             << "using ::xsde::cxx::gmonth_day;"
             << "using ::xsde::cxx::gyear_month;"
             << "using ::xsde::cxx::date;"
             << "using ::xsde::cxx::time;"
             << "using ::xsde::cxx::date_time;"
             << "using ::xsde::cxx::duration;"
             << endl;

          os << "// Base serializer skeletons." << endl
             << "//" << endl
             << "using ::xsde::cxx::serializer::serializer_base;"
             << "typedef " << impl << "::simple_content " <<
            "serializer_simple_content;"
             << "typedef " << impl << "::complex_content " <<
            "serializer_complex_content;"
             << endl;

          os << "// Serializer skeletons and implementations for the" << endl
             << "// XML Schema built-in types." << endl
             << "//" << endl;

          names (ns);

          os << "// Error codes." << endl
             << "//" << endl;

          if (!exceptions)
            os << "using xsde::cxx::sys_error;";

          os << "typedef xsde::cxx::serializer::genx::xml_error " <<
            "serializer_xml_error;";

          if (validation)
            os << "typedef xsde::cxx::schema_error " <<
              "serializer_schema_error;";

          os << endl;

          if (exceptions)
          {
            os << "// Exceptions." << endl
               << "//" << endl
               << "typedef xsde::cxx::serializer::exception " <<
              "serializer_exception;"
               << "typedef xsde::cxx::serializer::xml serializer_xml;";

            if (validation)
              os << "typedef xsde::cxx::serializer::schema " <<
                "serializer_schema;";

            os << endl;
          }
          else
            os << "// Error object." << endl
               << "//" << endl
               << "typedef xsde::cxx::serializer::error serializer_error;"
               << endl;

          os << "// Document serializer." << endl
             << "//" << endl
             << "using xsde::cxx::serializer::genx::writer;"
             << "using xsde::cxx::serializer::genx::document_simpl;"
             << endl;

          os << "// Serializer context." << endl
             << "//" << endl
             << "typedef xsde::cxx::serializer::context serializer_context;"
             << endl;

          post (ns);
        }
      };
    }

    Void
    generate_serializer_header (Context& ctx, String const& expr)
    {
      ctx.os << "#include <xsde/config.h>" << endl
             << endl;

      // std::string or xsde::cxx::string is used in wildcard API.
      //
      if (ctx.stl)
      {
        ctx.os << "#include <string>" << endl
               << endl;
      }
      else
      {
        ctx.os << "#include <xsde/cxx/string.hxx>" << endl
               << endl;
      }

      // Data types.
      //
      ctx.os << "#include <xsde/cxx/serializer/xml-schema.hxx>" << endl
             << endl;

      // Error handling.
      //
      if (ctx.exceptions)
        ctx.os << "#include <xsde/cxx/serializer/exceptions.hxx>" << endl
               << endl;
      else
      {
        ctx.os << "#include <xsde/cxx/sys-error.hxx>" << endl;

        if (ctx.validation)
          ctx.os << "#include <xsde/cxx/schema-error.hxx>" << endl;

        ctx.os << "#include <xsde/cxx/serializer/error.hxx>" << endl
               << "#include <xsde/cxx/serializer/genx/xml-error.hxx>" << endl;

        ctx.os << endl;
      }

      // Serializers.
      //
      if (ctx.validation)
        ctx.os << "#include <xsde/cxx/serializer/validating/serializer.hxx>" << endl
               << "#include <xsde/cxx/serializer/validating/xml-schema-sskel.hxx>" << endl
               << "#include <xsde/cxx/serializer/validating/xml-schema-simpl.hxx>" << endl
               << endl;
      else
        ctx.os << "#include <xsde/cxx/serializer/non-validating/serializer.hxx>" << endl
               << "#include <xsde/cxx/serializer/non-validating/xml-schema-sskel.hxx>" << endl
               << "#include <xsde/cxx/serializer/non-validating/xml-schema-simpl.hxx>" << endl
               << endl;

      // Document.
      //
      ctx.os << "#include <xsde/cxx/serializer/genx/document.hxx>" << endl
             << endl;

      // Generate includes that came from the type map.
      //
      if (ctx.schema_root.context ().count ("includes"))
      {
        typedef Cult::Containers::Set<String> Includes;

        Includes& is (
          ctx.schema_root.context ().get<Includes> ("includes"));

        for (Includes::ConstReverseIterator i (is.rbegin ());
             i != is.rend (); ++i)
        {
          ctx.os << "#include " << *i << endl;
        }

        ctx.os << endl;
      }

      // Generate fundamental types.
      //
      {
        Traversal::Schema schema, xsd;
        Traversal::Implies implies;
        Traversal::Names names;
        FundNamespace ns (ctx);

        schema >> implies >> xsd >> names >> ns;

        Traversal::Names ns_names;
        FundType type (ctx);

        ns >> ns_names >> type;

        schema.dispatch (ctx.schema_root);
      }

      // Generate user type mapping.
      //
      {
        Traversal::Schema schema;

        Traversal::Sources sources;
        Includes includes (ctx, expr);
        Traversal::Names schema_names;

        Namespace ns (ctx);
        Traversal::Names names;

        schema >> includes;
        schema >> sources >> schema;
        schema >> schema_names >> ns >> names;

        List list (ctx);
        Union union_ (ctx);
        Complex complex (ctx);
        Enumeration enumeration (ctx);

        names >> list;
        names >> union_;
        names >> complex;
        names >> enumeration;

        schema.dispatch (ctx.schema_root);
      }
    }
  }
}
