// file      : xsde/cxx/serializer/generator.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <type-map/lexer.hxx>
#include <type-map/parser.hxx>
#include <type-map/type-map.hxx>

#include <cxx/serializer/elements.hxx>
#include <cxx/serializer/generator.hxx>

#include <cxx/serializer/validator.hxx>
#include <cxx/serializer/name-processor.hxx>
//#include <cxx/serializer/state-processor.hxx>
#include <cxx/serializer/type-processor.hxx>

#include <cxx/serializer/serializer-forward.hxx>
#include <cxx/serializer/serializer-header.hxx>
#include <cxx/serializer/serializer-inline.hxx>
#include <cxx/serializer/serializer-source.hxx>

/*
#include <cxx/serializer/impl-header.hxx>
#include <cxx/serializer/impl-source.hxx>
#include <cxx/serializer/driver-source.hxx>
*/

#include <cxx/serializer/element-validation-source.hxx>
#include <cxx/serializer/attribute-validation-source.hxx>

#include <xsd-frontend/semantic-graph.hxx>

#include <backend-elements/regex.hxx>
#include <backend-elements/indentation/cxx.hxx>
#include <backend-elements/indentation/sloc.hxx>
#include <backend-elements/indentation/clip.hxx>

#include <cult/containers/set.hxx>
#include <cult/containers/vector.hxx>

#include <boost/filesystem/fstream.hpp>

#include <cctype>   // std::toupper
#include <iostream>

#include <usage.hxx>

#include "../../../libxsde/xsde/cxx/version.hxx"

using std::endl;
using std::wcerr;

using namespace XSDFrontend::SemanticGraph;

//
//
typedef
boost::filesystem::wifstream
WideInputFileStream;

typedef
boost::filesystem::wofstream
WideOutputFileStream;

typedef
boost::filesystem::ifstream
NarrowInputFileStream;

namespace CXX
{
  namespace
  {
    Char const copyright_gpl[] =
    "// Copyright (C) 2005-2007 Code Synthesis Tools CC\n"
    "//\n"
    "// This program was generated by CodeSynthesis XSD/e, an XML Schema\n"
    "// to C++ data binding compiler for embedded systems.\n"
    "//\n"
    "// This program is free software; you can redistribute it and/or modify\n"
    "// it under the terms of the GNU General Public License version 2 as\n"
    "// published by the Free Software Foundation.\n"
    "//\n"
    "// This program is distributed in the hope that it will be useful,\n"
    "// but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "// GNU General Public License for more details.\n"
    "//\n"
    "// You should have received a copy of the GNU General Public License\n"
    "// along with this program; if not, write to the Free Software\n"
    "// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA\n"
    "//\n"
    "//\n\n";

    Char const copyright_proprietary[] =
    "// Copyright (C) 2005-2007 Code Synthesis Tools CC\n"
    "//\n"
    "// This program was generated by CodeSynthesis XSD/e, an XML Schema to\n"
    "// C++ data binding compiler for embedded systems, in the Proprietary\n"
    "// License mode. You should have received a proprietary license from\n"
    "// Code Synthesis Tools CC prior to generating this code. See the\n"
    "// license text for conditions.\n"
    "//\n\n";

    Char const copyright_impl[] =
    "// Not copyrighted - public domain.\n"
    "//\n"
    "// This sample serializer implementation was generated by CodeSynthesis XSD/e,\n"
    "// an XML Schema to C++ data binding compiler for embedded systems. You may\n"
    "// use it in your programs without any restrictions.\n"
    "//\n\n";
  }

  namespace Serializer
  {
    namespace CLI
    {
      extern Key type_map                 = "type-map";
      extern Key no_stl                   = "no-stl";
      extern Key no_iostream              = "no-iostream";
      extern Key no_exceptions            = "no-exceptions";
      extern Key no_long_long             = "no-long-long";
      extern Key output_dir               = "output-dir";
      extern Key generate_inline          = "generate-inline";
      extern Key suppress_validation      = "suppress-validation";
      /*
      extern Key generate_noop_impl       = "generate-noop-impl";
      extern Key generate_print_impl      = "generate-print-impl";
      extern Key generate_test_driver     = "generate-test-driver";
      */
      extern Key force_overwrite          = "force-overwrite";
      extern Key root_element_first       = "root-element-first";
      extern Key root_element_last        = "root-element-last";
      extern Key root_element             = "root-element";
      extern Key skel_type_suffix         = "skel-type-suffix";
      extern Key skel_file_suffix         = "skel-file-suffix";
      extern Key impl_type_suffix         = "impl-type-suffix";
      extern Key impl_file_suffix         = "impl-file-suffix";
      extern Key namespace_map            = "namespace-map";
      extern Key namespace_regex          = "namespace-regex";
      extern Key namespace_regex_trace    = "namespace-regex-trace";
      extern Key include_with_brackets    = "include-with-brackets";
      extern Key include_prefix           = "include-prefix";
      extern Key include_regex            = "include-regex";
      extern Key include_regex_trace      = "include-regex-trace";
      extern Key hxx_suffix               = "hxx-suffix";
      extern Key ixx_suffix               = "ixx-suffix";
      extern Key cxx_suffix               = "cxx-suffix";
      extern Key hxx_regex                = "hxx-regex";
      extern Key ixx_regex                = "ixx-regex";
      extern Key cxx_regex                = "cxx-regex";
      extern Key hxx_prologue             = "hxx-prologue";
      extern Key ixx_prologue             = "ixx-prologue";
      extern Key cxx_prologue             = "cxx-prologue";
      extern Key prologue                 = "prologue";
      extern Key hxx_epilogue             = "hxx-epilogue";
      extern Key ixx_epilogue             = "ixx-epilogue";
      extern Key cxx_epilogue             = "cxx-epilogue";
      extern Key epilogue                 = "epilogue";
      extern Key hxx_prologue_file        = "hxx-prologue-file";
      extern Key ixx_prologue_file        = "ixx-prologue-file";
      extern Key cxx_prologue_file        = "cxx-prologue-file";
      extern Key prologue_file            = "prologue-file";
      extern Key hxx_epilogue_file        = "hxx-epilogue-file";
      extern Key ixx_epilogue_file        = "ixx-epilogue-file";
      extern Key cxx_epilogue_file        = "cxx-epilogue-file";
      extern Key epilogue_file            = "epilogue-file";
      extern Key show_anonymous           = "show-anonymous";
      extern Key show_sloc                = "show-sloc";
      extern Key proprietary_license      = "proprietary-license";
    }
  }

  Void Serializer::Generator::
  usage ()
  {
    std::wostream& e (wcerr);
    ::CLI::Indent::Clip< ::CLI::OptionsUsage, WideChar> clip (e);

    e << "--type-map <mapfile>" << endl
      << " Read XML Schema to C++ type mapping information\n"
      << " from <mapfile>. Repeat this option to specify\n"
      << " several type maps. Type maps are considered in\n"
      << " order of appearance and the first match is used."
      << endl;

    e << "--output-dir <dir>" << endl
      << " Write generated files to <dir> instead of the\n"
      << " current directory."
      << endl;

    e << "--no-stl" << endl
      << " Generate code that does not use STL."
      << endl;

    e << "--no-iostream" << endl
      << " Generate code that does not use the iostream\n"
      << " library."
      << endl;

    e << "--no-exceptions" << endl
      << " Generate code that does not use C++ exceptions."
      << endl;

    e << "--no-long-long" << endl
      << " Generate code that does not use the long long\n"
      << " and unsigned long long types."
      << endl;

    e << "--generate-inline" << endl
      << " Generate certain functions inline."
      << endl;

    e << "--suppress-validation" << endl
      << " Suppress generation of validation code."
      << endl;

    /*
    e << "--generate-noop-impl" << endl
      << " Generate a sample serializer implementation that\n"
      << " does nothing (no operation)."
      << endl;

    e << "--generate-print-impl" << endl
      << " Generate a sample serializer implementation that\n"
      << " prints the XML data to STDOUT."
      << endl;

    e << "--generate-test-driver" << endl
      << " Generate a test driver for the sample serializer\n"
      << " implementation."
      << endl;

    e << "--force-overwrite" << endl
      << " Force overwriting of the existing implementation\n"
      << " and test driver files."
      << endl;

    e << "--root-element-first" << endl
      << " Indicate that the first global element is the\n"
      << " document root."
      << endl;

    e << "--root-element-last" << endl
      << " Indicate that the last global element is the\n"
      << " document root."
      << endl;

    e << "--root-element <element>" << endl
      << " Indicate that <element> is the document root."
      << endl;
    */

    e << "--skel-type-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '_sskel' to\n"
      << " construct the names of generated serializer\n"
      << " skeletons."
      << endl;

    e << "--skel-file-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '-sskel' to\n"
      << " construct the names of generated serializer\n"
      << " skeleton files."
      << endl;

    e << "--impl-type-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '_simpl' to\n"
      << " construct the names of serializer implementations\n"
      << " for the built-in XML Schema types."
      << endl;

    /*

    Remove the previous version (above) when uncommenting.

    e << "--impl-type-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '_simpl' to\n"
      << " construct the names of serializer implementations\n"
      << " for the built-in XML Schema types and sample\n"
      << " serializer implementations."
      << endl;
    */

    /*
    e << "--impl-file-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '-simpl' to\n"
      << " construct the names of generated sample\n"
      << " serializer implementation files."
      << endl;
    */

    e << "--namespace-map <xns>=<cns>" << endl
      << " Map XML Schema namespace <xns> to C++ namespace\n"
      << " <cns>. Repeat this option to specify mapping for\n"
      << " more than one XML Schema namespace."
      << endl;

    e << "--namespace-regex <regex>" << endl
      << " Add <regex> to the list of regular expressions\n"
      << " used to translate XML Schema namespace names to\n"
      << " C++ namespace names. The first successful match\n"
      << " is used. The last provided expression is\n"
      << " considered first. Expression is a perl-like regex\n"
      << " in the form /pattern/replacement/."
      << endl;

    e << "--namespace-regex-trace" << endl
      << " Trace the process of applying regular expressions\n"
      << " specified with the --namespace-regex option."
      << endl;

    e << "--include-with-brackets" << endl
      << " Use angle brackets (<>) instead of quotes (\"\") in\n"
      << " generated #include directives."
      << endl;

    e << "--include-prefix <prefix>" << endl
      << " Add <prefix> to generated #include directive\n"
      << " paths."
      << endl;

    e << "--include-regex <regex>" << endl
      << " Add <regex> to the list of regular expressions\n"
      << " used to transform #include directive paths. The\n"
      << " first successful match is used. The last provided\n"
      << " expression is considered first. Expression is a\n"
      << " perl-like regex in the form /pattern/replacement/."
      << endl;

    e << "--include-regex-trace" << endl
      << " Trace the process of applying regular expressions\n"
      << " specified with the --include-regex option."
      << endl;

    e << "--hxx-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '.hxx' to\n"
      << " construct the name of the header file."
      << endl;

    e << "--ixx-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '.ixx' to\n"
      << " construct the name of the inline file."
      << endl;

    e << "--cxx-suffix <suffix>" << endl
      << " Use <suffix> instead of the default '.cxx' to\n"
      << " construct the name of the source file."
      << endl;

    e << "--hxx-regex <regex>" << endl
      << " Use <regex> to construct the name of the header\n"
      << " file. Expression is a perl regex in the form\n"
      << " /pattern/replacement/."
      << endl;

    e << "--ixx-regex <regex>" << endl
      << " Use <regex> to construct the name of the inline\n"
      << " file. Expression is a perl regex in the form\n"
      << " /pattern/replacement/."
      << endl;

    e << "--cxx-regex <regex>" << endl
      << " Use <regex> to construct the name of the source\n"
      << " file. Expression is a perl regex in the form\n"
      << " /pattern/replacement/."
      << endl;


    // Prologues.
    //
    e << "--hxx-prologue <text>" << endl
      << " Insert <text> at the beginning of the header file."
      << endl;

    e << "--ixx-prologue <text>" << endl
      << " Insert <text> at the beginning of the inline file."
      << endl;

    e << "--cxx-prologue <text>" << endl
      << " Insert <text> at the beginning of the source file."
      << endl;

    e << "--prologue <text>" << endl
      << " Insert <text> at the beginning of each generated\n"
      << " file for which there is no file-specific prologue."
      << endl;


    // Epilogues.
    //
    e << "--hxx-epilogue <text>" << endl
      << " Insert <text> at the end of the header file."
      << endl;

    e << "--ixx-epilogue <text>" << endl
      << " Insert <text> at the end of the inline file."
      << endl;

    e << "--cxx-epilogue <text>" << endl
      << " Insert <text> at the end of the source file."
      << endl;

    e << "--epilogue <text>" << endl
      << " Insert <text> at the end of each generated file\n"
      << " for which there is no file-specific epilogue."
      << endl;


    // Prologue files.
    //
    e << "--hxx-prologue-file <file>" << endl
      << " Insert the content of the <file> at the beginning\n"
      << " of the header file."
      << endl;

    e << "--ixx-prologue-file <file>" << endl
      << " Insert the content of the <file> at the beginning\n"
      << " of the inline file."
      << endl;

    e << "--cxx-prologue-file <file>" << endl
      << " Insert the content of the <file> at the beginning\n"
      << " of the source file."
      << endl;

    e << "--prologue-file <file>" << endl
      << " Insert the content of the <file> at the beginning\n"
      << " of each generated file for which there is no file-\n"
      << " specific prologue file."
      << endl;


    // Epilogue files.
    //
    e << "--hxx-epilogue-file <file>" << endl
      << " Insert the content of the <file> at the end of\n"
      << " the header file."
      << endl;

    e << "--ixx-epilogue-file <file>" << endl
      << " Insert the content of the <file> at the end of\n"
      << " the inline file."
      << endl;

    e << "--cxx-epilogue-file <file>" << endl
      << " Insert the content of the <file> at the end of\n"
      << " the source file."
      << endl;

    e << "--epilogue-file <file>" << endl
      << " Insert the content of the <file> at the end of\n"
      << " each generated file for which there is no file-\n"
      << " specific epilogue file."
      << endl;


    // Misc.
    //
    e << "--show-anonymous" << endl
      << " Show elements and attributes that are of anonymous\n"
      << " types."
      << endl;

    e << "--show-sloc" << endl
      << " Show the number of generated physical source lines\n"
      << " of code (SLOC)."
      << endl;

    e << "--sloc-limit <num>" << endl
      << " Check that the number of generated physical source\n"
      << " lines of code (SLOC) does not exceed <num>."
      << endl;

    e << "--proprietary-license" << endl
      << " Indicate that the generated code is licensed under\n"
      << " a proprietary license instead of the GPL."
      << endl;
  }

  Serializer::CLI::OptionsSpec Serializer::Generator::
  options_spec ()
  {
    CLI::OptionsSpec spec;

    spec.option<CLI::skel_file_suffix> ().default_value ("-sskel");
    spec.option<CLI::skel_type_suffix> ().default_value ("_sskel");
    spec.option<CLI::impl_file_suffix> ().default_value ("-simpl");
    spec.option<CLI::impl_type_suffix> ().default_value ("_simpl");

    spec.option<CLI::hxx_suffix> ().default_value (".hxx");
    spec.option<CLI::ixx_suffix> ().default_value (".ixx");
    spec.option<CLI::cxx_suffix> ().default_value (".cxx");

    return spec;
  }


  namespace
  {
    template <typename S>
    Void
    open (S& ifs, NarrowString const& path)
    {
      try
      {
        Path fs_path (path, boost::filesystem::native);
        ifs.open (fs_path, std::ios_base::in | std::ios_base::binary);

        if (!ifs.is_open ())
        {
          wcerr << path.c_str () << ": error: unable to open in read mode"
                << endl;

          throw Serializer::Generator::Failed ();
        }
      }
      catch (InvalidPath const&)
      {
        wcerr << "error: '" << path.c_str () << "' is not a valid "
              << "filesystem path" << endl;

        throw Serializer::Generator::Failed ();
      }
    }

    Void
    append (WideOutputFileStream& os,
            NarrowString const& path,
            WideInputFileStream& default_is)
    {
      using std::ios_base;

      if (path)
      {
        WideInputFileStream is;
        open (is, path);
        os << is.rdbuf ();
      }
      else if (default_is.is_open ())
      {
        os << default_is.rdbuf ();
        default_is.seekg (0, ios_base::beg);
      }
    }

    Void
    append (WideOutputFileStream& os,
            Cult::Containers::Vector<NarrowString> const& primary,
            Cult::Containers::Vector<NarrowString> const& def)
    {
      Cult::Containers::Vector<NarrowString> const& v (
        primary.empty () ? def : primary);

      for (Containers::Vector<NarrowString>::ConstIterator
             i (v.begin ()), e (v.end ()); i != e; ++i)
      {
        os << i->c_str () << endl;
      }
    }
  }


  UnsignedLong Serializer::Generator::
  generate (Serializer::CLI::Options const& ops,
            Schema& schema,
            Path const& file_path,
            AutoUnlinks& unlinks)
  {
    using std::ios_base;
    using BackendElements::Regex::perl_s;
    namespace Indentation = BackendElements::Indentation;

    try
    {
      /*
      Boolean impl (ops.value<CLI::generate_noop_impl> () ||
                    ops.value<CLI::generate_print_impl> ());

      Boolean driver (ops.value<CLI::generate_test_driver> ());
      */

      Boolean impl (false);
      Boolean driver (false);

      // Evaluate the graph for possibility of generating something useful.
      //
      {
        Validator validator;
        if (!validator.validate (ops, schema, file_path))
          throw Failed ();
      }

      // Process names.
      //
      {
        NameProcessor proc;
        proc.process (schema,
                      file_path,
                      impl || driver,
                      ops.value<CLI::skel_type_suffix> (),
                      ops.value<CLI::impl_type_suffix> ());
      }

      //
      //
      Boolean validation (!ops.value<CLI::suppress_validation> ());

      // Read-in type maps.
      //
      TypeMap::Namespaces type_map;
      {
        using namespace TypeMap;
        typedef Containers::Vector<NarrowString> Files;

        Files const& files (ops.value<CLI::type_map> ());

        for (Files::ConstIterator f (files.begin ()); f != files.end (); ++f )
        {
          NarrowInputFileStream ifs;
          open (ifs, *f);

          Lexer l (ifs, *f);
          TypeMap::Parser p (l, *f);

          if (!p.parse (type_map))
            throw Failed ();
        }

        // Add the built-in mappings at the end.
        //
        String xns;
        {
          Context ctx (std::wcerr, schema, ops);
          xns = ctx.xs_ns_name ();
        }

        if (ops.value<CLI::no_stl> ())
        {
          TypeMap::Namespace xsd_std (
            L"http://www\\.w3\\.org/2001/XMLSchema");

          String qname (L"const " + xns + L"::qname*");
          String string_seq (L"const " + xns + L"::string_sequence*");

          xsd_std.types_push_back ("string", "const char*", "const char*");
          xsd_std.types_push_back ("normalizedString", "const char*", "const char*");
          xsd_std.types_push_back ("token", "const char*", "const char*");
          xsd_std.types_push_back ("Name", "const char*", "const char*");
          xsd_std.types_push_back ("NMTOKEN", "const char*", "const char*");
          xsd_std.types_push_back ("NMTOKENS", string_seq, string_seq);
          xsd_std.types_push_back ("NCName", "const char*", "const char*");

          xsd_std.types_push_back ("ID", "const char*", "const char*");
          xsd_std.types_push_back ("IDREF", "const char*", "const char*");
          xsd_std.types_push_back ("IDREFS", string_seq, string_seq);

          xsd_std.types_push_back ("language", "const char*", "const char*");
          xsd_std.types_push_back ("anyURI", "const char*", "const char*");
          xsd_std.types_push_back ("QName", qname, qname);

          type_map.push_back (xsd_std);
        }
        else
        {
          TypeMap::Namespace xsd_std (
            L"http://www\\.w3\\.org/2001/XMLSchema");

          String qname (xns + L"::qname");
          String string_seq (xns + L"::string_sequence");

          xsd_std.types_push_back ("string", "::std::string");
          xsd_std.types_push_back ("normalizedString", "::std::string");
          xsd_std.types_push_back ("token", "::std::string");
          xsd_std.types_push_back ("Name", "::std::string");
          xsd_std.types_push_back ("NMTOKEN", "::std::string");
          xsd_std.types_push_back ("NMTOKENS", string_seq);
          xsd_std.types_push_back ("NCName", "::std::string");

          xsd_std.types_push_back ("ID", "::std::string");
          xsd_std.types_push_back ("IDREF", "::std::string");
          xsd_std.types_push_back ("IDREFS", string_seq);

          xsd_std.types_push_back ("language", "::std::string");
          xsd_std.types_push_back ("anyURI", "::std::string");
          xsd_std.types_push_back ("QName", qname);

          type_map.push_back (xsd_std);
        }

        String buffer (L"const " + xns + L"::buffer*");

        TypeMap::Namespace xsd (L"http://www\\.w3\\.org/2001/XMLSchema");

        xsd.types_push_back ("boolean", "bool", "bool");

        xsd.types_push_back ("byte", "signed char", "signed char");
        xsd.types_push_back ("unsignedByte", "unsigned char", "unsigned char");

        xsd.types_push_back ("short", "short", "short");
        xsd.types_push_back ("unsignedShort", "unsigned short", "unsigned short");

        xsd.types_push_back ("int", "int", "int");
        xsd.types_push_back ("unsignedInt", "unsigned int", "unsigned int");

        if (ops.value<CLI::no_long_long> ())
        {
          xsd.types_push_back ("long", "long", "long");
          xsd.types_push_back ("unsignedLong", "unsigned long", "unsigned long");
        }
        else
        {
          xsd.types_push_back ("long", "long long", "long long");
          xsd.types_push_back ("unsignedLong", "unsigned long long", "unsigned long long");
        }

        xsd.types_push_back ("integer", "long", "long");

        xsd.types_push_back ("negativeInteger", "long", "long");
        xsd.types_push_back ("nonPositiveInteger", "long", "long");

        xsd.types_push_back ("positiveInteger", "unsigned long", "unsigned long");
        xsd.types_push_back ("nonNegativeInteger", "unsigned long", "unsigned long");

        xsd.types_push_back ("float", "float", "float");
        xsd.types_push_back ("double", "double", "double");
        xsd.types_push_back ("decimal", "double", "double");

        xsd.types_push_back ("base64Binary", buffer, buffer);
        xsd.types_push_back ("hexBinary", buffer, buffer);

        xsd.types_push_back ("gDay", xns + L"::gday");
        xsd.types_push_back ("gMonth", xns + L"::gmonth");
        xsd.types_push_back ("gYear", xns + L"::gyear");
        xsd.types_push_back ("gMonthDay", xns + L"::gmonth_day");
        xsd.types_push_back ("gYearMonth", xns + L"::gyear_month");
        xsd.types_push_back ("date", xns + L"::date");
        xsd.types_push_back ("time", xns + L"::time");
        xsd.types_push_back ("dateTime", xns + L"::date_time");
        xsd.types_push_back ("duration", xns + L"::duration");

        type_map.push_back (xsd);

        // Everyhting else maps to void.
        //
        TypeMap::Namespace rest (".*");
        rest.types_push_back (".*", "void", "void");
        type_map.push_back (rest);
      }

      // Process types.
      //
      {
        TypeProcessor proc;
        proc.process (schema, type_map);
      }

      // Generate code.
      //
      NarrowString name (file_path.leaf ());
      NarrowString skel_suffix (ops.value <CLI::skel_file_suffix> ());
      NarrowString impl_suffix (ops.value <CLI::impl_file_suffix> ());

      NarrowString hxx_suffix (ops.value <CLI::hxx_suffix> ());
      NarrowString ixx_suffix (ops.value <CLI::ixx_suffix> ());
      NarrowString cxx_suffix (ops.value <CLI::cxx_suffix> ());


      NarrowString hxx_expr (
        ops.value <CLI::hxx_regex> ().empty ()
        ? "#^(.+?)(\\.[^./\\\\]+)?$#$1" + skel_suffix + hxx_suffix + "#"
        : ops.value <CLI::hxx_regex> ());

      NarrowString ixx_expr (
        ops.value <CLI::ixx_regex> ().empty ()
        ? "#^(.+?)(\\.[^./\\\\]+)?$#$1" + skel_suffix + ixx_suffix + "#"
        : ops.value <CLI::ixx_regex> ());

      NarrowString cxx_expr (
        ops.value <CLI::cxx_regex> ().empty ()
        ? "#^(.+?)(\\.[^./\\\\]+)?$#$1" + skel_suffix + cxx_suffix + "#"
        : ops.value <CLI::cxx_regex> ());

      NarrowString hxx_impl_expr;
      NarrowString cxx_impl_expr;
      NarrowString cxx_driver_expr;

      if (impl || driver)
      {
        hxx_impl_expr =
          "#^(.+?)(\\.[^./\\\\]+)?$#$1" + impl_suffix + hxx_suffix + "#";

        cxx_impl_expr =
          "#^(.+?)(\\.[^./\\\\]+)?$#$1" + impl_suffix + cxx_suffix + "#";

        cxx_driver_expr =
          "#^(.+?)(\\.[^./\\\\]+)?$#$1-driver" + cxx_suffix + "#";
      }

      NarrowString hxx_name (perl_s (name, hxx_expr));
      NarrowString ixx_name (perl_s (name, ixx_expr));
      NarrowString cxx_name (perl_s (name, cxx_expr));

      NarrowString hxx_impl_name;
      NarrowString cxx_impl_name;
      NarrowString cxx_driver_name;

      if (impl || driver)
      {
        hxx_impl_name = perl_s (name, hxx_impl_expr);
        cxx_impl_name = perl_s (name, cxx_impl_expr);
        cxx_driver_name = perl_s (name, cxx_driver_expr);
      }

      Path hxx_path (hxx_name);
      Path ixx_path (ixx_name);
      Path cxx_path (cxx_name);

      Path hxx_impl_path;
      Path cxx_impl_path;
      Path cxx_driver_path;

      if (impl || driver)
      {
        hxx_impl_path = hxx_impl_name;
        cxx_impl_path = cxx_impl_name;
        cxx_driver_path = cxx_driver_name;
      }

      if (NarrowString dir  = ops.value<CLI::output_dir> ())
      {
        try
        {
          Path path (dir, boost::filesystem::native);

          hxx_path = path / hxx_path;
          ixx_path = path / ixx_path;
          cxx_path = path / cxx_path;

          if (impl || driver)
          {
            hxx_impl_path = path / hxx_impl_path;
            cxx_impl_path = path / cxx_impl_path;
            cxx_driver_path = path /cxx_driver_path;
          }

        }
        catch (InvalidPath const&)
        {
          wcerr << dir.c_str () << ": error: invalid path" << endl;
          throw Failed ();
        }
      }

      // Open the impl files first so that if open fails, the skel files
      // are not deleted.
      //
      WideOutputFileStream hxx_impl;
      WideOutputFileStream cxx_impl;
      WideOutputFileStream cxx_driver;

      if (impl)
      {
        if (!ops.value<CLI::force_overwrite> ())
        {
          WideInputFileStream tmp (hxx_impl_path, ios_base::in);

          if (tmp.is_open ())
          {
            wcerr << hxx_impl_path << ": error: cowardly refusing to " <<
              "overwrite an existing file" << endl;
            throw Failed ();
          }

          tmp.close ();
        }

        hxx_impl.open (hxx_impl_path, ios_base::out);

        if (!hxx_impl.is_open ())
        {
          wcerr << hxx_impl_path << ": error: unable to open in write mode"
                << endl;
          throw Failed ();
        }

        unlinks.add (hxx_impl_path);

        if (!ops.value<CLI::force_overwrite> ())
        {
          WideInputFileStream tmp (cxx_impl_path, ios_base::in);

          if (tmp.is_open ())
          {
            wcerr << cxx_impl_path << ": error: cowardly refusing to " <<
              "overwrite an existing file" << endl;
            throw Failed ();
          }

          tmp.close ();
        }

        cxx_impl.open (cxx_impl_path, ios_base::out);

        if (!cxx_impl.is_open ())
        {
          wcerr << cxx_impl_path << ": error: unable to open in write mode"
                << endl;
          throw Failed ();
        }

        unlinks.add (cxx_impl_path);
      }

      if (driver)
      {
        if (!ops.value<CLI::force_overwrite> ())
        {
          WideInputFileStream tmp (cxx_driver_path, ios_base::in);

          if (tmp.is_open ())
          {
            wcerr << cxx_driver_path << ": error: cowardly refusing to " <<
              "overwrite an existing file" << endl;
            throw Failed ();
          }

          tmp.close ();
        }

        cxx_driver.open (cxx_driver_path, ios_base::out);

        if (!cxx_driver.is_open ())
        {
          wcerr << cxx_driver_path << ": error: unable to open in write " <<
            "mode" << endl;
          throw Failed ();
        }

        unlinks.add (cxx_driver_path);
      }

      // Open the skel files.
      //
      WideOutputFileStream hxx (hxx_path, ios_base::out);
      WideOutputFileStream ixx;
      WideOutputFileStream cxx (cxx_path, ios_base::out);

      if (!hxx.is_open ())
      {
        wcerr << hxx_path << ": error: unable to open in write mode" << endl;
        throw Failed ();
      }

      unlinks.add (hxx_path);

      Boolean inline_ (ops.value<CLI::generate_inline> ());

      if (inline_)
      {
        ixx.open (ixx_path, ios_base::out);

        if (!ixx.is_open ())
        {
          wcerr << ixx_path << ": error: unable to open in write mode" << endl;
          throw Failed ();
        }

        unlinks.add (ixx_path);
      }


      if (!cxx.is_open ())
      {
        wcerr << cxx_path << ": error: unable to open in write mode" << endl;
        throw Failed ();
      }

      unlinks.add (cxx_path);

      // Print copyright and license.
      //
      Char const* copyright (
        ops.value<CLI::proprietary_license> ()
        ? copyright_proprietary
        : copyright_gpl);

      hxx << copyright;

      if (inline_)
        ixx << copyright;

      cxx << copyright;

      if (impl)
      {
        hxx_impl << copyright_impl;
        cxx_impl << copyright_impl;
      }

      if (driver)
        cxx_driver << copyright_impl;

      // Prologue.
      //
      WideInputFileStream prologue;
      {
        NarrowString name (ops.value<CLI::prologue_file> ());

        if (name)
          open (prologue, name);
      }

      // Epilogue.
      //
      WideInputFileStream epilogue;
      {
        NarrowString name (ops.value<CLI::epilogue_file> ());

        if (name)
          open (epilogue, name);
      }


      // SLOC counter.
      //
      UnsignedLong sloc (0);
      Boolean show_sloc (ops.value<CLI::show_sloc> ());

      // HXX
      //
      {
        Context ctx (hxx, schema, ops);

        Indentation::Clip<Indentation::SLOC, WideChar> hxx_sloc (hxx);

        String guard (hxx_name);
        guard = perl_s (guard, L"/([a-z])([A-Z])/$1_$2/"); // Split words.
        std::transform (guard.begin (), guard.end(), guard.begin (), upcase);
        guard = ctx.escape (guard); // Make it a C++ id.

        hxx << "#ifndef " << guard << endl
            << "#define " << guard << endl
            << endl;

        // Copy prologue.
        //
        hxx << "// Begin prologue." << endl
            << "//" << endl;

        append (
          hxx, ops.value<CLI::hxx_prologue> (), ops.value<CLI::prologue> ());
        append (hxx, ops.value<CLI::hxx_prologue_file> (), prologue);

        hxx << "//" << endl
            << "// End prologue." << endl
            << endl;

        {
          // Version check.
          //
          hxx << "#include <xsde/cxx/version.hxx>" << endl
              << endl
              << "#if (XSDE_INT_VERSION != " << XSDE_INT_VERSION << "L)" << endl
              << "#error XSD/e runtime version mismatch" << endl
              << "#endif" << endl
              << endl;

          // Runtime/generated code compatibility checks.
          //

          hxx << "#include <xsde/cxx/config.hxx>" << endl
              << endl;

          if (ops.value<CLI::no_stl> ())
          {
            hxx << "#ifdef XSDE_STL" << endl
                << "#error the XSD/e runtime uses STL while the " <<
              "generated code does not (reconfigure the runtime or " <<
              "remove --no-stl)" << endl
                << "#endif" << endl
                << endl;
          }
          else
          {
            hxx << "#ifndef XSDE_STL" << endl
                << "#error the generated code uses STL while the " <<
              "XSD/e runtime does not (reconfigure the runtime or " <<
              "add --no-stl)" << endl
                << "#endif" << endl
                << endl;
          }

          if (ops.value<CLI::no_iostream> ())
          {
            hxx << "#ifdef XSDE_IOSTREAM" << endl
                << "#error the XSD/e runtime uses iostream while the " <<
              "generated code does not (reconfigure the runtime or " <<
              "remove --no-iostream)" << endl
                << "#endif" << endl
                << endl;
          }
          else
          {
            hxx << "#ifndef XSDE_IOSTREAM" << endl
                << "#error the generated code uses iostream while the " <<
              "XSD/e runtime does not (reconfigure the runtime or " <<
              "add --no-iostream)" << endl
                << "#endif" << endl
                << endl;
          }

          if (ops.value<CLI::no_exceptions> ())
          {
            hxx << "#ifdef XSDE_EXCEPTIONS" << endl
                << "#error the XSD/e runtime uses exceptions while the " <<
              "generated code does not (reconfigure the runtime or " <<
              "remove --no-exceptions)" << endl
                << "#endif" << endl
                << endl;
          }
          else
          {
            hxx << "#ifndef XSDE_EXCEPTIONS" << endl
                << "#error the generated code uses exceptions while the " <<
              "XSD/e runtime does not (reconfigure the runtime or " <<
              "add --no-exceptions)" << endl
                << "#endif" << endl
                << endl;
          }

          if (ops.value<CLI::no_long_long> ())
          {
            hxx << "#ifdef XSDE_LONGLONG" << endl
                << "#error the XSD/e runtime uses long long while the " <<
              "generated code does not (reconfigure the runtime or " <<
              "remove --no-long-long)" << endl
                << "#endif" << endl
                << endl;
          }
          else
          {
            hxx << "#ifndef XSDE_LONGLONG" << endl
                << "#error the generated code uses long long while the " <<
              "XSD/e runtime does not (reconfigure the runtime or " <<
              "add --no-long-long)" << endl
                << "#endif" << endl
                << endl;
          }

          if (ops.value<CLI::suppress_validation> ())
          {
            hxx << "#ifdef XSDE_VALIDATION_SERIALIZER" << endl
                << "#error the XSD/e runtime uses validation while the " <<
              "generated code does not (reconfigure the runtime or " <<
              "remove --suppress-validation)" << endl
                << "#endif" << endl
                << endl;
          }
          else
          {
            hxx << "#ifndef XSDE_VALIDATION_SERIALIZER" << endl
                << "#error the generated code uses validation while the " <<
              "XSD/e runtime does not (reconfigure the runtime or " <<
              "add --suppress-validation)" << endl
                << "#endif" << endl
                << endl;
          }

          //
          //

          hxx << "#include <xsde/cxx/pre.hxx>" << endl
              << endl;

          // Set auto-indentation.
          //
          Indentation::Clip<Indentation::CXX, WideChar> hxx_clip (hxx);

          // Generate.
          //
          generate_serializer_forward (ctx);
          generate_serializer_header (ctx, hxx_expr); //@@ move expr to ctx


          if (inline_)
            hxx << "#include " << ctx.process_include_path (ixx_name) << endl
                << endl;

          hxx << "#include <xsde/cxx/post.hxx>" << endl
              << endl;
        }

        // Copy epilogue.
        //
        hxx << "// Begin epilogue." << endl
            << "//" << endl;

        append (hxx, ops.value<CLI::hxx_epilogue_file> (), epilogue);
        append (
          hxx, ops.value<CLI::hxx_epilogue> (), ops.value<CLI::epilogue> ());

        hxx << "//" << endl
            << "// End epilogue." << endl
            << endl;

        hxx << "#endif // " << guard << endl;

        if (show_sloc)
        {
          wcerr << hxx_path << ": "
                << hxx_sloc.buffer ().count () << endl;

          sloc += hxx_sloc.buffer ().count ();
        }
      }


      // IXX
      //
      if (inline_)
      {
        Context ctx (ixx, schema, ops);

        Indentation::Clip<Indentation::SLOC, WideChar> ixx_sloc (ixx);


        // Copy prologue.
        //
        ixx << "// Begin prologue." << endl
            << "//" << endl;

        append (
          ixx, ops.value<CLI::ixx_prologue> (), ops.value<CLI::prologue> ());
        append (ixx, ops.value<CLI::ixx_prologue_file> (), prologue);

        ixx << "//" << endl
            << "// End prologue." << endl
            << endl;

        {
          // Set auto-indentation.
          //
          Indentation::Clip<Indentation::CXX, WideChar> ixx_clip (ixx);


          // Generate.
          //
          generate_serializer_inline (ctx);
        }

        // Copy epilogue.
        //
        ixx << "// Begin epilogue." << endl
            << "//" << endl;

        append (ixx, ops.value<CLI::ixx_epilogue_file> (), epilogue);
        append (
          ixx, ops.value<CLI::ixx_epilogue> (), ops.value<CLI::epilogue> ());

        ixx << "//" << endl
            << "// End epilogue." << endl
            << endl;

        if (show_sloc)
        {
          wcerr << ixx_path << ": "
                << ixx_sloc.buffer ().count () << endl;

          sloc += ixx_sloc.buffer ().count ();
        }
      }


      // CXX
      //
      {
        Context ctx (cxx, schema, ops);

        Indentation::Clip<Indentation::SLOC, WideChar> cxx_sloc (cxx);

        // Copy prologue.
        //
        cxx << "// Begin prologue." << endl
            << "//" << endl;

        append (
          cxx, ops.value<CLI::cxx_prologue> (), ops.value<CLI::prologue> ());
        append (cxx, ops.value<CLI::cxx_prologue_file> (), prologue);

        cxx << "//" << endl
            << "// End prologue." << endl
            << endl;

        {
          // Set auto-indentation.
          //
          Indentation::Clip<Indentation::CXX, WideChar> cxx_clip (cxx);

          cxx << "#include " << ctx.process_include_path (hxx_name) << endl
              << endl;

          if (!inline_)
            generate_serializer_inline (ctx);

          generate_serializer_source (ctx);

          if (validation)
          {
            generate_element_validation_source (ctx);
            generate_attribute_validation_source (ctx);
          }
        }

        // Copy epilogue.
        //
        cxx << "// Begin epilogue." << endl
            << "//" << endl;

        append (cxx, ops.value<CLI::cxx_epilogue_file> (), epilogue);
        append (
          cxx, ops.value<CLI::cxx_epilogue> (), ops.value<CLI::epilogue> ());

        cxx << "//" << endl
            << "// End epilogue." << endl
            << endl;

        if (show_sloc)
        {
          wcerr << cxx_path << ": "
                << cxx_sloc.buffer ().count () << endl;

          sloc += cxx_sloc.buffer ().count ();
        }
      }

      // HXX impl
      //
      if (impl)
      {
        Context ctx (hxx_impl, schema, ops);

        String guard (hxx_impl_name);
        guard = perl_s (guard, L"/([a-z])([A-Z])/$1_$2/"); // Split words.
        std::transform (guard.begin (), guard.end(), guard.begin (), upcase);
        guard = ctx.escape (guard); // Make it a C++ id.

        hxx_impl << "#ifndef " << guard << endl
                 << "#define " << guard << endl
                 << endl;

        {
          // Set auto-indentation.
          //
          Indentation::Clip<Indentation::CXX, WideChar> clip (hxx_impl);

          hxx_impl << "#include " << ctx.process_include_path (hxx_name)
                   << endl << endl;

          /* @@
          generate_impl_header (ctx, hxx_impl_expr);
          */
        }

        hxx_impl << "#endif // " << guard << endl;
      }

      // CXX impl
      //
      if (impl)
      {
        Context ctx (cxx_impl, schema, ops);

        // Set auto-indentation.
        //
        Indentation::Clip<Indentation::CXX, WideChar> clip (cxx_impl);

        cxx_impl << "#include " << ctx.process_include_path (hxx_impl_name)
                 << endl << endl;

        /* @@
        generate_impl_source (ctx);
        */
      }

      // CXX driver
      //
      if (driver)
      {
        Context ctx (cxx_driver, schema, ops);

        // Set auto-indentation.
        //
        Indentation::Clip<Indentation::CXX, WideChar> clip (cxx_driver);

        cxx_driver << "#include " << ctx.process_include_path (hxx_impl_name)
                   << endl << endl;

        /* @@
        generate_driver_source (ctx);
        */
      }

      return sloc;
    }
    catch (NoNamespaceMapping const& e)
    {
      wcerr << e.file () << ":" << e.line () << ":" << e.column ()
            << ": error: unable to map XML Schema namespace '" << e.ns ()
            << "' to C++ namespace" << endl;

      wcerr << e.file () << ":" << e.line () << ":" << e.column ()
            << ": info: use the --namespace-map or --namespace-regex option "
            << "to provide custom mapping" << endl;

      throw Failed ();
    }
    catch (InvalidNamespaceMapping const& e)
    {
      wcerr << "error: invalid XML to C++ namespace mapping specified: "
            << "'" << e.mapping () << "': " << e.reason () << endl;

      throw Failed ();
    }
  }
}
