/* $Header: /fridge/cvs/xscorch/sgtk/swidgets/sconsole.c,v 1.14 2001/12/08 22:33:02 justins Exp $ */
/*
   
   xscorch - sconsole.c       Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   Code for a text console
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <string.h>
#include <sconsole.h>
#include <sstr.h>
#include <sgtkfont.h>
#include <gdk/gdkkeysyms.h>


static ScDrawbufClass *parent_class;



#define  sc_console_get_text_buffer(cons)    (cons->text.buffer)
#define  sc_console_get_char_ptr(cons, x, y) (cons->text.buffer + y * sc_console_get_width(cons) + x)

#define  SC_CONSOLE_BORDER_WIDTH_CHARS       2
#define  SC_CONSOLE_BORDER_HEIGHT_CHARS      1



enum _ScConsoleSignals {
   PAINT_REGION_SIGNAL,
   LAST_SIGNAL
};
static gint _sc_console_signals[LAST_SIGNAL] = { 0 };



static void _sc_console_destroy_data(ScConsole *cons) {

   GList *cur;

   /* Delete the console text buffer */
   if(sc_console_get_text_buffer(cons) != NULL) {
      free(sc_console_get_text_buffer(cons));
      cons->text.buffer = NULL;
   }
   
   /* Delete the highlights list */
   cur = cons->highlights;
   while(cur != NULL) {
      if(cur->data != NULL) free(cur->data);
      cur->data = NULL;
      cur = cur->next;
   }
   g_list_free(cons->highlights);

}



static void _sc_console_destroy(GtkObject *obj) {

   _sc_console_destroy_data(SC_CONSOLE(obj));

   /* Check for a parent signal handler */
   if(GTK_OBJECT_CLASS(parent_class)->destroy != NULL) {
      GTK_OBJECT_CLASS(parent_class)->destroy(obj);
   } /* Does parent have default? */

}



static inline gint _sc_console_char_width(ScConsole *cons) {
/* Return the width of one character, in pixels */
   return(gdk_char_width(cons->screen_font, 'w'));

}



static inline gint _sc_console_char_height(ScConsole *cons) {
/* Return the height of one character, in pixels */

   return(cons->screen_font->ascent + cons->screen_font->descent);

}



static inline gint _sc_console_char_x(ScConsole *cons, gint x) {
/* Return the screen position of the X'th character. 
   Does not compensate for scrolling, so char position is REAL */

   return(x * _sc_console_char_width(cons));

}



static inline gint _sc_console_char_y(ScConsole *cons, gint y) {
/* Return the screen position of the Y'th character
   Does not compensate for scrolling, so char position is REAL */

   return(y * _sc_console_char_height(cons));

}



gint sc_console_get_row_height(ScConsole *cons) {
/* Return one row height, in pixels */

   return(_sc_console_char_height(cons));

}



gint sc_console_get_col_width(ScConsole *cons) {
/* Return one column width, in pixels */

   return(_sc_console_char_width(cons));

}



inline gint sc_console_get_char_from_pixel_x(ScConsole *cons, gint x, gboolean view) {
/* Returns the character X position which is under pixel position x, where 
   x is relative to the top-left of the widget.  This takes scrolling into
   account if view is TRUE, so you will get a valid position into the char 
   buffer if view is set.  This always compensates for borders.  */

   x = x / sc_console_get_col_width(cons);
   if(view) x += sc_console_get_view_x(cons);
   if(cons->style != CONSOLE_BORDERLESS) x -= SC_CONSOLE_BORDER_WIDTH_CHARS;
   return(x);

}



inline gint sc_console_get_char_from_pixel_y(ScConsole *cons, gint y, gboolean view) {
/* Returns the character Y position which is under pixel position y, where 
   y is relative to the top-left of the widget.  This takes scrolling into
   account if view is set to TRUE, so you will get a valid position into 
   the char buffer if view is set.  This always compensates for borders. */

   y = y / sc_console_get_row_height(cons);
   if(view) y += sc_console_get_view_y(cons);
   if(cons->style != CONSOLE_BORDERLESS) y -= SC_CONSOLE_BORDER_HEIGHT_CHARS;
   return(y);

}



inline void sc_console_get_char_from_pixel(ScConsole *cons, gint *x, gint *y, gboolean view) {
/* Interface that updates both X and Y */

   *x = sc_console_get_char_from_pixel_x(cons, *x, view);
   *y = sc_console_get_char_from_pixel_y(cons, *y, view);

}



void sc_console_get_char_from_pixel_rect(ScConsole *cons, GdkRectangle *r, gboolean view) {
/* Fancy rectangle version */

   r->x = sc_console_get_char_from_pixel_x(cons, r->x, view);
   r->y = sc_console_get_char_from_pixel_y(cons, r->y, view);
   r->width  = r->width  * sc_console_get_col_width(cons);
   r->height = r->height * sc_console_get_row_height(cons);

}



inline gint sc_console_get_pixel_from_char_x(ScConsole *cons, gint x, gboolean view) {
/* Returns the pixel X position which corresponds to character col x, where 
   x is relative to the top-left of the inner border.  This takes scrolling 
   into account if view is TRUE.  This always compensates for borders; if
   view is false, (0,0) will be inside the border, never outside. */

   if(cons->style != CONSOLE_BORDERLESS) x += SC_CONSOLE_BORDER_WIDTH_CHARS;
   if(view) x -= sc_console_get_view_x(cons);
   x = _sc_console_char_x(cons, x);
   return(x);

}



inline gint sc_console_get_pixel_from_char_y(ScConsole *cons, gint y, gboolean view) {
/* Returns the pixel Y position which corresponds to character col y, where 
   y is relative to the top-left of the inner border.  This takes scrolling 
   into account if view is TRUE.  This always compensates for borders; if
   view is false, (0,0) will be inside the border, never outside. */

   if(cons->style != CONSOLE_BORDERLESS) y += SC_CONSOLE_BORDER_HEIGHT_CHARS;
   if(view) y -= sc_console_get_view_y(cons);
   y = _sc_console_char_y(cons, y);
   return(y);

}



inline void sc_console_get_pixel_from_char(ScConsole *cons, gint *x, gint *y, gboolean view) {
/* Interface that updates both X and Y */

   *x = sc_console_get_pixel_from_char_x(cons, *x, view);
   *y = sc_console_get_pixel_from_char_y(cons, *y, view);

}



void sc_console_get_pixel_from_char_rect(ScConsole *cons, GdkRectangle *r, gboolean view) {
/* Fancy rectangle version */

   r->x = sc_console_get_pixel_from_char_x(cons, r->x, view);
   r->y = sc_console_get_pixel_from_char_y(cons, r->y, view);
   r->width  = r->width  * sc_console_get_col_width(cons);
   r->height = r->height * sc_console_get_row_height(cons);

}



static inline gboolean _sc_console_in_bounds(gint cx, gint cy, gint x, gint y, gint width, gint height) {

   return(cx >= x && cy >= y && cx < x + width && cy < y + height);

}



static void _sc_console_draw_char(ScConsole *cons, GdkGC *fg, GdkGC *bg, gboolean bold, gint x, gint y, char ch) {

   GdkFont *font;
   
   /* Which font to select? */
   if(bold) font = cons->screen_bold_font;
   else     font = cons->screen_font;

   /* Update X, Y to account for a window frame; also,
      set X, Y to the screen coordinates to write to. */
   sc_console_get_pixel_from_char(cons, &x, &y, FALSE);

   /* Write the text! */
   gdk_draw_rectangle(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)), 
                      bg, 
                      TRUE, 
                      x, y, 
                      _sc_console_char_width(cons), _sc_console_char_height(cons));
   gdk_draw_text(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)), 
                      font, fg,
                      x, y + font->ascent, 
                      &ch, 1);

}



static void _sc_console_draw_region(ScConsole *cons, gint x, gint y, gint width, gint height) {

   ScConsoleHighlight *high;
   GdkColor *oldfgcolor;
   GdkColor *oldbgcolor;
   GdkRectangle region;
   GdkColor *fgcolor;
   GdkColor *bgcolor;
   GdkGC *fg_gc;
   GdkGC *bg_gc;
   GList *cur;

   char *chptr;
   gint x1;                   /* Left X boundary */
   gint y1;                   /* Top Y boundary  */
   gint x2;                   /* Right X boundary */
   gint y2;                   /* Bottom Y boundary */
   gint cx;                   /* Current X iter */
   gint cy;                   /* Current Y iter */
   gboolean bold;             /* Draw this in bold? */

   if(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)) == NULL) return;

   /* Set up the region variable. */
   region.x = x;
   region.y = y;
   region.width = width;
   region.height = height;

   x1 = x;
   y1 = y;
   x2 = x + width - 1;
   y2 = y + height - 1;
   
   if(x1 > x2) {
      x1 = x1 + x2;
      x2 = x1 - x2;
      x1 = x1 - x2;
   }
   if(y1 > y2) {
      y1 = y1 + y2;
      y2 = y1 - y2;
      y1 = y1 - y2;
   }

   if(x1 < cons->text.viewx) x1 = cons->text.viewx;
   if(y1 < cons->text.viewy) y1 = cons->text.viewy;
   if(x1 >= cons->text.viewx + cons->text.vieww) return;
   if(y1 >= cons->text.viewy + cons->text.viewh) return;
   
   if(x2 < cons->text.viewx) return;
   if(y2 < cons->text.viewy) return;
   if(x2 >= cons->text.viewx + cons->text.vieww) x2 = cons->text.viewx + cons->text.vieww - 1;
   if(y2 >= cons->text.viewy + cons->text.viewh) y2 = cons->text.viewy + cons->text.viewh - 1;
   
   /* Request a GC */
   fg_gc = gdk_gc_new(((GtkWidget *)cons)->window);
   bg_gc = gdk_gc_new(((GtkWidget *)cons)->window);

   /* Setup default foreground, background colors */
   oldfgcolor = &cons->colors.foreground;
   oldbgcolor = &cons->colors.background;
   gdk_gc_set_foreground(fg_gc, oldfgcolor);
   gdk_gc_set_foreground(bg_gc, oldbgcolor);
   
   for(cy = y1; cy <= y2; ++cy) {

      /* Get the character to be drawn */
      chptr = sc_console_get_char_ptr(cons, x1, cy);
      for(cx = x1; cx <= x2; ++cx, ++chptr) {

         fgcolor = &cons->colors.foreground;
         bgcolor = &cons->colors.background;
         bold = cons->colors.bold;

         /* What highlight are we using? */
         cur = cons->highlights;
         while(cur != NULL) {
            /* Are we on this highlight? */
            high = cur->data;
            cur = cur->next;
            if(_sc_console_in_bounds(cx, cy, high->x, high->y, high->width, high->height)) {
               /* We are on this highlight; set new fg/bg color */
               if(!high->colors.colors_alloc) {
                  high->colors.colors_alloc = TRUE;
                  gdk_colormap_alloc_color(gtk_widget_get_colormap((GtkWidget *)cons), &high->colors.foreground, FALSE, TRUE);
                  gdk_colormap_alloc_color(gtk_widget_get_colormap((GtkWidget *)cons), &high->colors.background, FALSE, TRUE);
               }
               fgcolor = &high->colors.foreground;
               bgcolor = &high->colors.background;
               bold = high->colors.bold;
            } /* Coordinates in bound? */
         } /* Iterating thru installed highlights ... */

         /* Are we drawing the new cursor? */
         if(_sc_console_in_bounds(cx, cy, cons->cursor.x, cons->cursor.y, cons->cursor.width, cons->cursor.height)) {
            /* Welp, we're on the cursor.. all that work, for nothing */
            if(GTK_WIDGET_HAS_FOCUS((GtkWidget *)cons)) {
               if(cons->cursor.highlighted) {
                  fgcolor = &cons->colors.forelight;
                  bgcolor = &cons->colors.backlight;
               } else {
                  fgcolor = &cons->colors.forecursor;
                  bgcolor = &cons->colors.backcursor;
               }
            } else {
               fgcolor = &cons->colors.foreshadow;
               bgcolor = &cons->colors.backshadow;
            }
         } /* We be a cursor? */

         if(oldfgcolor != fgcolor) {
            oldfgcolor = fgcolor;
            gdk_gc_set_foreground(fg_gc, fgcolor);
         }
         if(oldbgcolor != bgcolor) {
            oldbgcolor = bgcolor;
            gdk_gc_set_foreground(bg_gc, bgcolor);
         }

         _sc_console_draw_char(cons, fg_gc, bg_gc, bold, cx - cons->text.viewx, cy - cons->text.viewy, *chptr);
      } /* Iterating thru X */
   } /* Iterate thru Y */

   /* Update X, Y to account for a window frame; also,
      set X, Y to the screen coordinates to write to. */
   sc_console_get_pixel_from_char(cons, &x, &y, TRUE);

   /* Update width, height to screen coordinates */
   width = width * _sc_console_char_width(cons);
   height= height* _sc_console_char_height(cons);

   /* Make sure this update is queued for display to screen. */
   sc_drawbuf_queue_draw(SC_DRAWBUF(cons), x, y, width, height);

   /* Propagate the draw request to the active console. */
   gtk_signal_emit_by_name(GTK_OBJECT(cons), "paint_region", &region, NULL);

   /* Release the GC's */
   gdk_gc_unref(fg_gc);
   gdk_gc_unref(bg_gc);

}



static void _sc_console_draw_horiz_scroll(ScConsole *cons) {

}



static inline void _sc_console_vert_scroll_extents(ScConsole *cons, gint *startx, gint *starty, gint *width, gint *height, gint *arrowh) {

   *width  = ((_sc_console_char_width(cons) + 2) & ~1);
   *height = GTK_WIDGET(cons)->allocation.height - 3 * _sc_console_char_height(cons);
   *startx = GTK_WIDGET(cons)->allocation.width - 2 * _sc_console_char_width(cons) + 1;
   *starty = _sc_console_char_height(cons) * 3 / 2;
   *arrowh = *width + 2;

}



static inline void _sc_console_vert_trough_extents(ScConsole *cons, gint starty, gint height, gint *pos, gint *size) {

   *pos    = starty + height * cons->text.viewy / cons->text.bufferh;
   *size   = height * cons->text.viewh / cons->text.bufferh;

}



static inline gboolean _sc_console_can_scroll_up(ScConsole *cons) {

   return(sc_console_get_view_y(cons) > 0);

}



static inline gboolean _sc_console_can_scroll_down(ScConsole *cons) {

   return(sc_console_get_view_y(cons) + sc_console_get_view_height(cons) < sc_console_get_height(cons));

}



static void _sc_console_draw_vert_scroll(ScConsole *cons) {

   GtkWidget *widget = (GtkWidget *)cons;
   GdkPoint points[3];
   GdkGC *foreground;
   gint arrowh;
   gint startx;
   gint starty;
   gint height;
   gint width;
   gint size;   
   gint pos;

   /* Can we even draw yet? */
   if(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)) == NULL) return;
   if(cons->style == CONSOLE_BORDERLESS) return;

   /* Request a GC */
   foreground = gdk_gc_new(widget->window);
   
   /* Determine vertical scrollbar extents */
   _sc_console_vert_scroll_extents(cons, &startx, &starty, &width, &height, &arrowh);
   _sc_console_vert_trough_extents(cons, starty, height, &pos, &size);

   /* erase any original bars */
   gdk_gc_set_foreground(foreground, &cons->colors.backscroll);
   gdk_draw_rectangle(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)),
                      foreground,
                      TRUE,
                      startx, starty - arrowh, 
                      width, height + 2 * arrowh);

   /* setup gc colors for bar */
   gdk_gc_set_foreground(foreground, &cons->colors.forescroll);

   /* Draw vertical slider */
   gdk_draw_rectangle(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)),
                      foreground,
                      TRUE,
                      startx, pos,
                      width, size);

   /* Determine if up-arrow is required */
   if(_sc_console_can_scroll_up(cons)) {
      points[0].x = startx + width / 2;
      points[0].y = starty - arrowh;
      points[1].x = points[0].x - width / 2;
      points[1].y = starty - 2;
      points[2].x = points[0].x + width / 2;
      points[2].y = starty - 2;
      gdk_draw_polygon(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)),
                       foreground,
                       TRUE,
                       points, 3);
   } /* Up arrow? */

   /* Determine if down-arrow is required */
   if(_sc_console_can_scroll_down(cons)) {
      points[0].x = startx + width / 2;
      points[0].y = starty + height + arrowh;
      points[1].x = points[0].x - width / 2;
      points[1].y = starty + height + 2;
      points[2].x = points[0].x + width / 2;
      points[2].y = starty + height + 2;
      gdk_draw_polygon(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)),
                       foreground,
                       TRUE,
                       points, 3);
   } /* Down arrow? */

   /* Release the GC's */
   gdk_gc_unref(foreground);

   /* Make sure everything is queued for draw */
   sc_drawbuf_queue_draw(SC_DRAWBUF(widget), startx, starty - arrowh, width, height + 2 * arrowh);
   
}



static void _sc_console_draw_frame(ScConsole *cons) {

   GtkWidget *widget = (GtkWidget *)cons;
   GdkGC *foreground;
   GdkGC *background;

   /* Can we even draw yet? */
   if(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)) == NULL) return;

   /* Request a GC */
   foreground = gdk_gc_new(widget->window);
   background = gdk_gc_new(widget->window);
   
   /* setup gc colors */
   gdk_gc_set_foreground(foreground, &cons->colors.foreground);
   gdk_gc_set_foreground(background, &cons->colors.background);

   /* Clear the screen; draw window border if appropriate */
   gdk_draw_rectangle(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)), 
                      background,
                      TRUE, 
                      0, 0, 
                      widget->allocation.width, widget->allocation.height);
   if(cons->style != CONSOLE_BORDERLESS) {
      /* Draw a border as well... */
      gdk_draw_rectangle(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)),
                         foreground,
                         FALSE, 
                         _sc_console_char_width(cons) - 4, _sc_console_char_height(cons) / 2 - 2, 
                         widget->allocation.width - 2 * _sc_console_char_width(cons) + 8, 
                         widget->allocation.height - _sc_console_char_height(cons) + 4);

      /* Did we need scrollers? */
      if(cons->text.scrollx) {
         _sc_console_draw_horiz_scroll(cons);
      } /* horizontal scrollbar */
      if(cons->text.scrolly) {
         _sc_console_draw_vert_scroll(cons);
      } /* vertical scrollbar */
   } /* Draw the window border? */

   /* Release the GC's */
   gdk_gc_unref(foreground);
   gdk_gc_unref(background);

}



static void _sc_console_draw_all(ScConsole *cons) {

   GtkWidget *widget = (GtkWidget *)cons;

   /* Can we even draw yet? */
   if(sc_drawbuf_get_buffer(SC_DRAWBUF(cons)) == NULL) return;

   /* Setup default foreground, background colors */
   /* (make sure to allocate them if not done already) */
   if(!cons->colors.colors_alloc) {
      cons->colors.colors_alloc = TRUE;
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.foreground, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.background, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.forecursor, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backcursor, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.foreshadow, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backshadow, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.forescroll, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backscroll, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.forelight, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backlight, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.foredisabled, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backdisabled, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.forestandard, FALSE, TRUE);
      gdk_colormap_alloc_color(gtk_widget_get_colormap(widget), &cons->colors.backstandard, FALSE, TRUE);
   }

   /* Redraw the window frame */
   _sc_console_draw_frame(cons);

   /* Redraw each line of text */   
   _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));
   
   /* Make sure everything is queued for draw */
   sc_drawbuf_queue_draw(SC_DRAWBUF(widget), 0, 0, widget->allocation.width, widget->allocation.height);
   
}



static gint _sc_console_configure(GtkWidget *widget, GdkEventConfigure *event) {

   /* Check for a parent signal handler */
   if(GTK_WIDGET_CLASS(parent_class)->configure_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->configure_event(widget, event)) {
         /* Oops; we must halt */
         return(TRUE);
      } /* Can we continue? */
   } /* Does parent have default? */

   /* Draw the console */
   _sc_console_draw_all(SC_CONSOLE(widget));

   /* Let other events run as well... */
   return(FALSE);
   
}



static gint _sc_console_draw_focus(GtkWidget *widget, GdkEventFocus *event) {

   ScConsole *cons = SC_CONSOLE(widget);

   GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
   _sc_console_draw_region(cons, cons->cursor.x, cons->cursor.y, cons->cursor.width, cons->cursor.height);

   /* Check for a parent signal handler */
   if(GTK_WIDGET_CLASS(parent_class)->focus_in_event != NULL) {
      return(GTK_WIDGET_CLASS(parent_class)->focus_in_event(widget, event));
   } /* Does parent have default? */
   
   /* Allow other events */
   return(FALSE);

}



static gint _sc_console_undraw_focus(GtkWidget *widget, GdkEventFocus *event) {

   ScConsole *cons = SC_CONSOLE(widget);

   GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
   _sc_console_draw_region(cons, cons->cursor.x, cons->cursor.y, cons->cursor.width, cons->cursor.height);

   /* Check for a parent signal handler */
   if(GTK_WIDGET_CLASS(parent_class)->focus_out_event != NULL) {
      return(GTK_WIDGET_CLASS(parent_class)->focus_out_event(widget, event));
   } /* Does parent have default? */

   /* Allow other events */
   return(FALSE);

}



static gint _sc_console_button_press(GtkWidget *widget, GdkEventButton *event) {

   ScConsole *cons = SC_CONSOLE(widget);
   gboolean needredraw = FALSE;
   gint arrowh;
   gint height;
   gint width;
   gint size;
   gint pos;
   gint x;
   gint y;

   /* Try out parent handler first */
   if(GTK_WIDGET_CLASS(parent_class)->button_press_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->button_press_event(widget, event)) {
         /* Crap. The signal's already been handled */
         return(TRUE);
      } /* Signal processed? */
   } /* Signal handler available? */

   /* Make sure this is a SINGLE click event */
   if(event->type != GDK_BUTTON_PRESS) return(FALSE);
   
   /* See if this was a click onto a scrollbar */
   if(cons->text.scrolly) {
      /* Check to see if we're clicking in vertical scrollbar region */
      _sc_console_vert_scroll_extents(cons, &x, &y, &width, &height, &arrowh);
      _sc_console_vert_trough_extents(cons, y, height, &pos, &size);
      if(event->x >= x && event->x < x + width && event->y >= y - arrowh && event->y < y + height + arrowh) {
         /* Click was onto the scrollbar itself */
         if(event->y >= y - arrowh && event->y < y) {
            /* Clicked up-arrow */
            if(_sc_console_can_scroll_up(cons)) {
               --cons->text.viewy;
               needredraw = TRUE;
            } /* Can we scroll down? */
            needredraw = TRUE;
         } else if(event->y >= y + height && event->y < y + height + arrowh) {
            if(_sc_console_can_scroll_down(cons)) {
               ++cons->text.viewy;
               needredraw = TRUE;
            } /* Can we scroll down? */
         } else if(event->y < pos) {
            /* Page up */
            if(_sc_console_can_scroll_up(cons)) {
               cons->text.viewy -= cons->text.viewh;
               if(cons->text.viewy < 0) cons->text.viewy = 0;
               needredraw = TRUE;
            } /* Can we page up? */
         } else if(event->y >= pos + size) {
            /* Page down */
            if(_sc_console_can_scroll_down(cons)) {
               cons->text.viewy += cons->text.viewh;
               if(cons->text.viewy > sc_console_get_height(cons) - cons->text.viewh) cons->text.viewy = sc_console_get_height(cons) - cons->text.viewh;
               needredraw = TRUE;
            } /* Can we page up? */
         } /* Checking Y coordinate */

         /* Redraw components? */
         if(needredraw) {
            _sc_console_draw_vert_scroll(cons);
            _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));
         }

         /* Click was in scrollbar, abort */
         cons->ignorerelease = TRUE;
         return(TRUE);
      } /* Checking X, Y coordinate */
   } /* Checking vertical scrollbar */

   /* Nothing interesting... */
   return(FALSE);

}



static gint _sc_console_button_release(GtkWidget *widget, GdkEventButton *event) {

   ScConsole *cons = SC_CONSOLE(widget);

   /* Try out parent handler first */
   if(GTK_WIDGET_CLASS(parent_class)->button_release_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->button_release_event(widget, event)) {
         /* Crap. The signal's already been handled */
         return(TRUE);
      } /* Signal processed? */
   } /* Signal handler available? */

   /* See if this was a click onto a scrollbar */
   if(cons->ignorerelease) {
      cons->ignorerelease = FALSE;
      return(TRUE);
   } /* Clicked a scrollbar */

   /* Nothing interesting... */
   return(FALSE);

}



static gint _sc_console_key_press(GtkWidget *widget, GdkEventKey *event) {

   ScConsole *cons = SC_CONSOLE(widget);
   
   /* Try out parent handler first */
   if(GTK_WIDGET_CLASS(parent_class)->key_press_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->key_press_event(widget, event)) {
         /* Crap. The signal's already been handled */
         return(TRUE);
      } /* Signal processed? */
   } /* Signal handler available? */
  
   switch(event->keyval) {
      case GDK_Page_Up:
      case GDK_KP_Page_Up:
         if(_sc_console_can_scroll_up(cons)) {
            cons->text.viewy -= cons->text.viewh;
            if(cons->text.viewy < 0) cons->text.viewy = 0;
            _sc_console_draw_vert_scroll(cons);
            _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));
         }
         return(TRUE);
      
      case GDK_Page_Down:
      case GDK_KP_Page_Down:
         if(_sc_console_can_scroll_down(cons)) {
            cons->text.viewy += cons->text.viewh;
            if(cons->text.viewy > sc_console_get_height(cons) - cons->text.viewh) cons->text.viewy = sc_console_get_height(cons) - cons->text.viewh;
            _sc_console_draw_vert_scroll(cons);
            _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));
         }
         return(TRUE);
   } /* Search for special keys */
   
   /* Fallthrough */
   return(FALSE);

}



static gint _sc_console_key_release(GtkWidget *widget, GdkEventKey *event) {

   /* Try out parent handler first */
   if(GTK_WIDGET_CLASS(parent_class)->key_release_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->key_release_event(widget, event)) {
         /* Crap. The signal's already been handled */
         return(TRUE);
      } /* Signal processed? */
   } /* Signal handler available? */
  
   switch(event->keyval) {
      case GDK_Page_Up:
      case GDK_KP_Page_Up:
      case GDK_Page_Down:
      case GDK_KP_Page_Down:
         return(TRUE);
   } /* Search for special keys */
   
   /* Fallthrough */
   return(FALSE);

}



static void _sc_console_class_init(ScConsoleClass *klass) {

   GtkObjectClass *object_class = (GtkObjectClass *)klass;
   
   /* Determine parent class */
   parent_class = gtk_type_class(sc_drawbuf_get_type());

   /* Construct new signals */
   _sc_console_signals[PAINT_REGION_SIGNAL] =
      gtk_signal_new("paint_region",
                     GTK_RUN_LAST,  
                     object_class->type,
                     GTK_SIGNAL_OFFSET(ScConsoleClass, paint_region),
                     gtk_marshal_NONE__POINTER,
                     GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

   /* Install the new signals */
   gtk_object_class_add_signals(object_class, _sc_console_signals, LAST_SIGNAL);
   
   /* Attach default signal handlers */
   klass->paint_region                          = NULL;
   GTK_OBJECT_CLASS(klass)->destroy             = _sc_console_destroy;
   GTK_WIDGET_CLASS(klass)->configure_event     = _sc_console_configure;
   GTK_WIDGET_CLASS(klass)->focus_in_event      = _sc_console_draw_focus;
   GTK_WIDGET_CLASS(klass)->focus_out_event     = _sc_console_undraw_focus;
   GTK_WIDGET_CLASS(klass)->button_press_event  = _sc_console_button_press;
   GTK_WIDGET_CLASS(klass)->button_release_event= _sc_console_button_release;
   GTK_WIDGET_CLASS(klass)->key_press_event     = _sc_console_key_press;
   GTK_WIDGET_CLASS(klass)->key_release_event   = _sc_console_key_release;
   
}



static void _sc_console_init_obj(ScConsole *cons) {

   cons->text.scrollx = 0;
   cons->text.scrolly = 0;
   cons->text.buffer  = NULL;
   cons->screen_font = gdk_font_load(SC_GTK_CONSOLE_FONT);
   cons->screen_bold_font = gdk_font_load(SC_GTK_CONSOLE_BOLD_FONT);
   cons->cursor.highlighted = FALSE;
   cons->highlights = NULL;
   cons->ignorerelease = FALSE;

   cons->cursor.x = 0;
   cons->cursor.y = 0;
   cons->cursor.width = 0;
   cons->cursor.height= 0;

   gdk_color_parse("#000000", &cons->colors.background);
   gdk_color_parse("#c0c0c0", &cons->colors.foreground);
   gdk_color_parse("#701010", &cons->colors.backcursor);
   gdk_color_parse("#ffffff", &cons->colors.forecursor);
   gdk_color_parse("#202020", &cons->colors.backshadow);
   gdk_color_parse("#e0e0e0", &cons->colors.foreshadow);
   gdk_color_parse("#101060", &cons->colors.backscroll);
   gdk_color_parse("#7090a0", &cons->colors.forescroll);
   gdk_color_parse("#a02020", &cons->colors.backlight);
   gdk_color_parse("#ffffff", &cons->colors.forelight);
   gdk_color_parse("#000000", &cons->colors.backdisabled);
   gdk_color_parse("#606060", &cons->colors.foredisabled);
   gdk_color_parse("#000000", &cons->colors.backstandard);
   gdk_color_parse("#c0c0c0", &cons->colors.forestandard);
   cons->colors.colors_alloc = FALSE;
   cons->colors.bold = FALSE;

   GTK_WIDGET_SET_FLAGS(cons, GTK_CAN_FOCUS);
   
}



GtkType sc_console_get_type(void) {

   static GtkType sc_console_type = 0;
   
   if(sc_console_type == 0) {
      static const GtkTypeInfo sc_console_info = {
         (char *)"ScConsole",
         sizeof(ScConsole),
         sizeof(ScConsoleClass),
         (GtkClassInitFunc)_sc_console_class_init,
         (GtkObjectInitFunc)_sc_console_init_obj,
         (GtkArgSetFunc)NULL,
         (GtkArgGetFunc)NULL
      };
      sc_console_type = gtk_type_unique(sc_drawbuf_get_type(), &sc_console_info);
   }

   return(sc_console_type);

}



void sc_console_buffer_size(ScConsole *cons, gint width, gint height) {

   g_return_if_fail(width > 0 && height > 0);

   if(width < cons->text.vieww) width = cons->text.vieww;
   if(height< cons->text.viewh) height= cons->text.viewh;
   cons->text.buffer = (char *)realloc(cons->text.buffer, width * height);
   if(cons->text.buffer) {
      memsetn(cons->text.buffer, ' ', width * height);
   }

   cons->text.viewx   = 0;
   cons->text.viewy   = 0;
   cons->text.bufferw = width;
   cons->text.bufferh = height;
   cons->text.scrollx = (cons->text.vieww < width);
   cons->text.scrolly = (cons->text.viewh < height);

}



void sc_console_init(ScConsole *cons, gint x, gint y, gint width, gint height, ScConsoleStyle style) {

   _sc_console_destroy_data(cons);

   cons->text.vieww   = width;
   cons->text.viewh   = height;
   sc_console_buffer_size(cons, width, height);

   cons->style = style;
   if(style != CONSOLE_BORDERLESS) {
      x -= SC_CONSOLE_BORDER_WIDTH_CHARS;
      y -= SC_CONSOLE_BORDER_HEIGHT_CHARS;
      width += SC_CONSOLE_BORDER_WIDTH_CHARS * 2;
      height+= SC_CONSOLE_BORDER_HEIGHT_CHARS * 2;
   }

   cons->req_alloc.x = _sc_console_char_x(cons, x);
   cons->req_alloc.y = _sc_console_char_y(cons, y);
   cons->req_alloc.width = width * _sc_console_char_width(cons);
   cons->req_alloc.height= height* _sc_console_char_height(cons);

   gtk_widget_set_usize(GTK_WIDGET(cons), cons->req_alloc.width, cons->req_alloc.height);
   gtk_widget_set_sensitive(GTK_WIDGET(cons), FALSE);
   
   sc_console_clear(cons);

}



void sc_console_clear(ScConsole *cons) {

   memset(cons->text.buffer, ' ', sc_console_get_width(cons) * sc_console_get_height(cons));
   sc_console_highlight_detach_all(cons);
   _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));

}



GtkWidget *sc_console_new(gint x, gint y, gint width, gint height, ScConsoleStyle style) {

   ScConsole *cons;

   cons = gtk_type_new(sc_console_get_type());
   g_return_val_if_fail(cons != NULL, NULL);

   sc_console_init(cons, x, y, width, height, style);
   return(GTK_WIDGET(cons));

}



void sc_console_write_char(ScConsole *cons, gint x, gint y, char ch) {

   if(x < 0 || x >= sc_console_get_width(cons)) return;
   if(y < 0 || y >= sc_console_get_height(cons)) return;

   if(ch < 0x20) ch = ' ';
   *sc_console_get_char_ptr(cons, x, y) = ch;
   _sc_console_draw_region(cons, x, y, 1, 1);
   
}



void sc_console_write_line(ScConsole *cons, gint x, gint y, const char *line) {

   gint width = sc_console_get_width(cons);
   const char *start;
   const char *end;

   if(y < 0 || y >= sc_console_get_height(cons)) return;
   
   for(start = line; *start != '\0' && (line - start) + x < 0; ++start) /* Just loop */;
   for(end = start; *end != '\0' && (end - start) + x < width; ++end) /* Just loop */;
   if(end == start) return;

   strncpy(sc_console_get_char_ptr(cons, x, y), start, end - start);
   _sc_console_draw_region(cons, x, y, end - start, 1);

}



void sc_console_clear_line(ScConsole *cons, gint y) {

   if(y < 0 || y >= sc_console_get_height(cons)) return;
   memset(sc_console_get_char_ptr(cons, 0, y), ' ', sc_console_get_width(cons));
   _sc_console_draw_region(cons, 0, y, sc_console_get_width(cons), 1);

}



void sc_console_write_line_wrap(ScConsole *cons, gint x, gint y, const char *line) {

   gint width = sc_console_get_width(cons);
   gint height= sc_console_get_height(cons);
   gint minx;
   char *p;
   
   if(y < 0 || y >= height || x < 0 || line == NULL) return;
   while(x >= width) {
      x -= width;
      ++y;
   }
   minx = x;

   p = sc_console_get_char_ptr(cons, x, y);
   while(y < height && *line != '\0') {
      *p = *line;
      ++line;
      ++p;
      ++x;
      
      if(x >= width) {
         _sc_console_draw_region(cons, minx, y, x - minx, 1);
         minx = 0;
         x = 0;
         ++y;
      }
   }

   if(x > minx) _sc_console_draw_region(cons, minx, y, x - minx, 1);

}



static void _sc_console_lock_view_to_cursor(ScConsole *cons) {
   
   ScConsoleCursor *cursor = &cons->cursor;
   gboolean needrewrite = FALSE;

   if(cursor->x < cons->text.viewx) {
      cons->text.viewx = cursor->x;
      _sc_console_draw_horiz_scroll(cons);
      needrewrite = TRUE;
   } else if(cursor->x + cursor->width > cons->text.viewx + cons->text.vieww) {
      cons->text.viewx = cursor->y - (cons->text.vieww - cursor->width);
      _sc_console_draw_horiz_scroll(cons);
      needrewrite = TRUE;
   }

   if(cursor->y < cons->text.viewy) {
      cons->text.viewy = cursor->y;
      _sc_console_draw_vert_scroll(cons);
      needrewrite = TRUE;
   } else if(cursor->y + cursor->height > cons->text.viewy + cons->text.viewh) {
      cons->text.viewy = cursor->y - (cons->text.viewh - cursor->height);
      _sc_console_draw_vert_scroll(cons);
      needrewrite = TRUE;
   }
   
   if(needrewrite) {
      _sc_console_draw_region(cons, 0, 0, sc_console_get_width(cons), sc_console_get_height(cons));
   }

}



void sc_console_set_cursor(ScConsole *cons, gint x, gint y, gint width, gint height) {

   ScConsoleCursor *cursor = &cons->cursor;
   gint oldx;
   gint oldy;
   gint oldw;
   gint oldh;
   
   oldx = cursor->x;
   oldy = cursor->y;
   oldw = cursor->width;
   oldh = cursor->height;
      
   cursor->x = x;
   cursor->y = y;
   cursor->width = width;
   cursor->height= height;

   _sc_console_lock_view_to_cursor(cons);
   _sc_console_draw_region(cons, oldx, oldy, oldw, oldh);
   _sc_console_draw_region(cons, cursor->x, cursor->y, cursor->width, cursor->height);

}



void sc_console_set_cursor_pos(ScConsole *cons, gint x, gint y) {

   ScConsoleCursor *cursor = &cons->cursor;
   gint oldx;
   gint oldy;

   oldx = cursor->x;
   oldy = cursor->y;
      
   cursor->x = x;
   cursor->y = y;

   _sc_console_lock_view_to_cursor(cons);
   _sc_console_draw_region(cons, oldx, oldy, cursor->width, cursor->height);
   _sc_console_draw_region(cons, cursor->x, cursor->y, cursor->width, cursor->height);

}



void sc_console_set_cursor_highlighted(ScConsole *cons, gboolean highlighted) {

   if((highlighted && !cons->cursor.highlighted) || (!highlighted && cons->cursor.highlighted)) {
      cons->cursor.highlighted = highlighted;
      _sc_console_draw_region(cons, 
                              sc_console_get_cursor_x(cons),
                              sc_console_get_cursor_y(cons),
                              sc_console_get_cursor_width(cons),
                              sc_console_get_cursor_height(cons));
   }

}



void sc_console_set_colors(ScConsole *cons, GdkColor *fg, GdkColor *bg) {

   if(fg != NULL) {
      cons->colors.foreground.red   = fg->red;
      cons->colors.foreground.green = fg->green;
      cons->colors.foreground.blue  = fg->blue;
   } 
   if(bg != NULL) {
      cons->colors.background.red   = bg->red;
      cons->colors.background.green = bg->green;
      cons->colors.background.blue  = bg->blue;
   }
   cons->colors.colors_alloc = FALSE;
   cons->colors.bold = FALSE;
   _sc_console_draw_all(cons);

}



GdkColor *sc_console_get_color(ScConsole *cons, ScConsoleColorId rqst) {

   switch(rqst) {
   case SC_CONSOLE_FOREGROUND:
      return(&cons->colors.foreground);
   case SC_CONSOLE_BACKGROUND:
      return(&cons->colors.background);
   case SC_CONSOLE_FORECURSOR:
      return(&cons->colors.forecursor);
   case SC_CONSOLE_BACKCURSOR:
      return(&cons->colors.backcursor);
   case SC_CONSOLE_FORESHADOW:
      return(&cons->colors.foreshadow);
   case SC_CONSOLE_BACKSHADOW:
      return(&cons->colors.backshadow);
   case SC_CONSOLE_FORESCROLL:
      return(&cons->colors.forescroll);
   case SC_CONSOLE_BACKSCROLL:
      return(&cons->colors.backscroll);
   case SC_CONSOLE_FORELIGHT:
      return(&cons->colors.forelight);
   case SC_CONSOLE_BACKLIGHT:
      return(&cons->colors.backlight);
   case SC_CONSOLE_FOREDISABLED:
      return(&cons->colors.foredisabled);
   case SC_CONSOLE_BACKDISABLED:
      return(&cons->colors.backdisabled);
   case SC_CONSOLE_FORESTANDARD:
      return(&cons->colors.forestandard);
   case SC_CONSOLE_BACKSTANDARD:
      return(&cons->colors.backstandard);
   }
   return(NULL);

}



void sc_console_highlight_attach(ScConsole *cons, GdkColor *fg, GdkColor *bg, gboolean bold, gint x, gint y, gint width, gint height) {

   ScConsoleHighlight *high;
   
   high = (ScConsoleHighlight *)malloc(sizeof(ScConsoleHighlight));
   g_return_if_fail(high != NULL);

   if(fg != NULL) {
      high->colors.foreground.red   = fg->red;
      high->colors.foreground.green = fg->green;
      high->colors.foreground.blue  = fg->blue;
   } else {
      high->colors.foreground.red   = cons->colors.foreground.red;
      high->colors.foreground.green = cons->colors.foreground.green;
      high->colors.foreground.blue  = cons->colors.foreground.blue;
   }
   if(bg != NULL) {
      high->colors.background.red   = bg->red;
      high->colors.background.green = bg->green;
      high->colors.background.blue  = bg->blue;
   } else {
      high->colors.background.red   = cons->colors.background.red;
      high->colors.background.green = cons->colors.background.green;
      high->colors.background.blue  = cons->colors.background.blue;
   }
   high->colors.colors_alloc = FALSE;
   high->colors.bold = bold;
   
   high->x = x;
   high->y = y;
   high->width = width;
   high->height= height;
   
   cons->highlights = g_list_append(cons->highlights, high);
   _sc_console_draw_region(cons, high->x, high->y, high->width, high->height);

}



void sc_console_highlight_attach_disabled(ScConsole *cons, gint x, gint y, gint width, gint height) {

   sc_console_highlight_attach(cons, &cons->colors.foredisabled, &cons->colors.backdisabled, FALSE, x, y, width, height);

}



gboolean sc_console_highlight_detach(ScConsole *cons) {

   ScConsoleHighlight *high;
   GList *cur;
   gint x;
   gint y;
   gint width;
   gint height;

   /* Is there anything to delete? */
   cur = cons->highlights;
   if(cur == NULL) return(FALSE);

   /* Delete the last highlight */
   while(cur->next != NULL) cur = cur->next;
   high = cur->data;
   cur->data = NULL;
   
   x = high->x;
   y = high->y;
   width = high->width;
   height= high->height;
   
   free(cur->data);
   cur->data = NULL;
   cons->highlights = g_list_remove_link(cons->highlights, cur);
   g_list_free(cur);
   
   /* Update display */
   _sc_console_draw_region(cons, x, y, width, height);
   return(TRUE);
 
}



void sc_console_highlight_detach_all(ScConsole *cons) {

   while(sc_console_highlight_detach(cons)) /* Just loop */;

}
