/*
 *   XPP - The X Printing Panel
 *   --------------------------
 *
 *   cupsHelper - This class was originally written by Michael Goffioul
 *   (e-mail: gofioul@emic.ucl.ac.be) for QTCUPS, another excellent
 *   printing frontend for CUPS. This version has the same functions,
 *   but is more general, it is not dependent on QT, and it also overloads
 *   printf to get rid of console messages completely and to take the
 *   information out of these messages
 *
 *   Thank you for your great work, Michael!
 *
 *   See the header file (cupshelper.h) for an overview of the fields and
 *   methods.
 *
 *   Copyright 2000 by Till Kamppeter and Michael Goffioul
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of the
 *   License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *   02111-1307  USA
 *
 */

#include "cupshelper.h"
#include "passworddialog.h"

#include <stdlib.h>
#include <stdarg.h>
#include <dlfcn.h>
#include <cups/language.h>

// Global variables

char printfstr[1024];

//**************************************************************
//* Supress console output when CUPS library functions ask for *
//* authentication and catch the login info to be put out by   *
//* overloading printf                                         *
//**************************************************************

//Comment out this function for debugging

int printf(const char *format, ...)
{
        va_list ap;
        va_start(ap, format);
        char* login = va_arg(ap, char *);
        char* host = va_arg(ap, char *);
        int r = sprintf(printfstr, "Authentication required on %s...", host);
        va_end(ap);
        return r;
}

//***************************************
//* CUPS libraries overloaded functions *
//***************************************

const char* cupsGetPassword(const char *prompt)
{
	return CupsHelper::cupsGetPassword();
}

const char* cupsUser()
{
	if (strcmp(CupsHelper::login(),"") == 0) return cupsUserFromLibrary();
	else return CupsHelper::login();
}

const char* cupsServer()
{
	if (strcmp(CupsHelper::host(),"") == 0) return cupsServerFromLibrary();
	else return CupsHelper::host();
}

int ippPort()
{
	if (CupsHelper::port() < 0) return ippPortFromLibrary();
	else return CupsHelper::port();
}

#define	BUFFER_SIZE	HTTP_MAX_HOST
static	char	server_buffer[BUFFER_SIZE];
static	char	user_buffer[BUFFER_SIZE];
static	char	password_buffer[BUFFER_SIZE];

const char* cupsServerFromLibrary()
{
	void	*handle = dlopen("libcups.so",RTLD_LAZY);

	memset(server_buffer,0,BUFFER_SIZE);
	handle = dlopen("libcups.so",RTLD_LAZY);
	if (handle) {
		const char* (*func)() = (const char*(*)())dlsym(handle,"cupsServer");
		if (dlerror() == NULL)
			strncpy(server_buffer,(*func)(),BUFFER_SIZE-1);
		dlclose(handle);
	}
	return server_buffer;
}

const char* cupsUserFromLibrary()
{
	void	*handle = dlopen("libcups.so",RTLD_LAZY);

	memset(user_buffer,0,BUFFER_SIZE);
	handle = dlopen("libcups.so",RTLD_LAZY);
	if (handle) {
		const char* (*func)() = (const char*(*)())dlsym(handle,"cupsUser");
		if (dlerror() == NULL)
			strncpy(user_buffer,(*func)(),BUFFER_SIZE-1);
		dlclose(handle);
	}
	return user_buffer;
}

const char* cupsGetPasswordFromLibrary(const char *prompt)
{
	void	*handle = dlopen("libcups.so",RTLD_LAZY);

	memset(password_buffer,0,BUFFER_SIZE);
	handle = dlopen("libcups.so",RTLD_LAZY);
	if (handle) {
		const char* (*func)(const char*) = (const char*(*)(const char*))dlsym(handle,"cupsGetPassword");
		if (dlerror() == NULL)
			strncpy(password_buffer,(*func)(prompt),BUFFER_SIZE-1);
		dlclose(handle);
	}
	return password_buffer;
}

int ippPortFromLibrary()
{
	void	*handle = dlopen("libcups.so",RTLD_LAZY);
	int	p(631);

	handle = dlopen("libcups.so",RTLD_LAZY);
	if (handle) {
		int (*func)() = (int (*)())dlsym(handle,"ippPort");
		if (dlerror() == NULL)
			p = (*func)();
		dlclose(handle);
	}
	return p;
}

//*****************************************************************************

char	CupsHelper::host_[256] = "";
int	CupsHelper::port_ = -1;
char    CupsHelper::login_[256] = "";
char    CupsHelper::password_[256] = "";
int     CupsHelper::count_ = 0;

CupsHelper::CupsHelper(){
}
CupsHelper::~CupsHelper(){
}

const char* CupsHelper::host(){
	return host_;
}

int CupsHelper::port(){
	return port_;
}

void CupsHelper::setHostInfo(const char *host, int port = 631){
	strcpy(host_,host);
	port_ = port;
}

void CupsHelper::setLoginInfo(const char *usr = 0, const char *pwd = 0){
	strcpy(login_,usr);
	strcpy(password_,pwd);
}

const char* CupsHelper::login(){
	return login_;
}

const char* CupsHelper::password(){
	return password_;
}

ipp_t* CupsHelper::newIppRequest(){
	ipp_t	*request = ippNew();
	request->request.op.request_id = 1;
	cups_lang_t	*lang = cupsLangDefault();
	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_CHARSET,"attributes-charset",NULL,cupsLangEncoding(lang));
	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_LANGUAGE,"attributes-natural-language",NULL,lang->language);
	return request;
}

ipp_t* CupsHelper::processRequest(ipp_t *req, const char *res){
	http_t	*HTTP = httpConnect(host_,port_);
	if (!HTTP)
	{
		ippDelete(req);
		return 0;
	}
	ipp_t	*answer = cupsDoRequest(HTTP,req,res);
	httpClose(HTTP);
	if (!answer)
		return 0;
	if (answer->state == IPP_ERROR || answer->state == IPP_IDLE)
	{
		ippDelete(answer);
		return 0;
	}
	return answer;
}

bool CupsHelper::checkHost(){
	http_t	*HTTP = httpConnect(host_,port_);
	if (!HTTP) return false;
	httpClose(HTTP);
	return true;
}

const char* CupsHelper::cupsGetPassword(){
        char login[256];
        const char *password;
        int noprintfstr = 0;
	if (count_ == 0 && strcmp(password_,"") != 0) return password_;
	else {
		count_++;
                strcpy(login,login_);
                if (printfstr[0] == '\0') {
		  noprintfstr = 1;
                  sprintf(printfstr,"Authentication required on %s...",
                          host());
                }
		// open a dialog to ask for password
                password = passworddialog(printfstr,login);
                if (noprintfstr == 1) printfstr[0] = '\0';
                if (password) {
                  setLoginInfo(login,password);
                  return password;
                } else return 0;
	}
}

void CupsHelper::setup(){
	if (strcmp(host_,"") == 0) strcpy(host_,cupsServerFromLibrary());
	if (strcmp(login_,"") == 0) strcpy(login_,cupsUserFromLibrary());
	if (port_ < 0) port_ = ippPortFromLibrary();
}

int CupsHelper::printerType(const char *name){
	ipp_t	  *request = newIppRequest();
	char      str[256];
	int	  value(-1);

	request->request.op.operation_id = IPP_GET_PRINTER_ATTRIBUTES;
	sprintf(str,"ipp://%s:%d/printers/%s",host(),port(),name);
	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_URI,"printer-uri",NULL,str);
	//sprintf(str,"/printers/%s",name);
	sprintf(str,"/printers/");
	request = processRequest(request,str);
	if (request && request->curtag == IPP_TAG_PRINTER) {
		ipp_attribute_t	*attr = ippFindAttribute(request,"printer-type",IPP_TAG_ENUM);
		if (attr) value = attr->values[0].integer;
	}
	ippDelete(request);
	return value;
}
