/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_pophelp.cc,v 1.4 2000/11/09 00:12:09 bbarrett Exp $
 *
 *	Function:	- popup help for buttons
 *
 *			- When the cursor enters a button with attached help,
 *			  after a short delay a help message will be popped up.
 *			  If the button is pressed or the cursor leaves the
 *			  button before the help message is popped up the
 *			  popup is cancelled.
 *			  Once help for a button is popped up it stays up
 *			  until the cursor leaves the button or the button
 *			  is pressed.
 */
#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/Frame.h>
#include <Xm/LabelG.h>
#include <string.h>

#include "all_list.h"

#define LABEL_MAX	127	       /* max. length of help msg. */
#define POPHELP_DELAY	(1000L)	       /* delay popup for 1 sec. */

/*
 * Private functions.
 */
extern "C" {
static void pophelp(Widget, void*, XEvent*, Boolean*);
static void popup(struct pophelp *);
static void create_popup(struct pophelp *);
static void destroy_cb(Widget);

static void arm_cb(Widget, struct pophelp*);
static int pophelp_cmp(void*, void*);

/*
 * Private variables.
 */
struct pophelp {
  Widget ph_button;		       /* button help attached to */
  Widget ph_shell;		       /* popup help message shell */
  XtIntervalId ph_timerid;	       /* delay timer for popping up */
  char ph_text[LABEL_MAX + 1];	       /* help message text */
  int ph_up;			       /* help currently popped up? */
};

static LIST *help = 0;		       /* list of popup helps */
static Pixel black;		       /* black colour */
}

/*
 *	xmpi_add_pophelp
 *
 *	Function:	- add popup help for a button
 *			- replaces previous help if any
 *	Accepts:	- button
 *			- help message
 */
void
xmpi_add_pophelp(Widget button, char *msg)
{
  // Lesstif seems to cause strange warnings.  Disable popups
#ifndef LESSTIF_VERSION
  struct pophelp newpop;
  struct pophelp *pop;

/*
 * Initialize application help list and colours if necessary.
 */
  memset(&newpop, 0, sizeof(newpop));
  if (help == 0) {
    help =  al_init(sizeof(struct pophelp), pophelp_cmp);
    if (help == 0)
      return;

    black = BlackPixel(XtDisplay(button),
		       DefaultScreen(XtDisplay(button)));
  }
/*
 * Check if button already has help.
 */
  newpop.ph_button = button;

  if ((pop = (struct pophelp*) al_find(help, &newpop))) {
/*
 * Is the new help different to the old?
 */
    if (strcmp(pop->ph_text, msg)) {
/*
 * Set new message text and destroy the old help popup if any.  A new
 * popup shell will be created when the help is triggered if ever.
 */
      strncpy(pop->ph_text, msg, LABEL_MAX);
      if (pop->ph_shell) {
	if (pop->ph_timerid) {
	  XtRemoveTimeOut(pop->ph_timerid);
	  pop->ph_timerid = 0;
	}
	XtDestroyWidget(pop->ph_shell);
	pop->ph_shell = 0;
	pop->ph_up = 0;
      }
    }
  } else {
/*
 * Add a new entry to the help list.  The popup help shell will be
 * created when the help is triggered if ever.
 */
    newpop.ph_shell = 0;
    newpop.ph_timerid = 0;
    newpop.ph_up = 0;
    strncpy(newpop.ph_text, msg, LABEL_MAX);

    pop = (struct pophelp*) al_append(help, &newpop);
    if (pop == 0)
      return;
/*
 * Add callbacks for removing the help when the button is destroyed and
 * for canceling help when it is pressed.
 */
    XtAddCallback(button, XmNdestroyCallback, 
		  (XtCallbackProc) destroy_cb, NULL);
    XtAddCallback(button, XmNarmCallback, 
		  (XtCallbackProc) arm_cb, (XtPointer) pop);
/*
 * Set up the button to generate enter/leave events.
 */
    XtAddEventHandler(button,
		      EnterWindowMask | LeaveWindowMask,
		      False, 
		      (XtEventHandler) pophelp, 
		      (XtPointer) pop);
  }
#endif // LESSTIFVERSION
}

/*
 *	pophelp
 *
 *	Function:	- handler for button enter and leave events
 *	Accepts:	- button
 *			- handler data
 *			- event
 */
static void
pophelp(Widget, void *data, XEvent *event, Boolean *)
{
  struct pophelp *pop = (struct pophelp*) data;

  if (event->type == EnterNotify) {

    if (event->xcrossing.subwindow
	|| event->xcrossing.mode == NotifyUngrab) {
      return;
    }
/*
 * Create the popup if necessary and pop it up.
 */
    if (pop->ph_shell == 0) {
      create_popup(pop);
    }
    pop->ph_timerid = XtAppAddTimeOut(
			    XtWidgetToApplicationContext(pop->ph_button),
			    POPHELP_DELAY, 
			    (XtTimerCallbackProc) popup,
			    (XtPointer) pop);
  } else if (event->type == LeaveNotify) {
/*
 * Cancel pending popup if any.
 */
    if (pop->ph_timerid) {
      XtRemoveTimeOut(pop->ph_timerid);
    }
    if (pop->ph_up) {
      XtPopdown(pop->ph_shell);
      pop->ph_up = 0;
    }
  }
}

/*
 *	create_popup
 *
 *	Function:	- create a popup help shell and message
 *	Accepts:	- popup help
 */
static void
create_popup(struct pophelp *pop)
{
  Widget frame;

  XmString xstr;

  pop->ph_shell = XtVaAppCreateShell("help_widget", "XMPI",
				     applicationShellWidgetClass,
				     XtDisplayOfObject(pop->ph_button),
				     XmNoverrideRedirect, True, NULL);

  xstr = XmStringCreateSimple(pop->ph_text);
/*
 * We want a simple black line edge around the label.
 */
  frame = XtVaCreateManagedWidget("popup_help",
				  xmFrameWidgetClass, pop->ph_shell,
				  XmNmarginWidth, 1,
				  XmNmarginHeight, 1,
				  XmNbottomShadowColor, black,
				  XmNtopShadowColor, black,
				  XmNshadowType, XmSHADOW_ETCHED_OUT,
				  XmNshadowThickness, 1, NULL);

  XtVaCreateManagedWidget("popup_help_label",
			  xmLabelGadgetClass, frame,
			  XmNlabelString, xstr,
			  NULL);

  XmStringFree(xstr);
}

/*
 *	popup
 *
 *	Function:	- pops up help message
 *			- this is only ever invoked by the help timer firing
 *	Accepts:	- popup help
 */
static void
popup(struct pophelp *pop)
{
  Position x, y;		       /* button position in root */

  Dimension border, width, height;     /* button sizes */

  pop->ph_timerid = 0;
/*
 * Find the position of the button in the root window and the button's
 * border size, width and height.  These are recalculated each time
 * because the button may change size or position.
 */
  XtTranslateCoords(pop->ph_button, (Position) 0, (Position) 0, &x, &y);

  XtVaGetValues(pop->ph_button,
		XmNborderWidth, &border,
		XmNheight, &height,
		XmNwidth, &width, NULL);
/*
 * Set the popup's position to have the top left corner half way along
 * and just below the button.
 */
  XtVaSetValues(pop->ph_shell,
		XmNx, x + border + width / 2,
		XmNy, y + 2 * border + height, NULL);

  pop->ph_up = 1;
  XtPopup(pop->ph_shell, XtGrabExclusive);
}

/*
 *	pophelp_cmp
 *
 *	Function:	- compare two buttons
 *	Accepts:	- ptr to two entries
 *	Returns:	- 0 if same button else 1
 */
static int
pophelp_cmp(void *tmp_p1, void *tmp_p2)
{
  struct pophelp *p1 = (struct pophelp *) tmp_p1;
  struct pophelp *p2 = (struct pophelp *) tmp_p2;
  return (p1->ph_button != p2->ph_button);
}

/*
 *      destroy_cb
 *
 *      Function:       - button destroy callback
 *			- cleans up any help associated with the button
 *      Accepts:        - button being destroyed
 */
static void
destroy_cb(Widget button)

{
  struct pophelp newpop;

  struct pophelp *pop;

/*
 * Find the help entry for the button, destroy the popup shell if any
 * and then delete the help entry.
 */
  newpop.ph_button = button;
  pop = (struct pophelp*) al_find(help, &newpop);

  if (pop->ph_timerid) {
    XtRemoveTimeOut(pop->ph_timerid);
  }
  if (pop->ph_shell) {
    XtDestroyWidget(pop->ph_shell);
  }
  al_delete(help, pop);
}

/*
 *	arm_cb
 *
 *	Function:	- button arm callback
 *			- cancel any pending or popped up help
 *	Accepts:	- button
 *			- popup help
 */
static void
arm_cb(Widget, struct pophelp *pop)
{
  if (pop->ph_up) {
    XtPopdown(pop->ph_shell);
    pop->ph_up = 0;
  } else if (pop->ph_timerid) {
    XtRemoveTimeOut(pop->ph_timerid);
    pop->ph_timerid = 0;
  }
}
