/*  XMMS STATUS PLUGIN - Status Docklet Plugin for XMMS
 *  Copyright (C) 2000  Ian Campbell.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "xmms-status-plugin.h"
#include "xmms_mini.xpm"

static void class_init(StatusDockletImageClass *klass);
static void init(StatusDockletImage *im);

static void draw (GtkWidget *widget, GdkRectangle *area);
static gint expose_event (GtkWidget *widget, GdkEventExpose *event);
static void realize (GtkWidget *widget);
static void unrealize (GtkWidget *widget);
static void size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void destroy (GtkObject *object);

static void state_changed (GtkWidget *widget, GtkStateType previous_state);
static void style_set (GtkWidget *widget, GtkStyle *previous_style);

static void draw_background_pixmap (StatusDockletImage *im);

static void load_image(StatusDockletImage *im, StatusDockletState state);
static void free_image(StatusDockletImage *im, StatusDockletState state);
static gboolean timeout_callback(StatusDockletImage *im);
static void update (StatusDockletImage *im);

struct _StatusDockletImageInfo
{
	gchar *path;
	gint frame_delay;
	gint frame_count;
	
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	
	gint current_frame;
};

GtkType status_docklet_image_get_type(void)
{
  static guint image_type = 0;

  if (!image_type)
    {
      GtkTypeInfo image_info =
      {
        "StatusDockletImage",
        sizeof (StatusDockletImage),
        sizeof (StatusDockletImageClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        NULL,
        NULL
      };

      image_type = gtk_type_unique (gtk_widget_get_type (), &image_info);
    }

  return image_type;

}

GtkWidget *status_docklet_image_new(void)
{
	StatusDockletImage *image;
	
	image = gtk_type_new (status_docklet_image_get_type ());

	image->images[STATUS_DOCKLET_STATE_PLAYING] = NULL;
	image->images[STATUS_DOCKLET_STATE_PAUSED] = NULL;
	image->images[STATUS_DOCKLET_STATE_STOPPED] = NULL;
	image->current_state = -1;
	image->timeout_tag = 0;
	image->default_pixmap = NULL;
	image->default_mask = NULL;
	image->offscreen_pixmap = NULL;
	image->background_pixmap = NULL;

	return GTK_WIDGET(image);
}

static void class_init(StatusDockletImageClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	widget_class = GTK_WIDGET_CLASS(klass);
	object_class = GTK_OBJECT_CLASS(klass);

	widget_class->draw = draw;
	widget_class->expose_event = expose_event;
	widget_class->realize = realize;
	widget_class->unrealize = unrealize;
	widget_class->size_allocate = size_allocate;
	widget_class->style_set = style_set;
	widget_class->state_changed = state_changed;

	object_class->destroy = destroy;
}

static void init(StatusDockletImage *im)
{
	GtkWidget *widget;

	GTK_WIDGET_SET_FLAGS(im, GTK_NO_WINDOW);

	widget = GTK_WIDGET(im);

	widget->requisition.width = 22;
	widget->requisition.height = 22;
}

static void destroy (GtkObject *object)
{

}

static void update (StatusDockletImage *im)
{
	GtkWidget *widget = GTK_WIDGET(im);

	if ( GTK_WIDGET_REALIZED (widget) )
		gtk_widget_queue_draw(widget);
}

static void paint (StatusDockletImage *im, GdkRectangle *area)
{
	GtkWidget *widget;
	StatusDockletImageInfo *image_info;

	g_return_if_fail ( GTK_WIDGET_REALIZED(im) );

	widget = GTK_WIDGET(im);

	if ( im->current_state == STATUS_DOCKLET_STATE_LAST )
		image_info = NULL;
	else
		image_info = im->images[im->current_state];

	gdk_draw_pixmap (im->offscreen_pixmap,
			 widget->style->black_gc,
			 im->background_pixmap,
			 0, 0, 0, 0, 22, 22);
	
	if ( image_info ) {
		
  		gdk_gc_set_clip_mask (widget->style->black_gc, image_info->mask);
   		gdk_gc_set_clip_origin (widget->style->black_gc, -image_info->current_frame*22, 0);
		
		gdk_window_copy_area (im->offscreen_pixmap,
				      widget->style->black_gc,
				      0, 0, image_info->pixmap,
				      image_info->current_frame*22, 0, 22, 22);
  		gdk_gc_set_clip_mask (widget->style->black_gc, NULL);


	} else {
		/* Draw the default image */
		/* Image is 16x16, so offset image and mask by 22-16/2 = 3 */

		gdk_gc_set_clip_mask (widget->style->black_gc, im->default_mask);
		gdk_gc_set_clip_origin (widget->style->black_gc, 3, 3);

		gdk_window_copy_area (im->offscreen_pixmap, 
				      widget->style->black_gc,
				      3, 3, im->default_pixmap,
				      0, 0, 16, 16);
		gdk_gc_set_clip_mask (widget->style->black_gc, NULL);
	}

	gdk_draw_pixmap (widget->window,
			 widget->style->black_gc,
			 im->offscreen_pixmap,
			 0, 0, 0, 0, 22, 22);
					      
}

static void draw (GtkWidget *widget, GdkRectangle *area)
{
	StatusDockletImage *im;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (STATUS_DOCKLET_IS_IMAGE (widget));
	g_return_if_fail (area != NULL);
	
	if ( GTK_WIDGET_DRAWABLE (widget)) {
		GdkRectangle p_area;
		
		im = STATUS_DOCKLET_IMAGE (widget);
		
		if (gdk_rectangle_intersect (area, &im->area, &p_area))
			paint(im, &p_area);

		gdk_flush ();
	}
}
	
static gint expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (STATUS_DOCKLET_IS_IMAGE (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (GTK_WIDGET_REALIZED (widget))
		draw(widget, &event->area);

	return FALSE;
}

static void realize (GtkWidget *widget)
{
	gint i;
	StatusDockletImage *im;
	GdkVisual *visual;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (STATUS_DOCKLET_IS_IMAGE(widget));

	GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

	widget->window = gtk_widget_get_parent_window(widget);
	gdk_window_ref(widget->window);

	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

	im = STATUS_DOCKLET_IMAGE (widget);
	visual = gtk_widget_get_visual (widget);

	im->default_pixmap = gdk_pixmap_create_from_xpm_d(widget->window, &im->default_mask,
							  &widget->style->bg[GTK_STATE_NORMAL],
							  xmms_mini_xpm);
	im->offscreen_pixmap = gdk_pixmap_new (widget->window,
					       widget->requisition.width,
					       widget->requisition.height,
					       visual->depth);
	im->background_pixmap = gdk_pixmap_new (widget->window,
					       widget->requisition.width,
					       widget->requisition.height,
					       visual->depth);

	draw_background_pixmap ( im );

	for ( i = 0 ; i < STATUS_DOCKLET_STATE_LAST; i++ )
		if ( im->images[i] )
			load_image(im, i);
}

static void unrealize (GtkWidget *widget)
{
	StatusDockletImage *im;
	gint i;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (STATUS_DOCKLET_IS_IMAGE(widget));

	im = STATUS_DOCKLET_IMAGE(widget);

	gtk_style_unref (widget->style);

	for ( i = 0 ; i < STATUS_DOCKLET_STATE_LAST ; i++ )
		free_image (im, i);

	gdk_pixmap_unref(im->default_pixmap);
	gdk_bitmap_unref(im->default_mask);
	im->default_pixmap = NULL;
	im->default_mask = NULL;

	gdk_window_unref(widget->window);

	GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
}

static void draw_background_pixmap (StatusDockletImage *im)
{
	GtkWidget *widget;

	widget = GTK_WIDGET(im);

	g_return_if_fail ( GTK_WIDGET_REALIZED(widget) );

	if ( widget->window != NULL ) {
		GtkStyle *style;
		guint state;
		GdkGC *gc;

		style = gtk_widget_get_style (widget->parent);
		state = GTK_WIDGET_STATE (widget->parent);
		gc = gdk_gc_new (widget->window);
		gdk_gc_copy (gc, style->bg_gc[state]);

		if (style->bg_pixmap[state] != NULL) {
			gdk_gc_set_tile (gc, style->bg_pixmap[state]);
			gdk_gc_set_fill (gc, GDK_TILED);
		} else
			gdk_gc_set_fill (gc, GDK_SOLID);

		gdk_gc_set_clip_origin (gc, -im->area.x,
				   -im->area.y);

		gdk_draw_rectangle (im->background_pixmap,
				    gc, TRUE, 0, 0,
				    widget->requisition.width,
				    widget->requisition.height);
		
		gdk_gc_destroy(gc);
	}
}


static void state_changed (GtkWidget *widget, GtkStateType previous_state)
{
	g_assert(STATUS_DOCKLET_IS_IMAGE(widget));

	if (GTK_WIDGET_REALIZED(widget))
		draw_background_pixmap(STATUS_DOCKLET_IMAGE(widget));
}

static void style_set (GtkWidget *widget, GtkStyle *previous_style)
{
	g_assert(STATUS_DOCKLET_IS_IMAGE(widget));

	if ( GTK_WIDGET_REALIZED(widget))
		draw_background_pixmap(STATUS_DOCKLET_IMAGE(widget));
}

static void size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	StatusDockletImage *im;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (STATUS_DOCKLET_IS_IMAGE(widget));
	g_return_if_fail (allocation != NULL);
	
	im = STATUS_DOCKLET_IMAGE (widget);

	if ( GTK_WIDGET_REALIZED(widget))
		gdk_window_clear_area (widget->window,
				       widget->allocation.x, widget->allocation.y,
				       widget->allocation.width, widget->allocation.height);

	widget->allocation = *allocation;

	if ( GTK_WIDGET_REALIZED(widget))
		gdk_window_clear_area (widget->window,
				       widget->allocation.x, widget->allocation.y,
				       widget->allocation.width, widget->allocation.height);

	im->area.x = (widget->allocation.x + (widget->allocation.x - widget->requisition.width ) / 2);
	im->area.y = (widget->allocation.y + (widget->allocation.height - widget->requisition.height) / 2);
	im->area.width = widget->requisition.width;
	im->area.height = widget->requisition.height;

	if ( GTK_WIDGET_REALIZED( widget ) )
		draw_background_pixmap(im);
	
	update(im);
}

static void load_image(StatusDockletImage *im, StatusDockletState state)
{
	GtkWidget *widget;
	GdkPixmap *p;
	GdkPixmap *b;
	gint width, height;
	gint frames;

	g_return_if_fail ( GTK_WIDGET_REALIZED(im) );
	g_return_if_fail (state < STATUS_DOCKLET_STATE_LAST);

	widget = GTK_WIDGET(im);

	p = gdk_pixmap_create_from_xpm(widget->window, &b, 
				       &widget->style->bg[GTK_STATE_NORMAL], 
				       im->images[state]->path);

	if ( p == NULL )
		return;

	gdk_window_get_size (p, &width, &height);
	
	g_assert( height == 22);
	g_assert( (width%22) == 0);

	frames = width / 22;

	im->images[state]->frame_count = frames;
	im->images[state]->pixmap = p;
	im->images[state]->mask = b;
	im->images[state]->current_frame = 0;
}

void status_docklet_image_load(StatusDockletImage *im, StatusDockletState state, const gchar *path, gint frame_delay)
{
	StatusDockletImageInfo *info;

	g_return_if_fail (state < STATUS_DOCKLET_STATE_LAST);

	free_image(im, state);

	if ( ( path == NULL ) || ( *path == '\0' ) )
		return;

	info = g_new0(StatusDockletImageInfo, 1);
	info->path = g_strdup(path);
	info->frame_delay = frame_delay;

	im->images[state] = info;

	if ( GTK_WIDGET_REALIZED(im) )
		load_image (im, state);
}

void status_docklet_image_state_set(StatusDockletImage *im, StatusDockletState state)
{

	if ( state == im->current_state )
		return;
	
	if (im->timeout_tag) gtk_timeout_remove(im->timeout_tag);

	im->timeout_tag = 0;
	im->current_state = state;

	if ( state < STATUS_DOCKLET_STATE_LAST && im->images[state] ) {
		im->images[state]->current_frame = 0;
		if ( im->images[state]->frame_count > 1 ) 
			im->timeout_tag = gtk_timeout_add(im->images[state]->frame_delay, (GtkFunction)timeout_callback, (gpointer)im);
	}

	update(im);
}

static gboolean timeout_callback(StatusDockletImage *im)
{
	StatusDockletImageInfo *image_info;

	image_info = im->images[im->current_state];
	image_info->current_frame++;
	if ( image_info->current_frame == image_info->frame_count )
		image_info->current_frame = 0;
	
	update(im);
	return TRUE;
}

static void free_image(StatusDockletImage *im, StatusDockletState state)
{
	StatusDockletImageInfo *info;

	g_return_if_fail (state < STATUS_DOCKLET_STATE_LAST);

	if ( im->current_state == state ) {
		status_docklet_image_state_set (im, STATUS_DOCKLET_STATE_LAST);
	}


	info = im->images[state];

	if ( info == NULL )
		return;

	g_free(info->path);

	gdk_pixmap_unref(info->pixmap);
	gdk_bitmap_unref(info->mask);

	g_free(im->images[state]);
	im->images[state] = NULL;

}

