/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <ctype.h>
#include <sys/stat.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit_song.h"
#include "singit_song_private.h"
#include "singit_macros.h"
#include "singit_macros_private.h"
#include "singit_config.h"
#include "singit_config_private.h"

#include "dlg_messagebox.h"
#include "dlg_input_number.h"
#include "dlg_singit_config.h"
#include "editor_singit_main.h"
#include "dlg_singit_about.h"
#include "dlg_singit_id3.h"
#include "singit_tools.h"
#include "singit_main.h"
#include "pixmaps.xpm.h"

#include "editor_singit_private.h"

#include "editor_clist_timestamps.h"
#include "editor_plain_text.h"
#include "editor_synced_lyrics.h"

//	DEBUG(("%s\n", gtk_type_name(GTK_OBJECT_TYPE(widget)));

static GtkWidget *singit_editor_win = NULL;
static GtkWidget
		*editor_vbox,
			*editor_menubar_handle_box, *editor_menubar,
			*editor_toolbar_handle_box, *editor_toolbar,
				*new_file_button,
				*load_file_button,
				*get_current_song_button,
				*reload_lyrics_button,

				*save_file_button,
				*save_file_as_button,
				*export_as_html_button,

//				*sync_mode_toggle_button,

				*strip_html_button,
				*strip_timetags_button,
				*nice_text_button,
				*change_time_button,
				*check_sync_consistency_button,

				#ifdef HAVE_ID3
				*id3_button,
				#endif
				*config_button,
				*help_button,
				*close_button,

			*editor_type_notebook,
				*editor_clist_timestamps, *editor_clist_timestamps_label,
				*editor_plain_text, *editor_plain_text_label,
				*editor_synced_lyrics, *editor_synced_lyrics_label,

			*editor_status_hbox,
				*editor_status_time_bar,
				*editor_status_mode_bar,
				*editor_status_consistency_bar,
				*editor_status_modified_bar,
			*editor_song_gtk_event_box, *editor_status_filename_bar;

static GtkTooltips *editor_tooltips = NULL;
static GtkWidget *save_question_win = NULL;
static GtkWidget *enter_number_win = NULL;
static GtkWidget *ok_cancel_win = NULL;

static GtkAccelGroup *editor_menubar_accel_group = NULL;
static GtkItemFactory *editor_menubar_item_factory = NULL;

EditorStatus editor_status;

extern VisPlugin singit_vp;
extern SingitStatus singit_status;

static void singit_editor_cleanup(void);
static void new_lyrics_event(GtkWidget *widget, gpointer data);
static void save_lyric_file_event(GtkWidget *widget, gpointer data);
static void show_save_error_message_box(gchar *filename);
static gint check_sync_consistency(gboolean invalidate);

gboolean singit_editor_is_realized()
{
	return ((singit_editor_win) && (singit_editor_win->window) && GTK_WIDGET_REALIZED(singit_editor_win));
}

static gchar* get_base_path()
{
        gchar *buffer, *fileDir;
	gchar **directories;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [get_base_path]\n"), 9);
#	endif

	directories = g_strsplit(getSCD(singit_config)->basePath, ",", 0);
	if (directories[0][0] == '~') {
		buffer = &directories[0][1];
		fileDir = g_strconcat(g_get_home_dir(), buffer, NULL);
	}
	else { fileDir = g_strdup(g_strconcat(directories[0], NULL)); }
	g_strfreev(directories);
	return fileDir;
}

static gchar* get_first_path(const gchar *filename)
{
        gchar *base_path, *fileDir;
	gchar **extensions;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [get_first_path]\n"), 9);
#	endif

	extensions = g_strsplit(getSCD(singit_config)->lyricExtension, ",", 0);
	base_path = get_base_path();
	fileDir = g_strconcat(base_path, (filename == NULL) ? "*" : filename, ".", extensions[0], NULL);
	g_free(base_path);
	g_strfreev(extensions);

	return fileDir;
}

static void set_filename(void)
{
	gchar *statusbarText, *file_name, *base_path;
	gint pos;

	if (!singit_editor_is_realized()) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [set_filename]\n"), 9);
#	endif

	if (editor_status.filename) { statusbarText = g_strdup(editor_status.filename); }
	else {
		pos = xmms_remote_get_playlist_pos(singit_vp.xmms_session);
		file_name = xmms_remote_get_playlist_file(singit_vp.xmms_session, pos);
		base_path = get_base_path();
		statusbarText = g_strconcat(base_path, g_basename(file_name), NULL);
		g_free(file_name);
	}

	if (editor_status.filename_context_id) {
		gtk_statusbar_pop
			(GTK_STATUSBAR(editor_status_filename_bar),
			editor_status.filename_context_id);
	}
	else {
		editor_status.filename_context_id =
			gtk_statusbar_get_context_id
				(GTK_STATUSBAR(editor_status_filename_bar), "filename");
	}
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_filename_bar),
		editor_status.filename_context_id, statusbarText);
	gtk_tooltips_set_tip (editor_tooltips, editor_song_gtk_event_box, statusbarText, NULL);
	g_free(statusbarText);
}

void singit_editor_set_time(guint time)
{
	gchar timeStamp[8];
	static guint last_time = 1;

	if (!(singit_editor_is_realized() && editor_status_time_bar)) { return; }
	if (time > 5999000) { return; }

	if (last_time != time / 1000) {
		last_time = time / 1000;
		timeStamp[7] = '\0';
		sprintf(timeStamp, " %.2i:%.2i ", last_time / 60, last_time % 60);

		GDK_THREADS_ENTER();
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_time_bar), editor_status.time_context_id);
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_time_bar), editor_status.time_context_id, timeStamp);
		GDK_THREADS_LEAVE();
	}
}

static void message_box_event(GtkWidget *widget, gint buttonNr, gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [message_box_event] : %i\n", buttonNr), 9);
#	endif

	switch (buttonNr) {
	case 1:
		editor_status.continue_func = FALSE;
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(save_question_win));
		break;
	case 2:
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(save_question_win));
	case 3:
		editor_status.continue_func = TRUE;
		break;
	}
}

static void ok_cancel_win_button_pressed_event
	(GtkWidget *widget, gint buttonNr, gpointer user_data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [inconsistence_question_button_pressed_event] : %i\n", buttonNr), 9);
#	endif

	switch (buttonNr) {
	case 1:
		editor_status.continue_func = FALSE;
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(ok_cancel_win));
		break;
	case 2:
		editor_status.continue_func = TRUE;
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(ok_cancel_win));
		break;
	}
}
/*
static gboolean check_user_save_sync(void)
{
	#ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [check_user_save_sync]\n"));
	#endif

	if (getSCD(singit_config)->checkSyncBeforeSave && (check_sync_consistency(0) != -1)) {
		editor_status.continue_func = FALSE;
		message_box_dialog_show(MESSAGE_BOX_DIALOG(ok_cancel_win),
			GTK_WINDOW (ok_cancel_win) );
		if (!editor_status.continue_func) { return FALSE; }
	}

	return TRUE;
}
*/
void toggle_modified_state (gboolean state)
{
	if (state && (check_sync_consistency(2) != -2)) {
		check_sync_consistency(1);
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_consistency_bar),
			editor_status.valid_consistency_id);
	}
	if (state == editor_status.text_changed) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [toggle_modified_state]\n"), 9);
#	endif

	if (state) {
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_modified_bar),
			editor_status.modified_context_id);
	}
	else {
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar),
			editor_status.modified_context_id, editor_status.changed_spaces);
	}
	editor_status.text_changed = state;
}

static gchar* get_text_from_editor (gint *carret, gint page)
{
	gchar *text = NULL;
	GtkWidget *editor;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [get_text_from_editor]\n"), 9);
#	endif

	if (page < 0) {
		editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook),
			gtk_notebook_get_current_page(GTK_NOTEBOOK(editor_type_notebook)));
	}
	else {
		editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook), page);
	}

	if (editor == editor_clist_timestamps) {
		text = editor_clist_timestamps_get_text
			(EDITOR_CLIST_TIMESTAMPS(editor_clist_timestamps), carret);
	}
	else if (editor == editor_plain_text) {
		text = editor_plain_text_get_text
			(EDITOR_PLAIN_TEXT(editor_plain_text), carret, 0, -1);
	}
	else if (editor == editor_synced_lyrics) {
		text = editor_synced_lyrics_get_text
			(EDITOR_SYNCED_LYRICS(editor_synced_lyrics), carret);
	}
	return text;
}

static void singit_editor_set_mode(gchar *mode)
{
	if (!singit_editor_is_realized()) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_set_mode]\n"), 9);
#	endif

	gtk_statusbar_pop(GTK_STATUSBAR(editor_status_mode_bar), editor_status.mode_context_id);
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_mode_bar), editor_status.mode_context_id, mode);
}

static void set_editor_from_text (gchar *text, gint carret, gint page)
{
	GtkWidget *editor;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [set_editor_from_text]\n"), 9);
#	endif

	if (page < 0) {
		editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook),
			gtk_notebook_get_current_page(GTK_NOTEBOOK(editor_type_notebook)));
	}
	else {
		editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook), page);
	}

	if (editor == editor_clist_timestamps) {
		editor_clist_timestamps_set_text
			(EDITOR_CLIST_TIMESTAMPS(editor_clist_timestamps), text, carret);
	}
	else if (editor == editor_plain_text) {
		editor_plain_text_set_text
			(EDITOR_PLAIN_TEXT(editor_plain_text), text, carret);
	}
	else if (editor == editor_synced_lyrics) {
		editor_synced_lyrics_set_text
			(EDITOR_SYNCED_LYRICS(editor_synced_lyrics), text, carret);
	}
}

// 0 - calc / 1 - invalidate / 2 - get last result
static gint check_sync_consistency(gint mode)
{
	static gint valid_consistency = -2;
	gchar *text;
	LSong *song;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [check_sync_consistency]\n"), 9);
#	endif

	switch (mode) {
	case 1:
		valid_consistency = -2;
	case 2:
		break;
	default:
		if (valid_consistency > -2) { return valid_consistency; }
		text = get_text_from_editor(NULL, -1);
		if (!text) { return -1; }

		song = l_song_new(NULL);
		l_song_read_text_stream(song, text);
		valid_consistency = l_song_check_sync_lyric_consistency(song);
		l_song_free(song);
		break;
	}
	return valid_consistency;
}

static void check_sync_consistency_event (GtkWidget *widget, gpointer data)
{
	gchar *output, *integer;
	gint result;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [check_sync_consistency_event]\n"), 9);
#	endif

	result = check_sync_consistency(FALSE);

	if (result == -1) {
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
			editor_status.valid_consistency_id, _(" Ok "));
	}
	else {
		integer = g_strdup_printf("%i", result);
		output = g_strconcat(_(" Error"), " : ", integer, " ", NULL);
		g_free(integer);
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
			editor_status.valid_consistency_id, output);
		g_free(output);
	}
}

void render_simple_html_tags(gchar* source_pos, gint length, gchar** pos_copy)
{
	/* Keep it sorted by length !!! */
	const gchar* translate_tag_table[9][2] =
		{ { "<P>", "\n\n" },	// 3
		  { "<BR>", "\n" },	// 4
		  { "<P/>", "\n\n" },
		  { "</P>", "\n\n" },
		  { "<BR/>", "\n" },	// 5
		  { "<P />", "\n\n" },
		  { "</BR>", "\n" },
		  { "<BR />", "\n" },	// 6
		  { 0, 0 } };

	gint i = 0;
	gint tag_length = 0;
/*	gchar *mystr = g_strndup(source_pos, length + 1);
	mystr[length] = '\0';
	g_print("%s / %i\n", mystr, length);*/

	while (!translate_tag_table[i][0] == 0) {
		tag_length = strlen(translate_tag_table[i][0]);
		if (tag_length > length) { break; }
		if ((length == tag_length) &&
			(g_strncasecmp(source_pos, translate_tag_table[i][0], length) == 0))
		{
			strncpy(*pos_copy, &translate_tag_table[i][1][0],
				strlen(&translate_tag_table[i][1][0]));
			(*pos_copy) += strlen(&translate_tag_table[i][1][0]);
			break;
		}
		i++;
	}
}

static void strip_html_tags_event (GtkWidget *widget, gpointer data)
{
	gchar *new_text = NULL, *pos_begin, *pos_end, *pos_copy, *text;
	gint len;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }
	if (!check_user_continue()) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [strip_html_tags_event]\n"), 9);
#	endif

	new_text = g_new(gchar, strlen(text) + 1);
	pos_copy = new_text;
	pos_begin = text;
	while (pos_begin) {
		pos_end = strstr(pos_begin, "<");
		if (!pos_end) {
			strcpy(pos_copy, pos_begin);
			pos_begin = NULL;
		}
		else {
			len = pos_end - pos_begin;
			if (len > 0) {
				strncpy(pos_copy, pos_begin, len);
				pos_copy += len;
			}
			pos_begin = strstr(pos_end, ">");
			if (pos_begin) {
				pos_begin++;
				render_simple_html_tags(pos_end, pos_begin - pos_end, &pos_copy);
			}
		}
	}
	g_free(text);

	text = new_text;
	new_text = g_new(gchar, strlen(text) + 1);
	pos_copy = new_text;
	pos_begin = text;
	while (pos_begin) {
		pos_end = strstr(pos_begin, "&nbsp;");
		if (!pos_end) {
			strcpy(pos_copy, pos_begin);
			pos_begin = NULL;
		}
		else {
			len = pos_end - pos_begin;
			if (len > 0) {
				strncpy(pos_copy, pos_begin, len);
				pos_copy += len;
			}
			pos_copy[0] = ' ';
			pos_copy++;
			pos_begin = pos_end + strlen("&nbsp;");
		}
	}
	g_free(text);

	set_editor_from_text(new_text, 0, -1);
	g_free(new_text);

	toggle_modified_state(TRUE);
}

static void nice_text_event (GtkWidget *widget, gpointer data)
{
	gchar *text;
	gint i = 0;
	gchar up_str[2] = " \0";
	gchar **lines;
	gchar *temp;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [nice_text_event]\n"), 9);
#	endif

	lines = g_strsplit(text, "\n", 0);
	g_free(text);

	while (lines[i]) {
		temp = g_strdup(g_strstrip(lines[i]));
		up_str[0] = temp[0];
		g_strup(&up_str[0]);
		temp[0] = up_str[0];
		g_free(lines[i]);
		lines[i] = temp;
		i++;
	}

	text = g_strjoinv("\n", lines);
	g_strfreev(lines);

	set_editor_from_text(text, 0, -1);
	g_free(text);

	toggle_modified_state(TRUE);
}

static void strip_time_tags_event (GtkWidget *widget, gpointer data)
{
	gchar *new_text = NULL, *pos_begin, *pos_end, *pos_copy, *text;
	gint len, tag_type;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }
	if (!check_user_continue()) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [strip_time_tags_event]\n"), 9);
#	endif

	new_text = g_new(gchar, strlen(text) + 1);
	pos_copy = new_text;
	pos_end = pos_begin = text;
	while (pos_begin) {
		pos_end = strstr(pos_end, "[");
		if (!pos_end) {
			strcpy(pos_copy, pos_begin);
			pos_begin = NULL;
		}
		else {
			tag_type = l_song_get_timetag_type(pos_end);
			switch (tag_type) {
			case 0:
				pos_end++;
				break;
			case 1:
				len = pos_end - pos_begin;
				strncpy (pos_copy, pos_begin, len);
				pos_copy += len;
				pos_end += 7;
				pos_begin = pos_end;
				break;
			case 2:
				len = pos_end - pos_begin;
				strncpy (pos_copy, pos_begin, len);
				pos_copy += len;
				pos_end += 11;
				pos_begin = pos_end;
				break;
			}
		}
	}
	g_free(text);

	set_editor_from_text(new_text, 0, -1);
	g_free(new_text);

	toggle_modified_state(TRUE);
}

static void ok_button_export_as_html_event (GtkWidget *widget, GtkWidget *fileSelector)
{
	FILE *file;
	gchar *new_text = NULL, *tmp = NULL, *text;
	gchar **stream, *connected = NULL;
	gchar timeToken[42];
	gint offset = 0, last_line = -1;
	gint i = 0;
	LSong *song = NULL;
	GList *item = NULL;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [ok_button_export_as_html_event] : %s\n", filename), 9);
#	endif

	if (!(file = fopen(filename, "w"))) {
		show_save_error_message_box(filename);
		gtk_widget_destroy(fileSelector);
//		g_free(filename);
		return;
	}

	song = l_song_new(NULL);
	text = get_text_from_editor(NULL, -1);
	l_song_read_text_stream(song, text);
	timeToken[37] = '\0';
	timeToken[41] = '\0';

	stream = g_new(gchar*, song->lyric_lines+1);
	stream[song->lyric_lines] = NULL;
	for (i = 0; i < song->lyric_lines; i++)
		{ stream[i] = g_strdup(song->lyrics[i]); }
	item = song->first_token;
	while (item) {
		if (GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active) {
			sprintf(timeToken, "<FONT CLASS=\"lyrictag\">[%.2i:%.2i:%.3i]</FONT>", tTime(item) / 60000, (tTime(item) / 1000) % 60, tTime(item) % 1000);
			if (tLine(item) != last_line) { offset = 0; }
			else { offset += 41; }
		}
		else {
			sprintf(timeToken, "<FONT CLASS=\"lyrictag\">[%.2i:%.2i]</FONT>", tTime(item) / 60000, (tTime(item) / 1000) % 60);
			if (tLine(item) != last_line) { offset = 0; }
			else { offset += 37; }
		}
		last_line = tLine(item);
		tmp = stream[tLine(item)];
		connected = tools_insert_string(stream[tLine(item)], &timeToken[0], tPos(item) + offset);
		if (connected) {
			stream[tLine(item)] = connected;
			g_free(tmp);
		}
		item = g_list_next(item);
	}
	for (i = 0; i < song->lyric_lines; i++) {
		tmp = stream[i];
		stream[i] = g_strconcat(stream[i], "<BR />", NULL);
		g_free(tmp);
	}
	tmp = new_text = g_strjoinv("\n", stream);
	new_text = g_strconcat(getSCD(singit_config)->htmlHeader, new_text, getSCD(singit_config)->htmlTail, NULL);
	g_free(tmp);

	if (fwrite(new_text, 1, strlen(new_text), file) != strlen(new_text))
       	{
		show_save_error_message_box(filename);
	}

	if (new_text) { g_free(new_text); }
	fclose(file);

	l_song_free(song);
	g_free(text);

//	if (filename) { g_free(filename); }

	gtk_widget_destroy(fileSelector);
}

static void export_as_html_event (GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [export_as_html_event]\n"), 9);
#	endif

	fileSelector = gtk_file_selection_new(_("Export lyric file as html"));

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(ok_button_export_as_html_event), fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), "");

	gtk_widget_show(fileSelector);
}

gboolean check_user_continue(void)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [check_user_continue]\n"), 9);
#	endif

	if (editor_status.text_changed) {
		message_box_dialog_show(MESSAGE_BOX_DIALOG(save_question_win),
			GTK_WINDOW (save_question_win) );
		if (!editor_status.continue_func) { return FALSE; }
	}

	return TRUE;
}

static void new_lyrics_event(GtkWidget *widget, gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [new_lyrics_event]\n"), 9);
#	endif

	if (check_user_continue())
		{ set_editor_from_text(NULL, 0, -1); }
}

static void change_time_dlg_button_event(EnterNumberDialog *mldlg, gint number, gpointer data)
{
	LSong *song;
	gchar *text;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [change_time_dlg_button_event]\n"), 9);
#	endif

	if (number != 0) {
		text = editor_plain_text_get_text
			(EDITOR_PLAIN_TEXT(editor_plain_text), NULL, -1, -1);
		if (text) {
			song = l_song_new("");
			l_song_read_text_stream(song, text);
			g_free(text);
			l_song_modify_overall_time(song, number);
			l_song_write_text_stream(song, &text,
				GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active);
			l_song_free(song);

			editor_plain_text_xchg_selection
				(EDITOR_PLAIN_TEXT(editor_plain_text), text);
			g_free(text);
		}
	}
	enter_number_dialog_hide(mldlg);
}

static void change_time_event(GtkWidget *widget, gpointer data)
{
	if (editor_plain_text_is_selected(EDITOR_PLAIN_TEXT(editor_plain_text))) {
		enter_number_dialog_show(ENTER_NUMBER_DIALOG(enter_number_win),
				GTK_WINDOW(singit_editor_win), TRUE);
	}
}

void lyrics_from_song_event(GtkWidget *widget, gpointer data)
{
	LSong *cur_song = NULL;
	gchar *text;

	set_filename();
	cur_song = l_song_attach(getSCD(singit_config)->song);

	if (!l_song_text_found(cur_song)) {
		editor_status.filename = g_strdup(cur_song->lyric_filename);
		set_filename();
		l_song_detach(cur_song, TRUE);
		return;
	}

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [lyrics_from_song_event]\n"), 9);
#	endif

	if (!check_user_continue()) {
		l_song_detach(cur_song, TRUE);
		return;
	}

	l_song_write_text_stream(cur_song, &text,
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active);
	editor_status.filename = g_strdup(cur_song->lyric_filename);
	l_song_detach(cur_song, TRUE);
	set_editor_from_text(text, 0, -1);
	g_free(text);

	set_filename();
	toggle_modified_state(FALSE);
}

void reload_lyrics_event(GtkWidget *widget, gpointer data)
{
	LSong *song;
	gchar *text = NULL;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [reload_lyrics_event]\n"), 9);
#	endif

	if (editor_status.filename != NULL)
	{
		if (singit_config) {
			song = l_song_attach(getSCD(singit_config)->song);
			if ((song != NULL) && (song->lyric_filename != NULL)) {
				if (strcmp(song->lyric_filename, editor_status.filename) == 0) {
					l_song_write_text_stream(song, &text,
						GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active);
				}
				l_song_detach(song, TRUE);
			}
		}
		if (text == NULL) {
			song = l_song_new(NULL);
			if (l_song_load_lyrics(song, editor_status.filename)) {
				l_song_write_text_stream(song, &text,
					GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active);
			}
			l_song_detach(song, TRUE);
		}
		if (text != NULL) {
			if (check_user_continue())
				{ set_editor_from_text(text, 0, -1); }
			g_free(text);
		}
	}
}

static void ok_button_open_file_selector_event(GtkWidget *widget, GtkWidget *fileSelector)
{
	LSong *song = NULL;
	gchar *text;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [ok_button_open_file_selector_event] : %s\n", filename), 9);
#	endif

	song = l_song_new(NULL);
	if (l_song_load_lyrics(song, filename)) {
		l_song_write_text_stream(song, &text,
			GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active);
		g_free(editor_status.filename);
		editor_status.filename = g_strdup(song->lyric_filename);
		set_editor_from_text(text, 0, -1);
		g_free(text);
		set_filename();
		toggle_modified_state(FALSE);
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(save_question_win));
	}
//	if (filename) { g_free(filename); }

	l_song_free(song);
	if (!getSCD(singit_config)->multiSelect)
		{ gtk_widget_destroy(fileSelector); }
}

static void open_lyric_file_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;
	gchar *file = NULL;
	FILE *r_file;
        struct stat stats;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [open_lyric_file_event]\n"), 9);
#	endif

	if (!check_user_continue()) { return; }

	fileSelector = gtk_file_selection_new(_("Lyrics selection"));

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(fileSelector));

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", (GtkSignalFunc) ok_button_open_file_selector_event, fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(fileSelector));

	file = get_first_path(NULL);
	if (lstat(file, &stats) == -1) {
		g_free(file);
		file = g_strdup("*");
	}
	else {
		if (!(r_file = fopen(file, "r"))) {
			g_free(file);
			file = g_strdup("*");
			fclose(r_file);
		}
	}
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), file);
	g_free(file);

	gtk_widget_show(fileSelector);
}

static void show_save_error_message_box(gchar* fileName)
{
	GtkWidget *message_box = NULL;
	GtkWidget *message_table, *message_label, *message_button_ok;
	gchar *text;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [show_save_error_message_box]\n"), 9);
#	endif

	message_box = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(message_box), _("Error / Failure"));
	gtk_window_set_modal(GTK_WINDOW(message_box), TRUE);
	gtk_signal_connect_object(GTK_OBJECT(message_box), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(message_box));

	message_table = gtk_table_new (1, 1, TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(message_box)->vbox), message_table);
	gtk_container_border_width(GTK_CONTAINER(message_table), 7);
	gtk_widget_show(message_table);

	if (fileName != NULL)
		{ text = g_strconcat(_("File save failed"), "\n\n", fileName, NULL); }
	else
		{ text = _("File save failed");	}
	message_label = gtk_label_new (text);
	g_free(text);
	gtk_table_attach_defaults (GTK_TABLE(message_table), message_label, 0, 1, 0, 1);
	gtk_widget_show(message_label);

	message_button_ok = gtk_button_new_with_label(_("Ok"));
	gtk_signal_connect_object(GTK_OBJECT(message_button_ok), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(message_box));
	GTK_WIDGET_SET_FLAGS(message_button_ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(message_box)->action_area), message_button_ok, TRUE, TRUE, 0);
	gtk_widget_show(message_button_ok);

	gtk_widget_show(message_box);
}

static void ok_button_save_as_file_selector_event(GtkWidget *widget, GtkWidget *fileSelector)
{
	LSong *song;
	gchar *text;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [ok_button_save_as_file_selector_event] : %s\n", filename), 9);
#	endif

	song = l_song_new(NULL);
	text = get_text_from_editor(NULL, -1);
	l_song_read_text_stream(song, text);

	pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
	if (!l_song_save_to_text_file(song, filename,
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active))
	{
		pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
		show_save_error_message_box(filename);
//		g_free(filename);
	}
	else {
		pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
		g_free(editor_status.filename);
		editor_status.filename = g_strdup(filename);
		set_filename();
		toggle_modified_state(FALSE);
	}
	l_song_free(song);
	g_free(text);

	gtk_widget_destroy(fileSelector);
}

static void save_as_lyric_file_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;
	gchar *file = NULL;
	gchar *selected_text = NULL;
	GtkWidget *editor;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [save_as_lyric_file_event]\n"), 9);
#	endif

	editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook),
		gtk_notebook_get_current_page(GTK_NOTEBOOK(editor_type_notebook)));

	if (editor == editor_plain_text) {
		selected_text = editor_plain_text_get_text
			(EDITOR_PLAIN_TEXT(editor), NULL,
			GTK_EDITABLE(EDITOR_PLAIN_TEXT(editor)->text)->selection_start_pos,
			GTK_EDITABLE(EDITOR_PLAIN_TEXT(editor)->text)->selection_end_pos);

		if (selected_text != NULL) {
			if ((strlen(selected_text) > 0) && (strstr(selected_text, "\n") == NULL)) {
				file = get_first_path(selected_text);
			}
			g_free(selected_text);
			selected_text = NULL;
		}
	}
	if (file == NULL)
		{ file = get_first_path(NULL); }

	fileSelector = gtk_file_selection_new(_("Lyric file save"));

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy), &fileSelector);

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(ok_button_save_as_file_selector_event), fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), file);
	g_free(file);

	gtk_widget_show(fileSelector);
}

static void save_lyric_file_event(GtkWidget *widget, gpointer data)
{
	LSong *song = NULL;
	gchar *text;

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [save_lyric_file_event]\n"), 9);
#	endif

//	if (!check_user_save_sync()) { return; }

	if (!editor_status.filename) { save_as_lyric_file_event(NULL, NULL); }
	else {
		text = get_text_from_editor(NULL, -1);
		song = l_song_new(NULL);
		l_song_read_text_stream(song, text);
		pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
		if (!l_song_save_to_text_file(song, editor_status.filename,
			GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active))
		{
			pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
			save_as_lyric_file_event(NULL, NULL);
		}
		else {
			pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
			toggle_modified_state(FALSE);
		}
		g_free(text);
		l_song_free(song);
	}
}

static gboolean singit_editor_main_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	gint page_nr;
	GtkWidget *editor;

	switch(event->keyval) {
		case GDK_Escape:
			if (check_user_continue())
				singit_editor_hide();
			break;
		case GDK_l:
			if (event->state & GDK_CONTROL_MASK) {
				lyrics_from_song_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_n:
			if (event->state & GDK_CONTROL_MASK) {
				new_lyrics_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_o:
			if (event->state & GDK_CONTROL_MASK) {
				open_lyric_file_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_s:
			if (event->state & GDK_CONTROL_MASK) {
				save_lyric_file_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_t:
			if (event->state & GDK_CONTROL_MASK) {
				page_nr = gtk_notebook_get_current_page(GTK_NOTEBOOK(editor_type_notebook));
				if (event->state & GDK_SHIFT_MASK) { page_nr--; }
				else { page_nr++; }
				editor = gtk_notebook_get_nth_page(GTK_NOTEBOOK(editor_type_notebook), page_nr);

				if (!editor) {
					if (page_nr > 0) { page_nr = 0; }
					else {
						page_nr = 0;
						while (gtk_notebook_get_nth_page
							(GTK_NOTEBOOK(editor_type_notebook), page_nr)) { page_nr++; }
					page_nr--;
					}
				}
				gtk_notebook_set_page(GTK_NOTEBOOK(editor_type_notebook), page_nr);
			}
			return TRUE;
		default:
			return FALSE;
	}
	return FALSE;
}

void singit_editor_switch_notebook_page (GtkNotebook *notebook,
		GtkNotebookPage *page, gint page_num, gpointer user_data)
{
	gchar *text;
	gint carret;

	if (gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data)) < 0) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_switch_notebook_page]\n"), 9);
	DEBUG(("   %i (new - page_num) / %i (old - gtk_notebook_get_current_page)\n", page_num,
		gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data))), 9);
#	endif

	text = get_text_from_editor
		(&carret, gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data)));
	set_editor_from_text(text, carret, page_num);

	gtk_widget_set_sensitive(change_time_button, (page_num == 1));
	switch (page_num) {
	case 0:
		singit_editor_set_mode(_(" Editor mode: timestamp"));
		break;
	case 1:
		singit_editor_set_mode(_(" Editor mode: text"));
		break;
	case 2:
		singit_editor_set_mode(_(" Editor mode: synced"));
		break;
	}

	g_free(text);
}

void singit_editor_show_config(GtkWidget *widget, gpointer user_data)
{
	singit_config_show();
}

static void singit_editor_init(void)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_init]\n"), 9);
#	endif

	editor_status.mode_context_id = 0;
	editor_status.time_context_id = 0;
	editor_status.filename_context_id = 0;
	editor_status.cleanedUp = FALSE;
	editor_status.filename = NULL;
	editor_status.text_changed = FALSE;
	editor_status.changed_spaces = NULL;
	editor_status.continue_func = TRUE;
	editor_status.valid_consistency = FALSE;
	editor_status.use_extended_timetags = getSCD(singit_config)->extendedLyrics;
}

static void singit_editor_cleanup(void)
{
	if (editor_status.cleanedUp) { return; }

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_cleanup]\n"), 9);
#	endif

	editor_status.cleanedUp = TRUE;

	if (editor_status.filename != NULL) {
		g_free(editor_status.filename);
		editor_status.filename = NULL;
	}

	if (editor_status.changed_spaces) {
		g_free(editor_status.changed_spaces);
		editor_status.changed_spaces = NULL;
	}

	editor_status.mode_context_id = 0;
	editor_status.time_context_id = 0;
	editor_status.filename_context_id = 0;

	if (save_question_win) { gtk_widget_destroy(save_question_win); }
	if (enter_number_win) { gtk_widget_destroy(enter_number_win); }
	if (ok_cancel_win) { gtk_widget_destroy(ok_cancel_win); }

	singit_config_save_editor();
	singit_config_detach(TRUE);
}

void singit_editor_hide(void)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_hide]\n"), 9);
#	endif

	if (singit_editor_win) {

		singit_editor_cleanup();
		gtk_widget_destroy(singit_editor_win);
		gtk_object_destroy(GTK_OBJECT(editor_menubar_item_factory));
		singit_main_finish(FALSE);
	}
}

static gint singit_editor_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	if (check_user_continue()) {

		singit_editor_cleanup();
		gtk_widget_destroy(singit_editor_win);
		gtk_object_destroy(GTK_OBJECT(editor_menubar_item_factory));
		singit_main_finish(FALSE);

		return (FALSE);
	}
	return (TRUE);
}

static void close_editor_event(GtkWidget *widget, gpointer user_data)
{
	if (check_user_continue()) {

		singit_editor_cleanup();
		gtk_widget_destroy(singit_editor_win);
		gtk_object_destroy(GTK_OBJECT(editor_menubar_item_factory));

		singit_main_finish(FALSE);
	}
}

void build_check_dialogs()
{
	save_question_win = message_box_dialog_new (_("Question"), _("Do you want to save your changes ?"),
		_("  Cancel  "), 100, 50);
	message_box_dialog_prepend_button(MESSAGE_BOX_DIALOG(save_question_win), _(" Continue "), NULL);
	message_box_dialog_prepend_button(MESSAGE_BOX_DIALOG(save_question_win), _("   Save   "),
		GTK_SIGNAL_FUNC(save_lyric_file_event));
	gtk_signal_connect(GTK_OBJECT(save_question_win), "button_pressed",
		GTK_SIGNAL_FUNC(message_box_event), NULL);
	gtk_signal_connect(GTK_OBJECT(save_question_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &save_question_win);

	ok_cancel_win = message_box_dialog_new (_("Question"),
		_("Your lyrics are inconsistent!\nThis may cause errors in the final file\nDo you want to proceed ?"),
		_("  Cancel  "), 100, 50);
	message_box_dialog_prepend_button(MESSAGE_BOX_DIALOG(ok_cancel_win),
		_(" Proceed "), NULL);
	gtk_signal_connect(GTK_OBJECT(ok_cancel_win), "button_pressed",
		GTK_SIGNAL_FUNC(ok_cancel_win_button_pressed_event), NULL);
	gtk_signal_connect(GTK_OBJECT(ok_cancel_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &ok_cancel_win);

	enter_number_win = enter_number_dialog_new(_("Select number"), 200, 100);
	gtk_signal_connect(GTK_OBJECT(enter_number_win), "new_number",
		GTK_SIGNAL_FUNC(change_time_dlg_button_event), NULL);
	gtk_signal_connect(GTK_OBJECT(enter_number_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &enter_number_win);
}

void tooltips_check_event (GtkWidget *widget, gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [tooltips_check_event]\n"), 9);
#	endif

	gtk_toolbar_set_tooltips(GTK_TOOLBAR(editor_toolbar),
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Show tooltips"))->active);
}

void extended_timetags_check_event (GtkWidget *widget, gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [extended_timetags_check_event]\n"), 9);
#	endif

	editor_status.use_extended_timetags =
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active;
}

void editor_main_show_about_event (GtkWidget *widget, gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [editor_main_show_about_event]\n"), 9);
#	endif

	singit_about_show_page((guint) data);
}

gchar* editor_main_menu_translate_func(const gchar *path, gpointer func_data)
{
	gchar **items = g_strsplit(path, "/", 0);
	gchar *tmp = NULL;
	gint i = 0;
	g_print("%s\n", path);

	i = 0;
	while (items[i]) {
		tmp = items[i];
		items[i] = g_strdup(_(items[i]));
		g_free(tmp);
		i++;
	}

	tmp = g_strjoinv("/", items);
	g_strfreev(items);
	return tmp;
}

static GtkItemFactoryEntry editor_menubar_items[] = {
	{ "/_File",                          NULL,                           NULL, 0,     "<Branch>" },
	{ "/File/_New",              "<control>N",               new_lyrics_event, 0,           NULL },
	{ "/File/_Open ...",         "<control>O",          open_lyric_file_event, 0,           NULL },
	{ "/File/_Save",             "<control>S",          save_lyric_file_event, 0,           NULL },
	{ "/File/Save _As ...",              NULL,       save_as_lyric_file_event, 0,           NULL },
	{ "/File/sep1",                      NULL,                           NULL, 0,  "<Separator>" },
	{ "/File/_Export HTML ...",          NULL,           export_as_html_event, 0,           NULL },
	{ "/File/sep2",                      NULL,                           NULL, 0,  "<Separator>" },
	{ "/File/Quit",              "<control>Q",             close_editor_event, 0,           NULL },
	{ "/_Edit",                          NULL,                           NULL, 0,     "<Branch>" },
	{ "/Edit/Strip _HTML",               NULL,          strip_html_tags_event, 0,           NULL },
	{ "/Edit/Strip _timtags",            NULL,          strip_time_tags_event, 0,           NULL },
	{ "/Edit/_Nice text",                NULL,                nice_text_event, 0,           NULL },
	{ "/Edit/sep1",                      NULL,                           NULL, 0,  "<Separator>" },
	{ "/Edit/_Change time ...",          NULL,              change_time_event, 0,       "<Item>" },
	{ "/_Options",                       NULL,                           NULL, 0,     "<Branch>" },
	{ "/Options/Ext. timetags",          NULL,  extended_timetags_check_event, 0, "<ToggleItem>" },
	{ "/Options/Show tooltips",          NULL,           tooltips_check_event, 0, "<ToggleItem>" },
	{ "/Options/sep1",                   NULL,                           NULL, 0,  "<Separator>" },
	{ "/Options/Strip HTML",             NULL,                           NULL, 0,     "<Branch>" },
	{ "/Options/Strip HTML/<BR> + \\n",  NULL,                           NULL, 1,  "<RadioItem>" },
	{ "/Options/Strip HTML/\\n",         NULL,                           NULL, 2,  "/Options/Strip HTML/<BR> + \\n" },
	{ "/Options/Strip HTML/<BR>",        NULL,                           NULL, 3,  "/Options/Strip HTML/<BR> + \\n" },
	{ "/Options/w <--> u",               NULL,                           NULL, 0,     "<Branch>" },
	{ "/Options/w <--> u/disabled",      NULL,                           NULL, 1,  "<RadioItem>" },
	{ "/Options/w <--> u/u -> w",        NULL,                           NULL, 2,  "/Options/w <--> u/disabled" },
	{ "/Options/w <--> u/w -> u",        NULL,                           NULL, 3,  "/Options/w <--> u/disabled" },
	{ "/Options/sep2",                   NULL,                           NULL, 0,  "<Separator>" },
	{ "/Options/_Save options",          NULL,                           NULL, 0,           NULL },
	{ "/_Help",                          NULL,                           NULL, 0, "<LastBranch>" },
	{ "/Help/_About ...",                NULL,   editor_main_show_about_event, 0,           NULL },
	{ "/Help/sep1",                      NULL,                           NULL, 0,  "<Separator>" },
	{ "/Help/_Keybindings ...",          NULL,   editor_main_show_about_event, 1,           NULL },
};

void get_main_menu(GtkWidget *window, GtkWidget **menubar)
{
	gint nmenu_items = sizeof (editor_menubar_items) / sizeof (editor_menubar_items[0]);

	editor_menubar_accel_group = gtk_accel_group_new ();

	editor_menubar_item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", editor_menubar_accel_group);

	gtk_item_factory_create_items (editor_menubar_item_factory, nmenu_items, editor_menubar_items, NULL);

	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags")),
		editor_status.use_extended_timetags);
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Show tooltips")),
		TRUE);
/*	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Strip HTML/\\n")),
		TRUE);
*/
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Edit/Change time ..."), FALSE);
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Strip HTML/\\n"), FALSE);
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Strip HTML/<BR>"), FALSE);
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Save options"), FALSE);

	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/w <--> u/u -> w"), FALSE);
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/w <--> u/w -> u"), FALSE);

	gtk_window_add_accel_group (GTK_WINDOW (window), editor_menubar_accel_group);

	if (menubar)
		*menubar = gtk_item_factory_get_widget (editor_menubar_item_factory, "<main>");
}

void singit_editor_show(void)
{
	GdkPixmap * icon = NULL;
	GdkBitmap * mask = NULL;
	GtkWidget * iconw = NULL;
	gchar *title;

	if (singit_editor_win) {
		if (singit_editor_is_realized())
			gdk_window_raise(singit_editor_win->window);
		return;
	}

	if (!singit_config_attach()) {
		singit_config_new();
		singit_config_load();
	}

#	ifdef CODEDEBUG
	DEBUG(("editor_singit_main.c [singit_editor_show]\n"), 9);
#	endif

	singit_editor_init();
	build_check_dialogs();

	editor_tooltips = gtk_tooltips_new ();

	singit_editor_win = gtk_window_new(GTK_WINDOW_DIALOG);
	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("Lyrics editor"), NULL);
	gtk_window_set_title(GTK_WINDOW(singit_editor_win), title);
	g_free(title);
	gtk_window_set_policy(GTK_WINDOW(singit_editor_win), TRUE, TRUE, TRUE);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &singit_editor_win);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "delete_event", GTK_SIGNAL_FUNC(singit_editor_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "key-press-event", GTK_SIGNAL_FUNC(singit_editor_main_keypress), NULL);

	gtk_container_border_width(GTK_CONTAINER(singit_editor_win), 2);
	gtk_widget_realize(singit_editor_win);

	editor_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(singit_editor_win), editor_vbox);
	gtk_widget_show(editor_vbox);

/*	Menu-Bar */
	editor_menubar_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_menubar_handle_box, FALSE, FALSE, 2 );
	gtk_widget_show(editor_menubar_handle_box);

/*	Tool-Bar */
	editor_toolbar_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_toolbar_handle_box, FALSE, FALSE, 2 );
	gtk_widget_show(editor_toolbar_handle_box);

	editor_toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_container_set_border_width(GTK_CONTAINER(editor_toolbar), 1);
	gtk_toolbar_set_space_size(GTK_TOOLBAR(editor_toolbar), 5);
	gtk_container_add(GTK_CONTAINER(editor_toolbar_handle_box), editor_toolbar);
	gtk_widget_show(editor_toolbar);

/*	Menu-Bar */
	get_main_menu(singit_editor_win, &editor_menubar);
	gtk_container_add(GTK_CONTAINER(editor_menubar_handle_box), editor_menubar);
	gtk_widget_show(editor_menubar);

/*	Tool-Bar */
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(editor_toolbar),
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Show tooltips"))->active);

	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, filenew_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	new_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("New lyrics"),
		_("Creates an empty lyrics file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(new_lyrics_event),
		NULL);

        gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, fileopen_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	load_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Load lyrics"),
		_("Loads a lyric file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(open_lyric_file_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, openlyrics_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	get_current_song_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Load song lyrics"),
		_("Loads the lyric file form current song"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(lyrics_from_song_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, reload_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	reload_lyrics_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Reload lyrics"),
		_("Reloads the lyrics from source"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(reload_lyrics_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, filesave_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	save_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Save lyrics"),
		_("Save the lyric file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(save_lyric_file_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, filesaveas_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	save_file_as_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Save lyrics as"),
		_("Save the lyric file with new filename"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(save_as_lyric_file_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, exporthtml_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	export_as_html_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Export html"),
		_("Export lyrics as html file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(export_as_html_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

/*	iconw = gtk_pixmap_new(icon, mask);
	sync_mode_toggle_button = gtk_toolbar_append_element(
		GTK_TOOLBAR(editor_toolbar),
		GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
		NULL,
		_("Sync lyric mode"),
		_("Toggle the editor mode to sync lyric"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC (toggle_sync_mode_event),
		editor_toolbar);
	gtk_widget_set_sensitive(sync_mode_toggle_button, FALSE);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar)); */

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, striphtml_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	strip_html_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Strip HTML"),
		_("Removes all HTML-tags from the text"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(strip_html_tags_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, striptags_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	strip_timetags_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Strip timetags"),
		_("Removes all timetags from the text"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(strip_time_tags_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, align_left_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	nice_text_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Nice text"),
		_("Strips whitespaces / Upper starting letter"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(nice_text_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, time_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	change_time_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Change Time"),
		_("Change Time of Selection"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(change_time_event),
		NULL);
	gtk_widget_set_sensitive(change_time_button, FALSE);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, search_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	check_sync_consistency_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Check consistency"),
		_("Check consistency of syncronized lyrics"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(check_sync_consistency_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

#	ifdef HAVE_ID3
	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, id3info_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	id3_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("ID3"),
		_("Show the id3 tag window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(singit_id3dlg_show),
		NULL);
#	endif

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, preferences_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	config_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Preferences"),
		_("Show the configuration window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(singit_config_show),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, help_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	help_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Help"),
		_("Show the keybindings window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(editor_main_show_about_event),
		(gpointer) 1);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, quit_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	close_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),   /* our toolbar */
		_("Close"),                    /* button label */
		_("Closes the editor window"), /* this button's tooltip */
		"Private",                     /* tooltip private info */
		iconw,                         /* icon widget */
		GTK_SIGNAL_FUNC(close_editor_event),  /* a signal */
		NULL);

	gdk_pixmap_unref(icon);

	/* editor type notebook */
	editor_type_notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(editor_type_notebook), GTK_POS_TOP);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(editor_type_notebook), FALSE);
	gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(editor_type_notebook), TRUE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(editor_type_notebook), TRUE);
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_type_notebook, TRUE, TRUE, 0);
	gtk_widget_show(editor_type_notebook);

	gtk_signal_connect_object(GTK_OBJECT(editor_type_notebook),
		"switch-page", GTK_SIGNAL_FUNC(singit_editor_switch_notebook_page),
		(gpointer) editor_type_notebook);

	editor_clist_timestamps = editor_clist_timestamps_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_clist_timestamps), 2);
	gtk_widget_show(editor_clist_timestamps);
	editor_clist_timestamps_label = gtk_label_new(_(" Timestamps "));
	gtk_widget_show(editor_clist_timestamps_label);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_clist_timestamps, editor_clist_timestamps_label);

	editor_plain_text = editor_plain_text_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_plain_text), 2);
	gtk_widget_show(editor_plain_text);
	editor_plain_text_label = gtk_label_new(_(" Plain text "));
	gtk_widget_show(editor_plain_text_label);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_plain_text, editor_plain_text_label);

	editor_synced_lyrics = editor_synced_lyrics_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_synced_lyrics), 2);
	gtk_widget_show(editor_synced_lyrics);
	editor_synced_lyrics_label = gtk_label_new(_(" Synced "));
	gtk_widget_show(editor_synced_lyrics_label);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_synced_lyrics, editor_synced_lyrics_label);

	/* status bars */
	editor_status_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_status_hbox, FALSE, FALSE, 0);
	gtk_widget_show(editor_status_hbox);

	editor_status_time_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_time_bar, FALSE, FALSE, 0);
	gtk_widget_show(editor_status_time_bar);

	editor_status_mode_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_mode_bar, TRUE, TRUE, 0);
	gtk_widget_show(editor_status_mode_bar);

	editor_status_modified_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_modified_bar, FALSE, FALSE, 0);
	editor_status.modified_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_modified_bar), "modified");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar), editor_status.modified_context_id,
		_(" modified "));
	editor_status.changed_spaces = g_strnfill(strlen(_(" modified ")), ' ');
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar), editor_status.modified_context_id,
		editor_status.changed_spaces);
	gtk_widget_show(editor_status_modified_bar);

	editor_status_consistency_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_consistency_bar, FALSE, FALSE, 0);
	editor_status.valid_consistency_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_consistency_bar), "validation");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
		editor_status.valid_consistency_id, _(" Not checked "));
	gtk_widget_show(editor_status_consistency_bar);

	editor_song_gtk_event_box = gtk_event_box_new ();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_song_gtk_event_box, FALSE, FALSE, 0);
	gtk_widget_show (editor_song_gtk_event_box);

	editor_status_filename_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (editor_song_gtk_event_box), editor_status_filename_bar);
	gtk_widget_show(editor_status_filename_bar);

	gtk_widget_show(singit_editor_win);

	editor_status.time_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_time_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_time_bar), editor_status.time_context_id, " 00:00 ");

	editor_status.mode_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_mode_bar), "modes");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_mode_bar), editor_status.mode_context_id,
		_(" Editor mode: timestamp"));

	gtk_object_set_data (GTK_OBJECT (singit_editor_win), "tooltips", editor_tooltips);

	if ((getSCD(singit_config)->editorWinPosX < 0) || (getSCD(singit_config)->editorWinPosY < 0)) {
		gtk_window_set_position(GTK_WINDOW(singit_editor_win), GTK_WIN_POS_MOUSE);
	}
	else {
		gtk_widget_set_uposition(singit_editor_win, getSCD(singit_config)->editorWinPosX, getSCD(singit_config)->editorWinPosY);
	}

	singit_main_init(FALSE);
}
