/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef __PLAYER_INTERFACE_H__
#define __PLAYER_INTERFACE_H__

#include <qstring.h>
#include <qstringlist.h>
#include <qobject.h>

//! PlayerInterface acts as an interface to music players.
/*!
  It contains virtual methods to control music players. The real functionality
  that differs from player to player has to be implemented in a class that
  inherits PlayerInterface (see XMMSPlayer, NoatunPlayer, SMPEGPlayer).
*/
class PlayerInterface:public QObject {
  
 public:
  //! Construct a new PlayerInterface.
  PlayerInterface();

  //! Destruct the PlayerInterface.
  virtual ~PlayerInterface();

  //! Type of the PlayerInterface.
  /*!
    There are currently 3 different implementations of the PlayerInterFace.
   */
  enum Player { XMMS, NOATUN, SMPEG, AMAROK, MPLAYER, NONE };
  
  //! Launch the player.
  virtual void startPlayer() {}

  //! Exit the player.
  virtual void exitPlayer() {}
  
  //! Jump to the previous song in the playlist.
  virtual void previous() {}

  //! Start playing.
  virtual void play() {}

  //! Pause playing.
  virtual void pause() {}

  //! Stop playing.
  virtual void stop() {}

  //! Jump to the next song in the playlist.
  virtual void next() {}

  //! Jump to the n'th song in the playlist.
  virtual void jumpto(int) {}

  //! Set the volume.
  /*!
    \param volume   The new volume.
  */
  virtual void setVolume(int volume) {}

  //! Seek to the given time.
  /*!
    \param time  The time to seek to.
   */
  virtual void setCurrentTime(int time) {}

  //! Toggle the repeat button.
  virtual void toggleRepeat() {}

  //! Toggle the shuffle button.
  virtual void toggleShuffle() {}
  
  //! Get the volume of the player.
  /*!
    \return  The volume
  */
  virtual int getVolume() { return 0; }

  //! Get the current elapsed time of the song.
  /*!
    \return  The elapsed time.
  */
  virtual int getCurrentTime() { return 0; }

  //! Get the total time of the song.
  /*!
    \return  The length of the song.
  */
  virtual int getTrackTime() { return 0; }

  //! Get the title of the current song.
  /*!
    \return  The title of the song.
  */
  virtual QString getTitle() { return QString("");}

  //! Get the state of the repeat button.
  virtual bool getRepeat() { return false; }

  //! Get the state of the shuffle button.
  virtual bool getShuffle() { return false; }

  //! Is the player playing at the moment?
  virtual bool isPlaying() { return false; }

  //! Is the player running at the moment?
  virtual bool isPlayerRunning() { return false; }

  //! Open the fileselector of the player.
  virtual void openFileSelector() {}

  //! Clear the playlist of the player.
  virtual void playlistClear() {}

  virtual int getPlaylistLength() { return 0; } 
  virtual int getPlaylistPos() { return 0; } 

  virtual QString getPlaylistTitle(int pos) { return QString(""); }

  //! Add a song to the playlist.
  /*!
    \param filename   The filename of the song to add.
   */
  virtual void playlistAdd(QString filename) {}

  //! Add songs to the playlist.
  /*!
    \param filenames  The filenames of the songs to add.
   */
  virtual void playlistAdd(QStringList filenames) {}

  //! Which player is this?
  virtual Player is() { return NONE; }
};

#endif
