/* $Id: dbaccess.c,v 1.5 1999/07/30 16:48:33 stano Exp $

   Database definitions

   This file contains the 

   (C) 1999 Stanislav Meduna <stano@eunet.sk>

*/

#include <dbaccess.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <stdio.h>
#include <libintl.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#define _(s) dgettext(PACKAGE, s)

DBM *sys_map  = NULL;  /* system database */
DBM *user_map = NULL;  /* user database */

static data_source_t cur_src;    /* current source */
static char cur_relfn[PATH_MAX+1]; /* current relative file name */

static char cur_redef_relfn[PATH_MAX+1]; /* current rel fn, as of redef file */

static char user_mode;           /* acting in user mode */

static int map_count;            /* count processed maps */
static int file_count;           /* count processed files */

extern FILE *xkbparserin;
extern int xkbparserlex();

/* === Forward declarations === */

static int parse_files(const char *reldir);

/* Caution: dptr in key and content point to static storage */
static int pack(const db_record_t *data, datum *key, datum *content);

static int unpack(datum key, datum content, db_record_t *data);

static void print_source(FILE *fp, data_source_t src);

/* === Public interface === */

/* Create/rebuild the database, system or user one */
int rebuild_db(int user)
{
	char cwd[PATH_MAX+1];

	static const data_source_t sys_sources[]  = { SYS_REDEF, SYS_DISTR, X11_DISTR, DEFAULT };
	static const data_source_t user_sources[] = { USER_REDEF, USER, DEFAULT };

	const data_source_t *src;

	int err = 0;

	mode_t old_umask;

	if (flag_verbose)
	{
		if (user)
			fprintf(stderr, _("Rebuilding user database\n"));
		else
			fprintf(stderr, _("Rebuilding system database\n"));
	}

	/* Save the cwd */
	if (getcwd(cwd, sizeof(cwd)) == NULL)
	{
		fprintf(stderr, _("cannot get cwd name: %s\n"), strerror(errno));
		return -1;
	}

	/* Set sane umask */
	old_umask = umask(022);

	/* For user mode, (re-)create the user database and 
	   open the system one r/o. Otherwise (re-)create the
	   system one. */

	if (user)
	{
		sys_map = dbm_open(sys_db, O_RDONLY, 0);
		if (sys_map == NULL)
		{
			fprintf(stderr, _("Cannot open database %s\n"), sys_db);
			return -1;
		}

		user_map = dbm_open(user_db, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		if (user_map == NULL)
		{
			fprintf(stderr, _("Cannot (re-)create database %s\n"), user_db);
			dbm_close(sys_map);
			return -1;
		}
	}
	else
	{
		sys_map = dbm_open(sys_db, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		if (sys_map == NULL)
		{
			fprintf(stderr, _("Cannot (re-)create database %s\n"), sys_db);
			return -1;
		}
	}


	user_mode = user;

	/* Process the data sources in the given priority */

	for (src = user ? user_sources : sys_sources; !err && *src != DEFAULT; src++)
	{
		const char *base;
		struct stat st;

		map_count = 0;
		file_count = 0;

		switch (*src)
		{
		case X11_DISTR:
			if (flag_verbose)
				fprintf(stderr, _("Processing X11 keymaps..."));
			base = x11_keymap_dir;
			break;
		case SYS_DISTR:
			if (flag_verbose)
				fprintf(stderr, _("Processing system keymaps..."));
			base = sys_keymap_dir;
			break;
		case SYS_REDEF:
			if (flag_verbose)
				fprintf(stderr, _("Processing system redefinitions..."));
			base = sys_redef_dir;
			break;
		case USER:
			if (flag_verbose)
				fprintf(stderr, _("Processing user keymaps..."));
			base = user_keymap_dir;
			break;
		case USER_REDEF:
			if (flag_verbose)
				fprintf(stderr, _("Processing user redefinitions..."));
			base = user_redef_dir;
			break;
		default:
			err = -1;
			break;
		}

		if (err)
			break;

		/* Nonexistent directory is no problem ... */
		if (access(base, R_OK) < 0)
		{
			if (flag_verbose)
				fprintf(stderr, _("nonexistent\n"));
			continue;
		}

		/* ... but it must be a directory */
	        if (stat(base, &st) < 0)
	        {
	                fprintf(stderr, _("cannot stat %s: %s\n"), base, strerror(errno));
	                err = -1;
			break;
		}

		if (! S_ISDIR(st.st_mode))
	        {
	                fprintf(stderr, _("%s is not a directory\n"), base);
	                err = -1;
			break;
		}

		if (chdir(base) < 0)
	        {
	                fprintf(stderr, _("cannot change to %s: %s\n"), base, strerror(errno));
	                err = -1;
			break;
		}

		cur_src = *src;
		err = parse_files("");

		if (err)
			break;

		if (flag_verbose)
		{
			fprintf(stderr, " ");

			if (file_count == 1)
				fprintf(stderr, _("1 file"));
			else
				fprintf(stderr, _("%d files"), file_count);

			fprintf(stderr, ", ");

			if (map_count == 1)
				fprintf(stderr, _("1 map"));
			else
				fprintf(stderr, _("%d maps"), map_count);

			fprintf(stderr, "\n");
		}
	}


	/* Restore cwd */
	if (chdir(cwd) < 0)
	{
		fprintf(stderr, _("cannot restore cwd %s: %s\n"), cwd, strerror(errno));
		return -1;
	}

	/* Close the databases */
	dbm_close(sys_map);
	if (user)
		dbm_close(user_map);

	/* Restore umask */
	umask(old_umask);

	if (flag_verbose && ! err)
		fprintf(stderr, _("Done.\n"));

	return err;
}

/* Open the system database and the user one (if present) for reading */
int open_db(void)
{
	sys_map = dbm_open(sys_db, O_RDONLY, 0);
	if (sys_map == NULL)
	{
		fprintf(stderr, _("Cannot open database %s\n"), sys_db);
		return -1;
	}

	user_map = dbm_open(user_db, O_RDONLY, 0);

	return 0;
}

/* Read the data for a specified map
   Expects valid map_name, the rest will be filled in
*/
int read_db(db_record_t *rec, int *found)
{
	datum key, content;
	int err;
	int foundusr = 0, foundsys = 0;
	db_record_t sysrec;

	*found = 0;

	/* Make a key */
	err = pack(rec, &key, 0);
	if (err < 0)
		return err;

	/* If user db exists, try there first */
	if (user_map != NULL)
	{
		content = dbm_fetch(user_map, key);
		foundusr = (content.dptr != NULL);
	}

	/* If found in user db, fill the record */
	if (foundusr)
	{
		err = unpack(key, content, rec);
		if (err < 0)
			return err;
	}

	/* Find in the system db */
	content = dbm_fetch(sys_map, key);
	foundsys = (content.dptr != NULL);

	/* If not found anywhere, return */
	if (! foundusr && ! foundsys)
	{
		*found = 0;
		return 0;
	}

	/* If found in usr and not in sys, we are done */
	if (! foundsys)
	{
		*found = 1;
		return 0;
	}

	/* If found in sys and not in usr, unpack and return */
	if (! foundusr)
	{
		err = unpack(key, content, rec);
		if (err < 0)
			return err;
		*found = 1;
		return 0;
	}

	/* Found in both - fill the missing data */
	*found = 1;

	err = unpack(key, content, &sysrec);
	if (err < 0)
		return err;

	if (rec->descr_src == DEFAULT && sysrec.descr_src != DEFAULT)
	{
		strcpy(rec->descr, sysrec.descr);
		rec->descr_src = sysrec.descr_src;
	}

	if (rec->keymap_src == DEFAULT && sysrec.keymap_src != DEFAULT)
	{
		strcpy(rec->keymap, sysrec.keymap);
		rec->keymap_src = sysrec.keymap_src;
	}

	if (rec->pixmap_src == DEFAULT && sysrec.pixmap_src != DEFAULT)
	{
		strcpy(rec->pixmap, sysrec.pixmap);
		rec->pixmap_src = sysrec.pixmap_src;
	}

	return 0;
}

/* Read the data for a specified map
*/
int read_db_name(const char *name, db_record_t *rec, int *found)
{
	int r;

	if (strlen(name) >= sizeof(rec->map_name))
	{
		fprintf(stderr, _("map name too long: %s\n"), name);
		return -1;
	}

	strcpy(rec->map_name, name);

	r = read_db(rec, found);

	return r;
}



/* Traverse the data (from the system or user view) */
int traverse_db(int userview, int first, db_record_t *rec, int *found)
{
	datum key, content;

	DBM *map = 0;
	int err;

	*found = 0;

	/* If the user db is open, traverse it first.
	   Then	traverse all sys records not found in user db.
	*/

	if (first)
		user_mode = (userview && user_map != NULL);

	map = user_mode ? user_map : sys_map;

	if (first)
	{
		key = dbm_firstkey(map);

		if (user_mode && key.dptr == NULL)
		{
			user_mode = 0;
			map = sys_map;
			key = dbm_firstkey(map);
		}

		*found = (key.dptr != NULL);
	}
	else
	{
		/* Remeber - we must skip things found in user db before */
		for (;;)
		{
			key = dbm_nextkey(map);

			if (user_mode && key.dptr == NULL)
			{
				user_mode = 0;
				map = sys_map;
				key = dbm_firstkey(map);
			}

			*found = (key.dptr != NULL);

			if (! *found || user_mode || ! user_map)
				break;

			/* Found in sys db - try user one */
			content = dbm_fetch(user_map, key);
			if (content.dptr == NULL)
				break;
		}
	}

	if (! *found)
		return 0;

	/* Fetch and unpack found data */
	content = dbm_fetch(map, key);

	if (content.dptr == NULL)
	{
		fprintf(stderr, _("Cannot fetch what first/nextkey found?!\n"));
		return -1;
	}

	err = unpack(key, content, rec);
	if (err < 0)
		return err;

	/* If this was in user data, we must process it */
	if (user_mode)
	{
		int f;

		err = read_db(rec, &f);
		if (err < 0)
			return err;

		if (! f)
		{
			fprintf(stderr, _("Cannot fetch what first/nextkey found?!\n"));
			return -1;
		}
	}

	/* Done */
	return 0;
}

/* Close the database(s) */
int close_db(void)
{
	if (user_map != NULL)
	{
		dbm_close(user_map);
		user_map = NULL;
	}

	if (sys_map != NULL)
	{
		dbm_close(sys_map);
		sys_map = NULL;
	}

	return 0;
}

/* Print the record */
void print_record(FILE *fp, const db_record_t *rec)
{
	fprintf(fp, _("Map name:    \"%s\""), rec->map_name);
	fprintf(fp, "\n");
	fprintf(fp, _("Description: \"%s\" from "), rec->descr);
	print_source(fp, rec->descr_src);
	fprintf(fp, "\n");
	fprintf(fp, _("Keymap file: \"%s\" from "), rec->keymap);
	print_source(fp, rec->keymap_src);
	fprintf(fp, "\n");
	fprintf(fp, _("Pixmap file: \"%s\" from "), rec->pixmap);
	print_source(fp, rec->pixmap_src);
	fprintf(fp, "\n");

	return;
}

/* === interface to lex === */

/* lex callback, called once for every matched map or redefinition */
void mapinfo_cb(const char *name, const char *descr, const char *pixmap)
{
	db_record_t data, data_old;
	datum key, content;
	int err;
	int found;

	DBM *map = user_mode ? user_map : sys_map;

	/* The redef files can specify a fake relative filename */
	char *rfn = ((cur_src == SYS_REDEF || cur_src == USER_REDEF) &&
	             *cur_redef_relfn) ?
		     	cur_redef_relfn :
			cur_relfn;

	/* Fill the record */
	if (strlen(rfn) + strlen(name) + 3 > sizeof(data.map_name))
	{
		fprintf(stderr, _("Map name %s(%s) too long - ignored\n"), rfn, name);
		return;
	}

	sprintf(data.map_name, "%s(%s)", rfn, name);

	/* Make a key and check whether we have something */
	err = pack(&data, &key, 0);
	if (err < 0)
		return;

	content = dbm_fetch(map, key);

	found = (content.dptr != NULL);

	/* If found, unpack to data_old */
	if (found)
	{
		if (flag_debug)
			fprintf(stderr, _("Record %s already exists in the database\n"), data.map_name);
		err = unpack(key, content, &data_old);
		if (err < 0)
			return;
	}

	/* Construct the new content */
	if (found && data_old.descr_src != DEFAULT)
	{
		strcpy(data.descr, data_old.descr);
		data.descr_src = data_old.descr_src;
	}
	else if (*descr)
	{
		if (strlen(descr) + 1 > sizeof(data.descr))
		{
			fprintf(stderr, _("Description %s too long - ignored\n"), descr);
			return;
		}
		strcpy(data.descr, descr);
		data.descr_src = cur_src;
	}
	else
	{
		*data.descr = 0;
		data.descr_src = DEFAULT;
	}

	if (found && data_old.keymap_src != DEFAULT)
	{
		strcpy(data.keymap, data_old.keymap);
		data.keymap_src = data_old.keymap_src;
	}
	else
	{
		if (strlen(rfn) + 1 > sizeof(data.keymap))
		{
			fprintf(stderr, _("File name %s too long - ignored\n"), descr);
			return;
		}
		strcpy(data.keymap, rfn);
		data.keymap_src = cur_src;
	}

	if (found && data_old.pixmap_src != DEFAULT)
	{
		strcpy(data.pixmap, data_old.pixmap);
		data.pixmap_src = data_old.pixmap_src;
	}
	else if (*descr)
	{
		if (strlen(pixmap) + 1 > sizeof(data.pixmap))
		{
			fprintf(stderr, _("Pixmap %s too long - ignored\n"), descr);
			return;
		}
		strcpy(data.pixmap, pixmap);
		data.pixmap_src = cur_src;
	}
	else
	{
		*data.pixmap = 0;
		data.pixmap_src = DEFAULT;
	}

	if (flag_debug)
	{
		fprintf(stderr, _("About to write following record:\n"));
		print_record(stderr, &data);
		fprintf(stderr, "\n");
	}

	err = pack(&data, &key, &content);
	if (err)
		return;

	err = dbm_store(map, key, content, DBM_REPLACE);

	if (err < 0)
	{
		fprintf(stderr, _("Database store failed\n"));
	}

	map_count++;

	return;
}

/* lex callback, called once for file name specification (for redef files) */
void filename_cb(const char *name)
{
	/* Fill the record */
	if (strlen(name) + 1 > sizeof(cur_redef_relfn))
	{
		fprintf(stderr, _("Map name %s too long - ignored\n"), name);
		return;
	}

	strcpy(cur_redef_relfn, name);
}

/* === Private functions === */

/* Recursively traverse all files in the specified directory
   and process them using the parser.

   It is assumed that the function is called with cwd
   pointing to the directory, where the maps reside.
*/
static int parse_files(const char *reldir)
{
	DIR *cwd;
	struct dirent *de;
	struct stat st;

	char path[PATH_MAX+1];
	int fnidx;

	char newrel[PATH_MAX+1];
	int relidx;

	if (getcwd(path, sizeof(path)) == NULL)
	{
		fprintf(stderr, _("cannot get cwd name: %s\n"), strerror(errno));
		return -1;
	}

	fnidx = strlen(path);

	/* The callback needs the relative file name */
	strncpy(newrel, reldir, sizeof(newrel)-1);
	strncpy(cur_relfn, reldir, sizeof(cur_relfn)-1);
	newrel[sizeof(newrel)-1] = 0;
	relidx = strlen(newrel);

	cwd = opendir(path);
	if (cwd == NULL)
	{
		fprintf(stderr, _("cannot open cwd %s: %s\n"), path, strerror(errno));
		return -1;
	}

	/* Process all regular files (and symlinks pointing to them) */
	for (de = readdir(cwd); de != NULL; de = readdir(cwd))
	{
		if (stat(de->d_name, &st) < 0)
		{
			fprintf(stderr, _("cannot stat %s: %s\n"), de->d_name, strerror(errno));
			closedir(cwd);
			return -1;
		}

		if (! S_ISREG(st.st_mode))
			continue;

		/* Build the full and relative filename... */
		if (fnidx  + 1 + strlen(de->d_name) >= sizeof(path) ||
		    relidx + 1 + strlen(de->d_name) >= sizeof(cur_relfn))
		{
			fprintf(stderr, _("name too long for dir %s, item %s\n"), path, de->d_name);
			closedir(cwd);
			return -1;
		}

		sprintf(path+fnidx, "/%s", de->d_name);
		sprintf(cur_relfn+relidx, "%s%s", *reldir ? "/" : "", de->d_name);

		/* Start without redefined filename */
		*cur_redef_relfn = 0;

		/* Open the file for lex */
		xkbparserin = fopen(path, "r");

		if (xkbparserin == NULL)
		{
			fprintf(stderr, _("cannot open %s: %s\n"), path, strerror(errno));
			closedir(cwd);
			return -1;
		}
		file_count++;

		/* Call the parser on the file */
		xkbparserlex();

		fclose(xkbparserin);
	}


	/* Process all directories (and symlinks pointing to them) */

	rewinddir(cwd);

	for (de = readdir(cwd); de != NULL; de = readdir(cwd))
	{
		if (stat(de->d_name, &st) < 0)
		{
			fprintf(stderr, _("cannot stat %s: %s\n"), de->d_name, strerror(errno));
			closedir(cwd);
			return -1;
		}

		if (! S_ISDIR(st.st_mode) || ! strcmp(de->d_name, ".") || ! strcmp(de->d_name, ".."))
			continue;

		/* Build the full and relative filename */
		if (fnidx  + 1 + strlen(de->d_name) >= sizeof(path) ||
		    relidx + 1 + strlen(de->d_name) >= sizeof(newrel))
		{
			fprintf(stderr, _("name too long for dir %s, item %s\n"), path, de->d_name);
			closedir(cwd);
			return -1;
		}

		sprintf(path+fnidx, "/%s", de->d_name);
		sprintf(newrel+relidx, "%s%s", *reldir ? "/" : "", de->d_name);

		if (chdir(path) < 0)
		{
			fprintf(stderr, _("cannot cd to %s\n"), path);
			closedir(cwd);
			return -1;
		}

		/* Recurse */
		if (parse_files(newrel) < 0)
		{
			closedir(cwd);
			return -1;
		}

		/* Restore cwd */
		path[fnidx] = 0;
		if (chdir(path) < 0)
		{
			fprintf(stderr, _("cannot restore cwd %s: %s\n"), path, strerror(errno));
			return -1;
		}
	}

	closedir(cwd);

	return 0;
}

/* Caution: dptr in key and content point to static storage */
static int pack(const db_record_t *data, datum *key, datum *content)
{
	/* Never longer than db_record_t itself */
	static char buf[sizeof(db_record_t)];

	char *kp, *cp;
	int   kl,  cl;

	/* Marshal the data - strings are null-terminated,
	   sources are 1-byte long */

	/* key */
	kp = buf;
	kl = 0;

	strcpy(&kp[kl], data->map_name);
	kl += strlen(data->map_name)+1;

	key->dptr = kp;
	key->dsize = kl;

	if (content != NULL)
	{
		/* content */
		cp = kp+kl;
		cl = 0;
	
		strcpy(&cp[cl], data->descr);
		cl += strlen(data->descr)+1;
	
		cp[cl++] = (char) data->descr_src;

		strcpy(&cp[cl], data->keymap);
		cl += strlen(data->keymap)+1;

		cp[cl++] = (char) data->keymap_src;
	
		strcpy(&cp[cl], data->pixmap);
		cl += strlen(data->pixmap)+1;

		cp[cl++] = (char) data->pixmap_src;

		content->dptr = cp;
		content->dsize = cl;
	}

	return 0;
}

static int unpack(datum key, datum content, db_record_t *data)
{
	char *p;

	/* Unmarshal the data */
	p = (char *) key.dptr;

	/* key */
	strncpy(data->map_name, p, sizeof(data->map_name));
	data->map_name[sizeof(data->map_name) - 1] = 0;

	/* content */
	p = (char *) content.dptr;
	strncpy(data->descr, p, sizeof(data->descr));
	data->descr[sizeof(data->descr) - 1] = 0;

	p += strlen(p) + 1;

	data->descr_src = *p++;

	strncpy(data->keymap, p, sizeof(data->keymap));
	data->keymap[sizeof(data->keymap) - 1] = 0;

	p += strlen(p) + 1;

	data->keymap_src = *p++;

	strncpy(data->pixmap, p, sizeof(data->pixmap));
	data->pixmap[sizeof(data->pixmap) - 1] = 0;

	p += strlen(p) + 1;

	data->pixmap_src = *p++;

	return 0;
}

static void print_source(FILE *fp, data_source_t src)
{
	switch (src)
	{
	case DEFAULT:
		fprintf(fp, _("default"));
		break;
	case X11_DISTR:
		fprintf(fp, _("X11 keymaps"));
		break;
	case SYS_DISTR:
		fprintf(fp, _("system keymaps"));
		break;
	case SYS_REDEF:
		fprintf(fp, _("system redefinitions"));
		break;
	case USER:
		fprintf(fp, _("user keymaps"));
		break;
	case USER_REDEF:
		fprintf(fp, _("user redefinitions"));
		break;
	default:
		fprintf(fp, _("unknown (%d)"), cur_src);
		break;
	}

	return;
}

