// This is adapted from 'textedit', a demo text viewer found
// in the early versions of the FOX library and written
// by Jeroen van der Zijp.

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>

#include <fox-1.2/fx.h>
#include <fox-1.2/fxkeys.h>
#include <fox-1.2/FXPNGIcon.h>

#include "icons.h"
#include "FileDialog.h"
#include "MessageBox.h"
#include "XFileView.h"

FXMainWindow *mainWindow;
char **args;

// Map
FXDEFMAP(XFileView) XFileViewMap[]={
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_ABOUT,XFileView::onCmdAbout),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_OPEN,XFileView::onCmdOpen),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_TITLE,XFileView::onUpdTitle),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_FIND,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_FIND_AGAIN,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_FIND_AGAIN,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_QUIT,XFileView::onCmdQuit),
	                                   FXMAPFUNC(SEL_COMMAND,XFileView::ID_PRINT,XFileView::onCmdPrint),
                                   };


// Object implementation
FXIMPLEMENT(XFileView,FXMainWindow,XFileViewMap,ARRAYNUMBER(XFileViewMap))


// Make some windows
XFileView::XFileView(FXApp* a):FXMainWindow(a,"Xfv ",NULL,NULL,DECOR_ALL,0,0,600,500,0,0)
{
    FXIcon *bigp=new FXPNGIcon(getApp(),xfv);
    setIcon(bigp);
    FXHotKey hotkey;
    FXButton *btn = NULL;
	
    // Create icons
	FXPNGIcon *fileopenicon=new FXPNGIcon(getApp(),fileopen);
	FXPNGIcon *quiticon=new FXPNGIcon(getApp(),quit);
	FXPNGIcon *printicon=new FXPNGIcon(getApp(),print);	
	FXPNGIcon *searchicon=new FXPNGIcon(getApp(),search);		
	FXPNGIcon *findicon=new FXPNGIcon(getApp(),find_again);

	// Make main window; set myself as the target
    setTarget(this);
    setSelector(ID_TITLE);

    // Make menu bar
    menubar=new FXMenuBar(this,FRAME_RAISED|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_THICK);

    // Tool bar
  	FXToolBarShell* dragshell=new FXToolBarShell(this,FRAME_RAISED);
  	toolbar=new FXToolBar(this,dragshell,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
  	new FXToolBarGrip(toolbar,toolbar,FXToolBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);
    int style=BUTTON_TOOLBAR;

    // Status bar
    statusbar=new FXStatusBar(this,FRAME_RAISED|LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X);

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Search Menu
    FXMenuPane *searchmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Search"),NULL,searchmenu);

    // Preferences Menu
    FXMenuPane *prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu);

    // Make contents
    contents=new FXHorizontalFrame(this,FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 2,2,2,2, 0,0/*,0,0,0,0, 0,0,0,0*/);

    // Make viewer window
    FXHorizontalFrame *cont=new FXHorizontalFrame(contents,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
    viewer=new FXText(cont,NULL,0,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Show column number in status bar
    new FXTextField(statusbar,4,viewer,FXText::ID_CURSOR_COLUMN,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,1,1,0,0);

    // Caption before number
    new FXLabel(statusbar,_("Col:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Show line number in status bar
    new FXTextField(statusbar,4,viewer,FXText::ID_CURSOR_ROW,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,1,1,0,0);

    // Caption before number
    new FXLabel(statusbar,_("Line:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Toolbar button: File open
    btn = new FXButton(toolbar,_("\tOpen document file (Ctrl-O)"),fileopenicon,this,ID_OPEN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_o;
    btn->addHotKey(hotkey);

    // Toolbar button: Print
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
	btn = new FXButton(toolbar,_("\tPrint document (Ctrl-P)"),printicon,this,ID_PRINT,style|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_p;
    btn->addHotKey(hotkey);

    // Toolbar buttons: Searching
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    btn = new FXButton(toolbar,_("\tFind a string (Ctrl-F)"),searchicon,this,ID_FIND,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_f;
    btn->addHotKey(hotkey);
    btn = new FXButton(toolbar,_("\tFind again (Ctrl-G)"),findicon,this,ID_FIND_AGAIN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_g;
    btn->addHotKey(hotkey);

    // Toolbar button: Quit
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    btn = new FXButton(toolbar,_("\tQuit Xfv (Ctrl-Q, F10)"),quiticon,this,ID_QUIT,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_q;
    btn->addHotKey(hotkey);
    btn->addHotKey(KEY_F10);

    // File Menu entries
    new FXMenuCommand(filemenu,_("Open file...\tCtrl-O"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("&Print file...\tCtrl-P"),printicon,this,ID_PRINT);
    new FXMenuCommand(filemenu,_("Quit\tCtrl-Q"),quiticon,this,ID_QUIT);

    // Search Menu entries
    new FXMenuCommand(searchmenu,_("Find a string...\tCtrl-F"),searchicon,this,ID_FIND);
    new FXMenuCommand(searchmenu,_("Find again\tCtrl-G"),findicon,this,ID_FIND_AGAIN);

    // Preferences menu
    new FXMenuCheck(prefsmenu,_("Toolbar\t\tDisplay toolbar"),toolbar,FXWindow::ID_TOGGLESHOWN);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("About X File View\tF1"),NULL,this,ID_ABOUT,0);

    // Make a tool tip
    new FXToolTip(getApp(),0);

    // Initialize file name
    filename="";
}


// About box
long XFileView::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXPNGIcon icon(getApp(),xfv);
    MessageBox about(this,_("About X File View"),_("X File View is a simple text viewer.\n\nCopyright (C) 2002-2004 Roland Baudin (roland65@free.fr)"),&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute();
    return 1;
}


// Load file
FXbool XFileView::loadfile(const FXString& file)
{
    struct stat info;
    char *text;
    FILE *fp;
    int n;
    if(stat(file.text(),&info)!=0)
    {
        MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Non-existing file: %s"),file.text());
        return FALSE;
    }
    fp=fopen(file.text(),"r");
    if(!fp)
    {
        MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Unable to open file: %s"),file.text());
        return FALSE;
    }
    FXMALLOC(&text,FXchar,info.st_size+1);
    n=fread(text,1,info.st_size,fp);
    if(n!=info.st_size)
    {
        MessageBox::error(this,MBOX_OK,_("Error Reading File"),_("Unable to load entire file: %s"),file.text());
        return FALSE;
    }
    else
    {
        text[info.st_size]='\0';
        viewer->setText(text);
        viewer->setModified(FALSE);
    }
    FXFREE(&text);
    fclose(fp);
    return TRUE;
}


// Open
long XFileView::onCmdOpen(FXObject*,FXSelector,void*)
{
    const FXchar *patterns[]=
        {
            _("All Files"),          "*",
            _("C++ Source Files"),   "*.cc",
            _("C Source Files"),     "*.c",
            _("C/C++ Header Files"),   "*.h",
            _("Text Files"),         "*.txt",
            _("HTML Header Files"),  "*.html",NULL
        };
    FileDialog open(this,_("Open Document"));
	open.setSelectMode(SELECTFILE_EXISTING);
    open.setPatternList(patterns);
    if(open.execute())
    {
        filename=open.getFilename();
        loadfile(filename);
        viewer->setFocus();
    }
    return 1;
}


// Print the text
long XFileView::onCmdPrint(FXObject*,FXSelector,void*)
{
	FXPrintDialog dlg(this,_("Print File"));
	FXPrinter printer;
	if(dlg.execute())
    	dlg.getPrinter(printer);
  	return 1;
}

// Save configuration when quitting
void XFileView::saveConfig()
{
    // Write new window size and position back to registry
	getApp()->reg().writeUnsignedEntry("SETTINGS","width",(FXuint)getWidth());
	getApp()->reg().writeUnsignedEntry("SETTINGS","height",(FXuint)getHeight());
	getApp()->reg().writeIntEntry("SETTINGS","xpos",getX());
	getApp()->reg().writeIntEntry("SETTINGS","ypos",getY());

    // Toolbar status
    if(toolbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",FALSE);

	// Write registry settings
	getApp()->reg().write();
}


// Quit
long XFileView::onCmdQuit(FXObject* sender,FXSelector sel,void* ptr)
{
	// Save settings 
    saveConfig();

    // Quit
    getApp()->exit(0);
    return 1;
}


// Update title
long XFileView::onUpdTitle(FXObject* sender,FXSelector,void*)
{
    FXString title="Xfv " + filename;
    if(viewer->isModified())
        title+=" (modified)";
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&title);
    return 1;
}


// Open find dialog
long XFileView::onCmdFind(FXObject* obj,FXSelector sel,void* ptr)
{
    int loc;

    static FXString str="";

    if(FXSELTYPE(sel)==SEL_UPDATE)
    {
        if(str=="")
            ((FXWindow*)obj)->disable();
        else
            ((FXWindow*)obj)->enable();
        return 1;
    }

    if(FXSELID(sel)==ID_FIND)
    {
        FXDialogBox dialog(this,_("Find"),DECOR_TITLE|DECOR_BORDER);
        FXVerticalFrame* content=new FXVerticalFrame(&dialog,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0,10,10,10,10,10,10);
        text=new FXTextField(content,30,NULL,0,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X);
        new FXHorizontalSeparator(content,SEPARATOR_GROOVE|LAYOUT_FILL_X);
        FXHorizontalFrame* buttons=new FXHorizontalFrame(content,LAYOUT_FILL_X|PACK_UNIFORM_WIDTH,0,0,0,0,0,0,0,0);
        new FXButton(buttons,_("&Cancel"),NULL,&dialog,FXDialogBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT,0,0,0,0,20,20);
        new FXButton(buttons,_("&Find"),NULL,&dialog,FXDialogBox::ID_ACCEPT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT);
        text->setText(str);
        text->setSelection(0,str.length());
        text->setFocus();

        if(!dialog.execute())
            return 0;
        str=text->getText();
    }
    loc=find(str,viewer->getCursorPos());
    if(loc==-1)
        MessageBox::information(this,MBOX_OK,_("Not Found"),_("String '%s' not found"),str.text());
    else
    {
        viewer->setCursorPos(loc+str.length());
        viewer->setTopLine(loc);
        viewer->setSelection(loc,str.length());
        viewer->setFocus();
    }
    return 1;
}


// Find string
FXint XFileView::find(const FXString str,FXint pos)
{
    FXString buf=viewer->getText();
    const char* text=buf.text();
    char *p = strstr(&text[pos],str.text());
    if(!p)
        return -1;
    return (p-text);
}


// Start the ball rolling
void XFileView::start(int& argc,char** argv)
{
    if(argc>1)
    {
        filename=argv[1];
        loadfile(filename);
    }
    else if(filename != "")
        loadfile(filename);
}


// Create and show window
void XFileView::create()
{
    // Read the Xfe registry
	FXRegistry* reg_xfe=new FXRegistry("Xfe",NULL);
	reg_xfe->read();
	
	// Set Xfv text font according to the Xfe registry
	FXString fontspec;
	fontspec=reg_xfe->readStringEntry("SETTINGS","textfont","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* font=new FXFont(getApp(),fontspec);
        viewer->setFont(font);
	}
	delete reg_xfe;

	// Wheel scrolling
    FXint wheellines=getApp()->reg().readUnsignedEntry("SETTINGS","wheellines",5);
	getApp()->setWheelLines(wheellines);
	
    // Get size and position
    FXuint width=getApp()->reg().readUnsignedEntry("SETTINGS","width",600);
    FXuint height=getApp()->reg().readUnsignedEntry("SETTINGS","height",400);
    FXint xpos=getApp()->reg().readIntEntry("SETTINGS","xpos",0);
    FXint ypos=getApp()->reg().readIntEntry("SETTINGS","ypos",0);

	// Get toolbar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showtoolbar",TRUE)==FALSE)
        toolbar->hide();
   
	// Reposition window
    position(xpos,ypos,width,height);

    FXMainWindow::create();
    show();
    viewer->setFocus();
    viewer->handle(this,FXSEL(SEL_COMMAND,FXText::ID_TOGGLE_EDITABLE),NULL);
}


// Standard X Window arrow cursor definition
#define left_ptr_width 16
#define left_ptr_height 16
#define left_ptr_x_hot 3
#define left_ptr_y_hot 1
static const FXuchar left_ptr_bits[] = {
                                  0x00, 0x00, 0x08, 0x00, 0x18, 0x00, 0x38, 0x00, 0x78, 0x00, 0xf8, 0x00,
                                  0xf8, 0x01, 0xf8, 0x03, 0xf8, 0x07, 0xf8, 0x00, 0xd8, 0x00, 0x88, 0x01,
                                  0x80, 0x01, 0x00, 0x03, 0x00, 0x03, 0x00, 0x00};

#define left_ptrmsk_width 16
#define left_ptrmsk_height 16
static const FXuchar left_ptrmsk_bits[] = {
                                     0x0c, 0x00, 0x1c, 0x00, 0x3c, 0x00, 0x7c, 0x00, 0xfc, 0x00, 0xfc, 0x01,
                                     0xfc, 0x03, 0xfc, 0x07, 0xfc, 0x0f, 0xfc, 0x0f, 0xfc, 0x01, 0xdc, 0x03,
                                     0xcc, 0x03, 0x80, 0x07, 0x80, 0x07, 0x00, 0x03};

// Start the whole thing
int main(int argc,char *argv[])
{
#ifdef HAVE_SETLOCALE
  // Set locale via LC_ALL.
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  // Set the text message domain.
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

    args=argv;

    // Make application
    FXApp* application=new FXApp("Xfv",NULL);

    // Open display
    application->init(argc,argv);

    // Redefine the default cursor to get the standard X Window arrow cursor
    FXCursor* left_ptr=new FXCursor(application,left_ptr_bits,left_ptrmsk_bits,left_ptr_width,left_ptr_height,left_ptr_x_hot,left_ptr_y_hot);
    application->setDefaultCursor(DEF_ARROW_CURSOR,left_ptr);

    // Read the Xfe registry
	FXRegistry* reg=new FXRegistry("Xfe",NULL);
	reg->read();
	
	// Set Xfv normal font according to the Xfe registry
	FXString fontspec;
	fontspec=reg->readStringEntry("SETTINGS","font","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* normalFont=new FXFont(application,fontspec);
        application->setNormalFont(normalFont);
    }

	// Set Xfv colors according to the Xfe registry
	FXColor basecolor,forecolor,backcolor,selforecolor,selbackcolor,bordercolor;
    basecolor=reg->readColorEntry("SETTINGS","basecolor",FXRGB(214,207,198));
    forecolor=reg->readColorEntry("SETTINGS","forecolor",FXRGB(0,0,0));
    backcolor=reg->readColorEntry("SETTINGS","backcolor",FXRGB(255,255,255));
    selforecolor=reg->readColorEntry("SETTINGS","selforecolor",FXRGB(255,255,255));
    selbackcolor=reg->readColorEntry("SETTINGS","selbackcolor",FXRGB(10,36,106));
    bordercolor=reg->readColorEntry("SETTINGS","bordercolor",FXRGB(0,0,0));
	application->setBaseColor(basecolor);
	application->setForeColor(forecolor);
	application->setBackColor(backcolor);
	application->setSelforeColor(selforecolor);
	application->setSelbackColor(selbackcolor);
	application->setBorderColor(bordercolor);

	// Delete the Xfe registry
	delete reg;
    
	// Make window
    XFileView* window=new XFileView(application);
	mainWindow=window;
	
    // Create it
    application->create();

    // Start
    window->start(argc,argv);

    // Run
    application->run();
}

