/*  xfce4
 *  
 *  Copyright (C) 2002 Olivier Fourdan (fourdan@xfce.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <errno.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <gdk/gdkx.h>
#include <glib.h>
#include <gmodule.h>
#include <gtk/gtk.h>

#include <libxfce4mcs/mcs-common.h>
#include <libxfce4mcs/mcs-manager.h>
#include <libxfce4util/i18n.h>
#include <libxfce4util/util.h>
#include "manager-plugin.h"
#include "xfce-mcs-dialog.h"

#define SOEXT           ("." G_MODULE_SUFFIX)	 
#define SOEXT_LEN       (strlen (SOEXT))

static GSList *plugin_list = NULL;
static GSList *module_list = NULL;

static McsManager *manager;

static enum {
	NOSIGNAL,
	RESTART,
	QUIT,
	NUM_SIGNALS
} sigstate = NOSIGNAL;

gchar *
mcs_plugin_check_version(gint version)
{
    if (XFCE_MCS_PLUGIN_VERSION == version)
        return(NULL);

    return(_("Incorrect module version"));
}

static gint
compare_plugins(gconstpointer plugin_a, gconstpointer plugin_b)
{
    /* XXX - This need not to be strcasecmp, strncmp should do the job! */
    return(g_ascii_strcasecmp(((const McsPlugin *)plugin_a)->plugin_name,
                ((const McsPlugin *)plugin_b)->plugin_name));
}

static gint
lookup_plugins(gconstpointer plugin, gconstpointer name)
{
    /* XXX - This need not to be strcasecmp, strncmp should do the job! */
    return(g_ascii_strcasecmp(((const McsPlugin *)plugin)->plugin_name, name));
}

static void
load_plugin(gchar *path)
{
    void (*init)(McsPlugin *);
    McsPlugin *plugin;
    GModule *module;

#ifdef DEBUG
    g_message("- attempting to load module %s", path);
#endif

    if ((module = g_module_open(path, 0)) == NULL) {
        g_warning("Module %s cannot be opened (%s)", path, g_module_error());
	return;
    }

    if (g_module_symbol(module, "mcs_plugin_init", (gpointer)&init)) {
    	plugin = g_new0(McsPlugin, 1);
    	plugin->manager = manager;

        init(plugin);

        if(g_slist_find_custom(plugin_list, plugin, compare_plugins)) {
            g_warning("  module %s (\"%s\") has already been loaded before",
                    path, plugin->plugin_name);

            g_module_close(module);
            g_free(plugin);
        }
        else {
#ifdef DEBUG
            g_message("  module %s (\"%s\") successfully loaded", path,
                    plugin->plugin_name);
#endif
            plugin_list = g_slist_append(plugin_list, plugin);
	    module_list = g_slist_append(module_list, module);
        }
    }
    else {
        g_warning("  incompatible module %s", path);
        g_module_close(module);
    }
}

static void
plugins_load_dir(const gchar *dir)
{
	const gchar *f;
	gchar *plugin;
	GDir *d;

	if ((d = g_dir_open(dir, 0, NULL)) == NULL)
		return;

	while ((f = g_dir_read_name(d)) != NULL) {
#if GLIB_CHECK_VERSION(2, 2, 0)
		if (g_str_has_suffix(f, SOEXT))
#else
		if (strncmp(f + strlen(f) - SOEXT_LEN, SOEXT, SOEXT_LEN) == 0)
#endif
		{
            		plugin = g_build_filename(dir, f, NULL);
			load_plugin(plugin);
			g_free(plugin);
		}
	}

	g_dir_close(d);
}

static void
plugin_load_all(void)
{
    gchar *pdir;

    /* load user's plugins */
    if((pdir = xfce_get_userfile("settings", "plugins", NULL)) != NULL) {
        plugins_load_dir(pdir);
        g_free(pdir);
    }

    /* load system plugins */
    plugins_load_dir(PLUGINSDIR);
}

static void
terminate_cb(void *data)
{
#ifdef DEBUG
    g_print("Releasing the selection and exiting");
#endif

    *(gboolean *)data = TRUE;
    gtk_main_quit();
}

static void
show_cb(char *name, void *cb_data)
{
    GSList *plugin_search;
    McsPlugin *plugin;

    if (!g_ascii_strcasecmp("", name) || !g_ascii_strcasecmp("all", name)) {
	    run_manager_dialog(plugin_list);
	    return;
    }
    
    plugin_search = g_slist_find_custom(plugin_list, name, lookup_plugins);

    if(plugin_search) {
        plugin = (McsPlugin *) plugin_search->data;
        plugin->run_dialog(plugin);
    }
    else {
        GtkWidget *dialog = gtk_message_dialog_new(NULL, 0,
                GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                _("Multi-Channel Manager Error:\nNo such plugin \"%s\""),
		name);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
	run_manager_dialog(plugin_list);
    }
}

static GdkFilterReturn
manager_event_filter(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	if (mcs_manager_process_event(manager, (XEvent *)xevent))
		return(GDK_FILTER_REMOVE);
	else
		return(GDK_FILTER_CONTINUE);
}

static void
sighandler(int signo)
{
	switch (signo) {
	case SIGUSR1:
		sigstate = RESTART;
		break;

	default:
		sigstate = QUIT;
	}
}

static gboolean
check_signal_state(void)
{
	if (sigstate == RESTART) {
		sigstate = NOSIGNAL;

		/*
		 * NOTE: Grabbing the X-Server while reloading the
		 * plugins will not work (Don't know why)
		 */
		g_slist_foreach(module_list, (GFunc)g_module_close, NULL);
		g_slist_free(module_list);
		module_list = NULL;

		g_slist_foreach(plugin_list, (GFunc)g_free, NULL);
		g_slist_free(plugin_list);
		plugin_list = NULL;

		plugin_load_all();
	}
	else if (sigstate == QUIT) {
		gtk_main_quit();
	}

	return(TRUE);
}

int
main(int argc, char **argv)
{
#ifdef HAVE_SIGACTION
    struct sigaction act;
#endif
    McsManagerCheck status;
    gboolean daemon_mode;
    gboolean std_mgr;
    int i;

    daemon_mode = TRUE;
    std_mgr = TRUE;

    xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

    gtk_set_locale();
    gtk_init(&argc, &argv);

    /* FIXME: Fake a SM client ID, so smproxy does not catch us */
    gdk_set_sm_client_id("FAKED CLIENTID");

    for (i = 1; i < argc; i++)
        if (strcmp(argv[i], "--no-daemon-debug") == 0)
            daemon_mode = FALSE;

    status = mcs_manager_check_running(gdk_display, DefaultScreen(gdk_display));

    switch (status) {
    case MCS_MANAGER_MULTI_CHANNEL:
    case MCS_MANAGER_BOTH:
        g_warning("Multi channel MCS manager already detected for screen %i",
                DefaultScreen(gdk_display));
        return(EXIT_FAILURE);

    case MCS_MANAGER_STD:
        g_message("Standard XSETTINGS manager already detected for screen %i",
                DefaultScreen(gdk_display));
        std_mgr = FALSE;

    default:
	break;
    }

    manager = mcs_manager_new(std_mgr, gdk_display, DefaultScreen(gdk_display),
            terminate_cb, show_cb, NULL);
    if (manager == NULL) {
        g_warning("Could not create manager!");
        return(EXIT_FAILURE);
    }

    gdk_window_add_filter(NULL, manager_event_filter, NULL);

    plugin_load_all();
    gdk_flush();

#ifdef HAVE_SIGACTION
    act.sa_handler = sighandler;
#ifdef SA_RESTART
    act.sa_flags = SA_RESTART;
#else
    act.sa_flags = 0;
#endif
    sigaction(SIGINT, &act, NULL);
    sigaction(SIGUSR1, &act, NULL);
#else /* !HAVE_SIGACTION */
    signal(SIGINT, sighandler);
    signal(SIGUSR1, sighandler);
#endif

    if (daemon_mode) {
	    switch (fork()) {
	    case -1:
        	g_warning("fork() failed");
        	return(EXIT_FAILURE);
        
	    case 0:
		/* child */
#ifdef HAVE_SETSID
		/* detach from session */
		(void)setsid();
#endif
		g_timeout_add(500, (GSourceFunc)check_signal_state, NULL);
        	break;

	    default:
		/* parent */
		_exit(EXIT_SUCCESS);
	    }
    }
    else
        g_message("Daemon mode disabled (for debug purpose only!)");

    gtk_main();
    mcs_manager_destroy(manager);

    return(EXIT_SUCCESS);
}
