/*
 *	Xenophilia GTK+ Theme Engine
 *	
 *	xeno_style.c:
 *		XenoStyle, XenoStyleData
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_rc_style.h"
#include "xeno_color.h"

#include "xeno_style.h"
#include "xeno_style_draw.h"
#include "xeno_style_fill.h"
#include "xeno_style_images.h"

#ifndef DEBUG
#define DEBUG 0 
#endif


/*
 *	Prototypes
 */
#if XENO_GTK2
static void xeno_style_class_init			(XenoStyleClass *klass);
static void	xeno_style_init					(XenoStyle		*style);
static void	xeno_style_copy					(GtkStyle		*dest,
											 GtkStyle		*src);
#else
static XenoStyleData * xeno_style_data_new (void);
#endif


/*
 *	Globals
 */
#if XENO_GTK2
static GtkStyleClass *	xeno_style_parent_class	= NULL;
#else
GMemChunk *				xeno_style_data_chunk	= NULL;
#endif

#if DEBUG
static gint style_count = 0;
#endif

/*
 *	XenoStyle / XenoStyleData
 */
#if XENO_GTK2
GType
xeno_style_get_type ()
{
	static GType style_type = 0;
	
	if (!style_type) {
		static const GTypeInfo style_info = {
			sizeof(XenoStyleClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) xeno_style_class_init,
			NULL,
			NULL,
			sizeof(XenoStyle),
			8,
			(GInstanceInitFunc) xeno_style_init,
		};
		style_type = g_type_register_static (GTK_TYPE_STYLE, "XenoStyle", &style_info, 0);
	}
	return style_type;
}

static void
xeno_style_class_init (XenoStyleClass *klass)
{
	GtkStyleClass	*style_class;
	GObjectClass	*object_class;
	
	xeno_style_parent_class = g_type_class_peek_parent (klass);
	
	object_class = G_OBJECT_CLASS(klass);
	
	style_class = GTK_STYLE_CLASS(klass);
	style_class->copy				= xeno_style_copy;
	style_class->realize			= xeno_style_realize;
	style_class->unrealize			= xeno_style_unrealize;
	
	style_class->draw_hline			= xeno_style_draw_hline;
	style_class->draw_vline			= xeno_style_draw_vline;
	style_class->draw_shadow		= xeno_style_draw_shadow;
	style_class->draw_polygon		= xeno_style_draw_polygon;
	style_class->draw_arrow			= xeno_style_draw_arrow;
	style_class->draw_diamond		= xeno_style_draw_diamond;
	style_class->draw_string		= xeno_style_draw_string;
	style_class->draw_box			= xeno_style_draw_box;
	style_class->draw_flat_box		= xeno_style_draw_flat_box;
	style_class->draw_check			= xeno_style_draw_check;
	style_class->draw_option		= xeno_style_draw_option;
	style_class->draw_tab			= xeno_style_draw_tab;
	style_class->draw_shadow_gap	= xeno_style_draw_shadow_gap;
	style_class->draw_box_gap		= xeno_style_draw_box_gap;
	style_class->draw_extension		= xeno_style_draw_extension;
	style_class->draw_focus			= xeno_style_draw_focus;
	style_class->draw_slider		= xeno_style_draw_slider;
	style_class->draw_handle		= xeno_style_draw_handle;
	style_class->draw_layout		= xeno_style_draw_layout;
	
	klass->fill_background			= xeno_style_real_fill_background;
	klass->fill_base				= xeno_style_real_fill_base;
}
#endif

#if XENO_GTK2
static void
xeno_style_init (XenoStyle *data)
{
	gint i;
	
	{
#else
XenoStyleData *
xeno_style_data_new ()
{
	XenoStyleData *data;
	gint i;

  #if DEBUG
	style_count += 1;
	g_print ("xeno_style_data_new(%d)\n", style_count);
  #endif

	if (xeno_style_data_chunk == NULL) {
		if ((xeno_style_data_chunk = g_mem_chunk_create(XenoStyleData, 256, G_ALLOC_AND_FREE)) == NULL)
			return NULL;
	}
	
	data = g_mem_chunk_alloc (xeno_style_data_chunk);
	if (data) {
#endif
		for (i = 0; i < XENO_STYLE_IMAGE_END; ++i)
			data->pixmaps[i] = NULL;
		
		for (i = 0; i < 5; ++i) {
			data->white_gc[i] = NULL;
			data->black_gc[i] = NULL;
		}
		data->focus_gc = NULL;
		
		xeno_gradient_set_init (&data->gradient_set);
		
		data->check_variant = XENO_STYLE_IMAGE_CHECK_BUTTON_DEFAULT;
		data->radio_variant = XENO_STYLE_IMAGE_RADIO_BUTTON_DEFAULT;
		data->menu_variant = XENO_STYLE_IMAGE_MENU_DEFAULT;
	}
	
  #if !XENO_GTK2
	return data;
  #endif
}

#if XENO_GTK2
static void
xeno_style_copy (GtkStyle *dest, GtkStyle *src)
{
	g_return_if_fail (XENO_IS_STYLE(src));
	g_return_if_fail (XENO_IS_STYLE(dest));
	
	xeno_style_parent_class->copy(dest, src);
	/* the rest will be recalculated when realize()'d */
}
#endif


/* Realization */

static void
xeno_realize_gc	(GdkGCValues	*gc_values,
				 GdkColormap	*colormap,
				 XenoColor		*color,
				 GdkColor		*color_p,
				 GdkGC			**gc_p)
{
	GdkColor *ptr;
	
	g_return_if_fail (gc_values != NULL);
	g_return_if_fail (colormap != NULL);
	g_return_if_fail (color != NULL || color_p != NULL);
	g_return_if_fail (gc_p != NULL);
	
	ptr = &gc_values->foreground;
	if (color) {
		xeno_color_to_gdk (color, ptr);
		if (color_p)
			*color_p = *ptr;
	} else {
		*ptr = *color_p;
	}
	
	if (!gdk_colormap_alloc_color(colormap, (ptr), FALSE, TRUE)) {
		g_warning ("unable to allocate color #%02x%02x%02x\n", ptr->red>>8, ptr->green>>8, ptr->blue>>8);
	} else if (gc_p) {
		if (*gc_p)
			gtk_gc_release (*gc_p);
		
		if (color_p)
			color_p->pixel = ptr->pixel;
		
		*gc_p = gtk_gc_get (xeno_theme_visual->depth, xeno_theme_colormap, gc_values, GDK_GC_FOREGROUND | GDK_GC_FONT );
	}
}

#if XENO_GTK2
void
xeno_style_realize (GtkStyle *style)
#else
void
xeno_realize_style (GtkStyle *style)
#endif
{
	GdkGCValues		gc_values;
	XenoColor		bg, light, dark, *white_p, *black_p;
	XenoRcData		*rc_data;
	XenoStyleData	*style_data;
	guint			order;
	
  #if !XENO_GTK2	
	extern GdkFont	*default_font;	/* exported from gtk/gtkstyle.c */
  #endif
	gint			i;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));

	rc_data = XENO_STYLE_RC_DATA(style);
  #if XENO_GTK2
    style_data = XENO_STYLE_DATA(style);
  #else
    style_data = NULL;
	if (rc_data)
		style->engine_data = style_data = xeno_style_data_new ();
  #endif
	
  #if XENO_GTK2
	xeno_style_parent_class->realize(style);
	
	style_data->font = gdk_font_from_description (style->font_desc);
	gc_values.font = style_data->font;
  #else
	if (!default_font)
		default_font = gdk_font_load ("-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*");
	
	if (style->font->type == GDK_FONT_FONT) {
		gc_values.font = style->font;
	} else if (style->font->type == GDK_FONT_FONTSET) {
		gc_values.font = default_font;
	}
  #endif
	order = ((XENO_STYLE_FONT_ASCENT(style) | 1) - 9) / 2;
	order = CLAMP(order, 0, 2);
	
	style_data->check_variant = order;
	style_data->radio_variant = order;
	style_data->menu_variant  = order;
	
	for (i = 0; i < 5; ++i) {
	  #if XENO_GTK2
		style_data->white_gc[i] = NULL;
		style_data->black_gc[i] = NULL;
	  #else
		xeno_color_from_gdk (&bg, &style->bg[i]);
		if (   style->bg_pixmap[i] != NULL
			&& style->bg_pixmap[i] != (GdkPixmap *)GDK_PARENT_RELATIVE
			&& gdk_color_equal (&style->bg[i], &style->rc_style->bg[i]))
		{
			xeno_color_from_pixmap (&bg, style->bg_pixmap[i]);
			xeno_color_to_gdk (&bg, &style->bg[i]);
		}
		
		if (style_data != NULL) {
	  #endif
			white_p = &style_data->white[i];
			black_p = &style_data->black[i];
			
			xeno_color_shade (&bg, rc_data->white[i], white_p);
			xeno_color_shade (&bg, rc_data->black[i], black_p);
			
			xeno_color_blend (&bg, white_p, rc_data->shine[i] - 1.0, &light);
			xeno_color_blend (&bg, black_p, 1.0 - rc_data->shade[i], &dark);
	  #if !XENO_GTK2
			xeno_realize_gc (&gc_values, style->colormap, white_p, NULL, &style_data->white_gc[i]);
			xeno_realize_gc (&gc_values, style->colormap, black_p, NULL, &style_data->black_gc[i]);
		} else {
			xeno_color_shade (&bg, XENO_DEFAULT_SHINE, &light);
			xeno_color_shade (&bg, XENO_DEFAULT_SHADE, &dark);
		}
		
		if (!xeno_theme_pseudocolor)
			xeno_color_blend (&light, &dark, 0.5, &bg);
		
		/* Allocate new gc's */
		xeno_realize_gc (&gc_values, style->colormap, &light, &style->light[i], &style->light_gc[i]);
		xeno_realize_gc (&gc_values, style->colormap, &dark,  &style->dark[i],  &style->dark_gc[i]);
		xeno_realize_gc (&gc_values, style->colormap, &bg,	  &style->mid[i],	&style->mid_gc[i]);
	  #endif
	}
	
	if (style_data != NULL) {
		xeno_realize_gc (&gc_values, style->colormap, NULL, &rc_data->focus_color, &style_data->focus_gc);
		xeno_gradient_set_realize (&style_data->gradient_set, style);
	}
}


#if XENO_GTK2
void
xeno_style_unrealize (GtkStyle *style)
{
	XenoStyleData	*style_data;
	gint i, variant;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	xeno_style_parent_class->unrealize(style);
	style_data = XENO_STYLE_DATA(style);
#else
void
xeno_style_data_destroy	(XenoStyleData *style_data)
{
	gint i, variant;
	
	g_return_if_fail (style_data != NULL);

  #if DEBUG
	style_count -= 1;
	g_print ("xeno_style_data_destroy(%d)\n", style_count);
  #endif	
#endif
	for (i = 0; i < XENO_STYLE_IMAGE_END; ++i) {
		if (style_data->pixmaps[i] != NULL) {
			if (i >= XENO_STYLE_IMAGE_CHECK_BUTTON_FIRST && i <= XENO_STYLE_IMAGE_CHECK_BUTTON_LAST) {
				variant = style_data->check_variant;
			} else if (i >= XENO_STYLE_IMAGE_RADIO_BUTTON_FIRST && i <= XENO_STYLE_IMAGE_RADIO_BUTTON_LAST) {
				variant = style_data->radio_variant;
			} else if (i >= XENO_STYLE_IMAGE_MENU_FIRST && i <= XENO_STYLE_IMAGE_MENU_LAST) {
				variant = style_data->menu_variant;
			} else {
				variant = 0;
			}
			
			xeno_pixmap_unref (style_data->pixmaps[i]);
			xeno_style_mask_unref (i, variant);
			style_data->pixmaps[i] = NULL;
		}
	}
	
	for (i = 0; i < 5; ++i) {
		if (style_data->white_gc[i]) {
			gtk_gc_release (style_data->white_gc[i]);
			style_data->white_gc[i] = NULL;
		}
		if (style_data->black_gc[i]) {
			gtk_gc_release (style_data->black_gc[i]);
			style_data->black_gc[i] = NULL;
		}
	}
	
	if (style_data->focus_gc) {
		gtk_gc_release (style_data->focus_gc);
		style_data->focus_gc = NULL;
	}
	xeno_gradient_set_unrealize (&style_data->gradient_set);
	
  #if !XENO_GTK2
	g_mem_chunk_free (xeno_style_data_chunk, style_data);
  #endif
}


#if !XENO_GTK2
/* The style classes */
XenoStyleClass xeno_style_classes[4] = {
	{
		{
			0,
			0,
			xeno_style_draw_hline,
			xeno_style_draw_vline,
			xeno_style_draw_shadow,
			xeno_style_draw_polygon,
			xeno_style_draw_arrow,
			xeno_style_draw_diamond,
			xeno_style_draw_oval,
			xeno_style_draw_string,
			xeno_style_draw_box,
			xeno_style_draw_flat_box,
			xeno_style_draw_check,
			xeno_style_draw_option,
			xeno_style_draw_cross,
			xeno_style_draw_ramp,
			xeno_style_draw_tab,
			xeno_style_draw_shadow_gap,
			xeno_style_draw_box_gap,
			xeno_style_draw_extension,
			xeno_style_draw_focus,
			xeno_style_draw_slider,
			xeno_style_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			1,
			1,
			xeno_style_draw_hline,
			xeno_style_draw_vline,
			xeno_style_draw_shadow,
			xeno_style_draw_polygon,
			xeno_style_draw_arrow,
			xeno_style_draw_diamond,
			xeno_style_draw_oval,
			xeno_style_draw_string,
			xeno_style_draw_box,
			xeno_style_draw_flat_box,
			xeno_style_draw_check,
			xeno_style_draw_option,
			xeno_style_draw_cross,
			xeno_style_draw_ramp,
			xeno_style_draw_tab,
			xeno_style_draw_shadow_gap,
			xeno_style_draw_box_gap,
			xeno_style_draw_extension,
			xeno_style_draw_focus,
			xeno_style_draw_slider,
			xeno_style_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			2,
			2,
			xeno_style_draw_hline,
			xeno_style_draw_vline,
			xeno_style_draw_shadow,
			xeno_style_draw_polygon,
			xeno_style_draw_arrow,
			xeno_style_draw_diamond,
			xeno_style_draw_oval,
			xeno_style_draw_string,
			xeno_style_draw_box,
			xeno_style_draw_flat_box,
			xeno_style_draw_check,
			xeno_style_draw_option,
			xeno_style_draw_cross,
			xeno_style_draw_ramp,
			xeno_style_draw_tab,
			xeno_style_draw_shadow_gap,
			xeno_style_draw_box_gap,
			xeno_style_draw_extension,
			xeno_style_draw_focus,
			xeno_style_draw_slider,
			xeno_style_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			3,
			3,
			xeno_style_draw_hline,
			xeno_style_draw_vline,
			xeno_style_draw_shadow,
			xeno_style_draw_polygon,
			xeno_style_draw_arrow,
			xeno_style_draw_diamond,
			xeno_style_draw_oval,
			xeno_style_draw_string,
			xeno_style_draw_box,
			xeno_style_draw_flat_box,
			xeno_style_draw_check,
			xeno_style_draw_option,
			xeno_style_draw_cross,
			xeno_style_draw_ramp,
			xeno_style_draw_tab,
			xeno_style_draw_shadow_gap,
			xeno_style_draw_box_gap,
			xeno_style_draw_extension,
			xeno_style_draw_focus,
			xeno_style_draw_slider,
			xeno_style_draw_handle
		},
		XENO_STYLE_MAGIC,
	}
};
#endif


/* end */

