/*
 *	Xenophilia GTK+ Theme Engine
 *
 *  xeno_theme_rc.c:
 *		XenoRcStyle, XenoRcData, parsing
 *
 *	Copyright  1999-2001 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_theme_rc.h"

#if XENO_GTK2
#include "xeno_theme_style.h"
#endif

/*
 *	Defaults
 */
#define XENO_DEFAULT_SHINE				1.5
#define XENO_DEFAULT_SHADE				0.6666667
#define XENO_DEFAULT_WHITE				2.0
#define XENO_DEFAULT_BLACK				0.0
#define XENO_DEFAULT_INSENSITIVE_SHINE	XENO_DEFAULT_SHINE
#define XENO_DEFAULT_INSENSITIVE_SHADE	XENO_DEFAULT_SHADE
#define XENO_DEFAULT_INSENSITIVE_WHITE	1.5
#define XENO_DEFAULT_INSENSITIVE_BLACK	0.5

#define XENO_DEFAULT_THICKNESS			2
#define XENO_MIN_THICKNESS				0
#define XENO_MAX_THICKNESS				3

#define XENO_DEFAULT_GRADIENT_FACTOR	1.0
#define XENO_DEFAULT_GRADIENT_GAMMA		0.41

#define XENO_DEFAULT_ORIGIN				XENO_ORIGIN_DEFAULT

#define XENO_DEFAULT_BUTTON_RELIEF		XENO_SHADOW_BLACK_OUT
#define XENO_DEFAULT_BUTTON_DEFAULT		XENO_SHADOW_XENO_IN
#define XENO_DEFAULT_SHADOW_MENU_ITEM	XENO_SHADOW_XENO_IN
#define XENO_DEFAULT_SHADOW_IN			XENO_SHADOW_XENO_IN
#define XENO_DEFAULT_SHADOW_OUT			XENO_SHADOW_XENO_OUT
#define XENO_DEFAULT_SHADOW_TEXT		XENO_SHADOW_NONE

#define XENO_DEFAULT_STEPPER_ENDS		FALSE
#define XENO_DEFAULT_STEPPER_ARROWS		FALSE
#define XENO_DEFAULT_STEPPER_BOX		TRUE
#define XENO_DEFAULT_SCROLLBAR_KNOB		XENO_KNOB_NONE
#define XENO_DEFAULT_SCROLLBAR_WIDTH	15
#define XENO_DEFAULT_SCROLLBAR_FLUSH	FALSE
#define XENO_DEFAULT_FLAT_WINDOWS		FALSE
#define XENO_DEFAULT_POPUP_ARROWS		FALSE
#define XENO_DEFAULT_HANDLE_KNOB		XENO_KNOB_DEFAULT
#define XENO_DEFAULT_PANED_KNOB         XENO_KNOB_BOX

/*
 *	Prototypes
 */
/* = rhs */
static gint				xeno_parse_eq_int		(GScanner *scanner, GTokenType *token_p, gint min, gint max);
static gfloat			xeno_parse_eq_float		(GScanner *scanner, GTokenType *token_p, gfloat min, gfloat max);
static gboolean			xeno_parse_eq_boolean	(GScanner *scanner, GTokenType *token_p);

static XenoShadowType	xeno_parse_eq_shadow	(GScanner *scanner, GTokenType *token_p, XenoShadowType default_shadow);
static XenoKnobType		xeno_parse_eq_knob		(GScanner *scanner, GTokenType *token_p,
												 XenoKnobType true_knob, XenoKnobType false_knob,
												 XenoKnobType default_knob);

static GTokenType		xeno_parse_eq_gradient	(GScanner *scanner, XenoRcGradient *output);
static GTokenType		xeno_parse_eq_origin	(GScanner *scanner, XenoRcOrigin *output);

/* lhs[] */
static GtkStateType		xeno_parse_state_type	(GScanner *scanner, GTokenType *token_p);

static GTokenType		xeno_parse_shading		(GScanner *scanner, gfloat *output, gfloat min, gfloat max, gboolean diminish,
												 XenoConfigMask *config, XenoConfigMask mask);
static GTokenType		xeno_parse_knobs		(GScanner *scanner, guint8 *output,
												 XenoKnobType true_knob, XenoKnobType false_knob, XenoKnobType default_knob,
												 XenoConfigMask *config, XenoConfigMask mask);
static GTokenType		xeno_parse_shadows		(GScanner *scanner, guint8 *output, XenoShadowType default_shadow,
												 XenoConfigMask *config, XenoConfigMask mask);
static GTokenType		xeno_parse_gradients	(GScanner *scanner, XenoRcGradient *output,
												 XenoConfigMask *config, XenoConfigMask mask);
static GTokenType		xeno_parse_origins		(GScanner *scanner, XenoRcOrigin *output,
												 XenoConfigMask *config, XenoConfigMask mask);

#if XENO_GTK2
static void				xeno_rc_style_class_init	(GtkRcStyleClass *klass);

static void				xeno_rc_style_init			(XenoRcStyle	*rc_style);
static GtkStyle *		xeno_rc_style_create_style	(GtkRcStyle		*rc_style);
static void				xeno_rc_style_merge			(GtkRcStyle		*dst_rc_style,
													 GtkRcStyle		*src_rc_style);
static guint			xeno_rc_style_parse			(GtkRcStyle		*rc_style,
													 GScanner		*scanner);
#endif

/*
 *	Globals
 */
#if !XENO_GTK2
GMemChunk * xeno_rc_data_chunk = NULL;
#endif

/*
 *	Parsing
 */
enum {
	XENO_TOKEN_INVALID = GTK_RC_TOKEN_LAST,

	XENO_TOKEN_TRUE,
	XENO_TOKEN_FALSE,
	
	XENO_TOKEN_NORMAL,
	XENO_TOKEN_PRELIGHT,
	XENO_TOKEN_ACTIVE,
	XENO_TOKEN_SELECTED,
	XENO_TOKEN_INSENSITIVE,

	XENO_TOKEN_SHINE,
	XENO_TOKEN_SHADE,
	XENO_TOKEN_WHITE,
	XENO_TOKEN_BLACK,

	XENO_TOKEN_DIMPLE,
	XENO_TOKEN_LINES,
	XENO_TOKEN_HOLES,
	XENO_TOKEN_BUDS,
	XENO_TOKEN_BOX,
	
	XENO_TOKEN_STEPPER_ENDS,
	XENO_TOKEN_STEPPER_ARROWS,
	XENO_TOKEN_STEPPER_BOX,
	XENO_TOKEN_SCROLLBAR_WIDTH,
	XENO_TOKEN_SCROLLBAR_FLUSH,
	XENO_TOKEN_SCROLLBAR_KNOB,
	XENO_TOKEN_PANED_KNOB,
	XENO_TOKEN_HANDLE_KNOB,
	XENO_TOKEN_POPUP_ARROWS,
	XENO_TOKEN_MENU_ITEM_SHADOW,
	XENO_TOKEN_FLAT_WINDOWS,
	XENO_TOKEN_THICKNESS,
	
	XENO_TOKEN_DEFAULT,
	XENO_TOKEN_ETCHED,
	XENO_TOKEN_THIN,
	XENO_TOKEN_NEXT,
	XENO_TOKEN_NOTE,
	XENO_TOKEN_IN,
	XENO_TOKEN_OUT,
	XENO_TOKEN_TEXT_SHADOW,
	
	XENO_TOKEN_BG_STYLE,
	XENO_TOKEN_GRADIENT,
	XENO_TOKEN_VERTICAL,
	XENO_TOKEN_HORIZONTAL,
	XENO_TOKEN_DIAGONAL,
	XENO_TOKEN_INTERLACED,
	
	XENO_TOKEN_ORIGIN,
	XENO_TOKEN_TOPLEVEL,
	XENO_TOKEN_ROOT,
	
	XENO_TOKEN_NO_NEXT_SHADOWS,
	XENO_TOKEN_INV_FLAT_WINDOWS,
	XENO_TOKEN_IGNORE,
	
	XENO_TOKEN_LAST,
	
	/* aliases */
	XENO_TOKEN_FIRST = XENO_TOKEN_TRUE
};

typedef struct {
	const gchar *name;
	GTokenType	token;
} XenoSymbol;

/* ordered so that they can be looked up by token */
const static XenoSymbol xeno_symbols[] = {
	{ "true",				XENO_TOKEN_TRUE				},
	{ "false",				XENO_TOKEN_FALSE			},

	{ "NORMAL",				XENO_TOKEN_NORMAL			},
	{ "PRELIGHT",			XENO_TOKEN_PRELIGHT			},
	{ "ACTIVE",				XENO_TOKEN_ACTIVE			},
	{ "SELECTED",			XENO_TOKEN_SELECTED			},
	{ "INSENSITIVE",		XENO_TOKEN_INSENSITIVE		},

	{ "shine",				XENO_TOKEN_SHINE			},
	{ "shade",				XENO_TOKEN_SHADE			},
	{ "white",				XENO_TOKEN_WHITE			},
	{ "black",				XENO_TOKEN_BLACK			},

	{ "dimple",				XENO_TOKEN_DIMPLE			},
	{ "lines",				XENO_TOKEN_LINES			},
	{ "holes",				XENO_TOKEN_HOLES			},
	{ "buds",				XENO_TOKEN_BUDS				},
	{ "box",				XENO_TOKEN_BOX				},
	
	{ "stepper_ends",		XENO_TOKEN_STEPPER_ENDS		},
	{ "stepper_arrows",		XENO_TOKEN_STEPPER_ARROWS	},
	{ "stepper_box",		XENO_TOKEN_STEPPER_BOX		},
	{ "scrollbar_width",	XENO_TOKEN_SCROLLBAR_WIDTH	},
	{ "scrollbar_flush",	XENO_TOKEN_SCROLLBAR_FLUSH	},
	{ "scrollbar_knob",		XENO_TOKEN_SCROLLBAR_KNOB	},
	{ "paned_knob",			XENO_TOKEN_PANED_KNOB		},
	{ "handle_knob",		XENO_TOKEN_HANDLE_KNOB		},
	{ "popup_arrows",		XENO_TOKEN_POPUP_ARROWS		},
	{ "menu_item_shadow",	XENO_TOKEN_MENU_ITEM_SHADOW },
	{ "flat_windows",		XENO_TOKEN_FLAT_WINDOWS		},
	{ "thickness",			XENO_TOKEN_THICKNESS		},
	
	{ "default",			XENO_TOKEN_DEFAULT			},
	{ "etched",				XENO_TOKEN_ETCHED			},
	{ "thin",				XENO_TOKEN_THIN				},
	{ "next",				XENO_TOKEN_NEXT				},
	{ "note",				XENO_TOKEN_NOTE				},
	{ "in",					XENO_TOKEN_IN				},
	{ "out",				XENO_TOKEN_OUT				},
	{ "label",				XENO_TOKEN_TEXT_SHADOW		},
	
	{ "bg_style",			XENO_TOKEN_BG_STYLE			},
	{ "gradient",			XENO_TOKEN_GRADIENT			},
	{ "vertical",			XENO_TOKEN_VERTICAL			},
	{ "horizontal",			XENO_TOKEN_HORIZONTAL		},
	{ "diagonal",			XENO_TOKEN_DIAGONAL			},
	{ "interlaced",			XENO_TOKEN_INTERLACED		},
	
	{ "align",				XENO_TOKEN_ORIGIN			},
	{ "window",				XENO_TOKEN_TOPLEVEL			},
	{ "root",				XENO_TOKEN_ROOT				},
	
	{ "use_xeno_shadows",	XENO_TOKEN_NO_NEXT_SHADOWS	},
	{ "window_shadow",		XENO_TOKEN_INV_FLAT_WINDOWS },
	{ "tooltip_shadow",		XENO_TOKEN_IGNORE			},
	
	/* aliases */
	{ "TRUE",				XENO_TOKEN_TRUE				},
	{ "FALSE",				XENO_TOKEN_FALSE			},
	{ "none",				XENO_TOKEN_FALSE			},
	{ "yes",				XENO_TOKEN_TRUE				},
	{ "no",					XENO_TOKEN_FALSE			},
	{ "y",					XENO_TOKEN_TRUE				},
	{ "n",					XENO_TOKEN_FALSE			},
	{ "on",					XENO_TOKEN_TRUE				},
	{ "off",				XENO_TOKEN_FALSE			},
	
	{ "scrollbar_dimple",	XENO_TOKEN_SCROLLBAR_KNOB	},
	{ "menu_style",			XENO_TOKEN_MENU_ITEM_SHADOW },
	{ "optionmenu_arrow",	XENO_TOKEN_POPUP_ARROWS		},
	{ "menuitem_shadow",	XENO_TOKEN_MENU_ITEM_SHADOW },
	{ "button_default",		XENO_TOKEN_DEFAULT			},
	{ "buttondef_shadow",	XENO_TOKEN_DEFAULT			},
	{ "use_xeno_text",		XENO_TOKEN_TEXT_SHADOW		},
	{ "menubar_shadow",		XENO_TOKEN_IGNORE			},
	{ "windows_combobox",	XENO_TOKEN_IGNORE			}
};

#define XENO_NSYMBOLS	(sizeof(xeno_symbols) / sizeof(XenoSymbol))

/* equal sign */
static GTokenType
xeno_parse_equal_sign (GScanner *scanner)
{
	GTokenType token;
	
	if ((token = g_scanner_peek_next_token(scanner)) == G_TOKEN_EQUAL_SIGN) {
		g_scanner_get_next_token(scanner);
		token = G_TOKEN_NONE;
	}
	
	return token;
}

static gint
xeno_parse_eq_ignore (GScanner *scanner)
{
	GTokenType token;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		return token;
	
	g_scanner_get_next_token (scanner);
	return G_TOKEN_NONE;
}

/* rhs */
static gint
xeno_parse_eq_int (GScanner *scanner, GTokenType *token_p, gint min, gint max)
{
	GTokenType token;
	gint i;

	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		goto end;
	
	token = g_scanner_get_next_token (scanner);
	if (token == G_TOKEN_INT) {
		i = scanner->value.v_int;
	} else if (token == G_TOKEN_FLOAT) {
		i = (gint)scanner->value.v_float;
	} else if (token == XENO_TOKEN_TRUE) {
		i = 1;
	} else if (token == XENO_TOKEN_FALSE) {
		i = 0;
	} else {
		goto end;
	}
	i = CLAMP(i, min, max);
	token = G_TOKEN_NONE;
	
  end:
	*token_p = token;
	
	return i;
}

static gfloat
xeno_parse_eq_float (GScanner *scanner, GTokenType *token_p, gfloat min, gfloat max)
{
	GTokenType token;
	gfloat ret = min;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		goto end;
	
	token = g_scanner_get_next_token(scanner);
	if (token == G_TOKEN_FLOAT) {
		ret = scanner->value.v_float;
	} else if (token == G_TOKEN_INT) {
		ret = scanner->value.v_int;
	} else {
		g_warning ("parse error\n");
		goto end;
	}
	ret = CLAMP(ret, min, max);
	token = G_TOKEN_NONE;
	
  end:
	*token_p = token;
	
	return ret;
}

static gboolean
xeno_parse_eq_boolean (GScanner *scanner, GTokenType *token_p)
{
	gboolean ret;
	
	if ((*token_p = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		return FALSE;
	
	switch (g_scanner_get_next_token(scanner)) {
	  case G_TOKEN_INT:
		ret = scanner->value.v_int > 0 ? TRUE : FALSE;
		break;
	  
	  case XENO_TOKEN_TRUE:
		ret = TRUE;
		break;
	  
	  case XENO_TOKEN_FALSE:
	  default:
		ret = FALSE;
		break;
	}
	
	*token_p = G_TOKEN_NONE;
	return ret;
}

static XenoShadowType
xeno_parse_eq_shadow (GScanner *scanner, GTokenType *token_p,
					  XenoShadowType default_shadow)
{
	GTokenType token;
	XenoShadowType shadow_type = XENO_SHADOW_NONE;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		goto end;
	
	switch ((token = g_scanner_peek_next_token(scanner))) {
	  case XENO_TOKEN_FALSE:
		shadow_type = XENO_SHADOW_NONE;
		goto done;
	  case XENO_TOKEN_IN:
		shadow_type = XENO_SHADOW_XENO_IN;
		goto done;
	  case XENO_TOKEN_OUT:
		shadow_type = XENO_SHADOW_XENO_OUT;
		goto done;
	  
	  case XENO_TOKEN_TRUE:
		shadow_type = default_shadow;
		break;
	  case XENO_TOKEN_ETCHED:
		shadow_type = XENO_SHADOW_ETCHED_IN;
		break;
	  case XENO_TOKEN_DEFAULT:
		shadow_type = XENO_SHADOW_XENO_IN;
		break;
	  case XENO_TOKEN_THIN:
		shadow_type = XENO_SHADOW_THIN_IN;
		break;
	  case XENO_TOKEN_NEXT:
		shadow_type = XENO_SHADOW_NEXT_IN;
		break;
	  case XENO_TOKEN_NOTE:
	    shadow_type = XENO_SHADOW_NOTE_IN;
		break;
	  case XENO_TOKEN_BLACK:
		shadow_type = XENO_SHADOW_BLACK_IN;
		break;
	
	  default:
		goto end;
	}
	g_scanner_get_next_token(scanner);
	
	token = g_scanner_peek_next_token(scanner);
	if (token == XENO_TOKEN_OUT) {
		shadow_type += 1;
	} else if (token != XENO_TOKEN_IN) {
		if (default_shadow == XENO_SHADOW_NONE)
			goto end;
		
		if ((default_shadow & 1) == 0)
			shadow_type += 1;
		
		goto done2;
	}
	
  done:
	g_scanner_get_next_token(scanner);
  done2:
	token = G_TOKEN_NONE;
  end:
	*token_p = token;
	return shadow_type;
}

/* FIXME: extended knob types */
static XenoKnobType
xeno_parse_eq_knob (GScanner *scanner, GTokenType *token_p,
					XenoKnobType true_value, XenoKnobType false_value,
					XenoKnobType default_value)
{
	GTokenType token;
	XenoKnobType knob_type = false_value;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		goto end;
	
	switch ((token = g_scanner_peek_next_token(scanner))) {
	  case XENO_TOKEN_TRUE:
		knob_type = true_value;
		break;
	  case XENO_TOKEN_FALSE:
		knob_type = false_value;
		break;
	  case XENO_TOKEN_DEFAULT:
		knob_type = default_value;
		break;

	  case XENO_TOKEN_DIMPLE:
		knob_type = XENO_KNOB_DIMPLE;
		break;
	  case XENO_TOKEN_LINES:
		knob_type = XENO_KNOB_LINES;
		break;
	  case XENO_TOKEN_HOLES:
		knob_type = XENO_KNOB_HOLES;
		break;
	  case XENO_TOKEN_BUDS:
		knob_type = XENO_KNOB_BUDS;
		break;
	  case XENO_TOKEN_BOX:
		knob_type = XENO_KNOB_BOX;
		break;
	  default:
		goto end;
	}
	g_scanner_get_next_token(scanner);
	token = G_TOKEN_NONE;

  end:
	*token_p = token;
	return knob_type;
}

static GTokenType
xeno_parse_eq_gradient (GScanner *scanner, XenoRcGradient *output)
{
	XenoGradientType gradient_type;
	GTokenType token;
	gfloat t;
	
	output->type = XENO_GRADIENT_NONE;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		return token;
	
	token = g_scanner_peek_next_token (scanner);
	if (token == XENO_TOKEN_GRADIENT) {
		g_scanner_get_next_token (scanner);
		token = g_scanner_peek_next_token (scanner);
	}
	switch (token) {
	  case XENO_TOKEN_FALSE:
		gradient_type = XENO_GRADIENT_NONE;
		break;
	  case XENO_TOKEN_HORIZONTAL:
		gradient_type = XENO_GRADIENT_HORIZONTAL;
		break;
	  case XENO_TOKEN_VERTICAL:
		gradient_type = XENO_GRADIENT_VERTICAL;
		break;
	  case XENO_TOKEN_DIAGONAL:
		gradient_type = XENO_GRADIENT_DIAGONAL;
		break;
	  default:
		return token;
	}
	g_scanner_get_next_token (scanner);
	
	output->type = gradient_type;
	if (gradient_type != XENO_GRADIENT_NONE) {
		token = g_scanner_peek_next_token (scanner);
		
		output->interlaced = FALSE;
		if (token == XENO_TOKEN_INTERLACED) {
			output->interlaced = TRUE;
			g_scanner_get_next_token (scanner);
			token = g_scanner_peek_next_token (scanner);
		}
		if (token != G_TOKEN_FLOAT)
			return token;
		
		g_scanner_get_next_token (scanner);
		output->factor = CLAMP(scanner->value.v_float, 0.0, 2.0);
		
		if ((token = g_scanner_peek_next_token (scanner)) == G_TOKEN_FLOAT)
			g_scanner_get_next_token (scanner);
	}
	return G_TOKEN_NONE;
}

static GTokenType
xeno_parse_eq_origin (GScanner *scanner, XenoRcOrigin *output)
{
	XenoOriginType origin_type;
	GTokenType token;
	gfloat x, y;
	
	if ((token = xeno_parse_equal_sign(scanner)) != G_TOKEN_NONE)
		return token;
	
	switch ((token = g_scanner_peek_next_token(scanner))) {
	  case G_TOKEN_FLOAT:
		origin_type = XENO_ORIGIN_ALIGNMENT;
		g_scanner_get_next_token(scanner);
		x = CLAMP(scanner->value.v_float, 0.0, 1.0);
		
		g_scanner_peek_next_token(scanner);
		y = CLAMP(scanner->value.v_float, 0.0, 1.0);
		break;
		
	  case XENO_TOKEN_TOPLEVEL:
		origin_type = XENO_ORIGIN_TOPLEVEL;
		break;
		
	  case XENO_TOKEN_ROOT:
		origin_type = XENO_ORIGIN_ROOT;
		break;
		
	  case XENO_TOKEN_DEFAULT:
		origin_type = XENO_ORIGIN_DEFAULT;
		x = 0.0;
		y = 0.0;
		break;
		
	  default:
		output->type = XENO_ORIGIN_DEFAULT;
		return token;
	}
	g_scanner_get_next_token(scanner);
	
	output->type = origin_type;
	output->x = x;
	output->y = y;
	
	return G_TOKEN_NONE;
}

/* lhs */
static GtkStateType
xeno_parse_state_type (GScanner *scanner, GTokenType *token_p)
{
	GtkStateType state_type;
	
	g_scanner_get_next_token(scanner); /* [ */
	switch (*token_p = g_scanner_peek_next_token(scanner)) {
	  case XENO_TOKEN_NORMAL: state_type = GTK_STATE_NORMAL; break;
	  case XENO_TOKEN_ACTIVE: state_type = GTK_STATE_ACTIVE; break;
	  case XENO_TOKEN_PRELIGHT: state_type = GTK_STATE_PRELIGHT; break;
	  case XENO_TOKEN_SELECTED: state_type = GTK_STATE_SELECTED; break;
	  case XENO_TOKEN_INSENSITIVE: state_type = GTK_STATE_INSENSITIVE; break;
	  default:
		
		return GTK_STATE_NORMAL;
	}
	g_scanner_get_next_token(scanner); /* state */
	
	if (g_scanner_peek_next_token(scanner) == G_TOKEN_RIGHT_BRACE) {
		g_scanner_get_next_token(scanner);
		*token_p = G_TOKEN_NONE;
	}
	
	return state_type;
}

/* parse variables */
static GTokenType
xeno_parse_shading (GScanner *scanner, gfloat *output,
					gfloat min, gfloat max, gboolean diminish,
					XenoConfigMask *config, XenoConfigMask mask)
{
	GTokenType token;
	gint state_type;
	
	token = g_scanner_peek_next_token (scanner);
	if (token == G_TOKEN_LEFT_BRACE) {
		state_type = xeno_parse_state_type (scanner, &token);
		if (token == G_TOKEN_NONE) {
			output[state_type] = xeno_parse_eq_float (scanner, &token, min, max);
			*config |= mask << state_type;
		}
	} else {
		gfloat t = xeno_parse_eq_float (scanner, &token, min, max);
        output[0] = t;
        output[1] = t;
        output[2] = t;
        output[3] = t;
		*config |= mask * 31; /* _not_ GTK_STATE_INSENSITIVE, tested below */
	}
	
	/* if GTK_STATE_NORMAL is set but not GTK_STATE_INSENSIIVE */
	if (diminish) {
		if ((((*config & (mask | (mask<<4))) == mask)))
			output[4] = (output[0] + 1.0) / 2.0;
		*config |= mask<<4;
	}
	
	
	return token;
}

static GTokenType
xeno_parse_knobs (GScanner *scanner, guint8 *output,
				  XenoKnobType true_knob, XenoKnobType false_knob, XenoKnobType default_knob,
				  XenoConfigMask *config, XenoConfigMask mask)
{
	GtkStateType state_type;
	XenoKnobType knob_type;
	GTokenType token;
	
	token = g_scanner_peek_next_token(scanner);
	if (token == G_TOKEN_LEFT_BRACE) {
		state_type = xeno_parse_state_type(scanner, &token);
		if (token != G_TOKEN_NONE)
			goto end;
		
		knob_type = xeno_parse_eq_knob (scanner, &token, true_knob, false_knob, default_knob);
		output[state_type] = knob_type;
		*config |= mask << state_type;
	} else if (token == G_TOKEN_EQUAL_SIGN) {
		knob_type = xeno_parse_eq_knob (scanner, &token, true_knob, false_knob, default_knob);
		output[0] = knob_type;
		output[1] = knob_type;
		output[2] = knob_type;
		output[3] = knob_type;
		output[4] = knob_type;
		*config |= mask * 31;
	}
  end:
	return token;
}

static GTokenType
xeno_parse_shadows (GScanner *scanner, guint8 *output,
					XenoShadowType default_shadow,
					XenoConfigMask *config, XenoConfigMask mask)
{
	GtkStateType state_type;
	XenoShadowType shadow_type;
	GTokenType token;
	
	token = g_scanner_peek_next_token(scanner);
	if (token == G_TOKEN_LEFT_BRACE) {
		state_type = xeno_parse_state_type(scanner, &token);
		if (token != G_TOKEN_NONE)
			goto end;
		
		shadow_type = xeno_parse_eq_shadow (scanner, &token, default_shadow);
		output[state_type] = shadow_type;
		*config |= mask << state_type;
	} else if (token == G_TOKEN_EQUAL_SIGN) {
		shadow_type = xeno_parse_eq_shadow (scanner, &token, default_shadow);
		output[0] = shadow_type;
		output[1] = shadow_type;
		output[2] = shadow_type;
		output[3] = shadow_type;
		output[4] = shadow_type;
		*config |= mask * 31;
	}
  end:
	return token;
}

static GTokenType
xeno_parse_origins (GScanner *scanner, XenoRcOrigin *output,
					XenoConfigMask *config, XenoConfigMask mask)
{
	GtkStateType state_type;
	GTokenType token;
	
	token = g_scanner_peek_next_token (scanner);
	if (token == G_TOKEN_LEFT_BRACE) {
		state_type = xeno_parse_state_type (scanner, &token);
		if (token != G_TOKEN_NONE);
			goto end;
		
		token = xeno_parse_eq_origin (scanner, &output[state_type]);
		*config |= mask << state_type;
	} else if (token == G_TOKEN_EQUAL_SIGN) {
		XenoRcOrigin rc_origin;
		token = xeno_parse_eq_origin (scanner, &rc_origin);
		for (state_type = 0; state_type < 5; ++state_type)
			output[state_type] = rc_origin;
		*config |= mask * 31;
	}
  end:
	return token;
}

/*
	bg_style =>			"bg_style" gradient_rhs
	gradient =>			"gradient" gradient_rhs
	gradient_rhs =>		['['state_type']'] '=' ["gradient" ]("horizontal"|"vertical"|'diagonal") factor bias
	factor =>			float
	bias =>				float
*/
static GTokenType
xeno_parse_gradients (GScanner *scanner, XenoRcGradient *output,
					  XenoConfigMask *config, XenoConfigMask mask)
{
	GtkStateType state_type;
	GTokenType token;
	
	token = g_scanner_peek_next_token (scanner);
	if (token == G_TOKEN_LEFT_BRACE) {
		state_type = xeno_parse_state_type (scanner, &token);
		if (token != G_TOKEN_NONE)
			goto end;
		
		token = xeno_parse_eq_gradient (scanner, &output[state_type]);
		*config |= mask << state_type;
	} else if (token == G_TOKEN_EQUAL_SIGN) {
		XenoRcGradient rc_gradient;
		token = xeno_parse_eq_gradient(scanner, &rc_gradient);
		for (state_type = 0; state_type < 5; ++state_type)
			output[state_type] = rc_gradient;
		*config |= mask * 31;
	}
  end:
	return token;
}

#if XENO_GTK2
static guint
xeno_rc_style_parse (GtkRcStyle *rc_style, GScanner *scanner)
{
	XenoRcData		*rc_data;
#else
GTokenType
xeno_rc_data_parse (XenoRcData *rc_data, GScanner *scanner)
{
#endif
	static GQuark	scope = 0;
	GQuark			old_scope;
	GTokenType		token;
	gint			i, t;

	g_return_val_if_fail (rc_data != NULL, G_TOKEN_NONE);
	g_return_val_if_fail (scanner != NULL, G_TOKEN_NONE);
  #if XENO_GTK2
	g_return_val_if_fail (XENO_IS_RC_STYLE(rc_style), G_TOKEN_NONE);
	
	rc_data = XENO_RC_STYLE(rc_style);
  #endif
	
	if (!scope)
		scope = g_quark_from_string("theme_engine");
	old_scope = g_scanner_set_scope(scanner, scope);
	
	/* add keywords */
	if (!g_scanner_lookup_symbol(scanner, xeno_symbols[0].name)) {
		g_scanner_freeze_symbol_table(scanner);
		for (i=0; i<XENO_NSYMBOLS; i++)
			g_scanner_scope_add_symbol (scanner, scope, xeno_symbols[i].name,
										GUINT_TO_POINTER(xeno_symbols[i].token));
		g_scanner_thaw_symbol_table(scanner);
	}
	
	/* parse
		FIXME: ensure correct error message for parse errors
	*/
	token = g_scanner_get_next_token (scanner);
	while (token != G_TOKEN_RIGHT_CURLY) {
		switch (token) {
		  case XENO_TOKEN_STEPPER_ENDS:
			rc_data->stepper_ends = xeno_parse_eq_boolean(scanner, &token);
			rc_data->config |= XENO_CONFIG_STEPPER_ENDS;
			break;
		  
		  case XENO_TOKEN_STEPPER_ARROWS:
			rc_data->stepper_arrows = xeno_parse_eq_boolean(scanner, &token);
			rc_data->config |= XENO_CONFIG_STEPPER_ARROWS;
			if (!(rc_data->config & XENO_CONFIG_STEPPER_BOX))
				rc_data->stepper_box = !(rc_data->stepper_arrows);
			break;

		  case XENO_TOKEN_STEPPER_BOX:
			rc_data->stepper_box = xeno_parse_eq_boolean(scanner, &token);
			rc_data->config |= XENO_CONFIG_STEPPER_BOX;
			break;

		  case XENO_TOKEN_SCROLLBAR_WIDTH:
			rc_data->scrollbar_width = xeno_parse_eq_int (scanner, &token, 4, 31);
			rc_data->config |= XENO_CONFIG_SCROLLBAR_WIDTH;
			break;

		  case XENO_TOKEN_SCROLLBAR_KNOB:
			token = xeno_parse_knobs (scanner, &rc_data->scrollbar_knobs[0],
									  XENO_KNOB_DIMPLE, XENO_KNOB_NONE, XENO_KNOB_NONE,
									  &rc_data->config_knobs, XENO_CONFIG_KNOB_SCROLLBAR);
			break;
		
		  case XENO_TOKEN_PANED_KNOB:
			token = xeno_parse_knobs (scanner, &rc_data->paned_knobs[0],
									  XENO_KNOB_BOX, XENO_KNOB_NONE, XENO_KNOB_BOX,
									  &rc_data->config_knobs, XENO_CONFIG_KNOB_PANED);
			break;
		
		  case XENO_TOKEN_HANDLE_KNOB:
			rc_data->handle_knob = xeno_parse_eq_knob (scanner, &token,
													XENO_KNOB_BUDS, XENO_KNOB_NONE, XENO_KNOB_BUDS);
			rc_data->config_knobs |= XENO_CONFIG_KNOB_HANDLE;
			break;
		  
		  case XENO_TOKEN_SHINE:
			token = xeno_parse_shading (scanner, &rc_data->shine[0], 1.0, 2.0, FALSE,
										&rc_data->chrome, XENO_CHROME_SHINE);
		    break;

		  case XENO_TOKEN_SHADE:
			token = xeno_parse_shading (scanner, &rc_data->shade[0], 0.0, 1.0, FALSE,
										&rc_data->chrome, XENO_CHROME_SHADE);
			break;
			
		  case XENO_TOKEN_WHITE:
			token = xeno_parse_shading (scanner, &rc_data->white[0], 1.0, 2.0, TRUE,
										&rc_data->chrome, XENO_CHROME_WHITE);
			break;
			
		  case XENO_TOKEN_BLACK:
			token = xeno_parse_shading (scanner, &rc_data->black[0], 0.0, 1.0, TRUE,
										&rc_data->chrome, XENO_CHROME_BLACK);
			break;
			
		  case XENO_TOKEN_THIN:
			if (xeno_parse_eq_boolean(scanner, &token))
				rc_data->thickness = 1;
			rc_data->config |= XENO_CONFIG_THICKNESS;
			break;
		
		  case XENO_TOKEN_IN:
			token = xeno_parse_shadows (scanner, &rc_data->shadow_in[0], XENO_SHADOW_XENO_IN,
										&rc_data->config_shadows, XENO_CONFIG_SHADOW_IN);
			break;
			
		  case XENO_TOKEN_OUT:
			token = xeno_parse_shadows (scanner, &rc_data->shadow_out[0], XENO_SHADOW_XENO_OUT,
										&rc_data->config_shadows, XENO_CONFIG_SHADOW_OUT);
			break;
			
		  case XENO_TOKEN_TEXT_SHADOW:
			token = xeno_parse_shadows (scanner, &rc_data->shadow_text[0], XENO_SHADOW_OUT,
										&rc_data->config_shadows, XENO_CONFIG_SHADOW_TEXT);
			break;
			
		  case XENO_TOKEN_SCROLLBAR_FLUSH:
			rc_data->scrollbar_flush = xeno_parse_eq_boolean(scanner, &token);
			rc_data->config |= XENO_CONFIG_SCROLLBAR_FLUSH;
			break;

		  case XENO_TOKEN_POPUP_ARROWS:
			rc_data->popup_arrows = xeno_parse_eq_boolean(scanner, &token);
			rc_data->config |= XENO_CONFIG_POPUP_ARROWS;
			break;
			
		  case XENO_TOKEN_FALSE: /* "none" */
			rc_data->shadow_button_relief = xeno_parse_eq_shadow (scanner, &token, XENO_SHADOW_OUT);
			rc_data->config_shadows |= XENO_CONFIG_SHADOW_BUTTON_RELIEF;
			break;
			
		  case XENO_TOKEN_DEFAULT:
			rc_data->shadow_button_default = xeno_parse_eq_shadow (scanner, &token, XENO_SHADOW_IN);
			rc_data->config_shadows |= XENO_CONFIG_SHADOW_BUTTON_DEFAULT;
			break;
			
		  case XENO_TOKEN_MENU_ITEM_SHADOW:
			rc_data->shadow_menu_item = xeno_parse_eq_shadow (scanner, &token, XENO_SHADOW_IN);
			rc_data->config_shadows |= XENO_CONFIG_SHADOW_MENU_ITEM;
			break;
		
		  case XENO_TOKEN_FLAT_WINDOWS:
			rc_data->flat_windows = xeno_parse_eq_boolean (scanner, &token);
			rc_data->config |= XENO_CONFIG_FLAT_WINDOWS;
			break;

		  case XENO_TOKEN_INV_FLAT_WINDOWS:
			rc_data->flat_windows = (xeno_parse_eq_boolean (scanner, &token)) ? FALSE : TRUE;
			rc_data->config |= XENO_CONFIG_FLAT_WINDOWS;
			break;
			
		  case XENO_TOKEN_BG_STYLE:
		  case XENO_TOKEN_GRADIENT:
			token = xeno_parse_gradients (scanner, &rc_data->gradient[0], &rc_data->chrome, XENO_CHROME_GRADIENT);
			break;
			
		  case XENO_TOKEN_ORIGIN:
			token = xeno_parse_origins (scanner, &rc_data->origin[0], &rc_data->chrome, XENO_CHROME_ORIGIN);
			break;
			
		  case XENO_TOKEN_THICKNESS:
			rc_data->thickness = xeno_parse_eq_int (scanner, &token, 0, 3);
			rc_data->config |= XENO_CONFIG_THICKNESS;
			break;
			
		  case XENO_TOKEN_NO_NEXT_SHADOWS: {
				gint in, out, i, mask = ((XENO_CONFIG_SHADOW_IN | XENO_CONFIG_SHADOW_OUT) * 31);
                if (xeno_parse_eq_boolean (scanner, &token)) {
					rc_data->config_shadows & ~mask;
					in	= XENO_SHADOW_XENO_IN;
					out	= XENO_SHADOW_XENO_OUT;
				} else {
					rc_data->config_shadows |= mask;
					in	= XENO_SHADOW_NEXT_IN;
					out	= XENO_SHADOW_NEXT_OUT;
				}
				for (i = 0; i < 5; ++i) {
					rc_data->shadow_in[i]	= in;
					rc_data->shadow_out[i]	= out;
				}
			} break;
		  
		  case XENO_TOKEN_IGNORE:
			token = xeno_parse_eq_ignore (scanner);
			break;
		
		  case G_TOKEN_RIGHT_CURLY:
			goto done;
			
		  default:
			token = G_TOKEN_RIGHT_CURLY;
		}
		
		if (token != G_TOKEN_NONE) {
			if (token >= XENO_TOKEN_FIRST && token < XENO_TOKEN_LAST) {
				g_warning ("parse error, expected \"%s\"\n", xeno_symbols[token - XENO_TOKEN_FIRST].name);
			} else {
				g_warning ("parse error\n");
			}
			return token;
		}
		
		token = g_scanner_get_next_token (scanner);
	}
  done:
	g_scanner_set_scope(scanner, old_scope);
	return G_TOKEN_NONE;
}


/*
 *	XenoRcData, XenoRcStyle
 */

#if XENO_GTK2
static void
xeno_rc_style_init (XenoRcStyle *rc_data)
{
	gint i;
	
	g_return_if_fail (rc_data != NULL);
	g_return_if_fail (XENO_IS_RC_STYLE(rc_data));
	
	{
#else
XenoRcData *
xeno_rc_data_new (void)
{
	XenoRcData *rc_data;
	gint i;
	
	if (xeno_rc_data_chunk == NULL) {
		if ((xeno_rc_data_chunk = g_mem_chunk_create (XenoRcData, 256, G_ALLOC_ONLY)) == NULL)
			return NULL;
	}
	
	rc_data = g_mem_chunk_alloc (xeno_rc_data_chunk);
	if (rc_data) {
		rc_data->ref_count			= 1;
#endif
		rc_data->thickness			= XENO_DEFAULT_THICKNESS;
		rc_data->stepper_ends		= XENO_DEFAULT_STEPPER_ENDS;
		rc_data->stepper_arrows		= XENO_DEFAULT_STEPPER_ARROWS;
		rc_data->stepper_box		= XENO_DEFAULT_STEPPER_BOX;
		rc_data->scrollbar_width	= XENO_DEFAULT_SCROLLBAR_WIDTH;
		rc_data->scrollbar_flush	= XENO_DEFAULT_SCROLLBAR_FLUSH;
		rc_data->flat_windows		= XENO_DEFAULT_FLAT_WINDOWS;
		rc_data->popup_arrows		= XENO_DEFAULT_POPUP_ARROWS;
		
		for (i=0; i<5; ++i) {
			xeno_rc_gradient_init (&rc_data->gradient[i]);
			xeno_rc_origin_init (&rc_data->origin[i]);
			
			rc_data->scrollbar_knobs[i]	= XENO_DEFAULT_SCROLLBAR_KNOB;
			rc_data->paned_knobs[i]		= XENO_DEFAULT_PANED_KNOB;
			
			rc_data->shadow_in[i]		= XENO_DEFAULT_SHADOW_IN;
			rc_data->shadow_out[i]		= XENO_DEFAULT_SHADOW_OUT;
			rc_data->shadow_text[i]		= XENO_DEFAULT_SHADOW_TEXT;
		}
		rc_data->handle_knob			= XENO_DEFAULT_HANDLE_KNOB;
		rc_data->shadow_button_relief	= XENO_DEFAULT_BUTTON_RELIEF;
		rc_data->shadow_button_default	= XENO_DEFAULT_BUTTON_DEFAULT;
		rc_data->shadow_menu_item		= XENO_DEFAULT_SHADOW_MENU_ITEM;
		
		for (i=0; i<4; ++i) {
			rc_data->shine[i] = XENO_DEFAULT_SHINE;
			rc_data->shade[i] = XENO_DEFAULT_SHADE;
			rc_data->white[i] = XENO_DEFAULT_WHITE;
			rc_data->black[i] = XENO_DEFAULT_BLACK;
		}
		rc_data->shine[4] = XENO_DEFAULT_INSENSITIVE_SHINE;
		rc_data->shade[4] = XENO_DEFAULT_INSENSITIVE_SHADE;
		rc_data->white[4] = XENO_DEFAULT_INSENSITIVE_WHITE;
		rc_data->black[4] = XENO_DEFAULT_INSENSITIVE_BLACK;
		
		rc_data->config = 0;
		rc_data->chrome = 0;
		rc_data->config_knobs = 0;
		rc_data->config_shadows = 0;
	}
  #if !XENO_GTK2
	return rc_data;
  #endif
}

#if !XENO_GTK2
void xeno_rc_data_destroy (XenoRcData *rc_data)
{
	g_return_if_fail (rc_data != NULL);
	g_return_if_fail (rc_data->ref_count == 0);
	
	g_mem_chunk_free (xeno_rc_data_chunk, rc_data);
}

XenoRcData *
xeno_rc_data_clone (XenoRcData *src)
{
	XenoRcData	*dst;
	
	g_return_val_if_fail (src != NULL, NULL);
	
	dst = g_mem_chunk_alloc (xeno_rc_data_chunk);
	if (dst) {
		*dst = *src;
		dst->ref_count = 1;
	}
	return dst;
}
#endif

#if XENO_GTK2
static void
xeno_rc_style_merge (GtkRcStyle *dest_rc_style, GtkRcStyle *src_rc_style)
{
	XenoRcData		*src_data;
	XenoRcData		*dest_data;
#else
void
xeno_rc_data_merge (XenoRcData *dest_data, XenoRcData *src_data)
{
#endif
	XenoConfigMask	config_diff;
	XenoConfigMask	chrome_diff;
	XenoConfigMask	knob_diff;
	XenoConfigMask	shadow_diff;
	guint i;
	
	g_return_if_fail (src_data != NULL);
	g_return_if_fail (dest_data != NULL);
  #if XENO_GTK2
	g_return_if_fail (XENO_IS_RC_STYLE(src_rc_style));
	g_return_if_fail (XENO_IS_RC_STYLE(dest_rc_style));
	
	src_data = XENO_RC_STYLE(src_rc_style);
	dest_data = XENO_RC_STYLE(dest_rc_style);
  #endif
	
	/* general configuration */
	config_diff = src_data->config & ~dest_data->config;
	if (config_diff) {
		if (config_diff & XENO_CONFIG_STEPPER_ENDS)
			dest_data->stepper_ends = src_data->stepper_ends;
		if (config_diff & XENO_CONFIG_STEPPER_ARROWS)
			dest_data->stepper_arrows = src_data->stepper_arrows;
		if (config_diff & XENO_CONFIG_STEPPER_BOX)
			dest_data->stepper_box = src_data->stepper_box;
		if (config_diff & XENO_CONFIG_SCROLLBAR_WIDTH)
			dest_data->scrollbar_width = src_data->scrollbar_width;
		if (config_diff & XENO_CONFIG_SCROLLBAR_FLUSH)
			dest_data->scrollbar_flush = src_data->scrollbar_flush;
		if (config_diff & XENO_CONFIG_POPUP_ARROWS)
			dest_data->popup_arrows = src_data->popup_arrows;
		if (shadow_diff & XENO_CONFIG_FLAT_WINDOWS)
			dest_data->flat_windows = src_data->flat_windows;
		if (config_diff & XENO_CONFIG_THICKNESS)
			dest_data->thickness = src_data->thickness;
		
		dest_data->config |= config_diff;
	}
	
	/* knobs */
	knob_diff = src_data->config_knobs & ~dest_data->config_knobs;
	if (knob_diff) {
		for (i=0; i<5; i++) {
			if (knob_diff & (XENO_CONFIG_KNOB_SCROLLBAR<<i))
				dest_data->scrollbar_knobs[i] = src_data->scrollbar_knobs[i];
			if (knob_diff & (XENO_CONFIG_KNOB_PANED<<i))
				dest_data->paned_knobs[i] = src_data->paned_knobs[i];
		}
		if (knob_diff & XENO_CONFIG_KNOB_HANDLE)
			dest_data->handle_knob = src_data->handle_knob;
		
		dest_data->config_knobs |= knob_diff;
	}

	/* shadows */
	shadow_diff	= src_data->config_shadows & ~dest_data->config_shadows;
	if (shadow_diff) {
		for (i=0; i<5; i++) {
			if (shadow_diff & (XENO_CONFIG_SHADOW_IN))
				dest_data->shadow_in[i] = src_data->shadow_in[i];
			if (shadow_diff & (XENO_CONFIG_SHADOW_OUT))
				dest_data->shadow_out[i] = src_data->shadow_out[i];
			if (shadow_diff & (XENO_CONFIG_SHADOW_TEXT))
				dest_data->shadow_text[i] = src_data->shadow_text[i];
		}
		if (shadow_diff & XENO_CONFIG_SHADOW_MENU_ITEM)
			dest_data->shadow_menu_item = src_data->shadow_menu_item;
		if (shadow_diff & XENO_CONFIG_SHADOW_BUTTON_RELIEF)
			dest_data->shadow_button_relief = src_data->shadow_button_relief;
		if (shadow_diff & XENO_CONFIG_SHADOW_BUTTON_DEFAULT)
			dest_data->shadow_button_default = src_data->shadow_button_default;
		
		dest_data->config_shadows |= shadow_diff;
	}
	
	/* chrome, yeah, stupid name I know */
	chrome_diff = src_data->chrome & ~dest_data->chrome;
	if (chrome_diff) {
		for (i=0; i<5; i++) {
			if (chrome_diff & (XENO_CHROME_SHINE<<i))
				dest_data->shine[i] = src_data->shine[i];
			if (chrome_diff & (XENO_CHROME_SHADE<<i))
				dest_data->shade[i] = src_data->shade[i];
			if (chrome_diff & (XENO_CHROME_WHITE<<i))
				dest_data->white[i] = src_data->white[i];
			if (chrome_diff & (XENO_CHROME_BLACK<<i))
				dest_data->black[i] = src_data->black[i];
			
			if (chrome_diff & (XENO_CHROME_GRADIENT<<i))
				dest_data->gradient[i] = src_data->gradient[i];
			
			if (chrome_diff & (XENO_CHROME_ORIGIN<<i))
				dest_data->origin[i] = src_data->origin[i];
		}
		dest_data->chrome |= chrome_diff;
	}
}

#if XENO_GTK2
/*
 *	XenoRcStyle (GTK 2.0)
 */

GType xeno_rc_style_get_type ()
{
	static GType rc_style_type = 0;
	
	if (!rc_style_type) {
		static const GTypeInfo rc_style_info = {
			sizeof(XenoRcStyleClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) xeno_rc_style_class_init,
			NULL,
			NULL,
			sizeof(XenoRcStyle),
			4, /* n_preallocs */
			(GInstanceInitFunc) xeno_rc_style_init,
		};
		rc_style_type = g_type_register_static (GTK_TYPE_RC_STYLE, "XenoRcStyle", &rc_style_info, 0);
	}
	return rc_style_type;
}

static void
xeno_rc_style_class_init (GtkRcStyleClass *klass) {
	g_return_if_fail (klass != NULL);
	g_return_if_fail (XENO_IS_RC_STYLE_CLASS(klass));
	
	/*
	klass->parse		= xeno_rc_style_parse;
	klass->merge		= xeno_rc_style_merge;
	*/
	klass->create_style = xeno_rc_style_create_style;
}

static GtkStyle *
xeno_rc_style_create_style (GtkRcStyle *rc_style)
{
	return xeno_style_new ();
}

#endif /* XENO_GTK2 */

/* end */

