// bond.cpp - Bond's implementation of functions

#include <qobject.h>
#include <qrect.h>
#include <iostream.h>
#include <math.h>

#include "render2d.h"
#include "drawable.h"
#include "bond.h"
#include "defs.h"
#include "bondedit.h"

Bond::Bond(Render2D *r1, QObject *parent, const char *name)
  : Drawable(parent, name)
{
  r = r1;
  order = 1;
  dashed = 0;
  highlighted = false;
  thick = 1;
  wside = 0;
}

bool Bond::Equals(Bond *a) {
  if ( (start == a->Start()) && (end == a->End()) ) return true;
  if ( (start == a->End()) && (end == a->Start()) ) return true;
  return false;
}

bool Bond::WithinRect(QRect n, bool shiftdown) {
  if ( DPointInRect(start, n) && DPointInRect(end, n) )
    highlighted = true;
  else
    highlighted = false;
  return highlighted;
}

void Bond::Edit() {
  cout << "edit bond" << endl;
  BondEditDialog be(r, "bond editor", start, end, TYPE_BOND, order, dashed, thick, 0, color);
  if ( !be.exec() ) return;
  cout << "change" << endl;
  color = be.Color();
  order = be.Order();
  dashed = be.Dash();
  thick = be.Thick();
}

void Bond::Render()
{
  double myangle, x2, y2, rise, run, lx, ly, cf = 0.02;
  QColor c1;
  if (highlighted)
    c1 = QColor(255,0,0);
  else
    c1 = color;
  // for single bonds
  if (order == 1) {
    if (dashed > 0)
      r->drawLine(start->toQPoint(), end->toQPoint(), 1, c1, 1);
    else
      r->drawLine(start->toQPoint(), end->toQPoint(), thick, c1);
    return;
  }
  // for stereo_up bonds
  if (order == 5) {
    r->drawUpLine(start->toQPoint(), end->toQPoint(), c1);
    return;
  }
  // for stereo_down bonds
  if (order == 7) {
    r->drawDownLine(start->toQPoint(), end->toQPoint(), c1);
    return;
  }
  // for double and triple bonds
  myangle = getAngle(start, end);
  myangle += 90.0;
  myangle = myangle * (M_PI / 180.0);
  double offs = start->distanceTo(end) / 25.0;
  if (offs > 2.0) offs = 2.0;
  //x1 = (int)(cos(myangle) * 2.5 * offs);
  //y1 = (int)(sin(myangle) * 2.5 * offs);
  x2 = (int)(cos(myangle) * 3.0 * offs);
  y2 = (int)(sin(myangle) * 3.0 * offs);
  //QPoint sp1(start->x + x2, start->y + y2);
  //QPoint ep1(end->x + x2, end->y + y2); 
  //QPoint sp2(start->x - x2, start->y - y2); 
  //QPoint ep2(end->x - x2, end->y - y2);
  // shorten lines by removing from each end
  double bl = start->distanceTo(end);
  if (bl < 100.0) cf = 0.03;
  if (bl < 51.0) cf = 0.05;
  if (bl < 27.0) cf = 0.07;
  // find sp1 and ep1
  rise = (end->y + y2) - (start->y + y2);
  run = (end->x + x2) - (start->x + x2);
  lx = start->x + x2 + run * cf;
  ly = start->y + y2 + rise * cf;
  QPoint sp1((int)lx, (int)ly);
  lx = start->x + x2 + run * (1.0 - cf);
  ly = start->y + y2 + rise * (1.0 - cf);
  QPoint ep1((int)lx, (int)ly);
  // find sp2 and ep2
  rise = (end->y - y2) - (start->y - y2);
  run = (end->x - x2) - (start->x - x2);
  lx = start->x - x2 + run * cf;
  ly = start->y - y2 + rise * cf;
  QPoint sp2((int)lx, (int)ly);
  lx = start->x - x2 + run * (1.0 - cf);
  ly = start->y - y2 + rise * (1.0 - cf);
  QPoint ep2((int)lx, (int)ly);

  if (order == 2) {
    QPoint sp, ep;
    if (wside == 1) {
      sp = sp1;
      ep = ep1;
    } else {
      sp = sp2;
      ep = ep2;
    }
    if (dashed > 1)
      r->drawLine(start->toQPoint(), end->toQPoint(), 1, c1, 1);
    else 
      r->drawLine(start->toQPoint(), end->toQPoint(), 1, c1);
    if (dashed > 0)
      r->drawLine(sp, ep, 1, c1, 1);
    else
      r->drawLine(sp, ep, 1, c1);
    return;
  }
  if (order == 3) {
    if (dashed > 2)
      r->drawLine(start->toQPoint(), end->toQPoint(), 1, c1, 1);
    else 
      r->drawLine(start->toQPoint(), end->toQPoint(), 1, c1);
    if (dashed > 1)
      r->drawLine(sp1, ep1, 1, c1, 1);
    else
      r->drawLine(sp1, ep1, 1, c1);
    if (dashed > 0) 
      r->drawLine(sp2, ep2, 1, c1, 1);
    else
      r->drawLine(sp2, ep2, 1, c1);
    return;
  }
}

int Bond::Type(void)
{
  return TYPE_BOND;
}

bool Bond::Find(DPoint *target) {
  if (start == target) return true;
  if (end == target) return true;
  return false;
}

// You *did* call Find() first to make sure target is in this Bond...?
DPoint *Bond::otherPoint(DPoint *target) {
  if (target == start) return end;
  if (target == end) return start;
  return 0;
}

DPoint * Bond::FindNearestPoint(DPoint *target, double &dist) {
  if (target->distanceTo(start) < target->distanceTo(end)) {
    dist = target->distanceTo(start);
    return start;
  } else {
    dist = target->distanceTo(end);
    return end;
  }
}

Drawable * Bond::FindNearestObject(DPoint *target, double &dist) {
  dist = DistanceToLine(start, end, target);
  return this;
}

void Bond::setPoints(DPoint *s, DPoint *e) {
  start = s;
  end = e;
}

QRect Bond::BoundingBox() {
  if (highlighted == false)
    return QRect( QPoint(999,999), QPoint(0,0) ); 
  int top, bottom, left, right, swp;
  top = (int)start->y;
  left = (int)start->x;
  bottom = (int)end->y;
  right = (int)end->x;
  if (bottom < top) { swp = top; top = bottom; bottom = swp; }
  if (right < left) { swp = left; left = right; right = swp; }
  return QRect( QPoint(left,top), QPoint(right,bottom) );
}
