/*
  XBubble - opponent.c

  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "utils.h"
#include "cell.h"
#include "bubble.h"
#include "board.h"
#include "data.h"
#include "opponent.h"

#ifdef DEBUG_OPPONENT
#define DEBUG 1
#else
#define DEBUG 0
#endif

extern int fps;
enum OpponentStage { COMPUTING_TARGETS, INITIAL_EVALUATION, SEARCHING_TREE };

#define MAX_DEPTH 3

struct _Opponent {
  enum OpponentStage stage;
  int launch_count, period, angle, nb_evaluations, thinking_speed;
  int depth, target_index[MAX_DEPTH], eval[MAX_DEPTH], best[MAX_DEPTH];
  int max_depth, line[MAX_DEPTH], color[MAX_DEPTH], target[NB_ANGLES];
  int cell_cache[NB_CELLS], color_eval[NB_COLORS], target_eval[NB_COLORS]; 
  int eval_cache[NB_CELLS][NB_COLORS], angle_error;
  double canon_x, canon_y, *vx, *vy;
  Set explosive_bubbles;
  Vector floating_cells;
  Vector path[NB_ANGLES], targets[MAX_DEPTH], perimeters[NB_CELLS];
  struct _Bubble bubble[MAX_DEPTH];
  CellArray array, board_array;
};

static void reset_targets( Opponent op ) {
  memset( op->cell_cache, 0, NB_CELLS*sizeof(int));
  empty_vector( op->targets[op->depth] );
}

Opponent new_opponent( Board board, int level ) {
  int i;
  Opponent op;
  CellArray ca = get_board_array(board);
  op = (Opponent) xmalloc( sizeof( struct _Opponent ));
  op->board_array = ca;
  op->array = new_cell_array( ca->moving_ceiling );
  op->explosive_bubbles = new_set( NB_CELLS );
  op->floating_cells = new_vector( NB_CELLS );
  for ( i = 0; i < MAX_DEPTH; i++ )
    op->targets[i] = new_vector( NB_CELLS );
  for ( i = 0; i < NB_CELLS; i++ )
    op->perimeters[i] = new_vector(6);
  for ( i = 0; i < NB_ANGLES; i++ ) {
    op->path[i] = new_vector( NB_CELLS );
  }
  switch( level ) {
  case VERY_EASY: 
    op->max_depth = 2;
    op->angle_error = 2;
    op->thinking_speed = 100/fps;
    break;
  case EASY:
    op->max_depth = 2;
    op->angle_error = 4;
    op->thinking_speed = 100/fps;
    break;
  case NORMAL:
    op->max_depth = 2;
    op->angle_error = 8;
    op->thinking_speed = 100/fps;
    break;
  case HARD:
    op->max_depth = 2;
    op->angle_error = 0;
    op->thinking_speed = 200/fps;
    break;
  case VERY_HARD:
    op->max_depth = 3;
    op->angle_error = 0;
    op->thinking_speed = 1000/fps;
    break;
  }
  if ( DEBUG )
    fprintf( stderr, "Start opponent - depth=%d, error=%d, speed=%d\n",
	     op->max_depth, op->angle_error, op->thinking_speed );
  reset_opponent( op, board );
  return op;
}

void delete_opponent( Opponent op ) {
  int i;
  delete_cell_array( op->array );
  delete_set( op->explosive_bubbles );
  delete_vector( op->floating_cells );
  for ( i = 0; i < NB_CELLS; i++ )
    delete_vector( op->perimeters[i] );
  for ( i = 0; i < NB_ANGLES; i++ )
    delete_vector( op->path[i] );
  for ( i = 0; i < MAX_DEPTH; i++ )
    delete_vector( op->targets[i] );
  free( op );
}

static int array_overflow( Opponent op, int depth ) {
  int i, first, last_row;
  last_row = ROWS - 1;
  if ( op->launch_count + depth >= op->period )
    last_row--;
  first = COLS*last_row;
  for ( i = first; i < NB_CELLS; i++ )
    if ( op->array->cell[i] != EMPTY_CELL ) {
      if ( DEBUG )
	fprintf( stderr, "*" );
      return 1;
    }
  return 0;
}

static void play_move( Opponent op, Bubble bubble, int target, int color ) {
  int count, i;
  Bubble bubble2;  
  /* stick bubble in target cell */
  bubble->state = STUCK;
  bubble->cell = target;
  bubble->color = color;
  op->array->cell[target] = bubble;  
  count = count_explosive_bubbles( op->array, target, 
				   op->explosive_bubbles, NULL );
  if ( count >= 3 ) {
    /* remove explosive bubbles */
    for ( i = 0; i < op->explosive_bubbles->size; i++ ) {
      bubble2 = op->explosive_bubbles->element[i];
      op->array->cell[bubble2->cell] = EMPTY_CELL;
    }
    /* remove dropped bubbles */
    count_floating_cells( op->array, op->floating_cells );
    for ( i = 0; i < op->floating_cells->size; i++ )
      op->array->cell[op->floating_cells->element[i]] = EMPTY_CELL;
  }
  empty_set( op->explosive_bubbles );
}

static void unplay_move( Opponent op ) {
  int i, target;
  target = op->line[op->depth];
  if ( op->array->cell[target] != EMPTY_CELL )
    op->array->cell[target] = EMPTY_CELL;
  else { /* XXX rebuild array from scratch */
    memcpy( op->array->cell, op->board_array->cell, sizeof(Bubble)*NB_CELLS );
    for ( i = 0; i < op->depth; i++ )
      play_move( op, &op->bubble[i], op->line[i], op->color[i] );
  }
}

static void compute_targets( Opponent op, int start, int end ) {
  double y, canon_y;
  int i, angle, target, cell, move;
  
  for ( i = start; i < end; i++ ) {
    angle = (i/2)*(1-2*(i%2)) + CANON_ANGLE_MAX;
    if ( op->depth == 0 ) {
      target = target_cell( op->array, op->canon_x, op->canon_y, 
			    op->vx[angle], op->vy[angle], &y, op->path[angle]);
      op->target[angle] = target;
    }
    else { /* depth >= 1 */
      canon_y = op->canon_y;
      /* move canon up to fake board going down */
      if ( op->launch_count + op->depth >= op->period + 1 )
	canon_y -= ROW_HEIGHT;
      /* check if target recomputation is really needed */
      target = op->target[angle];
      for ( move = 0; move < op->depth; move++ ) {
	cell = op->line[move];
	if (( canon_y != op->canon_y )||
	    ( op->array->cell[cell] == EMPTY_CELL )||
	    ( op->target[angle] == cell )||
	    ( element_is_in_vector( op->path[angle], cell ))) {
	  target = target_cell( op->array, op->canon_x, canon_y, 
				op->vx[angle], op->vy[angle], &y, NULL );
	  break;
	}
      }
    }
    if ( ! op->cell_cache[target] ) {
      op->cell_cache[target] = 1;
      add_element_to_vector( op->targets[op->depth], target );
    }
  }
  if ( DEBUG ) {
    if ( end == NB_ANGLES ) {
      fprintf(stderr,"\n");
      for( i = 1; i <= op->depth; i++ )
	fprintf( stderr, "\t");
      fprintf( stderr, "targets[%d]={ ", op->depth);
      for( i = 0; i < op->targets[op->depth]->size; i++ )
	fprintf( stderr, "%d,", op->targets[op->depth]->element[i]);
      fprintf( stderr, "} ");
    }
  }
}

static void add_to_perimeter( Opponent op, Vector perimeter, int cell ) {
  int neighbor;
  Quadrant q;
  for ( q = 0; q < 6; q++ ) {
    neighbor = neighbor_cell( op->array, cell, q );
    if (( neighbor != OUT_OF_BOUNDS )&&
	( op->array->cell[neighbor] == EMPTY_CELL )&&
	/* use op->array->tagged as a cache */
	( ! op->array->tagged[neighbor] )) {
      add_element_to_vector( perimeter, neighbor );
      op->array->tagged[neighbor] = 1;
    }
  }
}

void evaluate_target_potential( Opponent op, int target, int *eval,
				int search_depth, Vector perimeter ) {
  int cell, count, color, i;
  struct _Bubble temp_bubble;
  Bubble bubble;
  if ( perimeter != NULL )
    empty_vector(perimeter);

  /* stick temporary bubble in target cell */
  temp_bubble.state = STUCK;
  temp_bubble.cell = target;
  op->array->cell[target] = &temp_bubble;
  for ( color = 0; color < NB_COLORS; color++ ) {
    eval[color] = NB_CELLS * ROWS;
    temp_bubble.color = color;
    count = count_explosive_bubbles( op->array, target, 
				     op->explosive_bubbles, NULL );
    if ( count >= 3 ) { /* explosion */
      count_floating_cells( op->array, op->floating_cells );
      /* sum explosive bubbles */
      for ( i = 0; i < op->explosive_bubbles->size; i++ ) {
	bubble = op->explosive_bubbles->element[i];
	eval[color] += bubble->cell/COLS;
	if ( perimeter != NULL )
	  add_to_perimeter( op, perimeter, bubble->cell );
      }
      /* sum dropped bubbles */
      for ( i = 0; i < op->floating_cells->size; i++ ) {
	cell = op->floating_cells->element[i];
	eval[color] += cell/COLS + 1; /* dropping bonus */
	if ( perimeter != NULL )
	  add_to_perimeter( op, perimeter, cell);
      }
    }
    else { /* no explosion */
      if ( perimeter != NULL )
	for ( i = 0; i < op->explosive_bubbles->size; i++ ) {
	  bubble = op->explosive_bubbles->element[i];
	  add_to_perimeter( op, perimeter, bubble->cell );
	}
      eval[color] += ( count == 2 )? 2 : 0;
    }
    empty_set( op->explosive_bubbles );    
    /* check if move is losing */
    if ( array_overflow( op, search_depth ))
      eval[color] = 1;
  }
  if (( perimeter != NULL )&&( perimeter->size ))
    memset( op->array->tagged, 0, NB_CELLS*sizeof(int));
  /* remove temporary bubble */
  op->array->cell[target] = EMPTY_CELL;
}

int find_best_angle( Opponent op ) {
  Bubble bubble;
  int angle_max, target_index_max;
  int i, first, eval, cell, index, target, need_target_evaluation;
  switch( op->stage ) {

  case COMPUTING_TARGETS:
    angle_max = op->angle + 2*op->thinking_speed;
    if ( angle_max >= NB_ANGLES )
      angle_max = NB_ANGLES;
    compute_targets( op, op->angle, angle_max );
    if ( angle_max == NB_ANGLES )
      op->stage = INITIAL_EVALUATION;
    else
      op->angle = angle_max; 
    return 0;

  case INITIAL_EVALUATION:
    target_index_max = op->target_index[0] + op->thinking_speed;
    if ( target_index_max >= op->targets[0]->size )
      target_index_max = op->targets[0]->size;
    for ( i = op->target_index[0]; i < target_index_max; i++ ) {
      target = op->targets[0]->element[i];
      evaluate_target_potential( op, target, op->eval_cache[target], 0,
				 op->perimeters[target] );
    }
    if ( target_index_max == op->targets[0]->size ) {
      op->stage = SEARCHING_TREE;
      op->target_index[0] = 0;
    }
    else
      op->target_index[0] = target_index_max;
    return 0;

  case SEARCHING_TREE:
    while (1) {
      index = op->target_index[op->depth];
      if ( index < op->targets[op->depth]->size ) {
	/* process target */
	target = op->targets[op->depth]->element[index];
	op->target_index[op->depth]++;
	
	if ( op->depth == op->max_depth-1 ) {
	  /* process leaf node */
	  if ( ! index )
	    memset( op->color_eval, 0, sizeof(int)*NB_COLORS );
	  /* see if we can use cached info */
	  need_target_evaluation = 0;
	  if ( ! element_is_in_vector( op->targets[0], target ))
	    need_target_evaluation = 1;
	  else
	    for ( i = 0; i < op->depth; i++ ) {
	      cell = op->line[i];
	      if (( op->array->cell[cell] != &op->bubble[i] )||
		  ( element_is_in_vector( op->perimeters[target], cell ))) {
		need_target_evaluation = 1;
		break;
	      }
	    }
	  /* evaluation */
	  if ( need_target_evaluation ) { /* cache miss */
	    if ( DEBUG )
	      fprintf( stderr, "." );
	    evaluate_target_potential( op, target, op->target_eval, op->depth,
				       NULL);
	    op->nb_evaluations++;
	  }
	  else { /* cache hit */
	    if ( DEBUG )
	      fprintf( stderr, "!" );
	    for ( i = 0; i < NB_COLORS; i++ )
	      op->target_eval[i] = op->eval_cache[target][i];
	  }
	  /* end of evaluation */
	  /* compute max evaluations */
	  for ( i = 0; i < NB_COLORS; i++ )
	    if ( op->color_eval[i] < op->target_eval[i] )
	      op->color_eval[i] = op->target_eval[i];
	  /* stop if we did enough evaluations */
	  if (( need_target_evaluation )&&
	      ( op->nb_evaluations >= op->thinking_speed )) {
	    op->nb_evaluations = 0;
	    return 0;
	  }
	} /* end of leaf node processing */
	else { /* play move */
	  play_move( op, &op->bubble[op->depth], target, op->color[op->depth]);
	  op->line[op->depth] = target;
	  if ( ! array_overflow(op, op->depth ) ) {
	    /* compute new targets */
	    op->depth++;
	    op->target_index[op->depth] = 0;
	    reset_targets(op);
	    compute_targets( op, 0, NB_ANGLES );
	    op->eval[op->depth] = 0;
	  }
	  else { /* losing later is better than losing now */
	    if ( op->eval[op->depth] < op->depth + 1 ) {
	      op->eval[op->depth] = op->depth + 1;
	      op->best[op->depth] = op->line[op->depth];
	    }
	    unplay_move(op);
	  }
	} /* end of play move */
      } /* end of target processing */
      else { /* no targets left */
	if ( op->depth == 0 ) {
	  if ( DEBUG )
	    fprintf( stderr, "best target: %d (%d)\n", op->best[op->depth],
		     op->eval[op->depth]);
	  return 1;
	}
	else { /* walk up the tree */
	  if ( op->depth == op->max_depth-1 ) { /* leaf node */
	    /* compute average eval w.r.t. color */
	    eval = 0;
	    for ( i = 0; i < NB_COLORS; i++ )
	      eval += op->color_eval[i];
	    op->eval[op->depth] = eval/NB_COLORS;
	    if ( op->eval[op->depth] > 1 ) {
	      /* compute global sum on bubbles */
	      first = first_cell(op->array);
	      for ( i = first; i < NB_CELLS - COLS; i++ ) {
		bubble = op->array->cell[i];
		if ( bubble != EMPTY_CELL )
		  op->eval[op->depth] -= bubble->cell/COLS;
	      }
	    }
	  }
	  if ( op->eval[op->depth-1] < op->eval[op->depth] ) {
	    if ( DEBUG ) {
	      for( i = 0; i < op->depth; i++ )
		fprintf( stderr, "%d", op->line[i]);
	      fprintf( stderr, "->%d", op->eval[op->depth]);
	    }
	    op->eval[op->depth-1] = op->eval[op->depth];
	    op->best[op->depth-1] = op->line[op->depth-1];
	  }
	  op->depth--;
	  unplay_move(op);
	}
      }
    }
  default:
    return 1;
  }
}

int get_best_angle( Opponent op ) {
  int i, angle;
  static int counter = 0;
  for ( i = 0; i < NB_ANGLES; i++ ) {
    angle = (i/2)*(1-2*(i%2)) + CANON_ANGLE_MAX;
    if ( op->target[angle] == op->best[0] ) {
      angle -= CANON_ANGLE_MAX;
      if ( op->angle_error ) {
	counter++;
	/* add a random factor sometimes to simulate bad aiming */
	if ( counter % op->angle_error == 0 )
	  angle += rnd(8) - 4;
      }
      return angle;
    }
  }
  return 0;
}

int get_best_eval( Opponent op ) {
  return op->eval[0];
}

void reset_opponent( Opponent op, Board board ) {  
  /* make a clean copy of board array */
  memcpy( op->array->cell, op->board_array->cell, sizeof(void *)*NB_CELLS);
  op->array->first_row = op->board_array->first_row;
  op->array->parity = op->board_array->parity;
  op->depth = 0;
  op->nb_evaluations = 0;
  op->eval[0] = 0;
  get_board_info( board, &op->canon_x, &op->canon_y, &op->vx, &op->vy,
		  &op->color[0], &op->color[1], &op->launch_count,&op->period);
  op->target_index[0] = 0;
  op->stage = COMPUTING_TARGETS;
  op->angle = 0;
  reset_targets(op);
}
