/* partspace.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "partspace.h"

static void *PS_slavestart( void *arg )
{
  PartSpace *ps = (PartSpace*)arg;
  ps->slavehandler();
  return NULL;
}

PartSpace::PartSpace()
{
  spaceinfo = new List();
  spacerequest = new SpaceRequestList();
  spaceanswer = new SpaceRequestList();

  slave.running = 0;
  slave.stop = false;
  if ( pthread_create( &slave.th, NULL, PS_slavestart, this ) != 0 ) {
    fprintf( stderr, "Worker: Can't create thread, aborting!\n" );
    exit( EXIT_FAILURE );
  }
  info_lifetime = 10.0;  // force update after this time
  spaceinfo_maxsize = 100;
}

PartSpace::~PartSpace()
{
  SpaceRequestList::space_request_list_t *te;

  slave.stop = true;
  // wake up slave
  spacerequest->signal();
  pthread_join( slave.th, NULL );
  while ( spaceanswer->isEmpty_locked() == false ) {
    te = spaceanswer->remove_locked();
    if ( te != NULL ) {
      _freesafe( te->name );
      delete te;
    }
  }
  while ( spacerequest->isEmpty_locked() == false ) {
    te = spacerequest->remove_locked();
    if ( te != NULL ) {
      _freesafe( te->name );
      delete te;
    }
  }

  delete spacerequest;
  delete spaceanswer;

  te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement();
  while ( te != NULL ) {
    spaceinfo->removeFirstElement();
    _freesafe( te->name );
    delete te;
    te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement();
  }
  delete spaceinfo;
}

// So arbeitet es im Thread-Fall
// 1.Nachschauen, ob vom Slave Resultate da sind
// 1.1.Wenn ja, alle entnehmen und in spaceinfo einfgen
//     alte dabei loeschen
//     falls error == 0
// 2.Prfen, ob Elemente viel zu alt (mit lifetime)
// 2.1.Wenn ja, dann ber Liste gehen
//     Fr alle zu alten:
//       Entferne Eintrag aus liste
// 3.Name suchen
// 3.1.Wenn gefunden, dann stats in lokalen buffer kopieren und 0 zurckgeben
//     Damit wird bis zum nchsten readSpace auf diese Werte zugegriffen
//     Gltig ist dieser Wert wegen 2
//     Wenn allerdings zu alt, in slaverequest legen
// 3.2.sonst: neuen space_..._t besorgen, mit dupstring(name) belegen und 
//     in slaverequest einfgen und slave aufwecken
//     dann wird EAGAIN zurckgeliefert

int PartSpace::readSpace( const char* name )
{
  SpaceRequestList::space_request_list_t *te, *searchte;
  time_t ct;
  int id;
  bool do_update = false;
  worker_struct_stat buf1;
  dev_t name_device;

  if ( name == NULL ) return EFAULT;
  if ( worker_stat( name, &buf1 ) != 0 ) return EFAULT;
  name_device = buf1.st_dev;

  ct = time( NULL );
  spaceanswer->lock();
  while ( spaceanswer->isEmpty_locked() == false ) {
    te = spaceanswer->remove_locked();
    if ( te != NULL ) {
      if ( /*te->error*/ 0 == 0 ) {
        // valid
        // currently I accept even invalid entries as the VFS
        // doesn't support statvfs but I don't want to
        // start asking again and again

        // search for the same and remove it
        id = spaceinfo->initEnum();
        searchte = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
        while ( searchte != NULL ) {
          if ( searchte->device == te->device ) {
            spaceinfo->removeElement( searchte);
            _freesafe( searchte->name );
            delete searchte;
            break;
          }
          searchte = (SpaceRequestList::space_request_list_t*)spaceinfo->getNextElement( id );
        }
        spaceinfo->closeEnum( id );

        spaceinfo->addElement( te );
      } else {
        // invalid
        _freesafe( te->name );
        delete te;
      }
    }
  }
  spaceanswer->unlock();
    
  id = spaceinfo->initEnum();
  
  // remove oldest entries if there are too many entries
  if ( spaceinfo_maxsize > 0 ) {
    while( spaceinfo->size() > spaceinfo_maxsize ) {
      te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
      if ( te == NULL ) break;
      spaceinfo->removeFirstElement();
      _freesafe( te->name );
      delete te;
    }
  }

  // find name
  te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
  while ( te != NULL ) {
    if ( name_device == te->device ) {
      // entry found
      break;
    }
    te = (SpaceRequestList::space_request_list_t*)spaceinfo->getNextElement( id );
  }
  spaceinfo->closeEnum( id );
  
  blocksize = 0;
  space = 0;
  freespace = 0;
  if ( te != NULL ) {
    // found
    if ( te->error == 0 ) {
#ifdef HAVE_STATVFS
      blocksize = te->fsstatistic.f_frsize;
      space = te->fsstatistic.f_blocks;
      freespace = te->fsstatistic.f_bavail;
#elif defined( HAVE_STATFS )
      blocksize = te->fsstatistic.f_bsize;
      space = te->fsstatistic.f_blocks;
      freespace = te->fsstatistic.f_bavail;
#else
      blocksize = 0;
      space = 0;
      freespace = 0;
#endif
    }
    // check if this is too old
    if ( difftime( ct, te->readtime ) > info_lifetime ) {
      do_update = true;
    }
  }

  if ( ( te == NULL ) || ( do_update == true ) ) {  
    // not found
    te = new SpaceRequestList::space_request_list_t;
    te->name = dupstring( name );
    te->device = name_device;
    spacerequest->lock();
    spacerequest->put_locked( te );
    spacerequest->signal();
    spacerequest->unlock();
    if ( do_update == true ) return 0;  // we found it and fill the variables but want to reload
    else return EAGAIN;
  } else return 0;  // found and not too old
}

loff_t PartSpace::getBlocksize()
{
  return blocksize;
}

loff_t PartSpace::getFreeSpace()
{
  return freespace;
}

loff_t PartSpace::getSpace()
{
  return space;
}

// H stands for human readable, means in B/KB/MB/GB when great enough

loff_t PartSpace::getFreeSpaceH(char **unit_return)
{
  double dfree;
  char buffer[3];

  dfree = (double)freespace;
  dfree *= (double)blocksize;
  if(dfree>10.0*1024.0*1024.0*1024.0) {
    dfree+=512.0*1024.0*1024.0;
    dfree/=1024.0*1024.0*1024.0;
    strcpy(buffer,"GB");
  } else if ( dfree > 10.0*1024.0*1024.0 ) {
    dfree += 512.0*1024.0;
    dfree /= 1024.0*1024.0;
    strcpy(buffer,"MB");
  } else if ( dfree > 10.0*1024.0 ) {
    dfree += 512.0;
    dfree /= 1024.0;
    strcpy(buffer,"KB");
  } else {
    strcpy(buffer,"B");
  }
  if(unit_return!=NULL) {
    strcpy(*unit_return,buffer);
  }
  return (loff_t)dfree;
}

loff_t PartSpace::getSpaceH(char **unit_return)
{
  double dsize;
  char buffer[3];

  dsize = (double)space;
  dsize *= (double)blocksize;
  if(dsize>10.0*1024.0*1024.0*1024.0) {
    dsize+=512.0*1024.0*1024.0;
    dsize/=1024.0*1024.0*1024.0;
    strcpy(buffer,"GB");
  } else if ( dsize > 10*1024.0*1024.0 ) {
    dsize += 512*1024.0;
    dsize /= 1024.0*1024.0;
    strcpy(buffer,"MB");
  } else if ( dsize > 10.0*1024.0 ) {
    dsize += 512.0;
    dsize /= 1024.0;
    strcpy(buffer,"KB");
  } else {
    strcpy(buffer,"B");
  }
  if(unit_return!=NULL) {
    strcpy(*unit_return,buffer);
  }
  return (loff_t)dsize;
}

bool PartSpace::SpaceRequestList::isEmpty_locked()
{
  if ( elements < 1 ) return true;
  return false;
}

int PartSpace::SpaceRequestList::put_locked( space_request_list_t *elem )
{
  int pos;

  if ( elem == NULL ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
  } else {
    head = tail = elem;
  }

  pos = elements++;
  return pos;
}

PartSpace::SpaceRequestList::space_request_list_t *PartSpace::SpaceRequestList::remove_locked()
{
  space_request_list_t *te;

  if ( elements == 0 ) return NULL;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

PartSpace::SpaceRequestList::SpaceRequestList()
{
  head = NULL;
  tail = NULL;
  elements = 0;
}

PartSpace::SpaceRequestList::~SpaceRequestList()
{
  space_request_list_t *te;

  lock();
  while ( isEmpty_locked() == false ) {
    te = remove_locked();
    delete te;
  }
  unlock();
}

void PartSpace::SpaceRequestList::lock()
{
  ex.lock();
}

void PartSpace::SpaceRequestList::unlock()
{
  ex.unlock();
}

void PartSpace::SpaceRequestList::wait()
{
  ex.wait();
}

void PartSpace::SpaceRequestList::signal()
{
  ex.signal();
}

void PartSpace::slavehandler()
{
  SpaceRequestList::space_request_list_t *te;
  bool ende;
  
  if ( slave.running != 0 ) {
    fprintf( stderr, "Worker: another thread already running!\n");
    return;
  }
  slave.running = 1;
#ifdef DEBUG
  printf("entering slave handler\n");
#endif

  spacerequest->lock();
  for( ende = false; ende == false; ) {
    // wait for next element or stop
    while ( ( spacerequest->isEmpty_locked() == true ) && ( slave.stop == false ) )
      spacerequest->wait();

#ifdef DEBUG
    printf("wait finished\n");
#endif
    if ( slave.stop == false ) {
      // an element to check
      te = spacerequest->remove_locked();
      spacerequest->unlock();
      if ( te != NULL ) {
        te->error = getSpace( te->name, &(te->fsstatistic) );
        te->readtime = time(NULL);
        spaceanswer->lock();
        spaceanswer->put_locked( te );
        spaceanswer->unlock();
      }
      spacerequest->lock();
    } else {
      ende = true;
    }
  }
  spacerequest->unlock();
#ifdef DEBUG
  printf("leaving slave handler\n");
#endif
}

#ifdef HAVE_STATVFS
int PartSpace::getSpace( const char *name, worker_struct_statvfs *statbuf )
#elif defined( HAVE_STATFS )
int PartSpace::getSpace( const char *name, struct statfs *statbuf )
#else
int PartSpace::getSpace( const char *name, int *statbuf )
#endif
{
  if ( name == NULL ) return EFAULT;
  if ( statbuf == NULL ) return EFAULT;

#ifdef HAVE_STATVFS
  return worker_statvfs( name, statbuf );
#elif defined( HAVE_STATFS )
  return statfs( name, statbuf );
#else
  return 0;
#endif
}

void PartSpace::setLifetime( double t )
{
  // at least one second is minimum
  if ( t < 1.0 ) return;
  
  info_lifetime = t;
}
