/* bookmarkdbfilter.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2007-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "bookmarkdbfilter.hh"
#include "bookmarkdb.hh"
#include "bookmarkdbentry.hh"
#include <cctype>
#include <algorithm>

BookmarkDBFilter::BookmarkDBFilter( BookmarkDB &data ) : m_data( data )
{
}

void BookmarkDBFilter::setInfixFilter( const std::string &infix )
{
    m_infix = infix;
}

template <class T>
struct my_tolower : public std::unary_function<T,T> 
{
    T operator() (const T& c) const 
    {
        return std::tolower( c );
    }
};

struct my_sort : public std::binary_function< BookmarkDBEntry, BookmarkDBEntry, bool > 
{
    bool operator() ( const BookmarkDBEntry& c1, const BookmarkDBEntry& c2 ) const 
    {
        return c1.getName() < c2.getName();
    }
};

std::string tolower( const std::string &str )
{
    std::string res( str );

    std::transform( res.begin(),
                    res.end(),
                    res.begin(),
                    my_tolower<char>() );
    return res;
}

std::list< BookmarkDBEntry > BookmarkDBFilter::getEntries( const std::string &cat )
{
    std::list< BookmarkDBEntry > res, e = m_data.getEntries( cat );

    std::string infix = tolower( m_infix );

    for ( std::list< BookmarkDBEntry >::iterator it1 = e.begin();
          it1 != e.end();
          it1++ ) {
        std::string name = tolower( it1->getName() );
        if ( name.find( infix ) != std::string::npos ) {
            res.push_back( *it1 );
        } else {
            std::string alias = tolower( it1->getAlias() );
            if ( alias.find( infix ) != std::string::npos ) {
                res.push_back( *it1 );
            }
        }
    }

    res.sort( my_sort() );

    return res;
}

std::string BookmarkDBFilter::getInfixFilter() const
{
    return m_infix;
}
