#!/bin/sh
#----------------------------------------------------------------------------
#  build.sh
#   This script will perform the standard configure and make
#  instructions required to build the tcl/tk static support  
#  libraries generally used in CodeWeavers config tools
#  (e.g. winesetuptk, pluginsetup)
#----------------------------------------------------------------------------
rootdir=`pwd`
prefix=$rootdir/local

usage()
{
    cat <<EOF
usage: build [options]
 
options:
  --clean           Run make clean
  --distclean       Run make distclean
  --configure       Systematically redo a configure
  --shared          Build shared tcl libraries
EOF
    exit 2
}

doclean=""
dodistclean=""
doconfig=""
doshared=""
while [ $# -gt 0 ]
do
  case "$1" in
    --clean)       doclean="1"; shift ;;
    --distclean)   dodistclean="1"; shift ;;
    --configure)   doconfig="1"; shift ;;
    --shared)      doshared="1"; shift ;;

    -?|-h|--help)  usage ;;
    *)             echo "Unknown option: $1"; usage ;;
  esac
done

if [ -z "$MAKE" ]; then
    # Some Makefiles (especially the Loki installer) will only work 
    # with GNU make. So use gmake if it exists (helps on FreeBSD).
    MAKE=`which gmake 2>/dev/null`
fi
if [ $? -gt 0 -o -z "$MAKE" ]; then
    MAKE=make
fi

#----------------------------------------------------------------------------
#                                    Tcl
#----------------------------------------------------------------------------
echo "[Building tcl]"
if [ ! -f $rootdir/.tcl-bootstrapped ]
then
    # An ugly hack because Tcl assumes tclStubInit.c will not need to
    # be rebuilt during the bootstrap
    touch $rootdir/tcl8.3.2/generic/tclStubInit.c
    touch $rootdir/.tcl-bootstrapped
fi
cd $rootdir/tcl8.3.2/unix
if [ ! -f config.status -o -n "$doshared" -o -n "$doconfig" ]
then
    ./configure --disable-shared --prefix=$prefix
fi
if [ -n "$dodistclean" ]
then
    $MAKE distclean
elif [ -n "$doclean" ]
then
    $MAKE clean
else
    $MAKE install
    if [ $? != 0 ]; then
        echo "[Error] Static tcl library build failed"
        exit 1
    fi
    if [ -n "$doshared" ]
    then
        ./configure --enable-shared --prefix=$prefix
        $MAKE install-binaries
    fi
fi
if [ $? != 0 ]; then
    echo "[Error] Tcl library build failed"
    exit 1
fi


#----------------------------------------------------------------------------
#                                     Tk
#----------------------------------------------------------------------------
echo "[Building tk]"
cd $rootdir/tk8.3.2/unix
if [ ! -f config.status -o -n "$doshared" -o -n "$doconfig" ]
then
    ./configure --disable-shared --prefix=$prefix
fi
if [ -n "$dodistclean" ]
then
    $MAKE distclean
elif [ -n "$doclean" ]
then
    $MAKE clean
else
    $MAKE install
    if [ $? != 0 ]; then
        echo "[Error] Static tk library build failed"
        exit 1
    fi
    if [ -n "$doshared" ]
    then
        ./configure --enable-shared --prefix=$prefix
        $MAKE install-binaries
    fi
fi
if [ $? != 0 ]; then
    echo "[Error] Tk library build failed"
    exit 1
fi


#----------------------------------------------------------------------------
#                                    Itcl
#----------------------------------------------------------------------------
echo "[Building itcl]"
cd $rootdir/itcl3.1.0
if [ ! -f config.status -o -n "$doshared" -o -n "$doconfig" ]
then
    ./configure --disable-shared --prefix=$prefix \
                --with-tcl=$prefix/../tcl8.3.2/unix \
                --with-tk=$prefix/../tk8.3.2/unix
fi
if [ -n "$dodistclean" ]
then
    $MAKE distclean
elif [ -n "$doclean" ]
then
    $MAKE clean
else
    $MAKE && $MAKE install
    if [ $? != 0 ]; then
        echo "[Error] Static itcl library build failed"
        exit 1
    fi
    if [ -n "$doshared" ]
    then
        ./configure --enable-shared --prefix=$prefix \
                    --with-tcl=$prefix/../tcl8.3.2/unix \
                    --with-tk=$prefix/../tk8.3.2/unix
        $MAKE && $MAKE install
    fi
fi
if [ $? != 0 ]; then
    echo "[Error] Itcl library build failed"
    exit 1
fi


#----------------------------------------------------------------------------
#                                     Img
#----------------------------------------------------------------------------
echo "[Building img]"
cd $rootdir/img1.2.4
if [ ! -f config.status -o -n "$doshared" -o -n "$doconfig" ]
then
    ./configure --disable-shared --prefix=$prefix
fi
if [ -n "$dodistclean" ]
then
    $MAKE distclean
elif [ -n "$doclean" ]
then
    $MAKE clean
else
    $MAKE install
    if [ $? != 0 ]; then
        echo "[Error] Static img library build failed"
        exit 1
    fi
    if [ -n "$doshared" ]
    then
        $MAKE distclean
        ./configure --enable-shared --prefix=$prefix
        $MAKE install
    fi
fi
if [ $? != 0 ]; then
    echo "[Error] Shared img library build failed"
    exit 1
fi


#----------------------------------------------------------------------------
#                                  MkTclApp
#----------------------------------------------------------------------------
echo "[Building mktclapp]"
cd $rootdir/mktclapp
if [ -n "$dodistclean" ]
then
    $MAKE distclean
elif [ -n "$doclean" ]
then
    $MAKE clean
else
    $MAKE install
fi
if [ $? != 0 ]; then
    echo "[Error] mktclapp build failed"
    exit 1
fi
