/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "slider.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

namespace wftk {

Slider::Slider(Orientation orient, unsigned thickness) :
  buttonSurface_(NULL),
  button_(0, 0, 0, 0),
  orientation_(orient),
  thickness_(thickness),
  value_(0.0),
  delta_(10),
  min_(0),
  max_(100),
  moving_(false),
  delta_per_pixel_(0)
{
  buttonColor_ = Color::registry.find("slider_button_color");
  if(buttonColor_.a == Color::WFTK_TRANSPARENT)
    buttonColor_ = Color();
  setButtonSurface("slider_button_surface");

  getResourceBackground("slider");

  setPackingInfo();
}

Slider::~Slider()
{
  if(buttonSurface_)
    buttonSurface_->free();
}

bool
Slider::buttonEvent(Mouse::Button button, bool pressed, const Point& pos)
{
  if(Widget::buttonEvent(button, pressed, pos))
    return true;

  switch(button) {
    case Mouse::WHEEL_UP: 
      if(orientation_ == Slider::HORIZONTAL)
        increase();
      else
        decrease();
      break;
    case Mouse::WHEEL_DOWN:
      if(orientation_ == Slider::HORIZONTAL)
        decrease();
      else
        increase();
      break;
    case Mouse::LEFT:
      // move the slider if the click is inside the slider, or if
      // we're already moving.
      if(pressed && button_.contains(pos))
	{
	  position_ = pos;
	  moving_ = true;
	}

      if(moving_ && !pressed)
	{
	  moving_ = false;
	}
      return true;
      break;
    default:
      break;
  }

  return false;
}
  
bool
Slider::mouseEvent(const Point& pos, const Point& rel, Mouse::Button mask)
{
  Widget::mouseEvent(pos, rel, mask);

  if(!moving_)
    return false;

  int delta_x = pos.x - position_.x;
  int delta_y = pos.y - position_.y;

  if(orientation_ == Slider::HORIZONTAL)
    value_ += delta_x*delta_per_pixel_;
  else
    value_ += delta_y*delta_per_pixel_;

  //ensure value_ is still in valid range
  if(value_ < min_)
    {
      value_ = min_;
    }
  if(value_ > max_)
    {
      value_ = max_;
    }
  invalidate();
  valueChanged.emit((int)(value_+0.45));
  position_ = pos;

  return false;
}

void 
Slider::draw(Surface& surf, const Point& offset, const Region& r)
{
  Debug out(Debug::DRAWING);

  Widget::draw(surf, offset, r);

  int pos = (int)(value_ / delta_per_pixel_);
    
  if(orientation_ == Slider::HORIZONTAL)
    button_.warp(Point(pos,0));
  else
    button_.warp(Point(0,pos));

//   if(moving_)
//     Mouse::instance()->warp(globalCoord(button_).upperLeft()+
// 		Point(button_.width()/2, button_.height()/2));

  out << "Drawing slider button at " << button_ << Debug::endl;

  Region draw_region(button_);
  draw_region.offset(offset);
  draw_region &= r;

  if(draw_region.empty())
    out << "Button outside of update region" << Debug::endl;

  if(buttonColor_ != Color(0,0,0,0))
    surf.blend(draw_region, buttonColor_);

  if(buttonSurface_ && !buttonSurface_->res()->empty()) {
    if(!scaledSurface_.empty())
      scaledSurface_.blit(surf, offset + button_.origin(), r);
    else {
      assert(buttonSurface_->res()->width() == button_.width()
        && buttonSurface_->res()->height() == button_.height());
      buttonSurface_->res()->blit(surf, offset + button_.origin(), r);
    }
  }
}

void Slider::handleResize(Uint16 w, Uint16 h)
{
  Widget::handleResize(w, h);

  if(buttonSurface_) // need to redo scaledSurface_
    setScaledButtonSurface();

  if(orientation_ == Slider::HORIZONTAL)
    {
      button_ = Rect(0 , 0, h, h);
      delta_per_pixel_ = (w > h) ? (max_-min_)/(float)(w - h) : 0;
    }
  else
    {
      button_ = Rect(0, h > w ? h - w : 0, w, w);     
      delta_per_pixel_ = (h > w) ? (max_-min_)/(float)(h - w) : 0;
    }
}

void
Slider::setPackingInfo()
{
  packing_info_.x.min = packing_info_.y.min =
    packing_info_.x.pref = packing_info_.y.pref = thickness_;
  packing_info_.x.expand = packing_info_.y.expand = true;

  if(orientation_ == HORIZONTAL) {
    packing_info_.x.filler = PackingInfo::Expander::FILL;
    packing_info_.y.filler = 0;
    packing_info_.x.pref *= 5;
  }
  else {
    packing_info_.x.filler = 0;
    packing_info_.y.filler = PackingInfo::Expander::FILL;
    packing_info_.y.pref *= 5;
  }
}

void
Slider::setButtonColor(const Color& col)
{
  buttonColor_ = col;
  invalidate();
}

void
Slider::setButtonSurface(Surface::Resource* surface)
{
  if(surface == buttonSurface_)
    return;

  if(buttonSurface_)
    buttonSurface_->free();
  buttonSurface_ = surface;
  if(buttonSurface_)
    buttonSurface_->bind();
  setScaledButtonSurface();
  invalidate();
}

void
Slider::setScaledButtonSurface()
{
  if(buttonSurface_ && (buttonSurface_->res()->width() != button_.width()
    || buttonSurface_->res()->height() != button_.height())) {
    scaledSurface_ = *buttonSurface_->res();
    scaledSurface_.scale(button_.width(), button_.height());
  }
  else
    scaledSurface_ = Surface(); // invalid
}

void Slider::setButtonSurface(const Surface& surf)
{
  Surface* s = new Surface(surf);
  if(s->width() != button_.width() || s->height() != button_.height())
    s->scale(width(), height());

  Surface::Resource* r = new Surface::Resource(s);
  setButtonSurface(r);
  r->free();
}

void Slider::setButtonSurface(const std::string& name)
{
  Surface::Resource *res = Surface::registry.get(name);
  if(res)
    setButtonSurface(res);
}

void 
Slider::setRange(int min, int max)
{
  if(min < max)
    {
      min_ = min;
      max_ = max;
    }
  else
    {
      max_ = min;
      min_ = max;
    }

  //ensure value_ is still in valid range
  if(value_ < min_)
    {
      value_ = min_;
      valueChanged.emit((int)(value_+0.5));
    }
  if(value_ > max_)
    {
      value_ = max_;
      valueChanged.emit((int)(value_+0.5));
    }

  if(orientation_ == Slider::HORIZONTAL)
    delta_per_pixel_ = (max_-min_)/(float)(width()-button_.width());
  else
    delta_per_pixel_ = (max_-min_)/(float)(height()-button_.height());

  invalidate();
}

void Slider::setDelta(int delta)
{
  delta_ = delta;
  invalidate();
}

void
Slider::increase()
{
  if(value_ <= max_)
    {
      value_ += delta_;
      if(value_ > max_)
	{
	  value_ = max_;
	  valueChanged.emit((int)(value_));
	}
      else
	valueChanged.emit((int)(value_+0.5));
      invalidate();
    }
}

void
Slider::decrease()
{
  if(value_ >= min_)
    {
      value_ -= delta_;
      if(value_ < min_)
	{
	  value_ = min_;
	  valueChanged.emit((int)(value_));
	}
      else
	valueChanged.emit((int)(value_+0.5));
      invalidate();
    }
}

void
Slider::setValue(int value)
{
  float old_value_ = value_;

  if(value > max_)
    value_ = max_;
  else if(value < min_)
    value_ = min_;
  else 
    value_ = value;
  
  if(value_ != old_value_)
    valueChanged.emit((int)(value_+0.5));
  invalidate();
}

} // namespace wftk
