// Adds a user style sheet to the page
function webdeveloper_addUserStyleSheet(element, applyStyle)
{
    var addStyleSheet = element.getAttribute("checked");

    // If adding a style sheet and applying styles
    if(addStyleSheet && applyStyle)
    {
        var filePicker   = Components.classes["@mozilla.org/filepicker;1"].createInstance(Components.interfaces.nsIFilePicker);
        var stringBundle = document.getElementById("webdeveloper-string-bundle");

        filePicker.appendFilter(stringBundle.getString("webdeveloper_styleSheetDescription"), "*.css");
        filePicker.init(window, stringBundle.getString("webdeveloper_addUserStyleSheet"), filePicker.modeOpen);

        // If the user selected a style sheet
        if(filePicker.show() == filePicker.returnOK)
        {
            var inputStream      = Components.classes["@mozilla.org/network/file-input-stream;1"].createInstance(Components.interfaces.nsIFileInputStream);
            var scriptableStream = Components.classes["@mozilla.org/scriptableinputstream;1"].createInstance(Components.interfaces.nsIScriptableInputStream);

            inputStream.init(filePicker.file, 0x01, 0444, null);
            scriptableStream.init(inputStream);

            webdeveloper_userStyleSheet = scriptableStream.read(scriptableStream.available());

            scriptableStream.close();
            inputStream.close();
        }
        else
        {
            addStyleSheet               = false;
            webdeveloper_userStyleSheet = null;
        }
    }

    // If adding a style sheet and the style sheet is not empty
    if(addStyleSheet && webdeveloper_userStyleSheet)
    {
        var documentList    = webdeveloper_getDocuments(webdeveloper_getContentWindow());
        var documentLength  = documentList.length;
        var headElementList = null;
        var pageDocument    = null;
        var styleElement    = null;

        // Loop through the documents
        for(var i = 0; i < documentLength; i++)
        {
            pageDocument    = documentList[i];
            headElementList = pageDocument.getElementsByTagName("head");
            styleElement    = pageDocument.createElement("style");

            styleElement.setAttribute("id", "webdeveloper-add-user-style-sheet");
            styleElement.setAttribute("type", "text/css");
            styleElement.appendChild(pageDocument.createTextNode(webdeveloper_userStyleSheet));

            // If there is a head element
            if(headElementList.length > 0)
            {
                headElementList[0].appendChild(styleElement);
            }
            else
            {
                pageDocument.documentElement.appendChild(styleElement);
            }
        }

        // If applying styles
        if(applyStyle)
        {
            webdeveloper_addAppliedStyle("webdeveloper-add-user-style-sheet");
        }
    }
    else
    {
        webdeveloper_userStyleSheet = null;
        webdeveloper_removeStyleSheet("webdeveloper-add-user-style-sheet", true);
    }
}

// Display CSS for the given media type
function webdeveloper_displayCSSMediaType(type, reset)
{
    var documentList     = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength   = documentList.length;
    var media            = null;
    var pageDocument     = null;
    var styleSheet       = null;
    var styleSheetLength = null;
    var styleSheetList   = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument     = documentList[i];
        styleSheetList   = pageDocument.styleSheets;
        styleSheetLength = styleSheetList.length;

        // Loop through the style sheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If the style sheet is valid and not an alternate style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet) && !webdeveloper_isAlternateStyleSheet(styleSheet))
            {
                media = styleSheet.media;

                // If resetting
                if(reset)
                {
                    // If the style sheet has the webdeveloper-appended-screen media
                    if(webdeveloper_isMediaStyleSheet(styleSheet, "webdeveloper-appended-screen"))
                    {
                        media.deleteMedium("webdeveloper-appended-screen");
                        media.deleteMedium("screen");
                    }
                    else if(webdeveloper_isMediaStyleSheet(styleSheet, "webdeveloper-deleted-screen"))
                    {
                        media.appendMedium("screen");
                        media.deleteMedium("webdeveloper-deleted-screen");
                    }
                }
                else
                {
                    // If the style sheet matches this media
                    if(webdeveloper_isMediaStyleSheet(styleSheet, type))
                    {
                        // If the style sheet has the screen media
                        if(!webdeveloper_isMediaStyleSheet(styleSheet, "screen"))
                        {
                            media.appendMedium("webdeveloper-appended-screen");
                            media.appendMedium("screen");
                        }
                    }
                    else if(webdeveloper_isMediaStyleSheet(styleSheet, "screen"))
                    {
                        media.appendMedium("webdeveloper-deleted-screen");
                        media.deleteMedium("screen");
                    }
                }

                // Force the styles to reapply by disabling and enabling the style sheet
                styleSheet.disabled = true;
                styleSheet.disabled = false;
            }
        }
    }
}

// Display the disable individual style sheet menu
function webdeveloper_displayDisableIndividualStyleSheetMenu(menu)
{
    var currentDocument  = webdeveloper_getContentDocument();
    var mediaList        = null;
    var menuItem         = null;
    var ownerNode        = null;
    var styleSheet       = null;
    var styleSheetHref   = null;
    var styleSheetLabel  = null;
    var styleSheetList   = currentDocument.styleSheets;
    var styleSheetLength = styleSheetList.length;

    webdeveloper_removeGeneratedMenuItems(menu);

    // Loop through the style sheets
    for(var i = 0; i < styleSheetLength; i++)
    {
        styleSheet     = styleSheetList[i];
        ownerNode      = styleSheet.ownerNode;
        styleSheetHref = styleSheet.href;

        // If this is a valid style sheet, is not an line style sheet and is not an alternate style sheet
        if(webdeveloper_isValidStyleSheet(styleSheet) && styleSheetHref != currentDocument.documentURI)
        {
            menuItem = document.createElement("menuitem");

            // If the owner node is set, is not a processing instruction and has a title attribute
            if(ownerNode && ownerNode.nodeType != Components.interfaces.nsIDOMNode.PROCESSING_INSTRUCTION_NODE && ownerNode.hasAttribute("title"))
            {
                styleSheetLabel = ownerNode.getAttribute("title");
            }
            else
            {
                styleSheetLabel = styleSheetHref;
            }

            menuItem.setAttribute("class", "webdeveloper-generated-menu");
            menuItem.setAttribute("label", styleSheetLabel);
            menuItem.setAttribute("oncommand", "webdeveloper_toggleIndividualStyleSheet('" + styleSheetHref + "')");
            menuItem.setAttribute("type", "checkbox");

            // If the style sheet is disabled
            if(!styleSheet.disabled)
            {
                menuItem.setAttribute("checked", true);
            }

            menu.appendChild(menuItem);
        }
    }

    // If the menu has no children
    if(!menu.hasChildNodes())
    {
        menuItem        = document.createElement("menuitem");
        styleSheetLabel = document.getElementById("webdeveloper-string-bundle").getString("webdeveloper_noStyleSheets");

        menuItem.disabled = true;
        menuItem.setAttribute("class", "webdeveloper-generated-menu");
        menuItem.setAttribute("label", styleSheetLabel);
        menu.appendChild(menuItem);
    }
}

// Display CSS for the handheld media type
function webdeveloper_displayHandheldCSS(element, applyStyle)
{
    var printElement = document.getElementById("webdeveloper-display-print-css-menu");

    webdeveloper_configureElementByElement(printElement, "checked", "webdeveloper-display-print-css");

    // If the print element is checked
    if(printElement.getAttribute("checked"))
    {
        webdeveloper_displayCSSMediaType("print", true);
        webdeveloper_removeStyleSheet("webdeveloper-display-print-css", applyStyle);
    }

    webdeveloper_displayCSSMediaType("handheld", !element.getAttribute("checked"));
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-display-handheld-css", applyStyle);
}

// Display CSS for the print media type
function webdeveloper_displayPrintCSS(element, applyStyle)
{
    var handheldElement = document.getElementById("webdeveloper-display-handheld-css-menu");

    webdeveloper_configureElementByElement(handheldElement, "checked", "webdeveloper-display-handheld-css");

    // If the handheld element is checked
    if(handheldElement.getAttribute("checked"))
    {
        webdeveloper_displayCSSMediaType("handheld", true);
        webdeveloper_removeStyleSheet("webdeveloper-display-handheld-css", applyStyle);
    }

    webdeveloper_displayCSSMediaType("print", !element.getAttribute("checked"));
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-display-print-css", applyStyle);
}

// Allows the user to edit the CSS of the current page
function webdeveloper_editCSS(force)
{
    toggleSidebar("webdeveloper-edit-css-sidebar", force);
}

// Output style information for an element
function webdeveloper_outputStyleInformationForElement(displayInSidebar, styleSheetArray, title)
{
    var generatedDocument = null;
    var headerElement     = null;
    var linkElement       = null;
    var preElement        = null;
    var styleSheet        = null;

    // If displaying in the sidebar
    if(displayInSidebar)
    {
        var parent = null;

        generatedDocument = document.getElementById("sidebar-box").getElementsByAttribute("id", "sidebar")[0].contentDocument.getElementById("webdeveloper-view-style-information-iframe").contentDocument;
        parent            = generatedDocument.getElementsByTagName("head")[0];

        webdeveloper_removeAllChildNodes(parent);

        parent = generatedDocument.body;

        webdeveloper_removeAllChildNodes(parent);
    }
    else
    {
        generatedDocument = webdeveloper_generateDocument("");
    }

    generatedDocument.title = title;
    headerElement           = generatedDocument.createElement("h1");

    webdeveloper_addGeneratedStyles(generatedDocument);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the style sheets
    for(styleSheet in styleSheetArray)
    {
        linkElement = generatedDocument.createElement("a");
        preElement  = generatedDocument.createElement("pre");

        linkElement.setAttribute("href", styleSheet);
        linkElement.appendChild(generatedDocument.createTextNode(styleSheet));
        generatedDocument.body.appendChild(linkElement);
        preElement.appendChild(generatedDocument.createTextNode(styleSheetArray[styleSheet]));
        generatedDocument.body.appendChild(preElement);
    }
}

// Toggles the border box model
function webdeveloper_toggleBorderBoxModel(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/use_border_box_model.css", "webdeveloper-use-border-box-model", applyStyle);
}

// Toggles the browser default styles
function webdeveloper_toggleBrowserDefaultStyles(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/disable_browser_default_styles.css", "webdeveloper-disable-browser-default-styles", applyStyle);
}

// Toggles all the inline styles in elements in the document
function webdeveloper_toggleDocumentInlineStyles(node, disable)
{
    // If the node exists and is an element
    if(node && node.nodeType == Node.ELEMENT_NODE)
    {
        var childNodes       = node.childNodes;
        var childNodesLength = childNodes.length;

        // If disabling styles and the node has a style attribute
        if(disable && node.hasAttribute("style"))
        {
            node.setAttribute("webdeveloper-inline-style", node.getAttribute("style"));
            node.removeAttribute("style");
        }
        else if(!disable && node.hasAttribute("webdeveloper-inline-style"))
        {
            node.setAttribute("style", node.getAttribute("webdeveloper-inline-style"));
            node.removeAttribute("webdeveloper-inline-style");
        }

        // Loop through the child nodes
        for(var i = 0; i < childNodesLength; i++)
        {
            webdeveloper_toggleDocumentInlineStyles(childNodes[i], disable);
        }
    }
}

// Toggles all the embedded styles on the page
function webdeveloper_toggleEmbeddedStyles(element, applyStyle)
{
    var disable          = element.getAttribute("checked");
    var documentList     = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength   = documentList.length;
    var pageDocument     = null;
    var styleSheetLength = null;
    var styleSheetList   = null;
    var styleSheet       = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument     = documentList[i];
        styleSheetList   = pageDocument.getElementsByTagName("style");
        styleSheetLength = styleSheetList.length;

        // Loop through all the stylesheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j].sheet;

            // If this is a valid style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet))
            {
                styleSheet.disabled = disable;
            }
        }

    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-disable-embedded-styles", applyStyle);
}

// Toggles all the inline styles in elements on the page
function webdeveloper_toggleInlineStyles(element, applyStyle)
{
    var disable        = element.getAttribute("checked");
    var documentList   = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength = documentList.length;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        webdeveloper_toggleDocumentInlineStyles(documentList[i].documentElement, disable);
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-disable-inline-styles", applyStyle);
}

// Toggles all the linked styles on the page
function webdeveloper_toggleLinkedStyles(element, applyStyle)
{
    var disable          = element.getAttribute("checked");
    var documentList     = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength   = documentList.length;
    var ownerNode        = null;
    var pageDocument     = null;
    var styleSheet       = null;
    var styleSheetLength = null;
    var styleSheetList   = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument     = documentList[i];
        styleSheetList   = pageDocument.styleSheets;
        styleSheetLength = styleSheetList.length;

        // Loop through all the stylesheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If this is a valid style sheet, is not an inline style sheet and is not an alternate style sheet or style sheets are being disabled
            if(webdeveloper_isValidStyleSheet(styleSheet) && styleSheet.href != pageDocument.documentURI && (!webdeveloper_isAlternateStyleSheet(styleSheet) || disable))
            {
                styleSheet.disabled = disable;
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-disable-linked-styles", applyStyle);
}

// Toggles the print styles for the page
function webdeveloper_togglePrintStyles(element, applyStyle)
{
    var disable         = element.getAttribute("checked");
    var documentList    = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength  = documentList.length;
    var pageDocument    = null;
    var styleSheet      = null;
    var styleSheetLength = null;
    var styleSheetList  = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument     = documentList[i];
        styleSheetList   = pageDocument.styleSheets;
        styleSheetLength = styleSheetList.length;

        // Loop through the style sheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If the style sheet is valid, is not an alternate style sheet and is a print style sheet, but not a screen style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet) && !webdeveloper_isAlternateStyleSheet(styleSheet) && webdeveloper_isMediaStyleSheet(styleSheet, "print") && !webdeveloper_isMediaStyleSheet(styleSheet, "screen"))
            {
                styleSheet.disabled = disable;
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-disable-print-styles", applyStyle);
}

// Toggles the styles for the page
function webdeveloper_toggleStyles(element, applyStyle)
{
    var disable          = true;
    var contentWindow    = webdeveloper_getContentWindow();
    var documentList     = webdeveloper_getDocuments(contentWindow);
    var documentLength   = documentList.length;
    var key              = null;
    var ownerNode        = null;
    var pageDocument     = null;
    var styleElement     = null;
    var styleSheet       = null;
    var styleSheetLength = null;
    var styleSheetList   = null;

    // If the element is set
    if(element)
    {
        disable = element.getAttribute("checked");
    }
    else
    {
        var currentDocument = contentWindow.document;

        element = document.getElementById("webdeveloper-disable-all-styles-menu");

        // If the disable all styles element is set
        if(currentDocument.getElementById("webdeveloper-disable-all-styles"))
        {
            disable = false;
        }

        webdeveloper_configureElement(element, "checked", disable);
    }

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument     = documentList[i];
        styleSheetList   = pageDocument.styleSheets;
        styleSheetLength = styleSheetList.length;

        // Loop through all the stylesheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If this is a valid style sheet and is not an alternate style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet) && (!webdeveloper_isAlternateStyleSheet(styleSheet) || disable))
            {
                styleSheet.disabled = disable;
            }
        }

        webdeveloper_toggleDocumentInlineStyles(pageDocument.documentElement, disable);
    }

    // Toggle other CSS feature keyboard shortcuts
    document.getElementById("webdeveloper-edit-css-key").setAttribute("disabled", disable);
    document.getElementById("webdeveloper-view-style-information-key").setAttribute("disabled", disable);

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/empty.css", "webdeveloper-disable-all-styles", applyStyle);
}

// Updates the CSS menu
function webdeveloper_updateCSSMenu(suffix)
{
    var currentDocument         = webdeveloper_getContentDocument();
    var disableAllStylesChecked = false;

    // If the element exists
    if(currentDocument.getElementById("webdeveloper-disable-all-styles"))
    {
        disableAllStylesChecked = true;
    }

    webdeveloper_configureElement(document.getElementById("webdeveloper-add-user-style-sheet-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-display-media-type-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-edit-css-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-individual-style-sheet-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-use-border-box-model-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-view-style-information-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-add-user-style-sheet-" + suffix), "checked", "webdeveloper-add-user-style-sheet");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-use-border-box-model-" + suffix), "checked", "webdeveloper-use-border-box-model");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-view-style-information-" + suffix), "checked", "webdeveloper-view-style-information");
}

// Updates the CSS media type menu
function webdeveloper_updateCSSMediaTypeMenu(suffix)
{
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-handheld-css-" + suffix), "checked", "webdeveloper-display-handheld-css");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-print-css-" + suffix), "checked", "webdeveloper-display-print-css");
}

// Updates the disable styles menu
function webdeveloper_updateDisableStylesMenu(suffix)
{
    var currentDocument         = webdeveloper_getContentDocument();
    var disableAllStylesChecked = false;

    // If the element exists
    if(currentDocument.getElementById("webdeveloper-disable-all-styles"))
    {
        disableAllStylesChecked = true;
    }

    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-all-styles-" + suffix), "checked", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-browser-default-styles-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-embedded-styles-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-inline-styles-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-linked-styles-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElement(document.getElementById("webdeveloper-disable-print-styles-" + suffix), "disabled", disableAllStylesChecked);
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-disable-browser-default-styles-" + suffix), "checked", "webdeveloper-disable-browser-default-styles");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-disable-embedded-styles-" + suffix), "checked", "webdeveloper-disable-embedded-styles");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-disable-inline-styles-" + suffix), "checked", "webdeveloper-disable-inline-styles");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-disable-linked-styles-" + suffix), "checked", "webdeveloper-disable-linked-styles");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-disable-print-styles-" + suffix), "checked", "webdeveloper-disable-print-styles");
}

// View CSS
function webdeveloper_viewCSS()
{
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var inlineStylesText  = "";
    var linkElement       = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headerElement     = generatedDocument.createElement("h1");
    var pageDocument      = null;
    var preElement        = null;
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");
    var styleSheet        = null;
    var styleSheetHref    = null;
    var styleSheetList    = new Array();
    var styleSheetLength  = null;
    var title             = stringBundle.getFormattedString("webdeveloper_viewCSSTitle", [oldURL]);

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        headerElement    = generatedDocument.createElement("h2");
        inlineStylesText = "";
        linkElement      = generatedDocument.createElement("a");
        pageDocument     = documentList[i];
        documentURL      = pageDocument.documentURI;
        styleSheetList   = pageDocument.getElementsByTagName("style");
        styleSheetLength = styleSheetList.length;

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // Loop through the inline style sheets
        for(var j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If this is a valid style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet.sheet))
            {
                inlineStylesText += styleSheet.innerHTML.trim() + "\n\n";
            }
        }

        // If there are inline styles
        if(inlineStylesText != "")
        {
            headerElement = generatedDocument.createElement("h2");
            preElement    = generatedDocument.createElement("pre");

            headerElement.appendChild(generatedDocument.createTextNode(stringBundle.getFormattedString("webdeveloper_embeddedStylesFrom", [documentURL])));
            generatedDocument.body.appendChild(headerElement);
            preElement.appendChild(generatedDocument.createTextNode(inlineStylesText));
            generatedDocument.body.appendChild(preElement);
        }

        styleSheetList   = webdeveloper_getStyleSheetsForDocument(pageDocument, true, true);
        styleSheetLength = styleSheetList.length;

        // Loop through the style sheets
        for(j = 0; j < styleSheetLength; j++)
        {
            styleSheet = styleSheetList[j];

            // If this is a valid style sheet and is not an inline style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet) && styleSheet.href != documentURL)
            {
                headerElement  = generatedDocument.createElement("h2");
                linkElement    = generatedDocument.createElement("a");
                preElement     = generatedDocument.createElement("pre");
                styleSheetHref = styleSheet.href;

                linkElement.setAttribute("href", styleSheetHref);
                linkElement.appendChild(generatedDocument.createTextNode(styleSheetHref));
                headerElement.appendChild(linkElement);
                generatedDocument.body.appendChild(headerElement);

                preElement.appendChild(generatedDocument.createTextNode(webdeveloper_retrieveSource(styleSheetHref).replace(new RegExp("\r", "gi"), "\n")));
                generatedDocument.body.appendChild(preElement);
            }
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View style information
function webdeveloper_viewStyleInformation(element, applyStyle)
{
    var checked        = true;
    var contentWindow  = webdeveloper_getContentWindow();
    var documentList   = webdeveloper_getDocuments(contentWindow);
    var documentLength = documentList.length;
    var pageDocument   = null;

    // If the element is set
    if(element)
    {
        checked = element.getAttribute("checked");
    }
    else
    {
        var currentDocument = contentWindow.document;

        element = document.getElementById("webdeveloper-view-style-information-menu");

        // If the view style information element is set
        if(currentDocument.getElementById("webdeveloper-view-style-information"))
        {
            checked = false;
        }

        webdeveloper_configureElement(element, "checked", checked);
    }

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If viewing style information
        if(checked)
        {
            pageDocument.addEventListener("mouseover", webdeveloper_displayElementAncestors, true);
            pageDocument.addEventListener("click", webdeveloper_viewStyleInformationForElement, true);
        }
        else
        {
            var sidebar = document.getElementById("sidebar-box");

            // Try to remove the event listener
            try
            {
                pageDocument.removeEventListener("mouseover", webdeveloper_displayElementAncestors, true);
            }
            catch(exception)
            {
                // Do nothing
            }

            // Try to remove the event listener
            try
            {
                pageDocument.removeEventListener("click", webdeveloper_viewStyleInformationForElement, true);
            }
            catch(exception)
            {
                // Do nothing
            }

            getBrowser().contentWindow.status = null;

            // If the sidebar exists, is visible and is showing view style information
            if(sidebar && !sidebar.getAttribute("hidden") && sidebar.hasAttribute("src") && sidebar.getAttribute("src") == "chrome://webdeveloper/content/sidebar/view_style_information.xul")
            {
                toggleSidebar("webdeveloper-view-style-information-sidebar");
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/generated/view_style_information.css", "webdeveloper-view-style-information", applyStyle);
}

// View style information for an element
function webdeveloper_viewStyleInformationForElement(event)
{
    var eventTarget = event.target;

    // If there is an event target and the click was not a right click
    if(eventTarget && event.button != 2)
    {
        var displayInSidebar = webdeveloper_getBooleanPreference("webdeveloper.view.style.information.sidebar", true);
        var domUtils         = Components.classes["@mozilla.org/inspector/dom-utils;1"].getService(Components.interfaces.inIDOMUtils);
        var elementRule      = null;
        var headerElement    = null;
        var line             = null;
        var oldURL           = getBrowser().currentURI.spec;
        var ruleList         = domUtils.getCSSStyleRules(eventTarget);
        var stringBundle     = document.getElementById("webdeveloper-string-bundle");
        var styleRule        = null;
        var styleRuleLength  = null;
        var styleRuleList    = null;
        var styleSheet       = null;
        var styleSheetArray  = new Array();
        var styleSheetHref   = null;
        var styleText        = null;

        // Loop through the element rules
        for(var i = 0; i < ruleList.Count(); i++)
        {
            elementRule = ruleList.GetElementAt(i).QueryInterface(Components.interfaces.nsIDOMCSSStyleRule);
            line        = domUtils.getRuleLine(elementRule);

            // If there is a parent style sheet
            if(elementRule.parentStyleSheet)
            {
                styleSheet = elementRule.parentStyleSheet;
            }

            // If this is a valid style sheet
            if(webdeveloper_isValidStyleSheet(styleSheet))
            {
                styleRuleList   = elementRule.style;
                styleRuleLength = styleRuleList.length;
                styleSheetHref  = styleSheet.href;
                styleText       = elementRule.selectorText + "    (" + stringBundle.getString("webdeveloper_line").toLowerCase() + " " + line + ")\n{";

                // Loop through the style rules
                for(var j = 0; j < styleRuleLength; j++)
                {
                    styleRule  = styleRuleList[j];
                    styleText += "\n    " + styleRule + ": " + styleRuleList.getPropertyValue(styleRule) + ";";
                }

                styleText += "\n}\n\n";

                // If this style sheet has rules already stored
                if(styleSheetArray[styleSheetHref])
                {
                    styleSheetArray[styleSheetHref] += styleText;
                }
                else
                {
                    styleSheetArray[styleSheetHref] = styleText;
                }
            }
        }

        // If displaying in the sidebar
        if(displayInSidebar)
        {
            toggleSidebar("webdeveloper-view-style-information-sidebar", true);

            // This sets a small timeout to guarantee the sidebar is open
            window.setTimeout(webdeveloper_outputStyleInformationForElement, 500, displayInSidebar, styleSheetArray, stringBundle.getFormattedString("webdeveloper_viewStyleInformationTitle", [oldURL]));
        }
        else
        {
            var oldTab = getBrowser().selectedTab;

            webdeveloper_outputStyleInformationForElement(displayInSidebar, styleSheetArray, stringBundle.getFormattedString("webdeveloper_viewStyleInformationTitle", [oldURL]));

            // If the open tabs in background preference is set to true
            if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
            {
                getBrowser().selectedTab = oldTab;
            }
        }

        event.preventDefault();
    }
}
