#include <signal.h>
#include "fm.h"

#ifdef USE_MOUSE
#ifdef USE_GPM
#include <gpm.h>
#endif
#if defined(USE_GPM) || defined(USE_SYSMOUSE)
extern int do_getch();
#define getch()	do_getch()
#endif				/* USE_GPM */
#endif				/* USE_MOUSE */

#ifdef __EMX__
#include <sys/kbdscan.h>
#include <strings.h>
#endif
char *NullLine = "";
Lineprop NullProp[] = {0};

/* 
 * Buffer creation
 */
void
initBuffer(Buffer *n, int width)
{
    bzero((void *) n, sizeof(Buffer));
    n->reserved = (LineTruncated && *LineTruncated) ? 1 : 0;
    n->width = width - n->reserved;
    n->height = LASTLINE + 1;
    n->currentURL.scheme = SCM_UNKNOWN;
    n->baseURL = NULL;
    n->baseTarget = NULL;
    n->buffername = "";
    n->bufferprop = BP_NORMAL;
    n->linelen = 0;
    n->trbyte = 0;
#ifdef USE_SSL
    n->ssl_certificate = NULL;
#endif
    n->http_response_code = -1;
}

Buffer *
newBuffer(int width)
{
    Buffer *n;

    n = New(Buffer);
    initBuffer(n, width);
    return n;
}

/* 
 * Create null buffer
 */
Buffer *
nullBuffer(void)
{
    Buffer *b;

    b = newBuffer(INIT_BUFFER_WIDTH);
    b->buffername = "*Null*";
    return b;
}

/* 
 * clearBuffer: clear buffer content
 */
void
clearBuffer(Buffer * buf)
{
    buf->firstLine = buf->topLine = buf->currentLine = buf->lastLine = NULL;
    buf->allLine = 0;
}

/* 
 * discardBuffer: free buffer structure
 */

void
discardBuffer(Buffer * buf)
{
    int i;
    Buffer *b, *s;

    if (buf->bufferprop & BP_DISCARDED)
	return;

    clearBuffer(buf);
    buf->bufferprop |= BP_DISCARDED;
    s = buf->linkBuffer[LB_SOURCE];
    for (i = 0; i < MAX_LB; i++)
	if ((b = buf->linkBuffer[i]))
	    b->linkBuffer[REV_LB[i]] = NULL;
    if (buf->savecache)
	unlink(buf->savecache);
    if (s)
	return;

    if (buf->async_buf)
	discardAsyncRWBuf(buf->async_buf, TRUE);

    if ((buf->bufferprop & BP_ASYNC_MASK) != BP_ASYNC_DONE) {
	if (buf->sourcefile && (buf->real_scheme != SCM_LOCAL))
	    unlink(buf->sourcefile);
	while (buf->frameset) {
	    deleteFrameSet(buf->frameset);
	    buf->frameset = popFrameTree(&(buf->frameQ));
	}
    }
}

/* 
 * namedBuffer: Select buffer which have specified name
 */
Buffer *
namedBuffer(Buffer * first, char *name)
{
    Buffer *buf;

    if (!strcmp(first->buffername, name)) {
	return first;
    }
    for (buf = first; buf->nextBuffer != NULL; buf = buf->nextBuffer) {
	if (!strcmp(buf->nextBuffer->buffername, name)) {
	    return buf->nextBuffer;
	}
    }
    return NULL;
}

Buffer *
mkunlinkedBuffer(Buffer *first, Buffer *buf, Buffer **prev)
{
    if (prev)
	*prev = NULL;

    if (buf && (buf->bufferprop & BP_LINKED)) {
	if (buf == first)
	    first = buf->nextBuffer;
	else {
	    Buffer *cur;

	    if ((cur = prevBuffer(first, buf))) {
		setNextBuffer(cur, buf->nextBuffer);

		if (prev)
		    *prev = cur;
	    }
	}

	setNextBuffer(buf, NULL);
	buf->bufferprop &= ~BP_LINKED;
    }

    return first;
}

/* 
 * deleteBuffer: delete buffer
 */
Buffer *
deleteBuffer(Buffer * first, Buffer * delbuf)
{
    if (delbuf) {
	first = mkunlinkedBuffer(first, delbuf, NULL);
	discardBuffer(delbuf);
    }

    return first;
}

/* 
 * replaceBuffer: replace buffer
 */
Buffer *
replaceBuffer(Buffer * first, Buffer * delbuf, Buffer * newbuf)
{
    Buffer *prev;

    first = mkunlinkedBuffer(first, newbuf, NULL);
    first = mkunlinkedBuffer(first, delbuf, &prev);

    if (delbuf != newbuf)
	discardBuffer(delbuf);

    if (prev) {
	setNextBuffer(newbuf, prev->nextBuffer);
	setNextBuffer(prev, newbuf);
    }
    else {
	setNextBuffer(newbuf, first);
	first = newbuf;
    }

    newbuf->bufferprop |= BP_LINKED;
    return first;
}

Buffer *
insertBuffer(Buffer *first, Buffer *next, Buffer *newbuf)
{
    Buffer *prev;

    first = mkunlinkedBuffer(first, newbuf, NULL);

    if ((prev = prevBuffer(first, next))) {
	setNextBuffer(newbuf, prev->nextBuffer);
	setNextBuffer(prev, newbuf);
    }
    else {
	setNextBuffer(newbuf, first);
	first = newbuf;
    }

    newbuf->bufferprop |= BP_LINKED;
    return first;
}

Buffer *
nthBuffer(Buffer * firstbuf, int n)
{
    int i;
    Buffer *buf = firstbuf;

    if (n < 0)
	return firstbuf;
    for (i = 0; i < n; i++) {
	if (buf == NULL)
	    return NULL;
	buf = buf->nextBuffer;
    }
    return buf;
}

static void
writeBufferName(Buffer * buf, int n)
{
    Str msg;
    int all;

    all = buf->allLine;
    if (all == 0 && buf->lastLine != NULL)
	all = buf->lastLine->linenumber;
    move(n, 0);
    msg = Sprintf("<%s> [%d lines]", buf->buffername, all);
    if (buf->filename != NULL) {
	switch (buf->currentURL.scheme) {
	case SCM_LOCAL:
	case SCM_LOCAL_CGI:
	    if (strcmp(buf->currentURL.file, "-")) {
		Strcat_char(msg, ' ');
		Strcat_charp(msg, conv_from_system(buf->currentURL.real_file));
	    }
	    break;
	case SCM_UNKNOWN:
	case SCM_MISSING:
	    break;
	default:
	    Strcat_char(msg, ' ');
	    Strcat(msg, parsedURL2Str(&buf->currentURL));
	    break;
	}
    }
    addnstr_sup(msg->ptr, COLS - 1);
}


static void
incomplete_loading_info(Buffer *buf, char *msgbuf)
{
    if (buf->async_buf) {
	int rfd = buf->async_buf->rfd;

	msgbuf += strlen(msgbuf);

	if (is_recorded_read_fd(rfd))
	    sprintf(msgbuf, " (loading is incomplete, please wait)");
	else if (rfd >= 0 && is_hook_recorded_read_fd(rfd)) {
	    sprintf(msgbuf, " (loading is restarted, please retry)");
	    record_read_fd(rfd, NULL, NULL, NULL);
	}
    }
}

/* 
 * gotoLine: go to line number
 */
void
gotoLine(Buffer * buf, int n)
{
    char msg[128];
    Line *l = buf->firstLine;

    if (l == NULL)
	return;
    if (l->linenumber > n) {
	sprintf(msg, "First line is #%ld", l->linenumber);
	set_delayed_message(msg);
	buf->topLine = buf->currentLine = l;
	return;
    }
    if (buf->lastLine->linenumber < n) {
	l = buf->lastLine;
	sprintf(msg, "Last line is #%ld", l->linenumber);
	incomplete_loading_info(buf, msg);
	set_delayed_message(msg);
	if (l->linenumber - buf->topLine->linenumber >= buf->height - 2)
	    buf->topLine = lineSkip(buf, l, buf->topLine->linenumber - buf->currentLine->linenumber, 0);
	buf->currentLine = l;
	return;
    }
    for (; l != NULL; l = l->next) {
	if (l->linenumber >= n) {
	    if (n < buf->topLine->linenumber || buf->topLine->linenumber + buf->height - 1 <= n)
		buf->topLine = lineSkip(buf, l, buf->topLine->linenumber - buf->currentLine->linenumber, 0);
	    buf->currentLine = l;
	    break;
	}
    }
}

/* 
 * gotoRealLine: go to real line number
 */
void
gotoRealLine(Buffer * buf, int n)
{
    char msg[128];
    Line *l;

    if ((l = buf->firstLine) == NULL)
	return;
    if (l->real_linenumber > n) {
	sprintf(msg, "First line is #%ld", l->real_linenumber);
	set_delayed_message(msg);
	buf->topLine = buf->currentLine = l;
	return;
    }
    for (l = buf->lastLine ; l && !l->real_linenumber ; l = l->next)
	;
    if (!l)
	return;
    if (l->real_linenumber < n) {
	sprintf(msg, "Last line is #%ld", l->real_linenumber);
	incomplete_loading_info(buf, msg);
	set_delayed_message(msg);
	if (l->linenumber - buf->topLine->linenumber >= buf->height - 2)
	    buf->topLine = lineSkip(buf, l, buf->topLine->linenumber - buf->currentLine->linenumber, 0);
	buf->currentLine = l;
	return;
    }
    for (l = buf->firstLine; l != NULL; l = l->next) {
	if (l->real_linenumber >= n) {
	    Line *tl;

	    for (tl = buf->topLine ; tl && !tl->real_linenumber ; tl = tl->next)
		;
	    if (tl && tl->real_linenumber &&
		(n < tl->real_linenumber || tl->real_linenumber + buf->height - 1 <= n))
		buf->topLine = lineSkip(buf, l, buf->topLine->linenumber - buf->currentLine->linenumber, 0);
	    buf->currentLine = l;
	    break;
	}
    }
}


static Buffer *
listBuffer(Buffer * top, Buffer * current)
{
    int i, c = 0;
    Buffer *buf = top;

    move(0, 0);
#ifdef USE_COLOR
    if (useColor) {
	setfcolor(basic_color);
#ifdef USE_BG_COLOR
	setbcolor(bg_color);
#endif				/* USE_BG_COLOR */
    }
#endif				/* USE_COLOR */
    clrtobotx();
    for (i = 0; i < LASTLINE; i++) {
	if (buf == current) {
	    c = i;
	    standout();
	}
	writeBufferName(buf, i);
	if (buf == current) {
	    standend();
	    clrtoeolx();
	    move(i, 0);
	    toggle_stand();
	}
	else
	    clrtoeolx();
	if (buf->nextBuffer == NULL) {
	    move(i + 1, 0);
	    clrtobotx();
	    break;
	}
	buf = buf->nextBuffer;
    }
    standout();
    message("Buffer selection mode: SPC for select / D for delete buffer");
    standend();
    /* 
     * move(LASTLINE, COLS - 1); */
    move(c, 0);
    refresh();
    return buf->nextBuffer;
}


/* 
 * Select buffer visually
 */
Buffer *
selectBuffer(Buffer * firstbuf, Buffer * currentbuf, char *selectchar)
{
    int i, cpoint,		/* Current Buffer Number */
	spoint,			/* Current Line on Screen */
	maxbuf, sclimit = LASTLINE;	/* Upper limit of line * number in 
					 * the * screen */
    Buffer *buf, *topbuf;
    char c;

    i = cpoint = 0;
    for (buf = firstbuf; buf != NULL; buf = buf->nextBuffer) {
	if (buf == currentbuf)
	    cpoint = i;
	i++;
    }
    maxbuf = i;

    if (cpoint >= sclimit) {
	spoint = sclimit / 2;
	topbuf = nthBuffer(firstbuf, cpoint - spoint);
    }
    else {
	topbuf = firstbuf;
	spoint = cpoint;
    }
    listBuffer(topbuf, currentbuf);

    for (;;) {
	if ((c = getch()) == ESC_CODE) {
	    if ((c = getch()) == '[' || c == 'O') {
		switch (c = getch()) {
		case 'A':
		    c = 'k';
		    break;
		case 'B':
		    c = 'j';
		    break;
		case 'C':
		    c = ' ';
		    break;
		case 'D':
		    c = 'B';
		    break;
		}
	    }
	}
#ifdef __EMX__
	else if(!c)
	    switch(getch()){
	    case K_UP:
		c='k';
		break;
	    case K_DOWN:
		c='j';
		break;
	    case K_RIGHT:
		c=' ';
		break;
	    case K_LEFT:
		c='B';
	    }
#endif
	switch (c) {
	case CTRL_N:
	case 'j':
	    if (spoint < sclimit - 1) {
		if (currentbuf->nextBuffer == NULL)
		    continue;
		writeBufferName(currentbuf, spoint);
		currentbuf = currentbuf->nextBuffer;
		cpoint++;
		spoint++;
		standout();
		writeBufferName(currentbuf, spoint);
		standend();
		move(spoint, 0);
		toggle_stand();
	    }
	    else if (cpoint < maxbuf - 1) {
		topbuf = currentbuf;
		currentbuf = currentbuf->nextBuffer;
		cpoint++;
		spoint = 1;
		listBuffer(topbuf, currentbuf);
	    }
	    break;
	case CTRL_P:
	case 'k':
	    if (spoint > 0) {
		writeBufferName(currentbuf, spoint);
		currentbuf = nthBuffer(topbuf, --spoint);
		cpoint--;
		standout();
		writeBufferName(currentbuf, spoint);
		standend();
		move(spoint, 0);
		toggle_stand();
	    }
	    else if (cpoint > 0) {
		i = cpoint - sclimit;
		if (i < 0)
		    i = 0;
		cpoint--;
		spoint = cpoint - i;
		currentbuf = nthBuffer(firstbuf, cpoint);
		topbuf = nthBuffer(firstbuf, i);
		listBuffer(topbuf, currentbuf);
	    }
	    break;
	default:
	    *selectchar = c;
	    return currentbuf;
	}
	/* 
	 * move(LASTLINE, COLS - 1);
	 */
	move(spoint, 0);
	refresh();
    }
}


/* shallow copy */
void
copyBuffer(Buffer * a, Buffer * b)
{
    readBufferCache(b);
    bcopy((void *)b, (void *)a, sizeof(Buffer));
}

Buffer *
prevBuffer(Buffer * first, Buffer * buf)
{
    Buffer *b;

    for (b = first; b != NULL && b->nextBuffer != buf; b = b->nextBuffer);
    return b;
}

#define fwrite1(d, f) (fwrite(&d, sizeof(d), 1, f)==0)
#define fread1(d, f) (fread(&d, sizeof(d), 1, f)==0)

int
writeBufferCache(Buffer *buf)
{
    Str tmp;
    FILE *cache = NULL;
    Line *l;
#ifdef USE_ANSI_COLOR
    int colorflag;
#endif

    if (buf->savecache)
	return -1;

    if (buf->firstLine == NULL)
	goto _error1;

    tmp = tmpfname(TMPF_CACHE, NULL);
    buf->savecache = tmp->ptr;
    cache = fopen(buf->savecache, "w");
    if (!cache)
	goto _error1;

    if (fwrite1(buf->currentLine->linenumber, cache) ||
	fwrite1(buf->topLine->linenumber, cache))
	goto _error;

    for (l = buf->firstLine; l; l = l->next) {
	if (fwrite1(l->real_linenumber, cache) ||
	    fwrite1(l->usrflags, cache) ||
	    fwrite1(l->width, cache) ||
	    fwrite1(l->len, cache) ||
	    fwrite(l->lineBuf, 1, l->len, cache) < l->len ||
	    fwrite(l->propBuf, sizeof(Lineprop), l->len, cache) < l->len)
	    goto _error;
#ifdef USE_ANSI_COLOR
        colorflag = l->colorBuf ? 1 : 0;
	if (fwrite1(colorflag, cache))
	    goto _error;
	if (colorflag) {
	    if (fwrite(l->colorBuf, sizeof(Linecolor), l->len, cache) < l->len)
		goto _error;
	}
#endif
    }

    fclose(cache);
    return 0;
_error:
    fclose(cache);
    unlink(buf->savecache);
_error1:
    buf->savecache = NULL;
    return -1;
}

int
readBufferCache(Buffer *buf)
{
    FILE *cache;
    Line *l = NULL, *prevl = NULL;
    long lnum = 0, clnum, tlnum;
#ifdef USE_ANSI_COLOR
    int colorflag;
#endif

    if (buf->savecache == NULL)
	return -1;

    cache = fopen(buf->savecache, "r");
    if (cache == NULL ||
	fread1(clnum, cache) ||
	fread1(tlnum, cache)) {
	buf->savecache = NULL;
	return -1;
    }

    while (!feof(cache)) {
	lnum++;
	prevl = l;
	l = New(Line);
	l->prev = prevl;
	if (prevl)
	    prevl->next = l;
	else
	    buf->firstLine = l;
	l->linenumber = lnum;
	if (lnum == clnum)
	    buf->currentLine = l;
	if (lnum == tlnum)
	    buf->topLine = l;
	if (fread1(l->real_linenumber, cache) ||
	    fread1(l->usrflags, cache) ||
	    fread1(l->width, cache) ||
	    fread1(l->len, cache))
	    break;
	l->lineBuf = NewAtom_N(char, l->len + 1);
	fread(l->lineBuf, 1, l->len, cache);
	l->lineBuf[l->len] = '\0';
	l->propBuf = NewAtom_N(Lineprop, l->len);
	fread(l->propBuf, sizeof(Lineprop), l->len, cache);
#ifdef USE_ANSI_COLOR
	if (fread1(colorflag, cache))
	    break;
	if (colorflag) {
	    l->colorBuf = NewAtom_N(Linecolor, l->len);
	    fread(l->colorBuf, sizeof(Linecolor), l->len, cache);
	} else {
	    l->colorBuf = NULL;
	}
#endif
    }
    buf->lastLine = prevl;
    if (prevl) prevl->next = NULL;
    fclose(cache);
    unlink(buf->savecache);
    buf->savecache = NULL;
    return 0;
}

/* Local Variables:    */
/* c-basic-offset: 4   */
/* tab-width: 8        */
/* End:                */
