/*
 * Draw quadratics
 */

#include <GL/gl.h>

#include "global.h"
#include "zv.h"
#include "draw.h"

#ifndef VRENGD


void drawPoint(float x, float y)
{
    glBegin(GL_POINTS);
    glVertex2f(x, y);
    glEnd();
}

void drawLine(float x, float y)
{
    glBegin(GL_LINES);
    glVertex2f(x, 0);
    glVertex2f(0, y);
    glEnd();
}

void drawRect(float x, float y, int style)
{
    switch (style) {
    case STYLE_FILL:
      glBegin(GL_QUADS);
      break;
    case STYLE_LINES:
      glBegin(GL_LINE_LOOP);
      break;
    case STYLE_POINTS:
      glBegin(GL_POINTS);
      break;
    }
    glTexCoord2f(0,0); glVertex3f(-x, -y, 0);
    glTexCoord2f(1,0); glVertex3f(x,-y, 0);
    glTexCoord2f(1,1); glVertex3f(x, y, 0);
    glTexCoord2f(0,1); glVertex3f(-x, y, 0);
    glEnd();
}

/*
 * Draw a box (taken from Mesa)
 */
void drawBox(float x0, float x1, float y0, float y1, float z0, float z1, int type, int textures[], int style)
{
  static float normale[6][3] = {
    {-1.0, 0.0, 0.0},
    {0.0, 1.0, 0.0},
    {1.0, 0.0, 0.0},
    {0.0, -1.0, 0.0},
    {0.0, 0.0, 1.0},
    {0.0, 0.0, -1.0}
  };
  static int faces[6][4] = {
    { 0, 1, 2, 3 },
    { 3, 2, 6, 7 },
    { 7, 6, 5, 4 },
    { 4, 5, 1, 0 },
    { 5, 6, 2, 1 },
    { 7, 4, 0, 3 }
  };
  static float texcoords[6][4][2] = {
    { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
    { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
    { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
    { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
    { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } },
    { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } }
  };
  float v[8][3], tmp;
  int i,j;

  if (x0 > x1) {
    tmp = x0; x0 = x1; x1 = tmp;
  }
  if (y0 > y1) {
    tmp = y0; y0 = y1; y1 = tmp; 
  }
  if (z0 > z1) {
    tmp = z0; z0 = z1; z1 = tmp; 
  }
  v[0][0] = v[1][0] = v[2][0] = v[3][0] = x0;
  v[4][0] = v[5][0] = v[6][0] = v[7][0] = x1;
  v[0][1] = v[1][1] = v[4][1] = v[5][1] = y0;
  v[2][1] = v[3][1] = v[6][1] = v[7][1] = y1;
  v[0][2] = v[3][2] = v[4][2] = v[7][2] = z0;
  v[1][2] = v[2][2] = v[5][2] = v[6][2] = z1;

  for (i = 0; i < 6; i++) {
    if (textures[i] > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, textures[i]);
    }
    glBegin(type);
    for (j=0; j < 4; j++) {
      if (textures[i] > 0)
        glTexCoord2fv(&texcoords[i][j][0]);
      glNormal3fv(&normale[i][0]);
      zvVertex3fv(&v[faces[i][j]][0]);
    }
    glEnd();
    if (textures[i] > 0)
      glDisable(GL_TEXTURE_2D);
  }
}

void drawOctagon(float side, float height, int style)
{
  char j;
  float x = sin(0.785398163) * side, y = side / 2.0, z = height / 2.0, c;

  c = x + y;
  for (j = 0; j < 8; j++) {
    glTranslatef(-c, 0.0, 0.0);
    if (style != STYLE_FILL)
      glBegin(GL_LINE_LOOP);
    else
      glBegin(GL_QUADS);
    glNormal3f(-1.0, 0.0, 0.0);
    glVertex3f(0.0, -y, z);
    glVertex3f(0.0, y, z);
    glVertex3f(0.0, y, -z);
    glVertex3f(0.0, -y, -z);
    glEnd();
    glTranslatef(c, 0.0, 0.0);
    if (style == STYLE_FILL) {
      glBegin(GL_TRIANGLES);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(0.0, 0.0, z);
      glVertex3f(-c, -y, z);
      glVertex3f(-c, y, z);
      glNormal3f(0.0, 0.0, -1.0);
      glVertex3f(0.0, 0.0, -z);
      glVertex3f(-c, y, -z);
      glVertex3f(-c, -y, -z);
      glEnd();
    }
    glRotatef(45.0, 0.0, 0.0, 1.0);
  }
}

/*
 * Torus drawing (taken from Mesa)
 */
void drawTorus(float rc, int numc, float rt, int numt, int style)
{
  int i, j, k;
  double s, t;
  double x, y, z;

  for (i = 0; i < numc; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j <= numt; j++) {
      for (k = 1; k >= 0; k--) {
        s = (i + k) % numc + 0.5;
        t = j % numt;
        x = Cos(t*M_2PI/numt) * Cos(s*M_2PI/numc);
        y = Sin(t*M_2PI/numt) * Cos(s*M_2PI/numc);
        z = Sin(s*M_2PI/numc);
        glNormal3f(x, y, z);

        x = (rt + rc * Cos(s*M_2PI/numc)) * Cos(t*M_2PI/numt);
        y = (rt + rc * Cos(s*M_2PI/numc)) * Sin(t*M_2PI/numt);
        z = rc * Sin(s*M_2PI/numc);
        zvVertex3f(x, y, z);
      }
    }
    glEnd();
  }
}

/*
 * Sphere (taken from Mesa)
 */
void drawSphere(float radius, int slices, int stacks, int style)
{
  float rho, drho, theta, dtheta;
  float x, y, z;
  float s, t, ds, dt;
  int i, j;
  int normals = 1;
  float nsign = 1;
  
  drho = M_PI / (float) stacks;
  dtheta = M_2PI / (float) slices;

  /* draw +Z end as a triangle fan */
  glBegin(GL_TRIANGLE_FAN);
  glNormal3f(0.0, 0.0, 1.0);
  glTexCoord2f(0.5, 0.0);
  zvVertex3f(0.0, 0.0, nsign*radius);
  for (j=0; j <= slices; j++) {
    theta = (j==slices) ? 0.0 : j*dtheta;
     x = -Sin(theta) * Sin(drho);
     y = Cos(theta) * Sin(drho);
     z = nsign * Cos(drho);
     if (normals)
       glNormal3f(x*nsign, y*nsign, z*nsign);
     zvVertex3f(x*radius, y*radius, z*radius);
  }
  glEnd();

  ds = 1.0 / slices;
  dt = 1.0 / stacks;
  t = 1.0;  /* because loop now runs from 0 */

  /* draw intermediate stacks as quad strips */
  for (i=0; i < stacks; i++) {
     rho = i * drho;
     glBegin(GL_QUAD_STRIP);
     s = 0.0;
     for (j=0; j <= slices; j++) {
       theta = (j==slices) ? 0.0 : j*dtheta;
       x = -Sin(theta) * Sin(rho);
       y = Cos(theta) * Sin(rho);
       z = nsign * Cos(rho);
       if (normals)
         glNormal3f(x*nsign, y*nsign, z*nsign);
       glTexCoord2f(s, 1-t);
       zvVertex3f(x*radius, y*radius, z*radius);
       x = -Sin(theta) * Sin(rho+drho);
       y = Cos(theta) * Sin(rho+drho);
       z = nsign * Cos(rho+drho);
       if (normals)
         glNormal3f(x*nsign, y*nsign, z*nsign);
       glTexCoord2f(s, 1-(t-dt));
       s += ds;
       zvVertex3f(x*radius, y*radius, z*radius);
    }
    glEnd();
    t -= dt;
  }
  /* draw -Z end as a triangle fan */
  glBegin(GL_TRIANGLE_FAN);
  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(0.5, 1.0);
  zvVertex3f(0.0, 0.0, -radius*nsign);
  rho = M_PI - drho;
  s = 1.0;
  t = dt;
  for (j=slices; j >= 0; j--) {
    theta = (j==slices) ? 0.0 : j*dtheta;
    x = -Sin(theta) * Sin(rho);
    y = Cos(theta) * Sin(rho);
    z = nsign * Cos(rho);
    if (normals)
      glNormal3f(x*nsign, y*nsign, z*nsign);
    glTexCoord2f(s, 1-t);
    s -= ds;
    zvVertex3f(x*radius, y*radius, z*radius);
  }
  glEnd();
}

/*
 * Cylinder (taken from Mesa, Brian Paul)
 */
void drawCylinder(float baseR, float topR, float height, int slices, int stacks, int style)
{
  int i, j;
  float da, r, dr, dz;
  float z, nz, nsign;
  float ds = 1.0 / slices;
  float dt = 1.0 / stacks;
  float t = 0.0;

  nsign = 1.0;
  da = M_2PI / slices;
  dr = (topR-baseR) / stacks;
  dz = height / stacks;
  nz = (baseR-topR) / height;  /* Z component of normal vectors */

  switch (style) {

  case STYLE_FILL:
    {
      z =0.0;
      r = baseR;
      for (j=0; j < stacks; j++) {
        float s = 0.0;
        glBegin(GL_QUAD_STRIP);
        for (i=0; i <= slices; i++) {
          float x, y;
          if (i == slices) {
            x = Sin(0.0);
            y = Cos(0.0);
          }
          else {
            x = Sin(i*da);
            y = Cos(i*da);
          }
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glTexCoord2f(s, t);
          zvVertex3f(x*r, y*r, z);
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glTexCoord2f(s, t+dt);
          zvVertex3f(x*(r+dr), y*(r+dr), z+dz);
          s += ds;
        }
        glEnd();
        r += dr;
        t += dt;
        z += dz;
      }
    }
    break;
 
  case STYLE_LINES:
    {
      z =0.0;
      r = baseR;
      for (j=0; j <= stacks; j++) {
        glBegin(GL_LINE_LOOP);
        for (i=0; i < slices; i++) {
          float x, y;
          x = Cos(i*da);
          y = Sin(i*da);
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glVertex3f(x*r, y*r, z);
        }
        glEnd();
        z += dz;
        r += dr;
      }
      /* draw length lines */
      glBegin(GL_LINES);
      for (i=0;i<slices;i++) {
        float x, y;
        x = Cos(i*da);
        y = Sin(i*da);
        glNormal3f(x*nsign, y*nsign, nz*nsign);
        glVertex3f(x*baseR, y*baseR, 0.0);
        glVertex3f(x*topR, y*topR, height);
      }
      glEnd();
    }
    break;

  case STYLE_POINTS:
    {
      glBegin(GL_POINTS);
      for (i=0; i < slices; i++) {
        float x, y;
        x = Cos(i*da);
        y = Sin(i*da);
        glNormal3f(x*nsign, y*nsign, nz*nsign);
        z = 0.0;
        r = baseR;
        for (j=0; j <= stacks; j++) {
          glVertex3f(x*r, y*r, z);
          z += dz;
          r += dr;
        }
      }
      glEnd();
    }
    break;
  }
}

/*
 * Disk (taken from Mesa, Brian Paul)
 */
void drawDisk(float innerR, float outerR, int slices, int loops, int style)
{
  float da, dr;

  /* Normal vectors */
  /* glNormal3f( 0.0, 0.0, +1.0 ); */
  glNormal3f( 0.0, 0.0, -1.0 );

  da = M_2PI / slices;
  dr = (outerR-innerR) / (float) loops;

  switch (style) {
  case STYLE_FILL:
    {
      /* texture of a gluDisk is a cut out of the texture unit square
       * x, y in [-outerR, +outerR]; s, t in [0, 1]
       * (linear mapping)
       */
      float dtc = 2.0f * outerR;
      float sa,ca;
      float r1 = innerR;
      int l;
      for (l=0; l<loops; l++) {
        float r2 = r1 + dr;
        int s;
        glBegin( GL_QUAD_STRIP );
        for (s=slices;s>=0;s--) {
          float a;
          if (s==slices) a = 0.0;
          else  a = s * da;
          sa = Sin(a); ca = Cos(a);
          glTexCoord2f(0.5-sa*r2/dtc,0.5+ca*r2/dtc);
          glVertex3f( r2*sa, r2*ca, 0 );
          glTexCoord2f(0.5-sa*r1/dtc,0.5+ca*r1/dtc);
          glVertex3f( r1*sa, r1*ca, 0 );
        }
        glEnd();
        r1 = r2;
      }
    }
    break;
  case STYLE_LINES:
    {
      int l, s;
      /* draw loops */
      for (l=0; l<=loops; l++) {
        float r = innerR + l * dr;
        glBegin( GL_LINE_LOOP );
        for (s=0; s<slices; s++) {
          float a = s * da;
          glVertex3f( r*Sin(a), r*Cos(a), 0 );
        }
        glEnd();
      }
      /* draw spokes */
      for (s=0; s<slices; s++) {
        float a = s * da;
        float x = Sin(a);
        float y = Cos(a);
        glBegin( GL_LINE_STRIP );
          for (l=0; l<=loops; l++) {
            float r = innerR + l * dr;
            glVertex3f( r*x, r*y, 0 );
          }
        glEnd();
      }
    }
    break;
  case STYLE_POINTS:
    {
      int s;
      glBegin( GL_POINTS );
      for (s=0; s<slices; s++) {
        float a = s * da;
        float x = Sin(a);
        float y = Cos(a);
        int l;
        for (l=0; l<=loops; l++) {
          float r = innerR * l * dr;
          glVertex3f( r*x, r*y, 0 );
        }
      }
      glEnd();
    }
    break;
  }
}

#endif /* !VRENGD */
