#ifndef VRENGD

#include <GL/gl.h>
#include <GL/glu.h>

#include <ubit.hh>

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"	/* COL_ONCE */
#include "vnc.h"
#include "vnc/vncauth.h"

#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"

#include "zv.h"		/* parseGeometry */


const WClass Vnc::wclass(VNC_TYPE, "Vnc", Vnc::creator);

//Used in connection dialog
extern GUI gui;

// Put the focus on the object
void vncTakeFocus (Vnc *po, void *data, time_t sec, time_t usec);
void vncLeaveFocus (Vnc *po, void *data, time_t sec, time_t usec);


/* Parse everything concerning the server if defined in the config file.*/
/*     syntax:   serve=servename,port,password                          */
char * parseVNCServer(char *ptok, Vnc *po)
{
  int port = 5901;	// default
  char *p;
  char args[256];
  char server[80];
  char passwd[20];

  memset(server, 0, sizeof(server));
  memset(passwd, 0, sizeof(passwd));
  if (ptok) {
    if (sscanf(ptok, "server=%s", args) != 1)
      return ptok;
    if ((p = strchr(args, ',')) == 0) {
      warning("parsing: can't get a port from the Fileline:\n%s", ptok);
      strcpy(server, args);
    }
    else {
      strncpy(server, args, p-args);
      p++;
      //PD if (sscanf(p, "%d,%s", &port, passwd) != 2)
      //PD   fatal("parsing: can't get a passwd from the Fileline:\n%s", ptok);
      sscanf(p, "%d,%s", &port, passwd);
      if (port == 0)
        port = 5901;
    }
    trace(DBG_VNC, "parseVNCServer: %s:%d:%s", server, port, passwd);

    strcpy(po->serverName, server);
    po->port = port;

    sprintf(po->passwdFile, "%s-%s", vrengpasswdfile, server);
    if (strlen(passwd) > 8)
      passwd[8] = '\0';
    vncEncryptAndStorePasswd(passwd, po->passwdFile);

    po->isServerDefined = true;
  }
  ptok = strtok(NULL, SEP);
  return ptok;
}

/* Vnc initialization from a file */
void Vnc::creator(char *l)
{
  new Vnc(l);
}

Vnc::Vnc(char *l)
{ 
  isServerDefined = false;
  isConnected = false;
  texture_bytes = 0;
  static GLfloat colordef[] = {0.3, 0.3, 0.3};	// grey
 
  // Get the user parametres of this object
  l = parseName(l, this);
  l = parsePosition(l, this);
  l = parseVNCServer(l, this);
  for (int i=0; i<3; i++)
    color[i] = colordef[i];

  char s[256];
  sprintf(s, "bbox,%s", l);
  if ((soh = parseGeometry(s)) == 0)
    return;

  l = strtok(NULL, SEP);
  if (l) {
    // parse screen color
    if (strncmp(l, "color", 5) == 0) {
      l = strchr(l, '=');
      color[0] = (float) atof(++l);
      l = strchr(l, ',');
      color[1] = (float) atof(++l);
      l = strchr(l, ',');
      color[2] = (float) atof(++l);
      l = strtok(NULL, SEP);
    }
  }

  // Rendering done by the VNC class instead of VRENG
  nature.renderable = VR_SPECIAL_RENDER;
  nature.movable = VR_NO_ELEM_MOVE;
  nature.collision = COL_NEVER;

  // Include this object in the list representing the different VRENG objects.
  initializeObject(this, VNC_TYPE, VR_MOBILE);
  
  char sdbg[256];
  sprintf(sdbg, "VNC BBox: %.2f,%.2f,%.2f - %.2f - %.2f,%.2f,%.2f",
          pos.x, pos.y, pos.z, pos.az,
          soh->bbsize.v[0], soh->bbsize.v[1], soh->bbsize.v[2]);
  trace(DBG_VNC, sdbg);

  // Texture creation
  glGenTextures(1, &texture_num); // texture number given by OpenGL
  defaultTexture();
  trace(DBG_VNC, "Framebuffer size: %d %d", texture_width, texture_height);

  if (isServerDefined)
    vncConnect(this);
} 

void Vnc::defaultTexture()
{
  if (texture_bytes != 0)
    delete texture_bytes;

  texture_width = 2;
  texture_height = 2;

  if ((texture_bytes = new GLubyte[3 * texture_width * texture_height]) == 0) {
    trace(DBG_FORCE, "Vnc::defaultTexture: can't alloc screen");
    return;
  }

  GLubyte *pix = texture_bytes;
  for (int i=0; i < texture_width; i++) {
    for (int j=0; j < texture_height; j++) {
      *pix++ = (GLubyte)255;
      *pix++ = (GLubyte)255;
      *pix++ = (GLubyte)255;
    }
  }

  glEnable(GL_TEXTURE_2D); // We need to use a texture
  glBindTexture(GL_TEXTURE_2D, texture_num); // We use ours.
  // Put it in the video memory
  glTexImage2D(GL_TEXTURE_2D, 0, 3, texture_width, texture_height,
               0, GL_RGB, GL_UNSIGNED_BYTE, texture_bytes);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  //No interpolation
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  // No texture repetition
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glDisable(GL_TEXTURE_2D);

  u = v = 1;
  createVertices();
}

void Vnc::changePermanent(float lasting)
{
  if (isConnected) {
    FD_ZERO(&rmask);
    FD_SET(vncClient->getSock(), &rmask);

    int sel = select(32, &rmask, NULL, NULL, &delay);
    if (sel != 0) {
      if (FD_ISSET(vncClient->getSock(), &rmask)) {
        if (!vncClient->HandleRFBServerMessage()) {
          trace(DBG_FORCE, "could'nt handle RFB server message");
          return;
        }

        // We need to use a texture
        glEnable(GL_TEXTURE_2D);
        // We use ours
        glBindTexture(GL_TEXTURE_2D, texture_num);
        // Put it in the video memory
#if HAVE_LIBGLU
        gluBuild2DMipmaps(GL_TEXTURE_2D, 3, texture_width, texture_height,
                          GL_RGB, GL_UNSIGNED_BYTE, texture_bytes);
#else
        glTexImage2D(GL_TEXTURE_2D, 0, 3,texture_width, texture_height,
                     0, GL_RGB, GL_UNSIGNED_BYTE, texture_bytes);
#endif

        glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
        // no interpolation
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                        GL_LINEAR_MIPMAP_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        // No texture repetition
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
        glDisable(GL_TEXTURE_2D);
      }
    }
  }
}

void Vnc::createVertices()
{
  float xFront = soh->bbsize.v[0];
  float xBack = -soh->bbsize.v[0];
  float zTop = soh->bbsize.v[2];
  float zBot = -soh->bbsize.v[2];
  float yMax = soh->bbsize.v[1];
  float yMin = -soh->bbsize.v[1];
  float rTex = u * texture_width / (v * texture_height);
  float rSol = (float) soh->bbsize.v[1] / soh->bbsize.v[2];

  trace(DBG_VNC, "rTex:%f rSol: %f", rTex, rSol);

  float zScreen;
  float yScreen;

  if (rTex>rSol) {
    zScreen = -soh->bbsize.v[2] + 2*0.95*soh->bbsize.v[1]/rTex;
    yScreen = 0.95 * soh->bbsize.v[1];
  }
  else {
    zScreen = (2*0.95-1) * soh->bbsize.v[2];
    yScreen = 0.95 * soh->bbsize.v[2]*rTex;
  }
  float xScreen = 0.9 * soh->bbsize.v[0];

  vertices[ 0] = xScreen;
  vertices[ 1] = yScreen;
  vertices[ 2] = zBot;
  vertices[ 3] = xScreen;
  vertices[ 4] = yScreen;
  vertices[ 5] = zScreen;
  vertices[ 6] = xScreen;
  vertices[ 7] = -yScreen;
  vertices[ 8] = zScreen;
  vertices[ 9] = xScreen;
  vertices[10] = -yScreen;
  vertices[11] = zBot;

  vertices[12] = xFront;
  vertices[13] = yMax;
  vertices[14] = zBot;
  vertices[15] = xFront;
  vertices[16] = yMax;
  vertices[17] = zTop;
  vertices[18] = xFront;
  vertices[19] = yMin;
  vertices[20] = zTop;
  vertices[21] = xFront;
  vertices[22] = yMin;
  vertices[23] = zBot;

  vertices[24] = xBack;
  vertices[25] = yMax;
  vertices[26] = zBot;
  vertices[27] = xBack;
  vertices[28] = yMin;
  vertices[29] = zBot;
}

void Vnc::render()
{
  GLfloat glmat[16];
  int renderMode;

  glGetIntegerv(GL_RENDER_MODE, &renderMode);
  glEnable(GL_TEXTURE_2D); // We need to use a texture
  glBindTexture(GL_TEXTURE_2D, texture_num); // we use ours

  glEnable(GL_CULL_FACE);
   
  //createVertices();
  updateObjectIn3D(this);
  for (int i=0; i < 4; i++)
    for (int j=0; j < 4; j++)
      glmat[i*4+j] = soh->posmat.m[j][i];

  glPushMatrix();
  glMultMatrixf(glmat);

  // We show the screen
  glBegin(GL_QUADS);
    glTexCoord2f(u,v); glVertex3dv(vertices+0);
    glTexCoord2f(u,0); glVertex3dv(vertices+3);
    glTexCoord2f(0,0); glVertex3dv(vertices+6);
    glTexCoord2f(0,v); glVertex3dv(vertices+9);
  glEnd();
 
  glDisable(GL_LIGHTING); // disable ligthing
  glDisable(GL_TEXTURE_2D); // No use of textures anymore

  glColor3f(color[0], color[1], color[2]);

  // show the object on the screen
  glBegin(GL_QUAD_STRIP);
    glVertex3dv(vertices+ 0);
    glVertex3dv(vertices+12);
    glVertex3dv(vertices+ 3);
    glVertex3dv(vertices+15);
    glVertex3dv(vertices+ 6);
    glVertex3dv(vertices+18);
    glVertex3dv(vertices+ 9);
    glVertex3dv(vertices+21);
  glEnd();
  glBegin(GL_QUADS);
    glVertex3dv(vertices+24);
    glVertex3dv(vertices+27);
    glVertex3dv(vertices+18);
    glVertex3dv(vertices+15);
  glEnd();
  glBegin(GL_TRIANGLES);
    glVertex3dv(vertices+21);
    glVertex3dv(vertices+18);
    glVertex3dv(vertices+27);
    glVertex3dv(vertices+12);
    glVertex3dv(vertices+24);
    glVertex3dv(vertices+15);
  glEnd();

  glEnable(GL_LIGHTING);

  // Updating parametres for mouse gestion
  if (renderMode == GL_RENDER) {
    glGetIntegerv(GL_VIEWPORT, viewport);
    glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
    glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
  }
  glPopMatrix();
}

void Vnc::quit()
{
  if (texture_bytes != 0)
    delete texture_bytes;
  if (isConnected) {
    vncLeaveFocus(this, 0, 0, 0);
    if (vncClient->VNCClose() == false)
      warning("Vnc::~Vnc: disconnect failed");
    else {
      isConnected = false;
      trace(DBG_VNC, "Vnc::~Vnc: disconnect done");
      //PD defaultTexture();	// BUG: segmentation fault in createVertices
    }
  }
}

void Vnc::getVncCoords(int &x, int &y)
{
  GLdouble ox,oy,oz;
  GLdouble ax,ay,az;

  y = viewport[3]-y;

#if HAVE_LIBGLU
  // Get intersection with z=0 plan
  gluUnProject((GLdouble)x,(GLdouble)y,(GLdouble)0,
               modelMatrix,projMatrix,viewport,
               &ox,&oy,&oz );

  // Get intersectiom with z=1 plan
  gluUnProject((GLdouble)x,(GLdouble)y,(GLdouble)1,
               modelMatrix,projMatrix,viewport,
               &ax,&ay,&az );
#endif

  // Calculated intersection coordinates in vnc screen plane
  GLdouble dx,dy,dz;
  dx = ax - ox;
  if (dx >= 0) {
    x = -1;
    y = -1;
    return;
  }
  dy = ay - oy;
  dz = az - oz;

  GLdouble t = (vertices[0] - ox) / dx;
  GLdouble ey = t*dy + oy;
  GLdouble ez = t*dz + oz;

  ey = (ey-vertices[10]) / (vertices[4]-vertices[10]);
  ez = (vertices[5]-ez) / (vertices[5]-vertices[11]);

  // Built final coordinates
  if ((ey > 0) && (ey < 1) && (ez > 0) && (ez < 1)) {
    x = (int) (ey * vncClient->realScreenWidth);
    y = (int) (ez * vncClient->realScreenHeight);
  }
  else {
    x = -1;
    y = -1;
  }
}

/*
 * vncReconnect: open the dialog window
 */

static
void vncReconnect(Vnc *po, void *data, time_t sec, time_t usec)
{
  if (po->isConnected || po->isServerDefined)
    gui.guiWidgets->alert("VNC: Already connected, disconnect first");
  else
    gui.guiWidgets->launchVncConnect(po);
}

/*
 * updates server parameters from UStr from the dialog window
 */
void Vnc::vncConvert(const char *_server,
		     const char *_port,
		     const char *_passwd) {

  if (!_server || !_port || !_passwd) {
    gui.guiWidgets->alert("VNC connection: Wrong arguments");
    return;
  }

  strcpy(serverName, _server);
  port = atoi(_port);

  //if (strlen(passwd) > 8) passwd[8] = '\0';
  vncEncryptAndStorePasswd((char *) _passwd, passwdFile);
  vncConnect(this);
}
	
void Vnc::vncConnect(Vnc *po)
{
  po->vncClient = new VNCClientTextured(po->serverName, po->port, po->passwdFile);
  
  // VNC client initialisation 
  if (po->vncClient->VNCInit()) {
    po->isConnected = true;
    if (!po->isServerDefined)
      vncTakeFocus(po, 0, 0, 0);

    trace(DBG_VNC, "VNCClient OK");
    FD_ZERO(&po->rmask);
    FD_SET(po->vncClient->getSock(), &po->rmask); 

    po->delay.tv_sec =  0;
    po->delay.tv_usec = 1;

    po->vncClient->SendFramebufferUpdateRequest(0, 0,
	                                        po->vncClient->realScreenWidth,
                                                po->vncClient->realScreenHeight,
                                                false);
    po->move.perm_sec = 1;
  }
  else { // if connection failed
    po->isServerDefined = false;
    warning("VNClient connection failed");
    gui.guiWidgets->alert("VNC: VNClient connection failed");
  }
      
  // VNCClient init end

  // texture initialisation to the framebuffer
  po->texture_bytes = (GLubyte *) po->vncClient->FrameBuffer;
  po->texture_width = po->vncClient->framebufferWidth;
  po->texture_height = po->vncClient->framebufferHeight;
 
  glEnable(GL_TEXTURE_2D); // Need to use a texture
  glBindTexture(GL_TEXTURE_2D, po->texture_num); // we use ours
  // put it in the memory video
  glTexImage2D(GL_TEXTURE_2D, 0, 3, po->texture_width, po->texture_height,
               0, GL_RGB, GL_UNSIGNED_BYTE, po->texture_bytes);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  // No interpolation
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  // No texture repetition
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glDisable(GL_TEXTURE_2D);

  po->u = (float)po->vncClient->realScreenWidth / po->texture_width;
  po->v = (float)po->vncClient->realScreenHeight / po->texture_height;
}

static
void vncDisconnect(Vnc *po, void *data, time_t sec, time_t usec)
{
  if (po->isConnected) {
    if (po->vncClient->VNCClose() == false)
      warning("vncDisconnect: disconnect failed");
    else {
      po -> isConnected = false;
      po -> isServerDefined = false;
      po->defaultTexture();
      gui.guiWidgets->redirectToVnc(NULL);
    }
  }  
}

// redirects events to the Vnc object
boolean Vnc::redirectEvent(int x, int y, int button)
{
  if (!isConnected) 
    return FALSE;

  const Cardinal card = 4;
  const char *params[card];
  char p1[16], p2[16], p3[16]; 

  getVncCoords(x, y);  // Change coords

  params[0] = "ptr";
  sprintf(p1,"%d", x);
  params[1] = p1;
  sprintf(p2,"%d", y);
  params[2] = p2;
  sprintf(p3,"%d", button);
  params[3] = p3;

  vncClient->SendRFBEvent((char **) params, (Cardinal *) &card); // Send to VNC
  return TRUE;
}

boolean Vnc::redirectEvent(const char *key, boolean is_down)
{
  if (!isConnected)
    return FALSE;

  const Cardinal card = 2;
  const char *params[card];
  char p1[16]; 

  params[0] = (is_down ? "keydown" : "keyup");
  strcpy(p1, key);
  params[1] = p1;

  vncClient->SendRFBEvent((char **) params, (Cardinal *) &card); // Send to VNC
  return TRUE;
}

// Access to Gui parameters eventsCaptured and capturingObject
// give the focus to this object : All events redirected
void vncTakeFocus(Vnc *po, void *data, time_t sec, time_t usec)
{
  notice("Take Focus");
  gui.guiWidgets->redirectToVnc(po);
}


// Remove the focus from the object
void vncLeaveFocus(Vnc *po, void *data, time_t sec, time_t usec)
{
  notice("Leave Focus");
  gui.guiWidgets->redirectToVnc(NULL);
}

void vncInitFuncList(void)
{
  setMethodFunc(VNC_TYPE, 0, WO_ACTION vncTakeFocus,  "Take-focus");
  setMethodFunc(VNC_TYPE, 1, WO_ACTION vncLeaveFocus, "Leave-focus");
  setMethodFunc(VNC_TYPE, 2, WO_ACTION vncDisconnect, "Disconnect");
  setMethodFunc(VNC_TYPE, 3, WO_ACTION vncReconnect,  "Reconnect");
}

#endif /* !VRENGD */
