#ifndef VRENGD

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"	/* COL_ONCE */
#include "user.h"
#include "move.h"	/* GRAVITY */
#include "thing.h"

#include "zv.h"		/* parseGeometry */
#include "payload.h"    /* Payload */


const WClass Thing::wclass(THING_TYPE, "Thing", Thing::creator);

static u_int16 oid = 1;


/* thing creation from a file */
void Thing::creator(char *l)
{
  new Thing(l);
}

Thing::Thing(char *l)
{
  l = parseName(l, this);
  l = parsePosition(l, this);
  soh = parseGeometry(l);

  trace(DBG_WMGT, "thingCreateFromFile: x=%.2f y=%.2f z=%.2f a=%.2f", pos.x, pos.y, pos.z, pos.az);
  initializeObject(this, THING_TYPE, VR_MOBILE);
  maxlasting[THING_TYPE] = THING_LASTING;
  nature.collision = COL_ONCE;

  initializeNetObject(this, oid++, THING_PROPS, VR_PERMANENT);

  lspeed = THING_LSPEED;
  state.state = THING_INACTIVE;

  struct timeval te;
  gettimeofday(&te, NULL);
  move.perm_sec = te.tv_sec;
  move.perm_usec = te.tv_usec;
} 

void Thing::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = (float) (sec - move.sec) + (float) (usec - move.usec) / 1e6;
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl /* = THING_TTL */ ;
    move.ttl = 0;
    move.sec = 0;
    move.usec = 0;
  }
}

boolean Thing::change()
{
  return (move.ttl > 0.0005) ? TRUE : FALSE;
}

void Thing::changePosition(float lasting)
{
  if (state.state == THING_DROP)
    return;
  if (state.state != THING_INACTIVE) {
    if (pos.group) {	// group of objects
      WObject *po;

      for (po = prev; po && po->prev == prev; po = po->next) {
        trace(DBG_FORCE, "prev=%p po=%p po->prev=%p po->next=%p", prev, po, po->prev, po->next);
        po->pos.x += lasting * move.lspeed.v[0];
        po->pos.y += lasting * move.lspeed.v[1];
        po->pos.z += lasting * move.lspeed.v[2];
        po->pos.az += lasting * move.aspeed.v[0];
        trace(DBG_FORCE, "thingChangePosition: x=%.2f y=%.2f z=%.2f ttl=%.2f lasting=%.2f", po->pos.x, po->pos.y, po->pos.z, move.ttl, lasting);
      }
    }
    else {	// only one object
      pos.x += lasting * move.lspeed.v[0];
      pos.y += lasting * move.lspeed.v[1];
      pos.z += lasting * move.lspeed.v[2];
      pos.az += lasting * move.aspeed.v[0];
      trace(DBG_WMGT, "thingChangePosition: x=%.2f y=%.2f z=%.2f ttl=%.2f lasting=%.2f", pos.x, pos.y, pos.z, move.ttl, lasting);
    }
  }
}

void Thing::changePermanent(float lasting)
{
  static float lastz = 0.0;

  if (state.state == THING_DROP) {
    trace(DBG_WMGT, "thingChangePermanent: x=%.2f y=%.2f z=%.2f ttl=%.2f lasting=%.2f", pos.x, pos.y, pos.z, move.ttl, lasting);
    lastz = pos.z;
    /* drop this thing by gravity */
    if (pos.group) {	// group of objects
      WObject *po;

      for (po = prev; po && po->prev == prev; po = po->next) {
        trace(DBG_FORCE, "prev=%p po=%p po->prev=%p po->next=%p", prev, po, po->prev, po->next);
        po->pos.z -= lasting * GRAVITY;
      }
    }
    else	// only one object
      pos.z -= lasting * GRAVITY;

    /* test if finished */
    if (move.ttl == 0 /* || ABSF(lastz - pos.z) < 0.001 */) {
      state.state = THING_INACTIVE;
      lastz = 0.0;
    }
  }
}

boolean Thing::updateToNetwork(const Pos &oldpos)
{
  boolean change = FALSE;

  if (state.state != THING_INACTIVE) {
    if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
      if (pos.group) {	// group of objects
        WObject *po;
        for (po = prev; po && po->prev == prev; po = po->next) {
          trace(DBG_WMGT, "Thing::updateToNetwork: x=%.2f y=%.2f", po->pos.x, po->pos.y);
          declareObjDelta(&(po->noh), THING_PROPXY);
        }
      }
      else {
        trace(DBG_WMGT, "Thing::updateToNetwork: x=%.2f y=%.2f", pos.x, pos.y);
        declareObjDelta(&noh, THING_PROPXY);
      }
      change = TRUE;
    }
    if (ABSF(pos.z - oldpos.z) > 0.1) {
      if (pos.group) {	// group of objects
        WObject *po;
        for (po = prev; po && po->prev == prev; po = po->next) {
          declareObjDelta(&(po->noh), THING_PROPZ);
        }
      }
      else
        declareObjDelta(&noh, THING_PROPZ);
      change = TRUE;
    }
    if (pos.az != oldpos.az) {
      if (pos.group) {	// group of objects
        WObject *po;
        for (po = prev; po && po->prev == prev; po = po->next) {
          declareObjDelta(&(po->noh), THING_PROPAZ);
        }
      }
      else
        declareObjDelta(&noh, THING_PROPAZ);
      change = TRUE;
    }
  }
  return change;
}

static
void thingTake(Thing *po, void *data, time_t sec, time_t usec)
{
  trace(DBG_WMGT, "thingTake: x=%.2f y=%.2f z=%.2f", po->pos.x, po->pos.y, po->pos.z);
  po->move.perm_sec = 0;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = THING_TTL;
  po->move.lspeed.v[0] = -(double)(Cos((float)worlds->plocaluser->pos.az))* po->lspeed;
  po->move.lspeed.v[1] = -(double)(Sin((float)worlds->plocaluser->pos.az))* po->lspeed;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->state.state = THING_TAKEN;
}

static
void thingPush(Thing *po, void *data, time_t sec, time_t usec)
{
  trace(DBG_WMGT, "thingPush: x=%.2f y=%.2f z=%.2f", po->pos.x, po->pos.y, po->pos.z);
  po->move.perm_sec = 0;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = THING_TTL;
  po->move.lspeed.v[0] = (double)(Cos((float)worlds->plocaluser->pos.az))* po->lspeed;
  po->move.lspeed.v[1] = (double)(Sin((float)worlds->plocaluser->pos.az))* po->lspeed;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->state.state = THING_TAKEN;
}

static
void thingRot(Thing *po, void *data, time_t sec, time_t usec)
{
  trace(DBG_WMGT, "thingRot: az=%.2f", po->pos.az);
  po->move.perm_sec = 0;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = THING_TTL;
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0.88; /* 0.88 rd/s */
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->state.state = THING_TAKEN;
}

static
void thingDrop(Thing *po, void *data, time_t sec, time_t usec)
{
  if (po->state.state == THING_INACTIVE)
    return;
  po->move.perm_sec = sec;
  po->move.perm_usec = usec;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = THING_TTL;
  po->state.state = THING_DROP;
}

/* object intersection */
void Thing::whenIntersect(WObject *pcur, WObject *pold)
{
  trace(DBG_WMGT, "thingIntersect: x=%.2f y=%.2f z=%.2f", pos.x, pos.y, pos.z);
  //DAX copyPositionAndBB(pold, pcur);
}

void thingSet_xy(WObject *po, Payload *pp)
{
  trace(DBG_WMGT, "thingSet_xy: x=%.2f y=%.2f", po->pos.x, po->pos.y);
  set_xy(po, pp);
}

void thingSet_z(WObject *po, Payload *pp)
{
  set_z(po, pp);
}

void thingSet_az(WObject *po, Payload *pp)
{
  set_az(po, pp);
}

void thingSet_hname(WObject *po, Payload *pp)
{
  trace(DBG_FORCE, "thingSet_hname:");
  set_hname(po, pp);
}

void Thing::quit()
{
  oid = 1;
}

void thingInitFuncList(void)
{
  setFuncList[THING_PROPXY][THING_TYPE].pf = WO_PAYLOAD thingSet_xy;
  setFuncList[THING_PROPZ][THING_TYPE].pf = WO_PAYLOAD thingSet_z;
  setFuncList[THING_PROPAZ][THING_TYPE].pf = WO_PAYLOAD thingSet_az;
  setFuncList[THING_PROPHNAME][THING_TYPE].pf = WO_PAYLOAD thingSet_hname;

  getFuncList[THING_PROPXY][THING_TYPE].pf = WO_PAYLOAD get_xy;
  getFuncList[THING_PROPZ][THING_TYPE].pf = WO_PAYLOAD get_z;
  getFuncList[THING_PROPAZ][THING_TYPE].pf = WO_PAYLOAD get_az;
  getFuncList[THING_PROPHNAME][THING_TYPE].pf = WO_PAYLOAD get_hname;

  setMethodFunc(THING_TYPE, 0, WO_ACTION thingTake, "Take");
  setMethodFunc(THING_TYPE, 1, WO_ACTION thingPush, "Push");
  setMethodFunc(THING_TYPE, 2, WO_ACTION thingRot, "Rot");
  setMethodFunc(THING_TYPE, 3, WO_ACTION thingDrop, "Drop");
}

#endif /* !VRENGD */
