#include "global.h"

// Payload/ssrc
#include "net.h"
#include "netobj.h"

// Object
#include "wobject.h"
#include "wmgt.h"
#include "col.h"

// parse* functions
#include "solid.h"
#include "parse.h"

#include "ifcserver.h"


const WClass IfcServer::wclass(IFCSERVER_TYPE, "IfcServer", IfcServer::creator);
 
// Static class member initialization
IfcServer *IfcServer::server = NULL;


/* Create a ifc server from a fileline */
void IfcServer::creator(char *l)
{
   new IfcServer(l);
}

/*
 * IfcServer constructor.
 * Checks whether an instance of IfcServer has already
 * been created or not. If there already was a server,
 * it actually doesn't do anything else than warn the user
 * of an error. Otherwise, it starts up (opens the sockets),
 * and stores itself in the class static variable.
 *
 * Configuration line syntax:
 * servername[[:server_port[:local_port]]
 * - servername where the server lives
 * - optional port number for that server
 * - optional port number the server should call back to
 */
IfcServer::IfcServer(char *l)
{
  if (IfcServer::getServer() != NULL)
    warning("You can't have more than one IfcServer per world");
  else {
    l = strtok(l, " :\t");
    strcpy(host, l); l = strtok(NULL, " :\t");

    // default ports
    serverPort = IFC_SERVER_PORT;
    localPort = IFC_LOCAL_PORT;

    // asked ports
    if (l)
      serverPort = (int) atoi(l); l = strtok(NULL, " :\t");
    if (l)
      localPort = (int) atoi(l);

    nature.bbable = VR_NO_BB;
    nature.selectable = VR_NO_SELECTABLE;
    nature.collision = COL_NEVER;
    initializeObject(this, IFCSERVER_TYPE, VR_STILL);

    IfcServer::setServer(this);
    sock = NULL;
    lastMessage = NULL;
    lastping = 0;
    ssrc = my_ssrc_id;
    start();
  }
} 

/*
 * Destructor: closes the sockets after having notified the external server
 */
IfcServer::~IfcServer()
{
  IfcServer::stop();
}

/*
 * Returns the singleton instance, if it exists
 */
IfcServer *IfcServer::getServer(void)
{
  return server;
}

/*
 * Sets the singleton instance
 */
void IfcServer::setServer(IfcServer *srv)
{
  server = srv;
}

/*
 * Used to send pings to the server
 */
void IfcServer::render()
{
  IfcServer::getServer()->ping();
}

/*
 * Sends a ping to the server every IFC_PING_WAIT times
 */
void IfcServer::ping()
{
  lastping++;
  if (lastping >= IFC_PING_WAIT) {
    sendCommand(this, IFC_MSGV_PING);
    lastping = 0;
  }
}

/*
 * Shut down this IfcServer
 */
void IfcServer::quit()
{
  //PD done bye quitWorld// delete this;
}

/*
 * Starts the socket
 */
void IfcServer::start()
{
  sock = ifcInitSocket(localPort, host, serverPort);
  if (ifcOpenSocket(sock) == -1) {
    warning("Unable to open local comm. socket");
    sock = ifcCloseSocket(sock);
  }
  else {
    localPort = sock->listenPort;
    IfcMessage *msg = new IfcMessage(this, IFC_MSGT_CTRL, IFC_MSGV_INITIATE);
    sendData(msg);
    delete msg;
  }
}

/*
 * Sends a terminate notification to the external server
 * and closes the comm sockets. The class's singleton instance
 * is also removed.
 */
void IfcServer::stop()
{
  IfcServer *srv = IfcServer::getServer();
  if (srv != NULL) {
    trace(DBG_IFC, "Shutting down ifc server");
    sendCommand(NULL, IFC_MSGV_TERMINATE);
    srv->sock = ifcCloseSocket(srv->sock);
    srv->serverPort = -1;
    srv->localPort = -1;
    IfcServer::setServer(NULL);
  }
}

/*
 * Unregister an Ifc object with the external server
 */
void IfcServer::stopApp(Ifc *po)
{
  IfcServer *srv = IfcServer::getServer();
  if (srv == NULL) return;

  IfcServer::sendCommand(po, IFC_MSGV_UNREGISTER);
  if (srv->lastMessage != NULL) {
    delete srv->lastMessage;
    srv->lastMessage = NULL;
  }
}

/*
 * Register an Ifc object with the external server
 * The message contains:
 * - localhost name (string)
 * - local listen port (16bit)
 * - Ifc's given name (string)
 * - Ifc's class name (string)
 * - Ifc's codebase URL (string)
 * - With and heigh (32bit ints)
 */
void IfcServer::startApp(Ifc *po)
{
  char hostname[128];
  IfcMessage *msg;
  IfcServer *srv = IfcServer::getServer();
  if (srv == NULL) return;

  if (gethostname(hostname, 127) < 0) {
    warning("Ifcserver: Can't gethostname");
    return;
  }

  msg = new IfcMessage(po, IFC_MSGT_CTRL, IFC_MSGV_REGISTER);
  msg->putStr(hostname);
  msg->put16(srv->localPort);
  msg->putStr(po->name.given_name);
  msg->putStr(po->app);
  msg->putStr(po->url);
  msg->put32(po->incrx);
  msg->put32(po->incry);

  IfcServer::sendData(msg);
  delete msg;
}

void ifcServerInitFuncList(void) { }

// *******************************************************************
//                    Network management
// *******************************************************************
/* Read a header struct for the raw packet data */
void ifcCopyHeader(ifcHeader *hdr, ifcHeader cpy)
{
  hdr->proto     = cpy.proto;
  hdr->version   = cpy.version;
  hdr->app_ssrc  = cpy.app_ssrc;
  hdr->msg_type  = cpy.msg_type;
  hdr->msg_val   = cpy.msg_val;
  hdr->data_len  = cpy.data_len;
  hdr->obj_type  = cpy.obj_type;
  hdr->src_id    = cpy.src_id;
  hdr->port_id   = cpy.port_id;
  hdr->obj_id    = cpy.obj_id;
}

/* Copies a vector into another */
void ifcCopyV3(V3 *to, V3 from)
{
  to->v[0] = from.v[0];
  to->v[1] = from.v[1];
  to->v[2] = from.v[2];
}

/*
 * Reads in data from the child process, if any is availabe.
 * It is the caller's responsability to delete the message
 * received if it was non null.
 */
IfcMessage *IfcServer::getData(WObject *po)
{
  // Reception buffer
  int8 *buff;
  // File descriptor set and select timeout
  fd_set fds;
  struct timeval to;
  // Length of data read from the socket
  int read;

  /* Check that the socket is really open */
  IfcServer *srv = IfcServer::getServer();
  if ((srv == NULL) || (srv->sock->state != IFC_SOCK_OPEN)) {
    trace(DBG_FORCE, "ifcserver socket");
    return 0;
  }

  if (srv->lastMessage != NULL)
    goto haspack;
 
  /* Set a 0 sec timeout: the select will return
   * instantly if no data is available */
  FD_ZERO(&fds);
  FD_SET(srv->sock->recv, &fds);
  to.tv_sec = 0;
  to.tv_usec = 0;

  if (select(srv->sock->recv+1, &fds, NULL, NULL, &to) > 0) {

    buff = (int8 *) malloc(IFC_MAX_PACKET);
    if (buff == NULL) fatal("Can't allocate buffer for packet reception\n");

    if ((read = recv(srv->sock->recv, (void *) buff, IFC_MAX_PACKET, 0)) > 0) {
      if (read < IFC_HDR_LEN) {
	/* We didn't get a whole header */
	warning("Dropped incomplete packet (%d)", read);
      }
      else {
	/* Message size is big enough to contain a header */
	srv->lastMessage = new IfcMessage(buff, read);

	/* Check the header */
	// TODO: check protocol id and version number
	if (srv->lastMessage->getHeader().data_len > (IFC_MAX_PACKET-IFC_HDR_LEN)) {
	  /* Header and real length don't agree */
	  warning("Illegal data lenght");
	  delete srv->lastMessage;
	  srv->lastMessage = NULL;
	}
        else
	  goto haspack;
      }
    }
    // (r < 0) || (packet too short) || (invalid header)
    free(buff);
    return NULL;
  }
  // select <= 0
  return NULL;

haspack:
  // A message was read from the socket, or had been read previously
  if (srv->lastMessage->isForObject(po)) {
	IfcMessage *ret = srv->lastMessage;
	srv->lastMessage = NULL;
	return ret;
  }
  return NULL;
}

/* Creates a new ifcHeader */
ifcHeader ifcNewHeader(int ssrc, int type, int val, int len)
{
  ifcHeader hdr;

  hdr.proto = IFC_PROTO;
  hdr.version = IFC_VERSION;
  hdr.app_ssrc = ssrc;
  hdr.msg_type = type;
  hdr.msg_val  = val;
  hdr.data_len = len;
  return hdr;
}
  
/*
 * Send data over to the server.
 * The message is not deleted.
 */
int IfcServer::sendData(IfcMessage *msg)
{
  int sent = 0, total = 0;
  IfcServer *srv;
  int8 *buff;

  /* Check that the socket is ready */
  srv = IfcServer::getServer();
  if ((srv != NULL) && (srv->sock->state == IFC_SOCK_CONNECT)) {
    // The socket is still trying to connect. Check if it's done or not.
    if (ifcTestSendSocket(srv->sock) == 0) {
      // The connection is still in progress
      return 0;
    }
  }
  if ((srv != NULL) && (srv->sock->state == IFC_SOCK_OPEN)) {

    /* Check if an SSRC chang occured since we sent the REGISTER commands */
    if (my_ssrc_id != srv->ssrc) {
      if (!((msg->getHeader().msg_type == IFC_MSGT_CTRL)
         && (msg->getHeader().msg_val  == IFC_MSGV_REGISTER))) {
        IfcMessage *msg = new IfcMessage(srv, srv->ssrc, IFC_MSGT_CTRL, IFC_MSGV_UPDATE);
        trace(DBG_IFC, "Updating SSRC for IfcServer (old:%d, new:%d)", srv->ssrc, my_ssrc_id);
        msg->put32(my_ssrc_id);
        buff = msg->toBytes(&total);
        sent = send(srv->sock->send, buff, total, 0);
        delete msg;
      }
      srv->ssrc = my_ssrc_id;
    }

    /* Send the message */
    buff = msg->toBytes(&total);
    sent = send(srv->sock->send, buff, total, 0);
    if (debug)
      trace(DBG_IFC, "sending %d bytes with %02x %02x %d %08x (%d %d)",
	sent, buff[0], buff[1], buff[2], my_ssrc_id,
	msg->getHeader().msg_type,
	msg->getHeader().msg_val);
    return 1;
  }
  return 0;
}

/* Send a simple (no data) control command to the child */
int IfcServer::sendCommand(WObject *po, int val)
{
  IfcMessage *msg = new IfcMessage(po, IFC_MSGT_CTRL, val);
  int ret = IfcServer::sendData(msg);
  delete msg;
  return ret;
}

/*
 * Initialize a pair of sockets for communicating with the child app.
 * The sockets aren't actually opened. The addresses are initialized.
 */
ifcSocket *ifcInitSocket(int listenPort, char *destHost, int destPort)
{
  ifcSocket *sock;
  struct sockaddr_in *sa;
  struct hostent *hostinfo;

  if ((sock = (ifcSocket *) malloc(sizeof(ifcSocket))) == NULL)
    fatal("Malloc fails during socket creation");

  memset(sock, 0, sizeof(ifcSocket));

  // Set port fields
  sock->listenPort = listenPort;
  sock->destPort = destPort;

  // Set address field
  if ((sa = (struct sockaddr_in *) malloc(sizeof(struct sockaddr_in))) == NULL)
    fatal("Malloc fails during socket creation");

  memset((char *) sa, 0, sizeof(struct sockaddr_in));
  sa->sin_family = AF_INET; 
  sa->sin_port = htons(destPort);
  if ((hostinfo = gethostbyname(destHost)) == NULL) {
    warning("Host unknown while creating local socket");
    return NULL;
  }
  sa->sin_addr = *(struct in_addr *) hostinfo->h_addr;
  sock->destAddr = sa;

  // Set comm state
  sock->state = IFC_SOCK_CLOSED;
  sock->recv = -1;
  sock->send = -1;
  return sock;
}

/*
 * Open the receiver socket for this socket pair.
 * The socket is bound to the local port passed in when it was initialized.
 */
int ifcOpenRecvSock(ifcSocket *sock)
{
  struct sockaddr_in ad;
  int tries = 0;
  boolean bound = FALSE;

  if ((sock->recv = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "ifcOpenRecvSock socket: %s", strerror(errno));
    return 0;
  }

  memset((char *) &ad, 0, sizeof(struct sockaddr_in));
  ad.sin_family = AF_INET; 
  ad.sin_addr.s_addr = INADDR_ANY;

  // Bind it
  while ((tries < 10) && (bound == FALSE)) {
    ad.sin_port = htons(sock->listenPort+tries);
    if (bind(sock->recv, (struct sockaddr *)&ad, sizeof(struct sockaddr_in)) <0)
      tries++;
    else {
      bound = TRUE;
      sock->listenPort += tries;
      notice("Ifc receiver bound to port %d", sock->listenPort);
    }
  }
  if (!bound) {
    trace(DBG_FORCE, "ifcOpenRecvSock bind: %s", strerror(errno));
    close(sock->recv);
    return 0;
  }
  sock->state = IFC_SOCK_RCVOPEN;
  return 1;
}

/*
 * Opens the sender socket. IO is set to non-blocking on this one
 * so that the connect (done in the next step) does not lock up
 * the whole VRENG app.
 */
int ifcOpenSendSock(ifcSocket *sock)
{
  if ((sock->send = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "ifcOpenSendSock socket: %s", strerror(errno));
    return 0;
  }
  if (fcntl(sock->send, F_SETFL, O_NONBLOCK) == -1)
    trace(DBG_FORCE, "Couldn't set non-blocking io on socket");

  sock->state = IFC_SOCK_CONNECT;
  return 1;
}

/*
 * Tries a connect to the client app. Since the IO mode is NONBLOCK,
 * two types of errors can happen:
 * - EINPROGRESS : the connect didn't complet - the child app is not ready,
 * but is not dead either. The connect will have to be tried again.
 * - Other : the child app is not in a good state. Not worth retrying.
 */
int ifcConnectSendSock(ifcSocket *sock)
{
  // Try the connect
  if (connect(sock->send, (struct sockaddr *) sock->destAddr, sizeof(struct sockaddr_in)) < 0) {
    if ((errno == EINPROGRESS) || (errno == EALREADY)) {
	  // Socket set to non-blocking to prevent an app freeze
      return 1;
    }
    else {
      trace(DBG_FORCE, "ifcConnectSendSock connect: %s", strerror(errno));
      return 0;
    }
  }
  else
    sock->state = IFC_SOCK_OPEN;
  return 1;
}

/*
 * Check if the non-blocking connect call on the send socket
 * finished or not.
 */
int ifcTestSendSocket(ifcSocket *sock)
{
  // Setup the select fd_set
  fd_set set;
  struct timeval timeout;
  FD_ZERO(&set);
  FD_SET(sock->send, &set);
  timeout.tv_sec = 0;
  timeout.tv_usec = 0;
  if (select(sock->send+1, NULL, &set, NULL, &timeout)>0) {
    // The socket finished connecting
    sock->state = IFC_SOCK_OPEN;
    return 1;
  }
  return 0;
}

/*
 * Opens an ifcSocket. This call is 'non-blocking' : if the connect
 * does not work because the client is not ready yet, this function
 * returns 0. In this case, it should be called again to finish the connection.
 * If the socket is completly open, it returns 1.
 * If an error occured, it returns -1.
 */
int ifcOpenSocket(ifcSocket *sock)
{
  // check if it's not already open
  if (sock->state == IFC_SOCK_OPEN)
    return 1;
  
  // Open the receiver
  if (sock->state == IFC_SOCK_CLOSED) {
    if ( ifcOpenRecvSock(sock) == 0 )
      return -1;
  }
  // Open the sender
  if (sock->state == IFC_SOCK_RCVOPEN) {
    if ( ifcOpenSendSock(sock) == 0 )
      return -1;
  }
  // Connect the sender
  if (sock->state == IFC_SOCK_CONNECT) {
    if ( ifcConnectSendSock(sock) == 0 )
      return -1;
  }

  if (sock->state == IFC_SOCK_OPEN) {
    // Everything is connected
    return 1;
  }
  // One of the steps didn't finish yet.
  return 0;
}

/*
 * Closes and frees an ifcSocket.
 * The handle is invalidated by this call.
 */
ifcSocket *ifcCloseSocket(ifcSocket *sock)
{
  if (sock->state>=IFC_SOCK_RCVOPEN)
    close(sock->recv);
  if (sock->state>=IFC_SOCK_CONNECT)
    close(sock->send);
  free(sock->destAddr);
  free(sock);
  return NULL;
}

/*
 * Utility functions
 */
int isSockOpen(ifcSocket *sock)
{
  return ((sock && (sock->state == IFC_SOCK_OPEN)) ? 1 : 0);
}

int isSockClosed(ifcSocket *sock)
{
  return ((!sock || (sock->state == IFC_SOCK_CLOSED)) ? 1 : 0);
}


/*
 * IfcMessage
 * Data structure used to send messages to the controler
 */

/*
 * Constructor for outgoing messages.
 */
IfcMessage::IfcMessage(WObject *po, int ssrc, int type, int val)
{
  setup(po, ssrc, type, val);
}

/*
 * Constructor for outgoing messages.
 */
IfcMessage::IfcMessage(WObject *po, int type, int val)
{
  setup(po, (IfcServer::getServer() == NULL ? my_ssrc_id : IfcServer::getServer()->ssrc), type, val);
}

void IfcMessage::setup(WObject *po, int ssrc, int type, int val)
{
  // Set the cursor to the start of the buffer
  cursor = 0;
  // Create the header
  this->header = ifcNewHeader(ssrc, type, val, 0);
  // Set the sender object
  this->sender = po;
  // Prepare our data array
  this->data = (int8 *) malloc(IFC_MAX_PACKET);
  if (this->data == NULL) {
    warning("Can't allocate data buffer for outgoing IfcMessage");
    return;
  }
  // Write out the header
  putHeader();
}

/*
 * Constructor for incoming messages.
 */
IfcMessage::IfcMessage(int8 *data, int datalen)
{
  // Set the cursor to the start of the buffer
  cursor = 0;
  // Sender will always be null on incoming messages
  this->sender = NULL;
  // Allocate the buffer
  this->data = (int8 *) malloc(datalen);
  if (this->data == NULL)
    fatal("Can't allocate data buffer for incoming IfcMessage");

  // Copy the data
  memcpy(this->data, data, datalen);
  // Mark the maximum read length
  maxlen = datalen;
  // Read the header in
  this->header = readHeader();
}

/*
 * Destructor
 * Free the data buffer.
 */
IfcMessage::~IfcMessage()
{
  if (this->data != NULL) {
    free(this->data);
    this->data = NULL;
  }
}

/*
 * Checks whether the packet was for this WOject
 */
boolean IfcMessage::isForObject(WObject *po)
{
  return (header.src_id == po->noh.noid.src_id)
      && (header.port_id == po->noh.noid.port_id)
      && (header.obj_id == po->noh.noid.obj_id);
}

/*
 * Returns this message's header.
 * For outgoing messages, the header's data_len field is not set.
 */
ifcHeader IfcMessage::getHeader(void)
{
  return header;
}

#define IFC_CHECK_OVERFLOW_1(a) \
  if (cursor >= IFC_MAX_PACKET-a) fatal("Write past end of buffer (%d %d)", \
	getHeader().msg_type, \
	getHeader().msg_val);

/*
 * Add a 8bit int to the message.
 */
void IfcMessage::put8(int val)
{
  IFC_CHECK_OVERFLOW_1(IFC_INT8)
  data[cursor] = (int8) (0x000000FF) & val;
  cursor += IFC_INT8;
}

/*
 * Add a 16bit int to the message.
 */
void IfcMessage::put16(int val)
{
  IFC_CHECK_OVERFLOW_1(IFC_INT16)
  data[cursor]   = (val & 0xFF00) >> 8;
  data[cursor+1] = (val & 0x00FF);
  cursor += IFC_INT16;
}
 
/*
 * Add a 32bit int to the message.
 */
void IfcMessage::put32(int val)
{
  IFC_CHECK_OVERFLOW_1(IFC_INT32)
  data[cursor]   = (val & 0xFF000000) >> 24;
  data[cursor+1] = (val & 0x00FF0000) >> 16;
  data[cursor+2] = (val & 0x0000FF00) >> 8;
  data[cursor+3] = (val & 0x000000FF);
  cursor += IFC_INT32;
}

/*
 * Add a string to the message.
 * A 16b length field is set, followed by the caracters.
 * This method is pretty bad. No encoding is taken into account.
 * Results across heterogenous platforms are undefined.
 * Behavior on platforms where a char is more than 8 bits
 * is completly out of control.
 */
void IfcMessage::putStr(char *str)
{
  int i, len = strlen(str);
  put16(len);
  IFC_CHECK_OVERFLOW_1(len)
  for (i=0;i<len; i++)
    data[cursor+i] = (int8) str[i];
  cursor += len;
}
 
/*
 * Puts an object's net id.
 * Special case: the id passed in is the same as the sender's.
 *               We set the type identifier to 0.
 * Special case: NULL is passed as a parameter.
 *               We set all the fields to 0.
 */
void IfcMessage::putOID(WObject *po)
{
  if (po != NULL) {
    put8((po == sender ? 0 : po->noh.type));
    put32(po->noh.noid.src_id);
    put16(po->noh.noid.port_id);
    put16(po->noh.noid.obj_id);
  }
  else {
    put8( 0);
    put32(0);
    put16(0);
    put16(0);
  }
}
 
/*
 * Puts an object position.
 * This includes (x,y,z) coordinates, as well as angle data (az, ax, 0)
 * (Y angle never used).
 * All values are rounded to the 3rd decimal, and multiplied by 1000
 * so as not to have to worry about marshalling floating point values.
 */
void IfcMessage::putPos(WObject *po)
{
  put32((int32) (po->pos.x * 1000));
  put32((int32) (po->pos.y * 1000));
  put32((int32) (po->pos.z * 1000));
  put32((int32) (po->pos.az * 1000));
  put32((int32) (po->pos.ax * 1000));
  put32((int32) 0);
}

/*
 * Writes the message header.
 */
void IfcMessage::putHeader()
{
  // Write the header (proto part)
  put16(header.proto);
  put8( header.version);
  put32(header.app_ssrc);
  // Write the header (message part)
  put8( header.msg_type);
  put8( header.msg_val);
  put16(header.data_len);
  // Write the caller's id
  putOID(sender);
}

/*
 * Packs this message into an int8 array.
 * The total length of the array is stored in the len parameter.
 * DO NOT free the returned data. Free the message object when
 * you're done with it.
 */
int8 *IfcMessage::toBytes(int *len)
{
  // Data length = total length - header
  int datalen = cursor - IFC_HDR_LEN;
  // Total length is where our cursor currently is
  *len = cursor;
  // Set the data length in the packet
  cursor = IFC_HDR_DATALEN_IDX;
  put16(datalen);
  // Set the ssrc (could have been updated)
  cursor = IFC_HDR_SSRC_IDX;
  put32(IfcServer::getServer()->ssrc);
  // Restore the cursor. This should actually never be needed, except
  // if someone sends the same message twice, after updating it.
  cursor = *len;
  // return a pointer to the data.
  return data;
}

#define IFC_CHECK_OVERFLOW_2(a) \
  if (cursor > maxlen-a) fatal("Read past end of buffer (%d %d)", \
	getHeader().msg_type, \
	getHeader().msg_val);

/*
 * Read an 8 bit signed int.
 */
int8 IfcMessage::read8()
{
  IFC_CHECK_OVERFLOW_2(IFC_INT8)
  int8 val = data[cursor];
  cursor += IFC_INT8;
  return val;
}
 
/*
 * Read a 16 bit signed int
 */
int16 IfcMessage::read16()
{
  IFC_CHECK_OVERFLOW_2(IFC_INT16)
  int16 val = (data[cursor] << 8) + (0x00ff & data[cursor+1]);
  cursor += IFC_INT16;
  return val;
}
 
/*
 * Read a 32 bit signed int
 */
int32 IfcMessage::read32()
{
  IFC_CHECK_OVERFLOW_2(IFC_INT32)
  int32 val =
          (0xff000000 & (data[cursor]   << 24))
        + (0x00ff0000 & (data[cursor+1] << 16))
        + (0x0000ff00 & (data[cursor+2] << 8))
        + (0x000000ff &  data[cursor+3]);
  cursor += IFC_INT32;
  return val;
}

/*
 * Read a header struct for the raw packet data
 */
ifcHeader IfcMessage::readHeader()
{
  ifcHeader hdr;

  hdr.proto     = read16();
  hdr.version   = read8();
  hdr.app_ssrc  = read32();
  hdr.msg_type  = read8();
  hdr.msg_val   = read8();
  hdr.data_len  = read16();
  hdr.obj_type  = read8(); if (hdr.obj_type == 0) hdr.obj_type = IFC_TYPE;
  hdr.src_id    = read32();
  hdr.port_id   = read16();
  hdr.obj_id    = read16();
  return hdr;
}
 
/*
 * Read two 32bit ints, and return them as the x and y coords. of a V3.
 */
V3 IfcMessage::readPoint2D()
{
  V3 point;

  IFC_CHECK_OVERFLOW_2(2*IFC_INT32)
  point.v[0] = ((float) read32());
  point.v[1] = ((float) read32());
  point.v[2] = 0;
  return point;
}
 
/*
 * Read three 32bit ints, converted to floats by division by 1000,
 * and return them in a V3.
 */
V3 IfcMessage::readPoint3D()
{
  V3 point = readPoint2D();

  IFC_CHECK_OVERFLOW_2(IFC_INT32)
  point.v[2] = ((float) read32());
  point.v[0] /= 1000;
  point.v[1] /= 1000;
  point.v[2] /= 1000;
  return point;
}

/*
 * Returns TRUE if there is at least size bytes left to be read.
 */
boolean IfcMessage::hasData(int size)
{
  return ((cursor+size) <= maxlen);
}
