#ifndef VRENGD

#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"
#include "list.h"
#include "guy.h"

#include "zv.h"		/* parseGeometry */
#include "draw.h"


/* for cylinder models */
#define UPPER_LEG_SIZE 0.5
#define LOWER_LEG_SIZE 0.5
#define LEG_GIRTH 0.08
#define UPPER_LEG_GIRTH 0.1
#define UPPER_LEG_TAPER 0.8
#define LOWER_LEG_TAPER 0.8
#define LOWER_LEG_GIRTH 0.07

#define UPPER_ARM_SIZE 0.45
#define LOWER_ARM_SIZE 0.45
#define ARM_GIRTH 0.05
#define UPPER_ARM_GIRTH 0.05
#define LOWER_ARM_GIRTH 0.04
#define UPPER_ARM_TAPER 0.8
#define LOWER_ARM_TAPER 0.8

#define HIP_JOINT_SIZE 0.1
#define KNEE_JOINT_SIZE 0.09
#define SHOULDER_JOINT_SIZE 0.05
#define ELBOW_JOINT_SIZE 0.045

#define HEAD_SIZE 0.2
#define FOOT_SIZE 0.15

#define TORSO_HEIGHT 0.8
#define TORSO_WIDTH 0.35
#define TORSO_TAPER 0.7

#define STACKS 16
#define SLICES 16
#define GUY_NUM_BODY_PARTS 7


const WClass Guy::wclass(GUY_TYPE, "Guy", Guy::creator);


static
void store_head(Guy *po)
{ 
  glPushMatrix();
  glTranslatef(0.0, TORSO_HEIGHT+HEAD_SIZE, 0.0);
  glScalef(HEAD_SIZE, HEAD_SIZE, UPPER_LEG_GIRTH);
#if HAVE_LIBGLU
  gluSphere(po->quads, 1.0, SLICES, STACKS);
#else
  drawSphere(1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}

static
void store_torso(Guy *po)
{
  glPushMatrix();
  glScalef(TORSO_WIDTH, TORSO_HEIGHT, UPPER_LEG_GIRTH);
  glRotatef(-90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, TORSO_TAPER, 1.0, 1.0, SLICES, STACKS);
#else
  drawCylinder(TORSO_TAPER, 1.0, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}

static
void store_uleg(Guy *po)
{
  glPushMatrix();
  glTranslatef(0.0, -(HIP_JOINT_SIZE+UPPER_LEG_SIZE), 0.0);
#if HAVE_LIBGLU
  gluSphere(po->quads, KNEE_JOINT_SIZE, SLICES, STACKS);
#else
  drawSphere(KNEE_JOINT_SIZE, SLICES, STACKS, 0);
#endif
  glPopMatrix();
  glTranslatef(0.0, -HIP_JOINT_SIZE, 0.0);
#if HAVE_LIBGLU
  gluSphere(po->quads, HIP_JOINT_SIZE, SLICES, STACKS);
#else
  drawSphere(HIP_JOINT_SIZE, SLICES, STACKS, 0);
#endif
  glPushMatrix();
  glScalef(UPPER_LEG_GIRTH, UPPER_LEG_SIZE, UPPER_LEG_GIRTH);
  glRotatef(90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, 1.0, UPPER_LEG_TAPER, 1.0, SLICES, STACKS);
#else
  drawCylinder(1.0, UPPER_LEG_TAPER, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}
  
static
void store_lleg(Guy *po)
{
  glPushMatrix();
  glScalef(LOWER_LEG_GIRTH, LOWER_LEG_SIZE, LOWER_LEG_GIRTH);
  glRotatef(90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, 1.0, LOWER_LEG_TAPER, 1.0, SLICES, STACKS);
#else
  drawCylinder(1.0, LOWER_LEG_TAPER, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}
  
static
void store_foot(Guy *po)
{
  glPushMatrix();
  glTranslatef(0.0, 0.0, -FOOT_SIZE/2.0);
  glScalef(LOWER_LEG_GIRTH, LOWER_LEG_GIRTH, FOOT_SIZE);
  glRotatef(90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, 1.0, 1.0, 1.0, SLICES, STACKS);
#else
  drawCylinder(1.0, 1.0, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}
  
static
void store_uarm(Guy *po)
{
  glPushMatrix();
  glTranslatef(0.0, -(SHOULDER_JOINT_SIZE+UPPER_ARM_SIZE), 0.0);
#if HAVE_LIBGLU
  gluSphere(po->quads, ELBOW_JOINT_SIZE, SLICES, STACKS);
#else
  drawSphere(ELBOW_JOINT_SIZE, SLICES, STACKS, 0);
#endif
  glPopMatrix();
  glTranslatef(0.0, -SHOULDER_JOINT_SIZE, 0.0);
#if HAVE_LIBGLU
  gluSphere(po->quads, SHOULDER_JOINT_SIZE, SLICES, STACKS);
#else
  drawSphere(SHOULDER_JOINT_SIZE, SLICES, STACKS, 0);
#endif
  glPushMatrix();
  glScalef(UPPER_ARM_GIRTH, UPPER_ARM_SIZE, UPPER_ARM_GIRTH);
  glRotatef(90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, 1.0, UPPER_ARM_TAPER, 1.0, SLICES, STACKS);
#else
  drawCylinder(1.0, UPPER_ARM_TAPER, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}
  
static
void store_larm(Guy *po)
{
  glPushMatrix();
  glScalef(LOWER_ARM_GIRTH, LOWER_ARM_SIZE, LOWER_ARM_GIRTH);
  glRotatef(90.0, 1.0, 0.0, 0.0);
#if HAVE_LIBGLU
  gluCylinder(po->quadc, 1.0, LOWER_ARM_TAPER, 1.0, SLICES, STACKS);
#else
  drawCylinder(1.0, LOWER_ARM_TAPER, 1.0, SLICES, STACKS, 0);
#endif
  glPopMatrix();
}

static
void makeGuy(Guy *po)
{
#if HAVE_LIBGLU
  po->quadc = gluNewQuadric();
  po->quads = gluNewQuadric();
#endif

  po->body_lists = glGenLists(GUY_NUM_BODY_PARTS);
  glNewList(po->body_lists, GL_COMPILE);
  store_head(po);
  glEndList();
  glNewList(po->body_lists+1, GL_COMPILE);
  store_torso(po);
  glEndList();
  glNewList(po->body_lists+2, GL_COMPILE);
  store_uleg(po);
  glEndList();
  glNewList(po->body_lists+3, GL_COMPILE);
  store_lleg(po);
  glEndList();
  glNewList(po->body_lists+4, GL_COMPILE);
  store_foot(po);
  glEndList();
  glNewList(po->body_lists+5, GL_COMPILE);
  store_uarm(po);
  glEndList();
  glNewList(po->body_lists+6, GL_COMPILE);
  store_larm(po);
  glEndList();
}

static
void draw_head(Guy *po)
{
  glPushMatrix();
  glCallList(po->body_lists);
  glPopMatrix();
}

static
void draw_torso(Guy *po)
{
  glPushMatrix();
  glCallList(po->body_lists+1);
  glPopMatrix();
}

static
void draw_leg(int which, Guy *po)
{
  glPushMatrix();
  if (which == 0)
    glTranslatef(TORSO_TAPER*TORSO_WIDTH/2.0, 0.0, 0.0);
  else
    glTranslatef(-TORSO_TAPER*TORSO_WIDTH/2.0, 0.0, 0.0);

  /* UPPER leg: rotates about the x axis only */
  glRotatef(po->walk_cycle[which][0][po->step], 1, 0, 0);
  glPushMatrix();
  glCallList(po->body_lists+2);
  glPopMatrix();

  /* LOWER leg: rotates about the x axis only */
  glTranslatef(0.0, -(UPPER_LEG_SIZE+KNEE_JOINT_SIZE), 0.0);
  glRotatef(po->walk_cycle[which][1][po->step], 1, 0, 0);
  glPushMatrix();
  glCallList(po->body_lists+3);
  glPopMatrix();

  /* Foot: rotates about the x axis only */
  glTranslatef(0.0, -(UPPER_LEG_SIZE+LOWER_LEG_SIZE+LOWER_LEG_GIRTH)/2.0, 0.0);
  glRotatef(po->walk_cycle[which][2][po->step], 1, 0, 0);
  glPushMatrix();
  glCallList(po->body_lists+4);
  glPopMatrix();
  glPopMatrix();
}
 
static
void draw_arm(int which, Guy *po)
{
  int arm_which;

  if (which == 1)
    arm_which = 1;
  else arm_which = 0;

  glPushMatrix();
  glTranslatef(0.0, TORSO_HEIGHT, 0.0);
  if (which == 0)
    glTranslatef(TORSO_WIDTH, 0.0, 0.0);
  else
    glTranslatef(-TORSO_WIDTH, 0.0, 0.0);
  /* UPPER leg: rotates about the x axis only */
  glRotatef(po->walk_cycle[arm_which][3][po->step], 1, 0, 0);
  glPushMatrix();
  glCallList(po->body_lists+5);
  glPopMatrix();

  /* LOWER leg: rotates about the x axis only */
  glTranslatef(0.0, -(UPPER_ARM_SIZE+ELBOW_JOINT_SIZE), 0.0);
  glRotatef(po->walk_cycle[arm_which][4][po->step], 1, 0, 0);
  glPushMatrix();
  glCallList(po->body_lists+6);
  glPopMatrix();
  glPopMatrix();
}

static
void drawGuy(Guy *po)
{
  float head_diffuse[] = { 0.7, 0.4, 0.4, 1.0 };
  float torso_diffuse[] = { 0.0, 0.7, 0.7, 1.0 };
  float leg_diffuse[] = { 0.7, 0.0, 0.7, 1.0 };
  float arm_diffuse[] = { 0.7, 0.4, 0.4, 1.0 };

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, head_diffuse);
  draw_head(po);
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, torso_diffuse);
  draw_torso(po);
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, leg_diffuse);
  draw_leg(0, po);
  draw_leg(1, po);
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, arm_diffuse);
  draw_arm(0, po);
  draw_arm(1, po);
}

/* Matrix times a vector  dest = m*v */
static
void MultMV(float m[3][4], float v[4], float dest[3])
{
  for (int i = 0; i < 3; i++) {
    dest[i] = 0;
    for (int j = 0; j < 4; j++)
      dest[i] += m[i][j] * v[j];
  }
}

/* Matrix multiplication, dest = m1*m2 */
static
void MultM(float m1[3][4], float m2[4][4], float dest[3][4])
{
  for (int i = 0; i < 3; i++)
    for (int j = 0; j < 4; j++) {
      dest[i][j] = 0;
      for (int k = 0; k < 4; k++)
	  dest[i][j] += (m1[i][k] * m2[k][j]);
    }
}

static
void computeCurve(int joint, Guy *po)
{
  float prod[3][4], tm[4], pos[3];
  float t = 0, tinc = (float)GUY_CYCLE_STEP/GUY_OVERSAMPLE;
  int point, i;
  float BBasis[4][4] = {{-1, 3, -3, 1}, {3, -6, 3, 0},
                        {-3, 3,  0, 0}, {1,  0, 0, 0}};
  int lastindex, newindex;
  float pointset[3][4];

  for (i = 0; i < 4; i++)   /* z's are always zero, only 2-d */
    pointset[2][i] = 0; 

  lastindex = -1;
  for (point = 0; point < po->curve[joint].numpoints; point += 3) {
    t = 0;
    for (i = 0; i < 4; i++)
      pointset[0][i] = po->curve[joint].xcoords[point + i];
    for (i = 0; i < 4; i++)
      pointset[1][i] = po->curve[joint].angles[point + i];

    MultM(pointset, BBasis, prod);

    while (t <= 1) {
      tm[0] = t*t*t;
      tm[1] = t*t;
      tm[2] = t;
      tm[3] = 1;
      MultMV(prod, tm, pos);
      newindex = (int)(pos[0]*(GUY_CYCLE_SIZE-1));
      if ((int)(newindex > lastindex))  {  /* go at least one */
	po->walk_cycle[0][joint][newindex] = pos[1];
	lastindex++;
      } 
      t += tinc;
    }
  }
  for (i = 0; i < GUY_CYCLE_SIZE; i++) {  /* copy to other leg, out-o-phase */
    po->walk_cycle[1][joint][i] =
      po->walk_cycle[0][joint][(i+(GUY_CYCLE_SIZE/2))%GUY_CYCLE_SIZE];
  }
}

static
void computeCSet(Guy *po)
{
  for (int j = 0; j < po->numjoints; j++)
    computeCurve(j, po);
}

static
void flatCSet(Guy *po)
{
  for (int j = 0; j < po->numjoints; j++) {
    po->curve[j].numpoints = 4;
    po->curve[j].xcoords[0] = 0.0;
    po->curve[j].angles[0]  = 0.0;
    po->curve[j].xcoords[1] = 0.2;
    po->curve[j].angles[1]  = 0.0;
    po->curve[j].xcoords[2] = 0.8;
    po->curve[j].angles[2]  = 0.0;
    po->curve[j].xcoords[3] = 1.0;
    po->curve[j].angles[3]  = 0.0;
  }
}

static
void readCSet(void *oa, void *handle)
{
  int numjoints = 0, numpoints = 0, point;
  char *l, line[512];
  Guy *po = (Guy *) oa;

  if (handle == NULL)
    goto abort;
  httpClearBuf();
  httpGetLine(handle, line);
  if ((po->numjoints = numjoints = atoi(line)) >= GUY_MAX_JOINTS)
    goto abort;
  if ((po->curve = (tControlPts *) calloc(sizeof(tControlPts), numjoints)) == NULL) {
    trace(DBG_FORCE, "readCset: can't alloc curve");
    return;
  }
  httpGetLine(handle, line);	/* mirror_flag */
  for (int j = 0; j < po->numjoints; j++) {
    httpGetLine(handle, line);
    numpoints = atoi(line);
    if (numpoints < 4 || numpoints > GUY_MAX_CPOINTS)
      goto abort;
    po->curve[j].numpoints = numpoints;
    httpGetLine(handle, line);
    l = strtok(line, " ");
    for (point = 0; point < numpoints; point++) {
      po->curve[j].xcoords[point] = (float) atof(l);
      l = strtok(NULL, " ");
    }
    httpGetLine(handle, line);
    l = strtok(line, " ");
    for (point = 0; point < numpoints; point++) {
      po->curve[j].angles[point] = (float) atof(l);
      l = strtok(NULL, " ");
    }
  }
  return;

abort:
  trace(DBG_FORCE, "Something went wrong while reading CSet file, numjoints = %d, numpoint = %d", numjoints, numpoints);
  if (numjoints == 0) {
    numjoints = po->numjoints = GUY_NUM_JOINTS;
    if ((po->curve = (tControlPts *) calloc(sizeof(tControlPts), numjoints)) == NULL) {
      trace(DBG_FORCE, "readCset: can't alloc curve");
      return;
    }
  }
  flatCSet(po);
  return;
}

static
void guyInit(Guy *po)
{
  static float mat_ambient[] = { 1.0, 1.0, 1.0, 1.0 };
  
  glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
#if !defined(WITH_TINYGL)
  glDepthFunc(GL_LESS);
#endif
  glEnable(GL_NORMALIZE);
  glShadeModel(GL_SMOOTH);
  makeGuy(po);
}

/* create from a fileline */
void Guy::creator(char *l)
{
  new Guy(l);
}

Guy::Guy(char *l)
{
  l = parseName(l, this);
  l = parsePosition(l, this);
  l = parseURL(l, this);
  if (l) {
    anim = atoi(l);
    l = strtok(NULL, SEP);
  }
  if (l)
    walk = atoi(l);

  char geom[80];

  sprintf(geom,"bbox,size=%.2f,%.2f,%.2f",
          TORSO_WIDTH/2,
          UPPER_LEG_GIRTH,
          (HEAD_SIZE/2 + TORSO_HEIGHT + UPPER_LEG_TAPER + LOWER_LEG_TAPER)/2);
  if ((soh = parseGeometry(geom)) == NULL) {
    trace(DBG_FORCE, "guy: can't create solid");
    return;
  }
  nature.movable = VR_NO_ELEM_MOVE;
  nature.renderable = VR_SPECIAL_RENDER;
  nature.collision = COL_NEVER;
  initializeObject(this, GUY_TYPE, VR_MOBILE);

  rotz = (float) RADIAN2DEGREE(pos.az);
  rotx = (float) RADIAN2DEGREE(pos.ax);
  roty = 90.0;
  move.perm_sec = 1;
  step = GUY_CYCLE_SIZE/2;
  fstep = GUY_CYCLE_SIZE/2;
  incstep = 1.0;
  guyInit(this);
  httpOpen(name.url, readCSet, this, THREAD_NO_BLOCK);
  if (anim)
    computeCSet(this);
  else
    flatCSet(this);
}

/* system of equations handling permanent motion */
void Guy::changePermanent(float lasting)
{
  if (anim) {
    fstep = fmod(fstep + incstep, GUY_CYCLE_SIZE);
    step = (int) fstep;
  }
}

void Guy::render()
{
  float dx, dz;
  static       float guy_radian = 3 * M_PI/2; /* radian */
  static const float guy_step   = 72; /* number of steps */
  static const float guy_radius = 1.5; /* space unit */

  glPushMatrix();
  glTranslatef(pos.x, pos.y, pos.z);
  glRotatef(rotx, 1, 0, 0);
  glRotatef(roty, 0, 1, 0);
  glRotatef(rotz, 0, 0, 1);

  guy_radian -= M_2PI / guy_step;
  if (guy_radian <= 0.0)
    guy_radian = M_2PI;
  if (walk == 1) {
    dx =  guy_radius * cos(guy_radian);
    dz = -guy_radius * sin(guy_radian);
    glTranslatef(-dx, 0, -dz);
    glRotatef((float) RADIAN2DEGREE(guy_radian), 0, 1, 0);
  }
  if (walk == 2) {
    glRotatef((float) RADIAN2DEGREE(guy_radian), 0, 1, 0);
  }
  if (walk == 3) {
    glRotatef((float) RADIAN2DEGREE(guy_radian), 0, 0, 1);
  }
  if (walk == 4) {
    glRotatef((float) RADIAN2DEGREE(guy_radian), 1, 0, 0);
  }
  drawGuy(this);
  glPopMatrix();
}

void Guy::quit()
{
#if HAVE_LIBGLU
  gluDeleteQuadric(quadc);
  gluDeleteQuadric(quads);
#endif
  if (curve)
    free(curve);
}

void guyInitFuncList(void) { }

#endif /* !VRENGD */
