#ifndef VRENGD

#include <GL/gl.h>

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"	/* COL_ONCE */
#include "cinema.h"

#include "zv.h"		/* parseGeometry */
#include "texture.h"	/* getCurrentTextureNumber */
#include "images.h"	/* newImage */
#include "helpers.h"

#if HAVE_LIBMPEG
#include <mpeg.h>
#endif


const WClass Cinema::wclass(CINEMA_TYPE, "Cinema", Cinema::creator);

static boolean abortflag = FALSE;


void Cinema::creator(char *l)
{
  new Cinema(l);
}

Cinema::Cinema(char *l)
{
  fps = CINEMA_FPS;	/* 10 fps by default */
  l = parseName(l, this);
  l = parsePosition(l, this);
  l = parseURL(l, this);
  if (l && isdigit((int) *l)) {
    fps = atoi(l);
    l = strtok(NULL, SEP);
  }
  soh = parseGeometry(l);

  nature.movable = VR_NO_ELEM_MOVE;
  initializeObject(this, CINEMA_TYPE, VR_MOBILE);
  nature.collision = COL_ONCE;
  status = CINEMA_INACTIVE;

  texture = getCurrentTextureNumber();
  trace(DBG_WMGT, "cinema: texture_number = %d", texture);
} 

static
void mpegAbort(int sig)
{
#if HAVE_LIBMPEG
    CloseMPEG();
#endif
    abortflag = TRUE;
}

void Cinema::changePermanent(float lasting)
{
#if HAVE_LIBMPEG
  if (abortflag == TRUE)
    status = CINEMA_INACTIVE;
  if (status == CINEMA_INACTIVE)
    return;
  if (status == CINEMA_PAUSE)
    return;

  if (first) {
      gettimeofday(&this->start, NULL);
      first = 0;
  }

  int frame_inter = frame;
  struct timeval t;

  gettimeofday(&t, NULL);
  frame = (int) floor(((((float) (t.tv_sec - start.tv_sec) * 1000.) +
	                ((float) (t.tv_usec - start.tv_usec) / 1000.)
                       ) / 1000.) * (float) fps);

 /*
  * get Mpeg frame or not
  */
  for (int f=0; f < (frame - frame_inter); f++) {
    if (f >= rate)
      break;
    if (GetMPEGFrame((char *)pixels) == FALSE) {
      /* The End */
      if (status == CINEMA_LOOP) {
        RewindMPEG(fpmpeg, &img);
        frame = 0;
        first = 1;
        abortflag = FALSE;
        return;
      }
      CloseMPEG();
      status = CINEMA_INACTIVE;
      return;
    }
  }

  ColormapEntry *colorMap = img.Colormap;
  int woff = (screensize - imagewidth) / 2;
  int hoff = (screensize - imageheight) / 2;
  
#define R 0
#define G 1
#define B 2
#define A 3
 
  if (colorMap) {
    // case of Colormap Index
    for (int j=0; j < imageheight; j++) {
      for (int i=0; i < imagewidth; i++) {
        int colormapIndex = pixels[j * imagewidth + i];
        ColormapEntry * color = &colorMap[colormapIndex];
        int u = IMAGE_RGB * ((j+hoff) * screensize + (i+hoff));
        screen[u + R] = color->red % 255;	
        screen[u + G] = color->green % 255;	
        screen[u + B] = color->blue % 255;
      }
    }
  } else {
    for (int j=0; j < imageheight; j++) {
      for (int i=0; i < imagewidth; i++) {
        int u = IMAGE_RGB * (screensize * (j+hoff) + (i+woff));
        int v = IMAGE_RGBA * (imagewidth * j + i);
#if WORDS_BIGENDIAN
        screen[u + R] = pixels[v + A];
        screen[u + G] = pixels[v + B];
        screen[u + B] = pixels[v + G];
#else
        screen[u + R] = pixels[v + R];
        screen[u + G] = pixels[v + G];
        screen[u + B] = pixels[v + B];
#endif /* WORDS_BIGENDIAN */
      }
    }
  }
#undef R
#undef G
#undef B
#undef A

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexImage2D(GL_TEXTURE_2D, 0, VRENG_TEXTURE_TYPE, screensize, screensize, 0,
	       GL_RGB, GL_UNSIGNED_BYTE, screen);

#endif /* HAVE_LIBMPEG */
}

int mpegInit(Cinema *po)
{
#if HAVE_LIBMPEG
  int i, power = 0;
  struct stat bufstat;

  /*
   * download Mpeg file
   */
  if ((po->tmpmpeg = startwget(po->name.url)) == NULL) {
    trace(DBG_FORCE, "mpegInit: can't wget mpeg file");
    return 0;
  }
  po->fpmpeg = NULL;
  for (i=0; i < 5; i++) {
    if ((po->fpmpeg = fopen(po->tmpmpeg, "r")) != NULL)
      break;
    trace(DBG_FORCE, "downloading mpeg...");
    sleep(1);
  }
  if (po->fpmpeg == NULL)
    return 0;
  stat(po->tmpmpeg, &bufstat);
  if (bufstat.st_size == 0) {
    unlink(po->tmpmpeg);
    return 0;
  }

  /*
   * init Mpeg
   */
  SetMPEGOption(MPEG_DITHER, FULL_COLOR_DITHER); //ORDERED_DITHER);
  OpenMPEG(po->fpmpeg, &po->img);

  /*
   * get Mpeg infos
   */
  po->imagewidth = po->img.Width;
  po->imageheight = po->img.Height;
  po->rate = po->img.PictureRate;
  if ((po->pixels = (u_int8 *) malloc(po->img.Size)) == NULL) {
    trace(DBG_FORCE, "mpegInit: can't alloc pixels");
    return 0;
  }

  /*
   * compute screensize
   */
  if (po->imageheight > po->imagewidth)
    po->screensize = po->imageheight;
  else
    po->screensize = po->imagewidth;
  while ((po->screensize = po->screensize/2) > 0)
    power++;
  for (i=0, po->screensize = 1; i <= power; i++)
    po->screensize *= 2;

  trace(DBG_WMGT, "mpegInit: size=%d width=%d height=%d rate=%.2f screen=%d",
                   po->img.Size, po->imagewidth, po->imageheight, po->rate, po->screensize);

  /*
   * alloc screen
   */
  if ((po->screen = (GLubyte *) calloc(sizeof(GLubyte), IMAGE_RGB * po->screensize * po->screensize)) == NULL) {
    trace(DBG_FORCE, "mpegInit: can't alloc screen");
    free(po->pixels);
    return 0;
  }

  po->frame = 0;
  po->first = 1;
  signal(SIGABRT, mpegAbort);
  trace(DBG_WMGT, "mpegInit: done");
  return 1;
#else /* !HAVE_LIBMPEG */
  return 0;
#endif /* HAVE_LIBMPEG */
} 

void cinemaplay(Cinema *po, void *data, time_t sec, time_t usec)
{
  if (po->status == CINEMA_INACTIVE) {
    if (!mpegInit(po))
      return;
    po->status = CINEMA_PLAYING;
    po->move.perm_sec = 1;
  }
}

void cinemastop(Cinema *po, void *data, time_t sec, time_t usec)
{
  if (po->status != CINEMA_INACTIVE) {
    po->status = CINEMA_INACTIVE;
#if HAVE_LIBMPEG
    CloseMPEG();
#endif
    fclose(po->fpmpeg);
    abortflag = FALSE;
  }
  if (po->fpmpeg) {
    unlinktmp(po->tmpmpeg);
    po->fpmpeg = NULL;
  }
}

void cinemapause(Cinema *po, void *data, time_t sec, time_t usec)
{
  if (po->status == CINEMA_PLAYING || po->status == CINEMA_LOOP)
    po->status = CINEMA_PAUSE;
  else if (po->status == CINEMA_PAUSE)
    po->status = CINEMA_PLAYING;
}

void cinemarewind(Cinema *po, void *data, time_t sec, time_t usec)
{
#if HAVE_LIBMPEG
  if (po->status != CINEMA_PLAYING && po->status != CINEMA_LOOP && po->fpmpeg != NULL) {
    RewindMPEG(po->fpmpeg, &po->img);
    po->status = CINEMA_PLAYING;
    po->frame = 0;
    po->first = 1;
  }
#endif
}

void cinemaloop(Cinema *po, void *data, time_t sec, time_t usec)
{
  if (po->status != CINEMA_INACTIVE)
    po->status = CINEMA_LOOP;
}

void Cinema::whenIntersect(WObject *pcur, WObject *pold)
{
  copyPositionAndBB(pold, pcur);
}

void Cinema::quit()
{
  cinemastop(this, NULL, 0L, 0L);
}

void cinemaInitFuncList(void)
{
  setMethodFunc(CINEMA_TYPE, 0, WO_ACTION cinemaplay, "Play");
  setMethodFunc(CINEMA_TYPE, 1, WO_ACTION cinemastop, "Stop");
  setMethodFunc(CINEMA_TYPE, 2, WO_ACTION cinemapause, "Pause");
  setMethodFunc(CINEMA_TYPE, 3, WO_ACTION cinemaloop, "Loop");
  setMethodFunc(CINEMA_TYPE, 4, WO_ACTION cinemarewind, "Rewind");
}

#endif /* !VRENGD */
