
// glmutil.cpp
// Functions for glm design and manipulation
// Copyright (c) 2003-2009 by the VoxBo Development Team
//
// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// code contributed by Dongbo Hu, Dan Kimberg, and Tom King

using namespace std;

#include "glmutil.h"
#include <fstream>

/*****************************************************************************
 * Functions to read a condition function with strings 
 * (originally from gdw's condition function loading)
 * Added by Dongbo on March 16, 2004
 *****************************************************************************/

/* Main function to translate a condition function into vb_vector */
int getCondVec(const char *condFile, deque<string> &condKey, VB_Vector *condVec)
{
  tokenlist headerKey, output;
  int readStat = readCondFile(headerKey, output, condFile);
  if (readStat == -1)
    return -1;

  unsigned condLength = output.size();
  tokenlist contentKey = getContentKey(output);
  int cmpStat = cmpElement(headerKey, contentKey);
  if (cmpStat == -1) {
    sortElement(contentKey);
    for (int i = 0; i < contentKey.size(); i++)
      condKey.push_back(contentKey(i));
  }
  else if (cmpStat == -2)
    return -2;
  else if (cmpStat == 1)
    return 1;
  else {
    for (int i = 0; i < headerKey.size(); i++) 
      condKey.push_back(headerKey(i));
  }

  // condVec is the original vector converted from the condition function
  condVec->resize(condLength);
  for (unsigned i = 0; i < condLength; i++) {
    for (unsigned j = 0; j < condKey.size(); j++) {
      if (strcmp(output(i), condKey[j].c_str()) == 0) {
        condVec->setElement(i, j);
        break;
      }
      else
        continue;
    }
  }

  return 0;
}

/* readCondFile() reads an input condition function file and save each uncommented 
 * line into a tokenlist object. It also checks if the header comments have any 
 * specific condition label keys defined. */
int readCondFile(tokenlist &headerKey, tokenlist &output, const char *condFile)
{
  FILE *fp;
  char line[512]; // Assume each line has 512 charactoers or less

  fp = fopen(condFile,"r");
  if (!fp)
    return -1;    // returns -1 if file not readable

  string tmpString1, tmpString11, tmpString2, tmpString22, tmpString3;
  while (fgets(line,512,fp)) {
    if (strchr(";#%\n",line[0])) {
      stripchars(line,"\n");
      tmpString1 = &line[0];

      // If a comment line is long enough, check whether it has user-defined condition labels
      if (tmpString1.length() > 11) {
	tmpString11 = tmpString1.substr(1, tmpString1.length() - 1); // Remove comment character
	tmpString2 = xstripwhitespace(tmpString11);                  // Simplify white spaces

	// If tmpString2 started with "condition:", get the string after that and save as a condition label
	tmpString22 = tmpString2.substr(0, 10);
	tmpString22=vb_tolower(tmpString22);
	
	if (tmpString22 == "condition:") {
	  tmpString3 = xstripwhitespace(tmpString2.substr(10, tmpString2.length() - 10));
	  headerKey.Add(tmpString3);
	}
      }
      continue;
    }

    stripchars(line,"\n");
    // If the line starts with space (32) or tab (9) keys, delete them
    while(int(line[0]) == 32 || int(line[0]) == 9) { 
      for (int n = 0; n < (int)strlen(line); n++)
	line[n] = line[n + 1];
    }    
    output.Add(line); // Thanks to Dan's tokenlist class
  }

  fclose(fp);
  return 0;    // no error!

}

/* This function scans each element in condFunctand records unique condition keys in the 
 * order of their appearance. Keys are saved in condKeyInFile, which is another tokenlist. */
tokenlist getContentKey(tokenlist &inputLine)
{
  tokenlist condKeyInFile = tokenlist();
  string keyStr;
  int condLength = inputLine.size();
  // If one of the keys is "0" or "baseline", set it as the first condition key
  for (int i = 0; i < condLength; i++) {
    keyStr = inputLine(i);
    if (keyStr == "0" || keyStr == "baseline") {
      condKeyInFile.Add(keyStr);
      break;
    }
    else if (i == condLength - 1)
      condKeyInFile.Add(inputLine(0));
    else
      continue;
  }

  for (int i = 0; i < condLength; i++) {
    if (cmpString(inputLine(i), condKeyInFile)) 
      condKeyInFile.Add(inputLine(i));
  }

  return condKeyInFile;
}

/* This method is to compare an input string with elements in condKey, which is a tokenlist object. 
 * Returns 0 if the input string is equal to any of the elements in condKey
 * Returns 1 if the input string isn't equal to any of the elements in condKey */
int cmpString(const char *inputString, deque<string> &inputList)
{
  for (unsigned i = 0; i < inputList.size(); i++) {
    if (strcmp(inputList[i].c_str(), inputString) == 0)
      return 0;
  }

  return 1;
}

/* This function sorts the elements in a tokenlist object alphabetically.
 * First it checks whether the first element is "0" or "baseline". 
 * If yes, it will keep it and sort the rest of the elements alphabetically;
 * If not, it will sort all elements alphabetically. */
void sortElement(tokenlist &inputList)
{
  const char * element0 = inputList(0);
  if (strcmp(element0, "0") == 0 || strcmp(element0, "baseline") == 0) {
    tokenlist otherElmt;
    for (int i = 1; i < inputList.size(); i++)
      otherElmt.Add(inputList(i));
    otherElmt.Sort(alpha_sorter);

    inputList.clear();
    inputList.Add(element0);
    for (int i = 0; i < otherElmt.size(); i++)
      inputList.Add(otherElmt(i));
  }
  else
    inputList.Sort(alpha_sorter);
}

/* cmpElement() compares elements in two input tokenlists. 
 * If each element in input1 can be found in input2 and vice versa 
 * (without considering the order of elements), it will return 0; 
 * Otherwise, it will:
 * returns  0 if two input tokenlists are exactly same
 * returns -1 if the first tokenlist doesn't have any elements;
 * returns -2 if two tokenlist have different number of elements
 * returns  1 if some elements are different. */
int cmpElement(deque<string> &input1, deque<string> &input2)
{
  if (!input1.size())
    return -1;

  if (input1.size() != input2.size())
    return -2;

  for (unsigned i = 0; i < input1.size(); i++) {
    if (cmpString(input1[i].c_str(), input2) == 0 && cmpString(input2[i].c_str(), input1) == 0)
      continue;
    else 
      return 1;
  }

  return 0;
}

/* getCondLabel() reads a text file and saves each uncommented line into a tokenlist object 
 * It is written for loading condition labels from a certain file */
int getCondLabel(deque<string> &outputToken, const char *inputFile)
{
//  outputToken = tokenlist();
  const char *condFile = inputFile;
  FILE *fp;
  char line[512]; // Assume each line has 512 charactoers or less

  fp = fopen(condFile,"r");
  if (!fp)
    return -1; // returns -1 if file not readable

  while (fgets(line,512,fp)) {
    // Disregard the line if it starts with these characters: ;#%\n (notice that space is allowed!)
    if (strchr(";#%\n",line[0])) 
      continue;
    stripchars(line,"\n");

    // If the line starts with space (32) or tab (9) keys, delete them
    while(int(line[0]) == 32 || int(line[0]) == 9) { 
	for (int n = 0; n < (int)strlen(line); n++)
	  line[n] = line[n + 1];
    }    
    outputToken.push_back(line); // Thanks to Dan's tokenlist class
  }

  fclose(fp);
  return 0;    // no error!
}

/* Function to check an output filename's status, 
 * originally written for vbfit interactive mode */
int getInputStatus(const char *fitFilename)
{
  if (access(fitFilename,R_OK))
    return 2;  // returns 2 if the file can't be read
  else
    // FIXME this is atrocious!
    return 3;  // return 3 if it's a regular file and process-readable 
}

/* Function to check an output filename's status, 
 * originally written for vbfit interactive mode */
int checkOutputFile(const char *outFilename, bool ovwFlag)
{
  FileCheck inputCheck(outFilename);
  // FileCheck parentDir(inputCheck.getDirName());
  bool pw=0;
  if (!access(xdirname(outFilename).c_str(),W_OK))
    pw=1;

  // FIXME this is atrocious!
  if (inputCheck.getValid() && !pw)
    return 0;    // returns 0 if the file exists and parent directory isn't writable
  else if (inputCheck.getValid() && pw && !ovwFlag)
    return 1;    // returns 1 if the file exists, parent directory is writable, but overwrite isn't allowed

  else if (!inputCheck.getValid() && !pw)
    return 2;    // returns 2 if the file doesn't exist but parent directory isn't writable

  else if (inputCheck.getValid() && pw && ovwFlag) 
    return 3;    // returns 3 if the file exists, parent directory is writable and overwrite is allowed

  else
    return 4;    // returns 4 if the file doesn't exist and parent directory is writable
}

/* This function will upsample an input vector by a certain ratio 
 * and return the new vector */
VB_Vector * upSampling(VB_Vector *inputVector, int upRatio)
{
  int origSize = inputVector->getLength();
  int newSize = upRatio * origSize;
  VB_Vector *newVector = new VB_Vector(newSize);
  double tmpValue;
  int oldIndex;

  for (int newIndex = 0; newIndex < newSize; newIndex++) {
    oldIndex = newIndex / upRatio;
    tmpValue = inputVector->getElement(oldIndex);
    for (int j = 0; j < upRatio; j++) 
      newVector->setElement(newIndex, tmpValue);
  }
	
  return newVector;
}

/* This function will downsample an input vector by a certain ratio and returns the new vector */
VB_Vector * downSampling(VB_Vector *inputVector, int downRatio)
{
  int origSize =  inputVector->getLength();
  int newSize = origSize / downRatio;
  VB_Vector *newVector = new VB_Vector(newSize);
  double tmpValue;

  for (int newIndex = 0; newIndex < newSize; newIndex++) {
    tmpValue = inputVector->getElement(newIndex * downRatio);
    newVector->setElement(newIndex, tmpValue);
  }
  
  return newVector;
}

/* getConv() is a generic function to convolve an input VB_Vector with inputConv 
 * by the input sampling rate. It calls fftConv() for basic convolution calculation
 * using Geoff's fft/ifft combination algorithm and returns the new VB_Vector. */
VB_Vector getConv(VB_Vector *inputVector, VB_Vector *inputConv, int inputSampling, int tmpResolve)
{
  // Sampling is the time resolution in HRF 
  const unsigned int expFactor = (const unsigned int) (inputSampling / tmpResolve);
  VB_Vector *convVector = new VB_Vector(inputConv);
  convVector->sincInterpolation(expFactor);
  VB_Vector oldConvVec(convVector);

  int tmpLength = inputVector->getLength();
  convVector->resize(tmpLength);
  convVector->setAll(0);
  int orgLength = oldConvVec.getLength();
  // Make sure inputVector doesn't have less elements than inputConv
  if (orgLength > tmpLength) {
    printf("getConv() error: inputConv has more elements than inputVector, convolution not allowed\n");
    return convVector;
  }

  for (int i = 0; i < orgLength; i++)
    (*convVector)[i] = oldConvVec[i];

  convVector->meanCenter();
  convVector->normMag();

  return fftConv(inputVector, convVector, true);
}

/* fftConv() is another generic function for convolution calculation. 
 * It accepts two input vb_vector arguments and a "zeroFlag", which determines
 * whether this sentence will be added:
      realKernel->setElement(0, 1.0);
 * It returns the convolution vector. 
 * Note: 
 * When it is called by getConv() to modify a single covariate and efficiency
 * checking, zeroFlag = 1;
 * But when it is called by "Fourier set" or "Eigenvector set", zeroFlag = 1 */
VB_Vector fftConv(VB_Vector *inputVector, VB_Vector *convVector, bool zeroFlag)
{
  int tmpLength = inputVector->getLength();
  VB_Vector *realKernel = new VB_Vector(tmpLength);
  VB_Vector *imagKernel = new VB_Vector(tmpLength);
  convVector->fft(realKernel, imagKernel);

  if (zeroFlag) {
    realKernel->setElement(0, 1.0);
    imagKernel->setElement(0, 0.0);
  }

  VB_Vector *orgReal = new VB_Vector(tmpLength);
  VB_Vector *orgImag = new VB_Vector(tmpLength);
  inputVector->fft(orgReal, orgImag);

  VB_Vector *newReal = new VB_Vector(tmpLength);
  VB_Vector *newImag = new VB_Vector(tmpLength);
  double tmp1 = 0, tmp2 = 0;
  for (int i = 0; i < tmpLength; i++) {
    tmp1 = realKernel->getElement(i) * orgReal->getElement(i) - 
      imagKernel->getElement(i) * orgImag->getElement(i);
    tmp2 = imagKernel->getElement(i) * orgReal->getElement(i) + 
      realKernel->getElement(i) * orgImag->getElement(i);
    newReal->setElement(i, tmp1);
    newImag->setElement(i, tmp2);
  }
  
  VB_Vector *inverseReal1 = new VB_Vector(tmpLength);
  VB_Vector *inverseImag1 = new VB_Vector(tmpLength);
  VB_Vector *inverseReal2 = new VB_Vector(tmpLength);
  VB_Vector *inverseImag2 = new VB_Vector(tmpLength);

  newReal->ifft(inverseReal1, inverseImag1);
  newImag->ifft(inverseReal2, inverseImag2);

  VB_Vector newVec = VB_Vector(tmpLength);
  for(int i = 0; i < tmpLength; i++) {
    double tmp = inverseReal1->getElement(i) - inverseImag2->getElement(i);
    newVec.setElement(i, tmp);
  }

  delete realKernel;
  delete imagKernel;
  delete orgReal;
  delete orgImag;
  delete newReal;
  delete newImag;
  delete inverseReal1;
  delete inverseImag1;
  delete inverseReal2;
  delete inverseImag2;

  return newVec;
}

/* getDeterm() returns the determinant of the input G matrix 
 * Note: What it actually returns is the determinant of:
 *          inputG * Trnaspose(inputG)
 * DO NOT use this function to calculate determinant of a square matrix */
double getDeterm(VBMatrix &inputMat)
{
  int totalReps = inputMat.m;
  int colNum = inputMat.n;

  // a is gsl_matrix copied from inputMat
  gsl_matrix *a = gsl_matrix_calloc(totalReps, colNum);
  gsl_matrix_memcpy(a, &(inputMat.mview).matrix);

  // aTa: transpose(a) * a
  gsl_matrix *aTa = gsl_matrix_calloc(colNum, colNum);
  gsl_blas_dgemm(CblasTrans, CblasNoTrans, 1.0, a, a, 0.0, aTa);

  // Copied from gsl documentation (Chapter 13: Linear Algebra)
  int s;
  gsl_permutation *p = gsl_permutation_alloc(colNum);
  gsl_linalg_LU_decomp(aTa, p, &s); 
  double determ = gsl_linalg_LU_det(aTa, s);

  // time to free up the space
  gsl_permutation_free(p);
  gsl_matrix_free(a);
  gsl_matrix_free(aTa);

  return determ;
}

/* calcColinear() is a generic function for colinearity evaluation with two arguments: 
 * subA is a VBMatrix object,
 * dependVec is the dependent parameter, which is a vb_vector.
 * It calls multiplyBetas for matrix multiplycation calculation. */
double calcColinear(VBMatrix & subA, VB_Vector & dependVec)
{
  // Make sure dependVec isn't a constant vector
  if (!dependVec.getVariance()) {
    printf("calcColinear(): no colinearity calculated because the dependent parameter is constant.\n");
    return -1;
  }

  int vecLen = dependVec.getLength();
  VB_Vector * multiplyVec = multiplyBeta(subA, dependVec);
  if (!multiplyVec)
    return -1;

  VB_Vector residuals(vecLen);
  for (int i = 0; i < vecLen; i++)
    residuals[i] = dependVec[i] - multiplyVec->getElement(i);
  delete multiplyVec;

  double errorSquared = 0;
  for (int k = 0; k < vecLen; k++)
    errorSquared += residuals[k] * residuals[k];

  double totalSumSquares = (dependVec.getVariance()) * (double)(vecLen - 1);
  double foo = 1.0 - errorSquared / totalSumSquares;
  if (foo <= -0.0001) {
    printf("calcColinear: invalid colinearity value.\n");
    return -1;
  }
  else if (foo < 0 && foo > -0.0001)
    foo = 0;

  double rMul = sqrt(foo);
  return rMul;
}

/* multiplyBeta() is a function which accepts two arguments:
 * inMat: an input VBMatrix with m rows and n columns
 * inVec: an input VB_Vector with length of m
 * The returned VB_Vector is outVec: output VB_Vector with length of m

 * Their relationship is:
 *  outVecMat = inMat ## ( invert( transpose(inMat) ## inMat ) ## transpose(inMat) ## inVecMat )
 * in which outVecMat is a matrix with only m rows and one column; 
 * The elements in this column will build outVec; inVecMat is another matrix 
 * which has only one column with elements copied from inVec.

 * Note that gsl matrix multiplication functions are used here because the matrices
 * defined in this function are all intermediate ones which are not supposed to 
 * be written on the hard disk. 
 * The function MultiplyPartsXY() defined for VBMatrix are for matrices which have
 * a filename and will be written on disk. */
VB_Vector * multiplyBeta(VBMatrix &inMat, VB_Vector &inVec)
{
  int rowNum = inMat.m, colNum = inMat.n;
  int vecLen = inVec.getLength();

  // Make sure number of elements in inVec is same as number of rows in the matrix
  if (vecLen != rowNum) {
    printf("multiplyBeta(): dimentions of input matrix and vector don't match!\n");
    return 0;
  }

  // a is gsl_matrix copied from inMat
  gsl_matrix *a = gsl_matrix_calloc(rowNum, colNum);
  gsl_matrix_memcpy(a, &(inMat.mview).matrix);

  // aTa: transpose(a) * a
  gsl_matrix *aTa = gsl_matrix_calloc(colNum, colNum);
  gsl_blas_dgemm(CblasTrans, CblasNoTrans, 1.0, a, a, 0.0, aTa);

  // inv_aTa: the inverse of aTa
  int s;
  gsl_permutation *p = gsl_permutation_alloc(colNum);
  gsl_linalg_LU_decomp(aTa, p, &s);
  // Make sure aTa isn't a singular matrix so that the inverse exists
  if (!gsl_linalg_LU_det(aTa, s)) {
    printf("multiplyBeta(): singlar matrix found, inverse matrix invalid\n");
    return 0;
  }
  // gsl way to do invertion
  gsl_matrix *inv_aTa = gsl_matrix_calloc(colNum, colNum);
  gsl_linalg_LU_invert(aTa, p, inv_aTa);

  // invXaT: inv_aTa * tranpose(a)
  gsl_matrix * invXaT = gsl_matrix_calloc(colNum, rowNum);
  gsl_blas_dgemm(CblasNoTrans, CblasTrans, 1.0, inv_aTa, a, 0.0, invXaT);

  // Set a one-colomn gsl_matrix based on the inVec
  gsl_matrix *b = gsl_matrix_calloc(vecLen, 1);
  gsl_matrix_set_col(b, 0, inVec.getTheVector());

  // betas: invXaT * b 
  gsl_matrix * betas = gsl_matrix_calloc(colNum, 1);
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, invXaT, b, 0.0, betas);

  // aXbetas: a * betas
  gsl_matrix * aXbetas = gsl_matrix_calloc(rowNum, 1);
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, a, betas, 0.0, aXbetas);  

  VB_Vector *outVec = new VB_Vector(vecLen);
  for (int i = 0; i < vecLen; i++)
    outVec->setElement(i, gsl_matrix_get(aXbetas, i, 0));

  // time to free up the space
  gsl_permutation_free(p);
  gsl_matrix_free(a);
  gsl_matrix_free(b);
  gsl_matrix_free(aTa);
  gsl_matrix_free(inv_aTa);
  gsl_matrix_free(invXaT);
  gsl_matrix_free(betas);
  gsl_matrix_free(aXbetas);

  return outVec;
}

/* This method is to count the number of elements in 
 * a VB_Vector that equal to a certain value m */
int countNum(VB_Vector *inputVector, int m)
{
  int length = inputVector->getLength();
  int count = 0;
  int element;
  for (int i = 0; i < length; i++) {
    element = (int) inputVector->getElement(i);
    if (m == element)
      count++;
  }

  return count;
}

/* This function will count the number of non-zero elements in 
 * an input vector, Written for "Mean center non-zero" functionality */
int countNonZero(VB_Vector *inputVector)
{
  int length = inputVector->getLength();
  int count = 0;
  double element;
  for (int i = 0; i < length; i++) {
    element = inputVector->getElement(i);
    if (element)
      count++;
  }

  return count;  
}

/* This function accepts an input vb_vector and converts it by delta calculation.
 * It follows Geoff's G_Delta procedures. 
 * Note that although VB_Vector class has left shift (operator<<) and right shift (operator>>)
 * defined, they are different from IDL's shift function because shift() also wraps the elements
 * at the end (for right shift) or beginning (for left shift). */
void calcDelta(VB_Vector *inputVec)
{
  int vecLength = inputVec->getLength();
  VB_Vector *tmpVec = new VB_Vector(inputVec);
  VB_Vector *shifter = new VB_Vector(vecLength); 
  inputVec->setAll(0.0);

  // calculate shifter   
  (*shifter)[0] = (*tmpVec)[0] - (*tmpVec)[vecLength - 1];
  for (int i = 1; i < vecLength; i++) {
      (*shifter)[i] = (*tmpVec)[i] - (*tmpVec)[i - 1];
  }

  for (int i = 0; i < vecLength; i++) {
    /* Different from Geoff here: Instead of using (*shifter)[i] > 0, another criteria is used.
     * With this standard, it might be better to distinguish between the real element difference
     * and fluctuation from noise. */
    if ((*shifter)[i] > abs(tmpVec->getMaxElement()) * 0.00001) { 
      (*inputVec)[i] = 1;
    }
  }

  double vecSum = tmpVec->getVectorSum();
  double stdDev = sqrt(tmpVec->getVariance());

  if ((vecSum > -0.9) && (vecSum < 0.9))
    inputVec->meanCenter();

  if ((stdDev > 0.9) && (stdDev < 1.1))
    inputVec->unitVariance();

  delete tmpVec;
  delete shifter;
}

/* getTxtColNum() returns the number of elements 
 * on the first uncommented line from input file 
 * returns -1 if it is not readable */
int getTxtColNum(const char *inputFile)
{
  FILE *fp;
  char line[1024]; // Assume each line has 1024 characters or less
  fp = fopen(inputFile,"r");
  if (!fp)
    return -1;     // Returns 1 if file not readable

  int colNum = 0;
  tokenlist myRow;
  while (fgets(line,1024,fp)) {
    if (strchr(";#%\n",line[0]))
      continue;
    stripchars(line,"\n");
    const string tmpStr(line);
    myRow = tokenlist(tmpStr);
    colNum = myRow.size();
    myRow.clear();
    break;
  }

  fclose(fp);
  return colNum;
}

/* getTxtRowNum() returns the number of rows from input file 
 * returns -1 if it is not readable */
int getTxtRowNum(const char *inputFile)
{
  FILE *fp;
  char line[1024]; // Assume each line has 1024 charactoers or less
  fp = fopen(inputFile,"r");
  if (!fp)
    return -1;    // returns 1 if file not readable

  int rowNum = 0;
  while (fgets(line,1024,fp)) {
    if (strchr(";#%\n",line[0]))
      continue;
    rowNum++;
  }
  fclose(fp);
  return rowNum;
}

/* readTxt() checks the input file's format. 
 * If it is valid, read each column into the vb_vector array txtCov;
 * returns 0 when the input file is good; returns 1 when the input file 
 * doesn't have exactly the same number of elements in a row */
int readTxt(const char *inputFile, std::vector< VB_Vector *> txtCov)
{
  FILE *fp;
  char line[1024]; // Assume each line has 1024 charactoers or less

  fp = fopen(inputFile,"r");
  int currentRowNum = 0;
  int colNum = 0;
  tokenlist myRow;
  while (fgets(line,1024,fp)) {
    if (strchr(";#%\n",line[0]))
      continue;
    stripchars(line,"\n");
    const string tmpStr(line);
    myRow = tokenlist(tmpStr);
    if (currentRowNum == 0)
      colNum = myRow.size();
    if (colNum != myRow.size()) {
      fclose(fp);
      return 1;
    }
    for (int j = 0; j < colNum; j++)
      txtCov[j]->setElement(currentRowNum, atof(myRow(j)));
    currentRowNum++;
    myRow.clear();
  }

  fclose(fp);
  return 0;    // no error!
}

/* derivative() is a function that simulates Geoff's derivative() function in IDL 
 * Reference: VoxBo_Fourier.pro */
VB_Vector * derivative(VB_Vector *inputVec)
{
  unsigned vecLength = inputVec->getLength();
  if (vecLength % 2 != 0) {
    printf("Error in derivative(): odd number of elements in input vector: %d\n", vecLength);
    return 0;
  }

  VB_Vector *fftReal = new VB_Vector(vecLength);
  VB_Vector *fftImg = new VB_Vector(vecLength);
  inputVec->fft(fftReal, fftImg);

  VB_Vector *fftDevReal = new VB_Vector(vecLength);
  fftDevReal->setAll(0);
  VB_Vector *fftDevImg = new VB_Vector(vecLength);
  fftDevImg->setAll(0);

  double fundFreq = 2.0 * 3.14159 / (double) vecLength;
  double freq = 0, tmpReal = 0, tmpImg = 0;
  for (unsigned h = 1; h < vecLength / 2; h++) {
    freq = fundFreq * (double) h;

    tmpReal = fftReal->getElement(h);
    tmpImg = fftImg->getElement(h);
    fftDevReal->setElement(h, tmpImg * freq * (-1.0));
    fftDevImg->setElement(h, tmpReal * freq);

    tmpReal = fftReal->getElement(vecLength - h);
    tmpImg = fftImg->getElement(vecLength - h);
    fftDevReal->setElement(vecLength - h, tmpImg * freq);
    fftDevImg->setElement(vecLength - h, tmpReal * freq * (-1.0));
  }

  VB_Vector *ifftDevRealReal = new VB_Vector(vecLength);
  VB_Vector *ifftDevRealImg = new VB_Vector(vecLength);
  VB_Vector *ifftDevImgReal = new VB_Vector(vecLength);
  VB_Vector *ifftDevImgImg = new VB_Vector(vecLength);

  fftDevReal->ifft(ifftDevRealReal, ifftDevRealImg);
  fftDevImg->ifft(ifftDevImgReal, ifftDevImgImg);

  VB_Vector *finalVec = new VB_Vector(vecLength);
  for (unsigned i = 0; i < vecLength; i++) {
    tmpReal = ifftDevRealReal->getElement(i) - ifftDevImgImg->getElement(i);
    finalVec->setElement(i, tmpReal);
  }
  delete fftReal;
  delete fftImg;
  delete fftDevReal;
  delete fftDevImg;
  delete ifftDevRealReal;
  delete ifftDevRealImg;
  delete ifftDevImgReal;
  delete ifftDevImgImg;

  return finalVec;
}

/*
VB_Vector
GLMInfo::getFits(VBRegion &rr)
{
  VB_Vector fits;
  // build a vector of VB_Vectors with all the covariates
  // for each variable that's of interest, grab the scaled version
  int xx,yy,zz;
  vector<VB_Vector>vars;
  vector<VB_Vector>indices;
  string kgname=glmi.stemname+".KG";
  string gname=glmi.stemname+".G";
  string prmname=glmi.stemname+".prm";
  VBMatrix KG;
  Tes prm;

  if (KG.ReadMAT1(kgname))
    if (KG.ReadMAT1(gname))
      return fits;
  if (prm.Readheader(prmname))
    return fits;

  int ntimepoints=KG.m;
  // VB_Vector var=KG.getColumn(interestlist[i]);
  VB_Vector tmpfits;
  tmpfits.resize(ntimepoints);
  
  for (int j=0; j<myregion.size(); j++) {
    VB_Vector vf(ntimepoints);
    xx=myregion[j].x;
    yy=myregion[j].y;
    zz=myregion[j].z;
    // get betas for this voxel
    if (prm.ReadTimeSeries(prmname,x,y,z))
      return fits;
    VB_Vector weights(nvars);
    // copy ones of interest
    for (int i=0; i<(int)interestlist.size(); i++) {
      int ind=interestlist[i];
      vf+=prm.timeseries[ind]*KG.getColumn(i);
    }
  }
  return fits;
}
*/



// return the desired covariate, possibly scaled
VB_Vector
GLMInfo::getCovariate(int x,int y,int z,int paramindex,int scaledflag)
{
  VB_Vector covar;
  string prmname=xsetextension(stemname,"prm");
  string kgname=xsetextension(stemname,"KG");
  VBMatrix KG;

  int errs=0;
  int ntimepoints;
  int nbetas;
  // load KG
  KG.ReadMAT1(kgname);
  ntimepoints=KG.m;
  nbetas=KG.n;
  // if not valid, return the empty covariate
  if (!KG.dataValid())
    return covar;
  // fill the covariate with the right row of the KG matrix
  covar.resize(ntimepoints);
  for (int i=0; i<ntimepoints; i++)
    covar.setElement(i,KG(i,paramindex));
  // if scaled, get the beta
  if (scaledflag) {
    Tes prm;
    if (prm.ReadHeader(prmname))
      errs++;
    if (prm.ReadTimeSeries(prmname,x,y,z))
      errs++;
    if ((int)(prm.timeseries.getLength()) <=paramindex)
      errs++;
    if (errs==0)
      covar*=prm.timeseries[paramindex];
  }
  return covar;
}

int
GLMInfo::filterTS(VB_Vector &signal)
{
  if (!(exoFilt.getLength())) {
    string exoname=xsetextension(stemname,"ExoFilt");
    exoFilt.ReadFile(exoname);
    if (!(exoFilt.getLength()))
      return 101;
  }
  VB_Vector exo_real(exoFilt.getLength());
  VB_Vector exo_imag(exoFilt.getLength());
  VB_Vector sig_real(signal.getLength());
  VB_Vector sig_imag(signal.getLength());
  VB_Vector prod_real(signal.getLength());
  VB_Vector prod_imag(signal.getLength());
  exoFilt.fft(exo_real,exo_imag);
  exo_real[0]=1.0;
  exo_imag[0]=0.0;
  signal.fft(sig_real,sig_imag);
  VB_Vector::compMult(sig_real,sig_imag,exo_real,exo_imag,prod_real,prod_imag);
  VB_Vector::complexIFFTReal(prod_real,prod_imag,signal);
  return 0;
}

int
GLMInfo::makeF1()
{
  // see if F1 is already populated
  if (f1Matrix.m) return 0;
  // see if it's on disk
  string f1name=xsetextension(stemname,"F1");
  f1Matrix.ReadMAT1(f1name);
  if (f1Matrix.m) return 0;
  // see if we can load a KG to pinv
  string kgname=xsetextension(stemname,"KG");
  VBMatrix kg;
  kg.ReadMAT1(kgname);
  if (kg.m) {
    f1Matrix.init(kg.n,kg.m);
    pinv(kg,f1Matrix);
    return 0;
  }
  // see if we have or can load a G matrix to pinv
  if (!gMatrix.m) {
    string gname=xsetextension(stemname,"G");
    gMatrix.ReadMAT1(gname);
  }
  if (gMatrix.m) {
    f1Matrix.init(gMatrix.n,gMatrix.m);
    pinv(gMatrix,f1Matrix);
    return 0;
  }
  return 1;  // give up, return error
}

// adjustTS() adjusts for covariates of no interest by first using F1
// with the (possibly spatially averaged) time series to calculate the
// betas.  the time series should already have been filtered.

int
GLMInfo::adjustTS(VB_Vector &signal)
{
  string kgname=xsetextension(stemname,"KG");
  string gname=xsetextension(stemname,"G");
  VBMatrix KG;

  if (makeF1())
    return 190;
  KG.ReadMAT1(kgname);
  if (!KG.m)
    KG.ReadMAT1(gname);
  if (!KG.m)
    return 191;

  // grab betas, kg, and g matrix figure out which covariates are not
  // of interest, scale them, and subtract them from the signal.  note
  // that for non-autocorrelated designs, KG is just the G matrix

  // for convenience
  int ntimepoints=f1Matrix.n;
  int nvars=f1Matrix.m;
  // calculate betas
  VB_Vector betas(nvars);
  for (int i=0; i<nvars; i++) {
    betas[i]=0;
    for (int j=0; j<ntimepoints; j++) {
      betas[i] += f1Matrix(i, j) * signal[j];
    }
  }

  for (int i=0; i<(int)nointerestlist.size(); i++) {
    VB_Vector tmp(ntimepoints);
    for (int j=0; j<ntimepoints; j++)
      tmp.setElement(j,KG(j,nointerestlist[i])*betas[nointerestlist[i]]);
    signal-=tmp;
  }
  return 0;
}

VB_Vector
GLMInfo::getTS(int x,int y,int z,uint32 flags)
{
  return ::getTS(teslist,x,y,z,flags);
//   VB_Vector signal;
//   for (int i=0; i<(int)teslist.size(); i++) {
//     Tes mytes;
//     if (mytes.ReadTimeSeries(teslist[i],x,y,z)) {
//       signal.clear();
//       return signal;
//     }
//     if (flags & MEANSCALE)
//       mytes.timeseries.meanNormalize();
//     if (flags & DETREND)
//       mytes.timeseries.removeDrift();
//     signal.concatenate(mytes.timeseries);
//   }
//   return signal;
}

VBRegion
GLMInfo::restrictRegion(VBRegion &rr)
{
  return ::restrictRegion(teslist,rr);
  VBRegion newreg;
  Tes tlist[teslist.size()];
  // preload tes masks (in headers)
  for (int i=0; i<(int)teslist.size(); i++)
    if (tlist[i].ReadHeader(teslist[i]))
      return newreg;
  // remove missing voxels from region
  uint64 xx,yy,zz;
  for (VI myvox=rr.begin(); myvox!=rr.end(); myvox++) {
    // rr.getxyz(myvox.first,xx,yy,zz);
    int good=1;
    for (int j=0; j<(int)teslist.size(); j++) {
      if (!(tlist[j].GetMaskValue(myvox->second.x,myvox->second.y,myvox->second.z))) {
        good=0;
        break;
      }
    }
    if (!good) continue;
    newreg.add(xx,yy,zz,0.0);
  }
  return newreg;
}

VB_Vector
GLMInfo::getRegionTS(VBRegion &rr,uint32 flags)
{
  return ::getRegionTS(teslist,rr,flags);
}

VBMatrix
GLMInfo::getRegionComponents(VBRegion &rr,uint32 flags)
{
  return ::getRegionComponents(teslist,rr,flags);
}

// residual is R * transpose(filtered_time_series)

VB_Vector
GLMInfo::getResid(int x,int y,int z,uint32 flags)
{
  VBRegion rr;
  rr.add(x,y,z,0.0);
  return getResid(rr,flags);
}

VB_Vector
GLMInfo::getResid(VBRegion &rr,uint32 flags)
{
  VB_Vector resid;
  if (rMatrix.m==0)
    rMatrix.ReadMAT1(xsetextension(stemname,"R"));
//   if (!paramtes.data_valid)
//     paramtes.ReadFile(xsetextension(stemname,"prm"));
  if (exoFilt.size()==0)
    exoFilt.ReadFile(xsetextension(stemname,"ExoFilt"));
  if (rMatrix.m==0 || exoFilt.size()==0)
    return resid;

  // assemble signal vector
  VB_Vector signal=getRegionTS(rr,flags);
  int ntimepoints=signal.getLength();

  // filter signal matrix
  VB_Vector exo_real(exoFilt.getLength());
  VB_Vector exo_imag(exoFilt.getLength());
  VB_Vector sig_real(signal.getLength());
  VB_Vector sig_imag(signal.getLength());
  VB_Vector prod_real(signal.getLength());
  VB_Vector prod_imag(signal.getLength());
  exoFilt.fft(exo_real,exo_imag);
  exo_real[0]=1.0;
  exo_imag[0]=0.0;
  signal.fft(sig_real,sig_imag);
  VB_Vector::compMult(sig_real,sig_imag,exo_real,exo_imag,prod_real,prod_imag);
  VB_Vector::complexIFFTReal(prod_real,prod_imag,signal);
  
  // premult KX (filtered data) by residual forming matrix R
  resid.resize(ntimepoints);
  gsl_blas_dgemv(CblasNoTrans,1.0,&(rMatrix.mview.matrix),
                 signal.getTheVector(),0.0,resid.getTheVector());
  return resid;
  
}

// FIXME regionstat not used anywhere?

double
GLMInfo::regionstat(VBRegion &rr)
{
  VB_Vector vv;
  vv=getTS(0,0,0,glmflags);  // just to init vv
  // load time series
  for (VI myvox=rr.begin(); myvox!=rr.end(); myvox++) {
    vv+=getTS(myvox->second.x,myvox->second.y,myvox->second.z,glmflags);
  }
  vv /= rr.size();
  // NOTE: filtering gets done in the regression routine
  // do the regression
  int err=Regress(vv);
  if (err) {
    printf("[E] vbdumpstats: error %d regressing time series\n",err);
    exit(15);
  }
  calc_stat();
  return statval;
}

GLMInfo::GLMInfo()
{
  init();
}

void
GLMInfo::setup(string name)
{
  init();
  findstem(name);
  findanatomy();
  findtesfiles();
  getcovariatenames();
  loadcontrasts();
  loadtrialsets();
  getglmflags();
}

void
GLMInfo::init()
{
  stemname="";
  anatomyname="";
  teslist.clear();
  cnames.clear();
  contrasts.clear();
  trialsets.clear();
  nvars=0;
  dependentindex=0;
  interceptindex=0;
  glmflags=0;
  gMatrix.clear();
  f1Matrix.clear();
  rMatrix.clear();
  f3Matrix.clear();
  exoFilt.clear();
  residuals.clear();
  betas.clear();
  traceRV.clear();
  pseudoT.clear();
  keeperlist.clear();
  interestlist.clear();
  paramtes.init();
  statcube.init();
  rawcube.init();
  effdf=nan("nan");
}

void
GLMInfo::findstem(string name)
{
  struct stat st;
  // name exists
  if (!(stat(name.c_str(),&st))) {
    if (S_ISDIR(st.st_mode)) {
      vglob vg(name+"/*.glm");
      if (vg.size()) stemname=vg.names[0];
      stemname=xsetextension(stemname,"");
      return;
    }
    else {
      stemname=xdirname(name)+"/"+xsetextension(xfilename(name),"");
      return;
    }
  }
  stemname=name;
  return;
}

void
GLMInfo::findanatomy()
{
  Tes prmfile;
  Cube anat;
 
  anatomyname="";
  string prmname=stemname+".prm";
  if (prmfile.ReadHeader(prmname))
    return;
  string aname=xdirname(stemname)+"/Display.cub";
  if (!(anat.ReadHeader(aname))) {
    anatomyname=aname;
    return;
  }
  aname=xdirname(stemname)+".cub";
  if (!(anat.ReadHeader(aname))) {
    anatomyname=aname;
    return;
  }
  int maxscore=0;
  string apat=xdirname(xdirname(stemname))+"/Anatomy/*.cub";
  vglob vg(apat);
  // glob for adir/*.cub, make a list of the dimensions, prune it down to 1 if possible
  for (size_t i=0; i<vg.size(); i++) {
    if (anat.ReadHeader(vg.names[i]))
      continue;
    if (anat.dimz != prmfile.dimz)
      continue;
    if ((anat.dimx % prmfile.dimx) || (anat.dimy % prmfile.dimy))
      continue;
    int score=0;
    string nn=vg.names[i];
    if (nn.find("Display") != string::npos)
      score+=10;
    if ((anat.dimx / prmfile.dimx == 4) && (anat.dimy/prmfile.dimy==4))
      score+=200;
    if ((anat.dimx / prmfile.dimx == 3) && (anat.dimy/prmfile.dimy==3))
      score+=100;
    if (score > maxscore) {
      anatomyname=nn;
      maxscore=score;
    }
  }
  return;
}

void
GLMInfo::print()
{
  printf("          stem: %s\n",stemname.c_str());
  printf("       anatomy: %s\n",anatomyname.c_str());
  printf("     tes files: %d\n",(int)teslist.size());
  printf("     dependent: %d\n",dependentindex);
  printf("  n indep vars: %d\n",nvars);
  printf("   vars of int: %d\n",(int)interestlist.size());
  printf("    covariates: ");
  if (cnames.size())
    printf("%c %s\n",cnames[0][0],cnames[0].c_str()+1);
  else
    printf("<none>\n");
  for (int i=1; i<(int)cnames.size(); i++)
    printf("                %c %s\n",cnames[i][0],cnames[i].c_str()+1);
  printf("     contrasts: ");
  if (contrasts.size())
    printf("%s (%s)\n",contrasts[0].name.c_str(),contrasts[0].scale.c_str());
  else
    printf("<none>\n");
  for (int i=1; i<(int)contrasts.size(); i++)
    printf("                %s (%s)\n",contrasts[i].name.c_str(),contrasts[i].scale.c_str());
}

string
GLMInfo::statmapExists(string matrixStemName, VB_Vector& contrasts, string scale) {
  string clist, plist;
  char temp[STRINGLEN];
  for (int i = 0; i < contrasts.size(); i++) {
      sprintf(temp, "%.0f", contrasts[i]);
      clist+=temp;
      clist+=(" ");
  }
  Tes prm(matrixStemName+"/"+matrixStemName+".prm");
  string prmtimestamp = prm.GetHeader("TimeStamp:");
  Cube cub;
  string list=matrixStemName+"/map_*.cub";
  vglob vg(list);
  for (size_t i=0; i<vg.size(); i++) {
    cub.ReadFile(vg[i]);
    if (cub.GetHeader("contrast_scale:")==(scale) &&
        cub.GetHeader("contrast_vector:")==(clist) &&
        cub.GetHeader("TimeStamp:")==(prmtimestamp)){
          return string(vg[i]);
    }
  }
  return "";
}

void
GLMInfo::findtesfiles()
{
  ifstream subfile;
  tokenlist line;
  subfile.open((stemname+".sub").c_str());
  if (!subfile)
    return;
  char subline[STRINGLEN];
  while (subfile.getline(subline,STRINGLEN,'\n')) {
    line.ParseLine(subline);
    if (line.size()==0)
      continue;
    if (line[0][0]==';' || line[0][0]=='#')
      continue;
    if (line[0]=="VB98" || line[0]=="TXT1")
      continue;
    teslist.push_back(line[0]);
  }
  subfile.close();
}

void
GLMInfo::loadcombinedmask()
{
  if (mask.data)
    return;
  mask.init();
  tesgroup.resize(teslist.size());
  for (int i=0; i<(int)teslist.size(); i++) {
    if (tesgroup[i].ReadHeader(teslist[i])) {
      mask.init();
      return;
    }
    Cube tesmask;
    tesgroup[0].ExtractMask(tesmask);
    if (!(mask.data)) {
      mask=tesmask;
    }
    else
      mask.intersect(tesmask);
  }
}

void
GLMInfo::getcovariatenames()
{
  dependentindex=-1;
  interceptindex=-1;
  string gname=stemname+".G";
  int index;
  VBMatrix m(gname);
  tokenlist line;
  line.SetSeparator("\t");
  string tag,type,name;
  keeperlist.clear();
  interestlist.clear();
  nointerestlist.clear();
  nvars=0;
  for (int i=0; i<(int)m.header.size(); i++) {
    line.ParseLine(m.header[i]);
    tag=line[0];
    index=strtol(line[1]);
    type=line[2];
    tag=vb_tolower(tag);
    type=vb_tolower(type);
    name=vb_tolower(line[3]);  // FIXME only gets the first token in the name
    if (tag!="parameter:")
      continue;
    nvars++;
    if (type=="interest")
      cnames.push_back((string)"I"+line[3]);
    else if (type=="nointerest")
      cnames.push_back((string)"N"+line[3]);
    else if (type=="keepnointerest")
      cnames.push_back((string)"K"+line[3]);
    else if (type=="dependent")
      cnames.push_back((string)"D"+line[3]);
    else
      cnames.push_back((string)"U"+line[3]);
    if (type=="interest" || type=="keepnointerest")
      keeperlist.push_back(index);
    if (type=="interest")
      interestlist.push_back(index);
    if (type=="keepnointerest" || type=="nointerest")
      nointerestlist.push_back(index);
    if (type=="dependent")
      dependentindex=index;
    if (name=="intercept") {
      interceptindex=index;
    }
  }
}

void
GLMInfo::loadtrialsets()
{
  string fname=xdirname(stemname)+"/averages.txt";
  trialsets=parseTAFile(fname);
}

void
GLMInfo::loadcontrasts()
{
  contrasts.clear();
  ifstream contfile;
  tokenlist line,args;
  VBMatrix gmatrix;

  gmatrix.ReadMAT1Header(stemname+".G");
  if (nvars==0) {
    for (int i=0; i<(int)gmatrix.header.size(); i++) {
      args.ParseLine(gmatrix.header[i]);
      if (args[0]=="Parameter:")
        nvars++;
    }
  }
  if (nvars<1)
    return;

  vector<string> cfnames;
  cfnames.push_back(xdirname(stemname)+"/contrasts.txt");
  cfnames.push_back(xdirname(stemname)+"/contrast.txt");
  cfnames.push_back(stemname+".contrasts");
  cfnames.push_back(stemname+".contrast");
  
  for (int i=0; i<(int)cfnames.size(); i++) {
    contfile.open(cfnames[i].c_str());
    if (contfile) {
      char cline[STRINGLEN];
      while (contfile.getline(cline,STRINGLEN,'\n')) {
        line.ParseLine(cline);
        if (line.size()==0)
          continue; 
        if (line[0][0]==';' || line[0][0]=='#')
          continue;
        if (line[0]=="VB98" || line[0]=="TXT1")
          continue;
        if (line.size()<3)
          continue;
        VBContrast ss;
        if (ss.parsemacro(line,nvars,interestlist)==0)
          contrasts.push_back(ss);
      }
      contfile.close();
    }
  }
  if (contrasts.size()==0 && nvars>0) {
    VBContrast ss;
    tokenlist tmpt;
    tmpt.ParseLine("all t allspikes");
    ss.parsemacro(tmpt,nvars,interestlist);
    contrasts.push_back(ss);
    tmpt.ParseLine("first t spikes 0");
    ss.parsemacro(tmpt,nvars,interestlist);
    contrasts.push_back(ss);
  }
}

void
GLMInfo::getglmflags()
{
  glmflags=0;
  Tes tmp;
  tokenlist args;
  if (tmp.ReadHeader(stemname+".prm"))
    return;
  for (int i=0; i<(int)tmp.header.size(); i++) {
    args.ParseLine(tmp.header[i]);
    string tag=vb_tolower(xstripwhitespace(args[0]," \t\n:"));
    if (tag=="options" || tag=="option") {
      for (int j=1; j<(int)args.size(); j++) {
        if (vb_tolower(args[j])=="detrend")
          glmflags|=DETREND;
        else if (vb_tolower(args[j])=="meanscale")
          glmflags|=MEANSCALE;
      }
    }
    else if (tag=="datascale") {
      if (vb_tolower(args[1])=="mean")
        glmflags|=MEANSCALE;
    }
  }
}

int
GLMInfo::parsecontrast(const string &str)
{
  // first try to look it up
  for (int i=0; i<(int)contrasts.size(); i++) {
    if (vb_tolower(contrasts[i].name)==vb_tolower(str)) {
      contrast=contrasts[i];
      return 0;  // no error!
    }
  }
  // try to parse it as a contrast macro
  tokenlist line;
  line.ParseLine(str);
  if (!(contrast.parsemacro(line,nvars,interestlist)))
    return 0;  // no error!
  // try to parse it as a series of numbers
  contrast.name="mycontrast";
  contrast.scale="t";
  contrast.contrast.resize(nvars);
  for (int i=0; i<nvars; i++) contrast.contrast[i]=0.0;
  if (line.size()<1)
    return 101;
  if (validscale(line[0])) {
    contrast.scale=line[0];
    line.DeleteFirst();
  }
  if (line.size()!=(int)interestlist.size())
    return 102;
  for (int i=0; i<line.size(); i++) {
    if (!isdigit(line[i][0]) && !(strchr("-.",line[i][0])))
      return 102;
    contrast.contrast[interestlist[i]]=strtod(line[i]);
  }
  return 0;
}

int
VBContrast::parsemacro(tokenlist &line,int nvars,vector<int> &interestlist)
{
  if (nvars<1)
    return 102;
  name=line[0];
  scale=line[1];
  contrast.resize(nvars);
  if (line[2]=="allspikes") {
    contrast+=1.0;
  }
  else if (line[2]=="spikes" || line[2]=="spike") {
    vector<int> myspikes=numberlist(line[3]);
    for (int i=0; i<(int)myspikes.size(); i++) {
      if (myspikes[i]>(int)interestlist.size()-1)
        return 109;
      contrast[interestlist[myspikes[i]]]=1.0;
    }
  }
  else if (line[2]=="vec" && line.size()-3==(int)interestlist.size()) {
    if (line.size()-3!=(int)interestlist.size())
      return 105;
    for (int i=3; i<line.size(); i++)
      contrast[interestlist[i-3]]=strtod(line[i]);
  }
  else if (line[2]=="contrast") {
    // FIXME used at all?  validated?
    vector<int> myspikes=numberlist(line[3]);
    for (int i=0; i<(int)myspikes.size(); i++)
      contrast[interestlist[myspikes[i]]]=1;
    if (line[4]=="minus")
      myspikes=numberlist(line[5]);
    else
      myspikes=numberlist(line[4]);
    for (int i=0; i<(int)myspikes.size(); i++)
      contrast[interestlist[myspikes[i]]]=-1;
  }
  else
    return 101;

  return 0;  // no error!
}

void
VBContrast::print()
{
  printf("[I] VBContrast %s (%s):",name.c_str(),scale.c_str());
  for (int i=0; i<contrast.size(); i++)
    printf(" %.1f",contrast[i]);
  printf("\n");
}

int
validscale(string scale) {
  scale=xstripwhitespace(vb_tolower(scale));
  if (scale=="t"||scale=="f"|| scale=="tp"||scale=="fp"||scale=="tz"||scale=="fz")
    return 1;
  if (scale=="beta"||scale=="rawbeta"||scale=="rb"||scale=="b")
    return 1;
  if (scale=="intercept"||scale=="int"||scale=="i"||scale=="pct"||scale=="percent")
    return 1;
  if (scale=="tp"||scale=="fp"||scale=="tz"||scale=="fz")
    return 1;
  if (scale=="tp/1"||scale=="tp/2"||scale=="tp1"||scale=="tp2")
    return 1;
  if (scale=="tz/1"||scale=="tz/2"||scale=="tz1"||scale=="tz2")
    return 1;
  if (scale=="error"||scale=="err"||scale=="e")
    return 1;
  return 0;
}

// permutes the passed vector

void
GLMInfo::permute_if_needed(VB_Vector &vec)
{
  if (perm_signs.size()==vec.size()) {
    for (int i=0; i<vec.size(); i++)
      vec[i]*=perm_signs[i];
  }
  if (perm_order.size()==vec.size()) {
    VB_Vector tmp(vec.size());
    for (int i=0; i<vec.size(); i++)
      tmp[i]=vec[(int)perm_order[i]];
    vec=tmp;
  }
}
