/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <NavigationWindow.h>

#include <Strings.h>
#include <Setup.h>
#include <MainWindow.h>
#include <GraphDefinition.h>
#include <ModifyGridWindow.h>
#include <ModifySmithWindow.h>
#include <ModifySweepWindow.h>
#include <SetDimensionValueWindow.h>
#include <ModifyOutputParameterWindow.h>

#include "images/tray.xpm"
#include "images/circuit.xpm"
#include "images/graph.xpm"
#include "images/smith.xpm"
#include "images/table.xpm"
#include "images/outputs.xpm"
#include "images/generator.xpm"
#include "images/variables.xpm"
#include "images/dimensions.xpm"
#include "images/selected.xpm"
#include "images/unselected.xpm"
#include "images/substrates.xpm"
#include "images/datafiles.xpm"

#include <qlistview.h>
#include <qfileinfo.h>
#include <qpixmap.h>
#include <qpopupmenu.h>

#include <iostream.h>

NavigationWindow* NavigationWindow::instance_ = 0;

//-----------------------------------------------------------------
NavigationWindow::NavigationWindow( QWidget* parent ) :
  QListView(parent, Strings::NavigationWindowName),
  root_(0),
  schematics_(0),
  graphs_(0),
  variables_(0),
  sweepWindow_(0),
  editWindow_(0),
  gridWindow_(0),
  smithWindow_(0),
  menu_(0),
  newMenuID_(-1),
  changeMenuID_(-1),
  renameMenuID_(-1),
  deleteMenuID_(-1)
{
  ASSERT( instance_ == 0 );
  instance_ = this;

  QString name = Strings::translate(Strings::LabelNavigation);
  addColumn( name );
  addColumn( "" );
  setTreeStepSize( 15 );
  setSorting(-1, TRUE);

  createNavigationTree();
  createPopupMenu();

}

//-----------------------------------------------------------------
void NavigationWindow::createNavigationTree()
{
  root_ = new ListViewItem( ListViewItem::rootItem, this,
			    MainWindow::instance()->getFilename() );
  root_->setPixmap( 0, QPixmap( tray ) );
  root_->setSelectable( FALSE );

  QString dimensionsName = Strings::translate( Strings::TreeItemDimensions);
  dimensions_ = new ListViewItem( ListViewItem::unitItem, root_,
				  root_, dimensionsName );
  dimensions_->setPixmap( 0, QPixmap( dimensions_xpm ) ); 
  dimensions_->setSelectable( FALSE );
  createDimensionList();

  QString variablesName = Strings::translate( Strings::TreeItemVariables);
  variables_ = new ListViewItem( ListViewItem::variableItem, root_,
				  dimensions_, variablesName );
  variables_->setPixmap( 0, QPixmap( variables_xpm ) ); 
  variables_->setSelectable( FALSE );

  QString schematicsName = Strings::translate( Strings::TreeItemSchematics);
  schematics_ = new ListViewItem( ListViewItem::schematicItem, root_, 
				  variables_, schematicsName );
  schematics_->setPixmap( 0, QPixmap( circuit_xpm ) );
  schematics_->setSelectable( FALSE );

  QString sweepName = Strings::translate( Strings::TreeItemSweep);
  sweep_ = new ListViewItem( ListViewItem::frequencyItem, root_,
			     schematics_, sweepName );
  sweep_->setPixmap( 0, QPixmap( generator_xpm ) ); 
  sweep_->setSelectable( FALSE );
  createSweepList();
  updateSweep();

  QString graphsName = Strings::translate( Strings::TreeItemGraphs);
  graphs_ = new ListViewItem( ListViewItem::graphItem, root_,
			      sweep_, graphsName );
  graphs_->setPixmap( 0, QPixmap( outputs_xpm ) ); 
  graphs_->setSelectable( FALSE );

  QString substrateName = Strings::translate( Strings::TreeItemSubstrates);
  substrates_ = new ListViewItem( ListViewItem::substrateItem, root_,
				  graphs_, substrateName );
  substrates_->setPixmap( 0, QPixmap( substrates_xpm ) ); 
  substrates_->setSelectable( FALSE );
  
  QString dataFilesName = Strings::translate( Strings::TreeItemDataFiles);
  dataFiles_ = new ListViewItem( ListViewItem::dataFileItem, root_,
				 substrates_, dataFilesName );
  dataFiles_->setPixmap( 0, QPixmap( datafiles_xpm ) ); 
  dataFiles_->setSelectable( FALSE );

  root_->setOpen( TRUE );

  connect( this, SIGNAL( doubleClicked( QListViewItem * ) ),
	   this, SLOT( itemSelectedSlot( QListViewItem* ) ) );

}

//-----------------------------------------------------------------
void NavigationWindow::createPopupMenu()
{
  menu_ = new QPopupMenu( this );
  QString newMenuName = Strings::translate(Strings::MenuLabelNewItem);
  QString changeMenuName = Strings::translate(Strings::MenuLabelChangeItem);
  QString renameMenuName = Strings::translate(Strings::MenuLabelRenameItem);
  QString deleteMenuName = Strings::translate(Strings::MenuLabelDeleteItem);
  newMenuID_ = menu_->insertItem( newMenuName, this,
				  SLOT( newItemSlot() ) );
  changeMenuID_ = menu_->insertItem( changeMenuName, this,
				     SLOT( changeItemSlot() ) );
  renameMenuID_ = menu_->insertItem( renameMenuName, this,
				     SLOT( renameItemSlot() ) );
  deleteMenuID_ = menu_->insertItem( deleteMenuName, this,
				     SLOT( deleteItemSlot() ) );
  menu_->setItemEnabled( newMenuID_, FALSE );
  menu_->setItemEnabled( changeMenuID_, FALSE );
  menu_->setItemEnabled( renameMenuID_, FALSE );
  menu_->setItemEnabled( deleteMenuID_, FALSE );

  connect( this, SIGNAL( rightButtonPressed( QListViewItem*, const QPoint&, int ) ),
	   this, SLOT( popupMenuSlot( QListViewItem*, const QPoint&, int  ) ) );

}

//-----------------------------------------------------------------
NavigationWindow::~NavigationWindow()
{
  instance_ = 0;
}

//-----------------------------------------------------------------
NavigationWindow* NavigationWindow::instance()
{
  return instance_;
}

//-----------------------------------------------------------------
void NavigationWindow::createDimensionList()
{
  emptyBranch( dimensions_ );
  DimensionMap& map =  Setup::instance()->getDimensionMap();
  DimensionMap::Iterator it;
  for( it = map.begin(); it != map.end(); ++it )
    {
      QString entry = Strings::translate( it.key());
      ListViewItem* item = new ListViewItem(  ListViewItem::unitChildItem,
					      dimensions_, entry );
      item->setSelectable( FALSE );
      const QString& value = it.data().getActiveValueName();
      item->setText( 1, value );
    }
  clearSelection();
  setCurrentItem( 0 );
}

//-----------------------------------------------------------------
void NavigationWindow::updateDimension( const QString& name, const QString& value )
{
  QString trName = Strings::translate( name);
  QListViewItem* item = findListItemInBranch( dimensions_, trName );
  ASSERT( item != 0 );
  item->setText( 1, value );
}

//-----------------------------------------------------------------
void NavigationWindow::createSweepList()
{
  QString entry = Strings::translate( Strings::SweepType);
  sweepType_ = new ListViewItem(  ListViewItem::frequencyChildItem,
				  sweep_, entry );
  sweepType_->setSelectable( FALSE );
  entry = Strings::translate( Strings::SweepStartFrequency);
  sweepStart_ = new ListViewItem( ListViewItem::frequencyChildItem,
				  sweep_, sweepType_, entry );
  sweepStart_->setSelectable( FALSE );
  entry = Strings::translate( Strings::SweepStopFrequency);
  sweepStop_ = new ListViewItem( ListViewItem::frequencyChildItem,
				 sweep_, sweepStart_, entry );
  sweepStop_->setSelectable( FALSE );
  entry = Strings::translate( Strings::SweepNumberOfPoints);
  sweepPoints_ = new ListViewItem( ListViewItem::frequencyChildItem,
				   sweep_, sweepStop_, entry );
  sweepPoints_->setSelectable( FALSE );
}

//-----------------------------------------------------------------
void NavigationWindow::updateSweep()
{
  QString tmp;
  Setup& setup = *Setup::instance();
  if ( setup.isLinearSweep() )
    {
      tmp = Strings::translate( Strings::SweepLinearType );
    }
  else
    {
      tmp = Strings::translate( Strings::SweepLogType );
    }
  sweepType_->setText( 1, tmp );
  tmp.setNum( setup.getStartFrequency() );
  sweepStart_->setText( 1, tmp );
  tmp.setNum( setup.getStopFrequency() );
  sweepStop_->setText( 1, tmp );
  tmp.setNum( setup.getNumberOfFrequencyPoints() );
  sweepPoints_->setText( 1, tmp );
}

//-----------------------------------------------------------------
void NavigationWindow::resizeEvent( QResizeEvent* e ) 
{
  QListView::resizeEvent( e );
}

//-----------------------------------------------------------------
void NavigationWindow::filenameChanged( const QString& filename )
{
  QFileInfo fileInfo( filename );
  root_->setText( 0, fileInfo.fileName() );
}

//-----------------------------------------------------------------
void NavigationWindow::addSchematic( const QString& name )
{
  ListViewItem* item = new ListViewItem( ListViewItem::schematicChildItem,
					 schematics_, name );
  itemSelectedSlot( item );
  item->setSelectable( FALSE );
  root_->setOpen( TRUE );
  schematics_->setOpen( TRUE );
  ensureItemVisible(item);
  triggerUpdate();
}

//-----------------------------------------------------------------
void NavigationWindow::itemSelectedSlot( QListViewItem* selectedItem )
{
  
  QPixmap selected( selected_xpm );
  QPixmap unselected( unselected_xpm );

  ListViewItem* item = dynamic_cast <ListViewItem*>( selectedItem );
  ASSERT( item != 0 );

  switch ( item->type() )
    {
    case ListViewItem::schematicChildItem:
      {
	QListViewItem* it = schematics_->firstChild();
	while ( it )
	  {
	    if ( it == item )
	      {
		it->setPixmap( 0, selected );
		MainWindow::instance()->setActiveSchematic( it->text(0) );
	      }
	    else
	      {
		it->setPixmap( 0, unselected );
	      }
	    it = it->nextSibling();
	  }
      }
      break;

    case ListViewItem::unitChildItem:
      {
	if ( editWindow_.isNull() )
	  {
	    QString title = Strings::translate( Strings::ChangeDimensionWindowTitle );
	    editWindow_ = new SetDimensionValueWindow( MainWindow::instance(), title );
	  }
	QString name = item->text( 0 );
	editWindow_->initialize( name );
	editWindow_->show();
      }
      break;

    case ListViewItem::variableChildItem:
      {
	QString name = item->text( 0 );
	MainWindow::instance()->changeVariable( name );
      }
      break;
   
    case ListViewItem::frequencyChildItem:
      {
	if ( sweepWindow_.isNull() )
	  {
	    QString title = Strings::translate( Strings::ModifySweepWindowTitle );
	    sweepWindow_ = new ModifySweepWindow( MainWindow::instance(), title );
	  }
	sweepWindow_->initialize();
	sweepWindow_->show();
      }
      break;
 
    case ListViewItem::graphChildItem:
      {
	QString name = item->text(0);
	modifyGraph( name );
      }
      break;
   
    case ListViewItem::graphResultItem:
      {
	long int id = item->id();
	ResultDefinition* definition = ResultDefinition::findResultDefinitionById( id );
	QListViewItem* parent =  selectedItem->parent();
	ASSERT( parent != 0 );
	showModifyOutputParameterWindow( parent->text(0), definition );
      }
      break;

    case ListViewItem::substrateChildItem:
      {
	QString name = item->text(0);
	MainWindow::instance()->changeSubstrate( name );
      }
      break;  
      
    default:
      //Do nothing
      break;
    }
} 

//-----------------------------------------------------------------
void NavigationWindow::reset()
{
  emptyBranch( schematics_ );
  emptyBranch( variables_ );
  emptyBranch( graphs_ );
  emptyBranch( substrates_ );
  emptyBranch( dataFiles_ );
}

//-----------------------------------------------------------------
void NavigationWindow::emptyBranch( ListViewItem* parent )
{
  ListViewItem* it = dynamic_cast<ListViewItem*> ( parent->firstChild() );
  while ( it )
    {
      delete it;
      it = dynamic_cast<ListViewItem*> ( parent->firstChild() );
    }
}

//-----------------------------------------------------------------
void NavigationWindow::popupMenuSlot( QListViewItem* item,
				      const QPoint& point, int )
{
  if ( item )
    {
      QListViewItem* parent = item->parent();
      if ( (parent == variables_) || (item == variables_) )
	{ 
	  bool newVariable = ( item == variables_ );
	  menu_->setItemEnabled( newMenuID_, newVariable );
	  menu_->setItemEnabled( changeMenuID_, !newVariable );
	  menu_->setItemEnabled( renameMenuID_, !newVariable );
	  menu_->setItemEnabled( deleteMenuID_, !newVariable );
	  menu_->popup( point );
	}
      if ( (parent == schematics_) || (item == schematics_) )
	{ 
	  bool newSchematic = ( item == schematics_ );
	  menu_->setItemEnabled( newMenuID_, newSchematic );
	  menu_->setItemEnabled( changeMenuID_, !newSchematic );
	  menu_->setItemEnabled( renameMenuID_, !newSchematic );
	  menu_->setItemEnabled( deleteMenuID_, !newSchematic );
	  menu_->popup( point );
	}
      if ( (parent == graphs_) || (item == graphs_) )
	{ 
	  bool newGraph = ( item == graphs_ );
	  menu_->setItemEnabled( newMenuID_, TRUE );
	  menu_->setItemEnabled( changeMenuID_, !newGraph );
	  menu_->setItemEnabled( renameMenuID_, !newGraph );
	  menu_->setItemEnabled( deleteMenuID_, !newGraph );
	  menu_->popup( point );
	}

      ListViewItem* listItem = dynamic_cast<ListViewItem*>( item );
      ASSERT( listItem != 0 );
      ListViewItem::ItemType itemType = listItem->type();
      switch ( itemType )
	{
	case ListViewItem::graphResultItem :
	  {
	    menu_->setItemEnabled( newMenuID_, FALSE );
	    menu_->setItemEnabled( changeMenuID_, TRUE );
	    menu_->setItemEnabled( renameMenuID_, FALSE );
	    menu_->setItemEnabled( deleteMenuID_, TRUE );
	    menu_->popup( point );
	  }
	break;

	case ListViewItem::substrateItem:
	case ListViewItem::substrateChildItem:
	  {
	    bool childItem = (itemType == ListViewItem::substrateChildItem );
	    menu_->setItemEnabled( newMenuID_, !childItem );
	    menu_->setItemEnabled( changeMenuID_, childItem );
	    menu_->setItemEnabled( renameMenuID_, childItem );
	    menu_->setItemEnabled( deleteMenuID_, childItem );
	    menu_->popup( point );
	  }
	  break;

	case ListViewItem::dataFileItem:
	case ListViewItem::dataFileChildItem:
	  {
	    bool childItem = (itemType == ListViewItem::dataFileChildItem);
	    menu_->setItemEnabled( newMenuID_, !childItem );
	    menu_->setItemEnabled( changeMenuID_, FALSE );
	    menu_->setItemEnabled( renameMenuID_, FALSE );
	    menu_->setItemEnabled( deleteMenuID_, childItem );
	    menu_->popup( point );
	  }
	  
	default:
	  {
	    //Do nothing
	  }
	}

    }
}

//-----------------------------------------------------------------
void NavigationWindow::newItemSlot()
{
  ListViewItem* item = dynamic_cast<ListViewItem*>( currentItem() );
  ASSERT( item != 0 );
  MainWindow* mainWindow = MainWindow::instance();
  switch ( item->type() )
    {
    case ListViewItem::variableItem:
      mainWindow->newVariableSlot();
      break;
    case ListViewItem::schematicItem:
      mainWindow->newSchematicSlot();
      break;
    case ListViewItem::graphItem:
      mainWindow->newGraphSlot();
      break;
    case ListViewItem::graphChildItem:
      showModifyOutputParameterWindow( item->text(0) );
      break;
    case ListViewItem::substrateItem:
      mainWindow->newSubstrateSlot();
      break;
    case ListViewItem::dataFileItem:
      mainWindow->newDataFileSlot();
      break;
    default:
      cout << "Unhandled switch!" << endl;
    }
}

//-----------------------------------------------------------------
void NavigationWindow::changeItemSlot()
{
  ListViewItem* item = dynamic_cast<ListViewItem*>( currentItem() );
  ASSERT( item != 0 );

  QString name = item->text( 0 );
 
  switch ( item->type() )
    {
    case ListViewItem::variableChildItem:
      MainWindow::instance()->changeVariable( name );
      break;

    case ListViewItem::schematicChildItem:
      MainWindow::instance()->changeSchematicSize( name );
      break;
      
    case ListViewItem::substrateChildItem:
      MainWindow::instance()->changeSubstrate( name );
      break;
   
    case ListViewItem::graphChildItem:
      modifyGraph( name );
      break;
      
    case ListViewItem::graphResultItem:
      {
	long int id = item->id();
	QListViewItem* parent = item->parent();
	ResultDefinition* definition = ResultDefinition::findResultDefinitionById( id );
	showModifyOutputParameterWindow( parent->text(0), definition );
      }
      break;

    default:
      break;
    }
}

//-----------------------------------------------------------------
void NavigationWindow::renameItemSlot()
{
  ListViewItem* item = dynamic_cast<ListViewItem*> ( currentItem() );
  if ( item )
    {
      QString name = item->text( 0 );
      switch ( item->type() )
	{
	case ListViewItem::variableChildItem:
	  MainWindow::instance()->renameVariable( name );
	  break;

	case ListViewItem::schematicChildItem:
	  MainWindow::instance()->renameSchematic( name );
	  break;

	case ListViewItem::substrateChildItem:
	  MainWindow::instance()->renameSubstrate( name );
	  break;

	case ListViewItem::graphChildItem:
	  MainWindow::instance()->renameGraph( name );
	  break;

	default:
	  //DO NOTHING
	  break;
	}
    }
}

//-----------------------------------------------------------------
void NavigationWindow::deleteItemSlot()
{
  ListViewItem* item = dynamic_cast<ListViewItem*>( currentItem() );
  ASSERT( item != 0 );

  QString name = item->text( 0 );
  switch ( item->type() )
    {
    case ListViewItem::variableChildItem: 
      MainWindow::instance()->deleteVariable( name );
      break;
      
    case ListViewItem::schematicChildItem:
      MainWindow::instance()->deleteSchematic( name );
      break;
    
    case ListViewItem::substrateChildItem:
      MainWindow::instance()->deleteSubstrate( name );
      break;

    case ListViewItem::graphChildItem:
      MainWindow::instance()->deleteGraph( name );
      break;
    
    case ListViewItem::graphResultItem:
      {
	long int id = item->id();
	ResultDefinition* definition = ResultDefinition::findResultDefinitionById( id );
	QListViewItem* parent = item->parent();
	QString parentName = parent->text( 0 );
	GraphDefinition* graph = MainWindow::instance()->findGraph( parentName );
	graph->removeOutputDefinition( definition );
	delete item;
      }      
      break;

    case ListViewItem::dataFileChildItem:
      MainWindow::instance()->deleteDataFile( name );
      break;
      
    default:
      break;
    }
}

//-----------------------------------------------------------------
void NavigationWindow::addVariable( const QString& name, const QString& value )
{
  ListViewItem* item = new ListViewItem( ListViewItem::variableChildItem,
					 variables_, name );
  item->setSelectable( FALSE );
  item->setText( 1, value );
  variables_->sortChildItems( 0, TRUE );
}

//-----------------------------------------------------------------
void NavigationWindow::updateVariable( const QString& name, const QString& value )
{
  QListViewItem* item = findListItemInBranch( variables_ , name );
  if ( item )
    {
      item->setText( 1, value );
    }
}

//-----------------------------------------------------------------
void NavigationWindow::removeVariable( const QString& name )
{
  removeListItemFromBranch( variables_, name );
}

//-----------------------------------------------------------------
void NavigationWindow::updateSubstrate( const QString& name,
					SubstrateDefinition::SubstrateType type )
{
  ListViewItem* item = findListItemInBranch( substrates_ , name );
  if ( item == 0 )
    {
      item = new ListViewItem( ListViewItem::substrateChildItem,
			       substrates_, name );
      item->setSelectable( FALSE );
    }
  QString typeStr(0);
  switch ( type )
    {
    case SubstrateDefinition::microstrip:
      typeStr = Strings::translate( Strings::LabelMicroStripType );
      break;
    case SubstrateDefinition::stripline:
      typeStr = Strings::translate( Strings::LabelStripLineType );
      break;
    default:
      ASSERT( "Unknown substrate type!" == 0 );
    }
  item->setText( 1, typeStr );
  substrates_->sortChildItems( 0, TRUE );
}

//-----------------------------------------------------------------
void NavigationWindow::removeSubstrate( const QString& name )
{
  removeListItemFromBranch( substrates_, name );
}

//-----------------------------------------------------------------
void NavigationWindow::removeSchematic( const QString& name )
{
  removeListItemFromBranch( schematics_, name );
}

//-----------------------------------------------------------------
void NavigationWindow::removeListItemFromBranch( ListViewItem* branch,
						 const QString& name )
{
  ListViewItem* it = findListItemInBranch( branch, name );
  if ( it )
    {
      delete it;
    }
}

//-----------------------------------------------------------------
QListViewItem* NavigationWindow::addListItemToBranch( ListViewItem::ItemType type,
						      QListViewItem* branch,
						      const QString& name )
{
  ListViewItem* item =  new ListViewItem( type, branch, name );
  item->setSelectable( FALSE );
  root_->setOpen( TRUE );
  branch->setOpen( TRUE );
  ensureItemVisible(item);
  branch->sort();
  triggerUpdate();
  return item;
}

//-----------------------------------------------------------------
ListViewItem* NavigationWindow::findListItemInBranch( ListViewItem* branch,
						      const QString& name)
{
  ListViewItem* it = dynamic_cast<ListViewItem*> ( branch->firstChild() );
  while ( it )
    {
      if ( it->text( 0 ) == name )
	{
	  break;
	}
      it = dynamic_cast<ListViewItem*> ( it->nextSibling() );
    }
  return it;
}

//-----------------------------------------------------------------
void NavigationWindow::addGraph( const GraphDefinition& graph )
{
  QListViewItem* item = addListItemToBranch( ListViewItem::graphChildItem,
					     graphs_, graph.getName() );
  QPixmap pixmap;
  switch ( graph.graphType() )
    {
    case GraphDefinition::gridType:
      pixmap = QPixmap( graph_xpm );
      break;
    case GraphDefinition::smithType:
      pixmap = QPixmap( smith_xpm );
      break;
    case GraphDefinition::tableType:
      pixmap = QPixmap( table_xpm );
      break; 
    default:
      ASSERT( "Not implemented yet!" == 0 );
    }
  item->setPixmap( 0, pixmap );
}

//-----------------------------------------------------------------
void NavigationWindow::modifyGraph( const QString& name )
{
  GraphDefinition* graph = MainWindow::instance()->findGraph( name );
  switch ( graph->graphType() )
    {
    case GraphDefinition::gridType:
      if ( gridWindow_.isNull() )
	{
	  QString title = Strings::translate( Strings::ModifyGridWindowTitle );
	  gridWindow_ = new ModifyGridWindow( MainWindow::instance(), title );
	}
      gridWindow_->initialize( name );
      gridWindow_->show();
      break;
    case GraphDefinition::smithType:
      if ( smithWindow_.isNull() )
	{
	  QString title = Strings::translate( Strings::ModifySmithWindowTitle );
	  smithWindow_ = new ModifySmithWindow( MainWindow::instance(), title );
	}
      smithWindow_->initialize( name );
      smithWindow_->show();
	  break;
    case GraphDefinition::tableType:
      //No modifiable settings for table, just ignore
      break;
    default:
      ASSERT( "Unknown view type!" == 0 );
    }
}

//-----------------------------------------------------------------
void NavigationWindow::removeGraph( const QString& name )
{
  removeListItemFromBranch( graphs_, name );
}

//-----------------------------------------------------------------
void NavigationWindow::renameGraph( const QString& oldName, 
				    const QString& newName )
{
  QListViewItem* it = findListItemInBranch( graphs_, oldName );
  it->setText( 0, newName );
}

//-----------------------------------------------------------------
void NavigationWindow::updateOutputDefinition( const QString& graphName,
					       ResultDefinition& definition )
{
  GraphDefinition* graph = MainWindow::instance()->findGraph( graphName );
  bool isSmith = ( graph->graphType() == GraphDefinition::smithType );
  QString text = ResultDefinition::buildResultDefinitionStr( isSmith, definition );
  QListViewItem* parent = NavigationWindow::findListItemInBranch( graphs_, graphName);
  ASSERT( parent != 0 );
  QListViewItem* item = parent->firstChild();
  while ( item )
    {
      ListViewItem* listItem = dynamic_cast <ListViewItem*> (item);
      ASSERT( listItem != 0 );
      if ( listItem->id() == definition.id() )
	{
	  break;
	}
      item = item->nextSibling();
    }
  if ( item != 0 )
    {
      item->setText( 0, definition.source() );
      item->setText( 1, text );
    }
  else
    {
      ListViewItem* listView = new ListViewItem( ListViewItem::graphResultItem,
						 parent, definition.source() );
      listView->setText( 1, text );
      listView->setId( definition.id() );
    }
}

//-----------------------------------------------------------------
void NavigationWindow::showModifyOutputParameterWindow( const QString& graphname,
							ResultDefinition* definition )
{
  if ( outputWindow_.isNull() )
    {
      QString title = Strings::translate( Strings::OutputParameterDefinitionWindowTitle );
      outputWindow_ = new ModifyOutputParameterWindow( MainWindow::instance(),
						       title );
    }
  if ( outputWindow_->initialize( graphname, definition ) )
    {
      outputWindow_->show();
    }
}

//-----------------------------------------------------------------
void NavigationWindow::addFileBlock( const QString& name )
{
  ListViewItem* item = new ListViewItem( ListViewItem::dataFileChildItem,
					 dataFiles_, name );
  item->setSelectable( FALSE );
  variables_->sortChildItems( 0, TRUE );
}

//-----------------------------------------------------------------
void NavigationWindow::removeFileBlock( const QString& name )
{
  removeListItemFromBranch( dataFiles_, name );
}
  
