/*
 *   +----------------------------------------------------------------------+
 *   |                                                                      |
 *   |   videogen - a simple X modeline calculator by Szabolcs Rumi         |
 *   |                                                                      |
 *   |   (c) 1997-2001, Szabolcs Rumi                                       |   
 *   |                                                                      |
 *   |   this whole package is distributed under the                        |
 *   |   GNU General Public License (GPLv2)                                 |
 *   |                                                                      |
 *   +----------------------------------------------------------------------+   
 */





#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"





#define ERR_RES		101
#define	ERR_CLK		102
#define ERR_HFREQ	103
#define ERR_VFREQ	104





void
banner (void) {
	fprintf (stdout, "       +--------------------------------------------------------------+\n");
	fprintf (stdout, "       |                                                              |\n");
	fprintf (stdout, "       |       videogen %5.5s     simple X mode line calculator       |\n", CFG_VIDEOGEN_VERSION);
	fprintf (stdout, "       |                                                              |\n");
	fprintf (stdout, "       |              by Szabolcs Rumi, (c) 1997 - 2001               |\n");
	fprintf (stdout, "       |        THIS PROGRAM COMES WITH ABSOLUTELY NO WARRANTY        |\n");
	fprintf (stdout, "       |    for details see the GNU General Public License (GPLv2)    |\n");
	fprintf (stdout, "       |                                                              |\n");
	fprintf (stdout, "       |                  calculations are based on                   |\n");
	fprintf (stdout, "       |     The Hitchhiker's Guide to X386/XFree86 Video Timing      |\n");
	fprintf (stdout, "       |          by Chin Fang, Bob Crosson, Eric S. Raymond          |\n");
	fprintf (stdout, "       |                                                              |\n");
	fprintf (stdout, "       +--------------------------------------------------------------+\n\n");
}





int
main (int argc, char **argv) {

	const float hvis = CFG_HORIZONTAL_VISIBLE;
	const float vvis = CFG_VERTICAL_VISIBLE;
	
	const unsigned int hfp = CFG_HORIZONTAL_FRONT_PORCH;
	const unsigned int hbp = CFG_HORIZONTAL_BACK_PORCH;
	const unsigned int vfp = CFG_VERTICAL_FRONT_PORCH;
	const unsigned int vbp = CFG_VERTICAL_BACK_PORCH;

	float dcmax = 0;	/* MHz */
	float hfmax = 0;	/* kHz */
	float vfmax = 0;	/* Hz */
	float hspmin = 0;	/* usec */
	unsigned int hr = 0;
	unsigned int vr = 0;

	float dc; 		/* MHz */
	float hf; 		/* kHz */
	float vf; 		/* Hz */
	unsigned int hsp;
	unsigned int vsp;
	unsigned int hfl, vfl;
	
	int interactive = 0;
	int verbose = 1;
	int framebuffer = 0;
	
	int i;


		
	/*
	 * reading command line parameters
	 */


	 
#ifdef CFG_DEBUG
	fprintf (stderr, "\nDEBUG: command line parameter count: %u", (argc-1) );
#endif  

	if ( (argc == 1) || ( (argc == 2) && (strcmp (argv[1], "--help") == 0) ) ) {
		banner ();
		fprintf (stdout, "\nUSAGE: videogen <-i>   (for interactive mode)\n       videogen [-s] <-fb> <-clk=n> <-width=n> [-height=n] <-hf=n> <-vf=n> <-hsync=n>\n\n");
		exit (0);
	}
	
	i = 1;
	
	while (i < argc) {

		if (strcmp (argv[i], "-s") == 0) {
			verbose = 0;
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter silent mode read from string", argv[i]);
#endif
		}
       
		if (strcmp (argv[i], "-i") == 0) {
			interactive++;
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter interactive mode read from string", argv[i]);
#endif
		}
		
		if (strcmp (argv[i], "-fb") == 0) {
			framebuffer++;
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter framebuffer device format read from string %s", argv[i]);
#endif         
		}
       
		if (strncmp (argv[i], "-clk=", 5) == 0) {
			dcmax = atof (argv[i] + 5);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter clk=%f read from string %s", dcmax, argv[i]);
#endif
		}

		if (strncmp (argv[i], "-hf=", 4) == 0) {
			hfmax = atof (argv[i] + 4);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter hf=%f read from string %s", hfmax, argv[i]);
#endif  
		}
       
		if (strncmp (argv[i], "-vf=", 4) == 0) {
			vfmax = atof (argv[i] + 4);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter vf=%f read from string %s", vfmax, argv[i]);
#endif 
		}
       
		if (strncmp (argv[i], "-hsync=", 7) == 0) {
			hspmin = atof (argv[i] + 7);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter hsync=%f read from string %s", hspmin, argv[i]);
#endif 
		}
       
		if (strncmp (argv[i], "-width=", 7) == 0) {
			hr = atof (argv[i] + 7);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter width=%u read from string %s", hr, argv[i]);
#endif
			hr = (hr >> 3) << 3; 
		}

		if (strncmp (argv[i], "-height=", 8) == 0) {
			vr = atof (argv[i] + 8);
#ifdef CFG_DEBUG
			fprintf (stderr, "\nDEBUG: parameter height=%u read from string %s", vr, argv[i]);
#endif
			vr = (vr >> 3) << 3;
		}
		
		i++;
	}    
	
	if (verbose > 0)
		banner();

	if (interactive > 0) {
		fprintf (stdout, "your video card's maximum dot clock [in MHz]? ");
		scanf ("%f", &dcmax); 
  
		fprintf (stdout, "\nyour monitor's maximum horizontal refresh [in kHz]? ");
		scanf ("%f", &hfmax);

		fprintf (stdout, "\nyour monitor's maximum vertical refresh [in Hz]? ");
		scanf ("%f", &vfmax);
  
		fprintf (stdout, "\nyour monitor's minimum horizontal sync pulse [in microsecs]? ");
		scanf ("%f", &hspmin);
  
		fprintf (stdout, "\nthe desired horizontal resolution [in pixels]? ");
		scanf ("%d", &hr);
  
		fprintf (stdout, "\nthe desired vertical resolution [in pixels]? ");
		scanf ("%d", &vr);
	}

	
	
	/*
	 * checking and correcting parameters before calculation
	 */



	if ((hr < CFG_WIDTH_MIN) || (hr > CFG_WIDTH_MAX)) {
		if (verbose > 0)
			fprintf (stderr, "\nERROR(101): invalid horizontal resolution (%u)\n\n", hr);
		exit (ERR_RES);
	}
  
	if ((vr < CFG_HEIGHT_MIN) || (vr > CFG_HEIGHT_MAX)) {
		if (verbose > 0) {
			fprintf (stderr, "\nWARNING: invalid vertical resolution (%u)", vr);
			fprintf (stdout, "\ncorrecting vertical resolution using the default aspect ratio\n\n");
		}
		vr = (unsigned int) hr * CFG_ASPECT_RATIO;
		vr = (vr >> 3) << 3;
	}
  
	if ((dcmax < CFG_CLK_MIN) || (dcmax > CFG_CLK_MAX)) {
		if (verbose > 0)
			fprintf (stderr, "\nERROR(102): invalid maximum dot clock frequency (%-6.2f)\n\n", dcmax);
		exit (ERR_CLK);
	}
  
	if ((hfmax < CFG_HFREQ_MIN) || (hfmax > CFG_HFREQ_MAX)) {
		if (verbose > 0)
			fprintf (stderr, "\nERROR(103): invalid horizontal refresh frequency (%-6.2f)\n\n", hfmax);
		exit (ERR_HFREQ);
	}
  
	if ((vfmax < CFG_VFREQ_MIN) || (vfmax > CFG_VFREQ_MAX)) {
		if (verbose > 0)
			fprintf (stderr, "\nERROR(104): invalid vertical refresh frequency (%-6.2f)\n\n", vfmax);
		exit (ERR_VFREQ);
	}  
  
	if ((hspmin < CFG_HSYNC_MIN) || (hspmin > CFG_HSYNC_MAX)) {
		if (verbose > 0) {
			fprintf (stderr, "\nWARNING: invalid horizontal sync pulse duration (%-6.2f)", hspmin);
			fprintf (stdout, "\ncorrecting horizontal sync pulse duration\n\n");
		}
		hspmin = CFG_HORIZONTAL_SYNC_PULSE;
	}

	
	
	/*
	 * calculating
	 */


	
	hf = hfmax;
	hfl = (unsigned int) hr / hvis;
	hfl = (hfl >> 3) << 3;
	
	dc = (hfl * hf) / 1000;
	
	if (dc > dcmax) {
#ifdef CFG_DEBUG
		fprintf (stderr, "\nDEBUG: dc=%6.2f too high", dc);
#endif
		hf = (dcmax * 1000) / hfl;
	}
		
	hsp = hfl - hr - hfp - hbp;
	
	if ((hsp / dc) < hspmin) {
#ifdef CFG_DEBUG
		fprintf (stderr, "\nDEBUG: hsp=%u (%u usec) too short", hsp, (unsigned int) hsp / dc);
#endif
		dc = hsp / hspmin;
		hf = (1000 * dc) / hfl;
	}
	
	vfl = (unsigned int) vr / vvis;
	vsp = vfl - vr - vfp - vbp;
	
	vf = (1000000 * dc) / (hfl * vfl);
	
	if (vf > vfmax) {
#ifdef CFG_DEBUG
		fprintf (stderr, "\nDEBUG: vf=%6.2f too high", vf);
#endif
		dc = (vfmax * (hfl * vfl)) / 1000000;
		hf = (1000 * dc) / hfl;
		vf = (1000000 * dc) / (hfl * vfl);
	}


	
	/*
	 * printing out results
	 */


  
#ifdef CFG_DEBUG
	fprintf (stderr, "\nDEBUG: dc:%6.2f  hf:%6.2f  vf:%6.2f", dc, hf, vf);
	fprintf (stderr, "\nDEBUG: hr:%u  hfp:%u  hsp:%u  hbp:%u", hr, hfp, hsp, hbp);
	fprintf (stderr, "\nDEBUG: hfl:%u  hfl-check:%u", hfl, (hr + hfp + hsp + hbp));
	fprintf (stderr, "\nDEBUG: vr:%u  vfp:%u  vsp:%u  vbp:%u", vr, vfp, vsp, vbp);
	fprintf (stderr, "\nDEBUG: vfl:%u  vfl-check:%u", vfl, (vr + vfp + vsp + vbp));
#endif
	
	if (verbose > 0) {
		fprintf (stdout, "\n\nPLEASE CHECK THESE CALCULATED VALUES:\n\n");
		fprintf (stdout, "        visible resolution:  %u x %u pixels\n", hr, vr);
		fprintf (stdout, "            dot clock rate:  %6.2f MHz\n", dc);
		fprintf (stdout, "     vertical refresh rate:  %6.2f Hz\n", vf);
		fprintf (stdout, "     horizontal sync pulse:  %6.2f microsecs\n\n", hsp / dc);
		fprintf (stdout, "\n\nMake sure that none of the above exceeds the specifications\n");
		fprintf (stdout, "of your hardware. If one does then DO NOT use the calculated\n");
		fprintf (stdout, "Modeline, because such an overload can damage your system!\n\n");
		fprintf (stdout, "If they are OK then use this Modeline in your XF86Config file:\n\n");
	}


	
	/* 
	 * The calculated Modeline will now be printed
	 */


	 
	if (framebuffer == 0) {
		fprintf (stdout, "\"%ux%u\"  %-6.2f  %u  %u  %u  %u  %u  %u  %u  %u",
				    hr,
				    vr,
				    dc,
				    hr,
				    hr + hfp,
				    hr + hfp + hsp,
				    hr + hfp + hsp + hbp,
				    vr,
				    vr + vfp,
				    vr + vfp + vsp,
				    vr + vfp + vsp + vbp);
	}

	else if (verbose == 0) {
		fprintf (stdout, "%0.0f %u %u %u %u %u %u",
		(unsigned int) 1000000 / dc, hbp, hfp, vbp, vfp, hsp, vsp);
	}
	
	if (verbose > 0) {
		fprintf (stdout, "\n\nfor the kernel framebuffer device use these timings with fbset:");
		fprintf (stdout, "\n\n%0.0f %u %u %u %u %u %u\n\n",
		(unsigned int) 1000000 / dc, hbp, hfp, vbp, vfp, hsp, vsp);
	}
	
	return (0);
}





/* EOF */
