/*
		  VMA - Undo (and Redo) Record Management

	These are structures which will be contained as lists on
	the editors. They record the operation that will perform an
	undo (or redo).

	These functions mearly facility easy management of the
	structures as lists.

	When adding new types, make sure you update VMAUndoCreate() to
	allocate the correct size of the data type structure and
	update VMAUndoDestroy() to deallocate and members that may point
	to allocated resources. Externally, you may need to update;
	editorundo.c

	All undo structures have the first two members as:

		int type;
		char *name;
		int repeats;

	Members that are dynamically allocated in the undo structure
	will be so noted.

 */

#ifndef VMAUNDO_H
#define VMAUNDO_H

#include <sys/types.h>
#include <gtk/gtk.h>

#include "v3dmp.h"
#include "v3dmodel.h"

/*
 *	Undo structure type codes:
 */
#define VMA_UNDO_TYPE_SET_NORMAL		20
#define VMA_UNDO_TYPE_SET_TEXCOORD		21
#define VMA_UNDO_TYPE_SET_VERTEX		22
#define VMA_UNDO_TYPE_SET_TEXORIENT		23
#define VMA_UNDO_TYPE_SET_HEIGHTFIELD		24

#define VMA_UNDO_TYPE_FLIP_WINDING		30

#define VMA_UNDO_TYPE_CREATE_PRIMITIVE		40
#define VMA_UNDO_TYPE_DELETE_PRIMITIVE		41
#define VMA_UNDO_TYPE_TRANSLATE_PRIMITIVE	42

#define VMA_UNDO_TYPE_ADD_VERTEX		50
#define VMA_UNDO_TYPE_REMOVE_VERTEX		51


/*
 *	All undo structures must contain this #define to define the
 *	first sequence of common members.
 */
#define VMA_UNDO_PREPEND_MEMBERS	\
	gint type;	\
	gchar *name;	\
	gint repeats;


/*
 *	Undo structures:
 */
typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gint vertex_num;
	mp_vertex_struct n;

} vma_undo_set_normal_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gint vertex_num;
	mp_vertex_struct tc;

} vma_undo_set_texcoord_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gint vertex_num;
	mp_vertex_struct v;

} vma_undo_set_vertex_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gdouble i, j, di, dj;

} vma_undo_set_texorient_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	/* Note, only sets heightfield values, not its path. */
	gpointer editor;
	gint model_num;
	gint primitive_num;
	gdouble x_length, y_length, z_length;
	gdouble x, y, z;
	gdouble heading, pitch, bank;	/* In radians. */

} vma_undo_set_heightfield_struct;


typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;

} vma_undo_flip_winding_struct;



typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint insert_at;		/* Create primitive at specified index. */
	gpointer p;		/* Dynamically allocated. */

} vma_undo_create_primitive_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint delete_at;		/* Delete primitive at specified index. */

} vma_undo_delete_primitive_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gdouble dx, dy, dz;	/* Movements for each vertex. */

} vma_undo_translate_primitive_struct;


typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gint insert_at;	/* Vertex number on primitive to insert at. */

	mp_vertex_struct *v, *n, *tc;	/* Dynamically allocated. */

} vma_undo_add_vertex_struct;

typedef struct {

VMA_UNDO_PREPEND_MEMBERS

	gpointer editor;
	gint model_num;
	gint primitive_num;
	gint delete_at;	/* Vertex number on primitive. */

} vma_undo_remove_vertex_struct;


/*
 *	Undo structure containing the minimal common members:
 */
typedef struct {

	VMA_UNDO_PREPEND_MEMBERS

} vma_undo_common_struct;


extern gpointer VMAUndoNew(gint type, const gchar *name);
extern void VMAUndoDelete(gpointer u);
extern void VMAUndoSetRepeats(gpointer u, gint repeats);

extern gpointer VMAUndoListGetPtr(gpointer *list, gint total, gint i);
extern gpointer VMAUndoListInsert(
	gpointer **list, gint *total, gint i,
	gint type, const gchar *name
);
extern void VMAUndoListDelete(gpointer **list, gint *total, gint i);
extern void VMAUndoListDeleteAll(gpointer **list, gint *total);


#endif	/* VMAUNDO_H */
