/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 0.4
 * Revision 0.4.2
 * November 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _vdk_grid_h
#define _vdk_grid_h
#include <vdk/vdkobj.h>
#include <vdk/gtksheet.h>
#include <vdk/gtkitementry.h>
#include <vdk/vdkstring.h>
#include <vdk/vdkarray.h>
#include <vdk/vdkutils.h>
#define  FIT_COLUMN_WIDTH  0
#define  FIT_ROW_HEIGHT    0
enum
{
grid_select_cell = 1024,
grid_unselect_cell,
grid_changed_cell,
grid_select_row,
grid_select_column,
grid_hor_scrolled,
grid_vert_scrolled
};

class VDKForm;
class VDKGrid;
#ifndef _string_array_def
typedef VDKArray<VDKString> StringArray;
#endif
///////////// VDKCELL CLASS ///////////////////////
/*!
  \class VDKGridCell
  \brief provides a single grid cell wrapper
 */
class VDKGridCell
{
  // properties
 public:
  /*!
    Sets/gets cell text
   */
  VDKReadWriteValueProp<VDKGridCell, VDKString>        Text;
  /*!
    Sets/gets cell text alignement
   */
  VDKReadWriteValueProp<VDKGridCell, GtkJustification> Align;
  /*!
    Gets cell attached widget
   */
  VDKReadOnlyValueProp<VDKGridCell, VDKObject*>   AttachedWidget;
  /*!
   */
  VDKReadWriteValueProp<VDKGridCell, bool> Editable;
 protected:
  friend class VDKGrid;
  GtkSheet* sheet;
  VDKObject* grid;
  int row,col;
  void SetText(VDKString text)
    { 
      if (sheet)
	  gtk_sheet_set_cell(sheet,row,col,Align,(char*) text); 
    }
  void SetAlign(GtkJustification j)
    {
      if(GTK_WIDGET_VISIBLE(sheet))
	{
	  VDKString s = Text;
	  gtk_sheet_set_cell(sheet,row,col,j,(char*) s);
	}
    }
  bool GetEditable();
  void SetEditable(bool flag);

 public:
  VDKGridCell(GtkSheet *sheet, int row, int col);
  virtual ~VDKGridCell();
  /*!
    Sets cell background
    \param rgb color
   */
  void SetBackground(VDKRgb& rgb);
  /*!
    Sets cell foreground
    \param rgb color
   */
  void SetForeground(VDKRgb& rgb);
  /*!
    Attaches a generic widget to cell
   */
  void Attach(VDKObject* object,float x_align, float y_align);
  /*!
    Returns cell row
   */
  int Row() { return row; }
  /*!
    Returns cell columns
   */
  int Column() { return col; }
  /*!
    Set cell border
    \param mask can be any ored combination of:
    \arg \b GTK_SHEET_LEFT_BORDER
    \arg \b GTK_SHEET_RIGHT_BORDER
    \arg \b GTK_SHEET_TOP_BORDER
    \arg \b GTK_SHEET_BOTTOM_BORDER
    \param width line thickness in pixels
    \param line_style can be one of:
    \arg \b GDK_LINE_SOLID
    \arg \b GDK_LINE_ON_OFF_DASH
    \arg \b GDK_LINE_DOUBLE_DASH
   */
  void SetBorder(int mask, int width, int line_style);
}; 
///////////////// ROW/COLUMN CLASS //////////////
class VDKGridSlice
{

 protected:
  friend class VDKGrid;
  GtkSheet* sheet;
 public:
  VDKGridSlice(GtkSheet *sheet):sheet(sheet){}
  virtual ~VDKGridSlice() {}
  void SetRangeForeground(VDKRgb& rgb,GtkSheetRange& range);
  void SetRangeBackground(VDKRgb& rgb,GtkSheetRange& range);
};

/*!
  \class VDKGridColumn
  \brief provides a grid column header
 */
class VDKGridColumn : public VDKGridSlice
{
 public:
  /*!
    Sets/gets column title
   */
  VDKReadWriteValueProp<VDKGridColumn, VDKString> Title;
  /*!
    Sets/gets column width
   */
  VDKReadWriteValueProp<VDKGridColumn, int>       Width;
  /*!
    Sets/gets column enabled flag
   */
  VDKReadWriteValueProp<VDKGridColumn, bool>      Enabled;
  /*!
    Sets/gets column background
   */
  VDKReadWriteValueProp<VDKGridColumn, VDKRgb>  Background;
  /*!
    Sets/gets column forground
   */
  VDKReadWriteValueProp<VDKGridColumn, VDKRgb>  Foreground;
  /*!
    Sets/gets column visible flag
   */
  VDKReadWriteValueProp<VDKGridColumn, bool>    Visible;
 protected:
  friend class VDKGrid;
  int col;
  void SetTitle(VDKString s) 
    {
      gtk_sheet_column_button_add_label(sheet,col, (char*) s);
    }
  VDKString GetTitle()
    {
      VDKString p(sheet->column[col].button.label);
      return p;
    }
  int  GetWidth() { return sheet->column[col].width; }
  void SetWidth(int w);
  void SetEnabled(bool f)
    {
      gtk_sheet_column_set_sensitivity(sheet,col,f);
    }
  void SetVisible(bool flag)
    {
      gtk_sheet_column_set_visibility	(sheet, col, flag);
    }
  void SetForeground(VDKRgb c);
  void SetBackground(VDKRgb c);
 public:
  VDKGridColumn(GtkSheet *sheet,int col):
    VDKGridSlice(sheet),
    Title("Title",this,
	  VDKString(""),&VDKGridColumn::SetTitle,&VDKGridColumn::GetTitle),
    Width("Width",this,0,&VDKGridColumn::SetWidth,&VDKGridColumn::GetWidth),
    Enabled("Enabled",this,false,&VDKGridColumn::SetEnabled),
    Background("Background",this,
	       VDKRgb(0,0,0),&VDKGridColumn::SetBackground),
    Foreground("Foreground",this,
	       VDKRgb(0,0,0),&VDKGridColumn::SetForeground),
    Visible("Visible",this,false,&VDKGridColumn::SetVisible),
    col(col)
    {}
  virtual ~VDKGridColumn() {};
};
/*!
  \class VDKGridRow
  \brief provides a grid row header
 */
class VDKGridRow : public VDKGridSlice
{
 public:
  /*!
    Sets/gets row title
   */
  VDKReadWriteValueProp<VDKGridRow, VDKString> Title;
  /*!
    Sets/gets row height
    \par Tip
    Setting this property to 0 makes row height be adapted
    using font dimension
   */
  VDKReadWriteValueProp<VDKGridRow, int>       Height;
  /*!
    Sets/gets row visible flag
   */
  VDKReadWriteValueProp<VDKGridRow, bool>  Visible;
  /*!
    Sets/gets row enabled flag
   */
  VDKReadWriteValueProp<VDKGridRow, bool>  Enabled;
 protected:
  friend class VDKGrid;
  int row;
  void SetTitle(VDKString s) 
    {
      gtk_sheet_row_button_add_label(sheet, row, (char*) s);
    }
  VDKString GetTitle()
    {
      VDKString p(sheet->row[row].button.label);
      return p;
    }    
  void SetVisible(bool flag)
    {
      gtk_sheet_row_set_visibility(sheet, row, flag);
    }
  void SetEnabled(bool f)
    {
      gtk_sheet_row_set_sensitivity(sheet,row,f);
    }
  void SetHeight(int w);
  int  GetHeight() { return sheet->row[row].height; }
 public:
  VDKGridRow(GtkSheet* sheet, int row):
    VDKGridSlice(sheet),
    Title("Title",this,
	  VDKString(""),&VDKGridRow::SetTitle,&VDKGridRow::GetTitle),
    Height("Height",this,0,&VDKGridRow::SetHeight,&VDKGridRow::GetHeight),
    Visible("Visible",this,false,&VDKGridRow::SetVisible),
    Enabled("Enabled",this,false,&VDKGridRow::SetEnabled),
    row(row) 
    {}
  virtual ~VDKGridRow() {};
};

////////////////////////////////////
/*!
  \class VDKGrid
  \brief VDKGrid is a wrapper of gtksheet widget.
  \author Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
  
  Basically is a table of cells that can be edited or contain 
  others widgets as well. Table can be done with or without 
  column/row headers.
  \par Signals
  \arg \b grid_select_cell emitted when a cell is selected 
  \arg \b grid_unselect_cell emitted when a cell is unselected 
  \arg \b grid_changed_cell emitted when a cell content is changed by user
  \arg \b grid_select_row emitted when a row is selected
  \arg \b grid_select_column emitted when a column is selected
  \par EXAMPLES
  In ./testvdk/vdkgridwin.cc
 */
class VDKGrid: public VDKObject
{
 private:
  VDKGridCell* aCell;
  VDKGridColumn* aColumn;
  VDKGridRow *aRow;
 public:
  /*!
    gets active cell
   */
  VDKReadOnlyValueProp<VDKGrid,VDKGridCell*> ActiveCell;
  /*!
    gets previous active cell
   */
  VDKReadOnlyValueProp<VDKGrid,VDKGridCell*> PreviousActiveCell;
  /*!
    gets changed cell
   */
  VDKReadOnlyValueProp<VDKGrid,VDKGridCell*> ChangedCell;
  /*!
    gets selected row
   */
  VDKReadOnlyValueProp<VDKGrid, int>     SelectedRow;
  /*!
    gets selected column
   */
  VDKReadOnlyValueProp<VDKGrid, int>     SelectedColumn;
  /*!
    Sets/gets row title visible flag
   */
  VDKReadWriteValueProp<VDKGrid, bool>   RowTitleVisible;
  /*!
    Sets/gets column title visible flag
   */
  VDKReadWriteValueProp<VDKGrid, bool>   ColumnTitleVisible;
  /*!
    gets grid row number
   */
  VDKReadOnlyValueProp<VDKGrid,int>      MaxRows;
  /*!
    gets grid column number
   */
  VDKReadOnlyValueProp<VDKGrid,int>      MaxColumns;
 protected:
  static int ActivateCell(GtkWidget*, int row, int col, gpointer);
  static int DeactivateCell(GtkWidget*, int row, int col, gpointer);
  static int OnChangedCell(GtkWidget* wid, int row, int col, gpointer gp);
  static void SelectRow(GtkWidget* , int, gpointer );
  static void SelectColumn(GtkWidget* , int, gpointer );
  static void HAdjChanged(GtkWidget* wid, gpointer gp);
  static void VAdjChanged(GtkWidget* wid, gpointer gp);
  virtual void    SetForeground(VDKRgb color, 
				GtkStateType state = GTK_STATE_NORMAL);
  virtual void    SetBackground(VDKRgb color, 
				GtkStateType state = GTK_STATE_NORMAL);
  void SetRowTitleVisible(bool flag)
    {
      if(flag)
	gtk_sheet_show_row_titles(GTK_SHEET(widget));
      else
	gtk_sheet_hide_row_titles(GTK_SHEET(widget));
    }
  void SetColumnTitleVisible(bool flag)
    {
      if(flag)
	gtk_sheet_show_column_titles(GTK_SHEET(widget));
      else
	gtk_sheet_hide_column_titles(GTK_SHEET(widget));
    }
  int GetMaxRows() { return GTK_SHEET(widget)->maxrow+1; }
  int GetMaxColumns() { return GTK_SHEET(widget)->maxcol+1; }

 public:
  /*!
    Constructor
    \param owner
    \param rows number of row
    \param cols number of columns
    \param editable
    \param title
   */
  VDKGrid(VDKForm* owner, int row, int cols, bool editable = false,
	  char* title = "title");
  /*!
    Destructor
   */
  virtual ~VDKGrid() 
    {
      if(aCell)
	delete aCell;
    }
  /*!
    Return a cell
    \param row number
    \param column number
   */
  VDKGridCell* Cell(int row, int col);
  /*!
    Return a cell
    \param p cell cordinates
   */
  VDKGridCell* Cell(VDKPoint p)
    {
      return Cell(p.X(),p.Y());
    }
  /*!
    Returns grid column
    \param col column number
   */
  VDKGridColumn* Column(int col);
  /*!
    Returns grid row
    \param row number
   */
  VDKGridRow* Row(int row);
  /*!
    Clears grid
   */
  void Clear()
    { gtk_sheet_range_clear(GTK_SHEET(widget), NULL); }
  /*!
    Add a row to grid
    \param array an array of strings
   */
  void AddRow(StringArray& array);
  /*!
    Add blanks rows to grid
    \param n how much rows to add
   */
  void AddBlankRows(int n);
  /*!
    Delete all grid rows.
   */
  void Flush()
    { gtk_sheet_delete_rows(GTK_SHEET(widget),0,
			    GTK_SHEET(widget)->maxrow); }
  /*!
    \internal
   */
  virtual void Setup();
#ifdef USE_SIGCPLUSPLUS
 public:
  /* emitted when a cell is selected, correspondes to grid_select_cell. 
     The first parameter is the row, the second one the column. */
  VDKSignal2<void,int,int> OnCellSelect;
  /* emitted when a cell is unselected, correspondes to grid_unselect_cell. 
     The first parameter is the row, the second one the column. */
  VDKSignal2<void,int,int> OnCellUnselect;
  /* emitted when a cells content is changed, correspondes to 
     grid_changed_cell. 
     The first parameter is the row, the second one the column. */
  VDKSignal2<void,int,int> OnCellChanged;
  /* emitted when a row is selected, correspondes to 
     grid_row_select. The parameter indicates the selected row. */
  VDKSignal1<void,int> OnRowSelect;
  /* emitted when a column is selected; correspondets to
     grid_select_column. The parameter indicates the selected column. */
  VDKSignal1<void,int> OnColSelect;

 protected:
  
#endif USE_SIGCPLUSPLUS
};

#endif




 
