/*
 * Copyright (C) 1999 Jonathan R. Hudson
 * Developed by Jonathan R. Hudson <jrhudson@bigfoot.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// Example to show VDKInput, multiple input streams, 
// user defined VDKsignals, interaction with Unix signals etc

#define _GNU_SOURCE

#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/wait.h>
#include <ctime>
#include <fstream>
#include "iodemo0.h"

DEFINE_SIGNAL_MAP(MyForm,VDKForm)
    ON_SIGNAL(inp1,INP_SIGNAL,DoIO),
    ON_SIGNAL(text, realize_signal,runcmd),
    ON_SIGNAL(entry, activate_signal,sendstuff),
    ON_SIGNAL(timer,timer_tick_signal,Quit),
    ON_SIGNAL(quit,clicked_signal,Quit)
END_SIGNAL_MAP       
DEFINE_SIGNAL_LIST(MyForm,VDKForm);

void MyForm::ShowInfanticide()
{
    io = 0;
    SignalEmit("KillSignal");
    Pid(0);
}

bool MyForm::sendstuff(VDKObject* obj)
{
    char txt[1024];
    char *t;
    
    strcpy(txt, entry->GetText());
    t = txt+strlen(txt);
    *t++ = '\n';
    
    if(xmit != -1)
    {
        write(xmit, txt, (t - txt));
    }
    entry->SetText("");
    return true;
}


bool MyForm::CanClose(void)
{
    return Stop(0);
}

bool MyForm::Stop (VDKObject *obj)    
{
    if(pid)
    {
        kill(pid, SIGTERM);
    }
    return true;
}
       
bool MyForm::DoIO (VDKObject *obj)
{
    VDKInput *ip = static_cast<VDKInput*>(obj);
    int res;
    char buf[1024];
        
    res = read (ip->getfd(), buf, sizeof(buf));
    
    if(res > 0)
    {
        text->TextInsert(buf, res);
    }
    else
    {
        close(ip->getfd());
        ShowInfanticide();
        ip->Destroy();
    }
    return true;
}

bool MyForm::runcmd(VDKObject *obj)
{
    if(timer)
    {
        timer->Destroy();
        timer = 0;
    }
    
    if(io == 0)
    {
        MyApp *a = static_cast<MyApp *>(Application());
        int inpipe[2], outpipe[2];
        
        pipe(inpipe);
        pipe(outpipe);

        pid = fork();
        switch(pid)
        {
                // Child
            case 0:
                dup2(inpipe[0], 0);
                dup2(outpipe[1], 1);
                dup2(outpipe[1], 2); 
                execvp(a->args[0], a->args);
                break;
            case -1:
                break;
            default:
                    // Parent
                close(outpipe[1]);      // Important! Close my end here
                close(inpipe[0]);      // Important! Close my end here
                xmit = inpipe[1];
                    // Set pipes none blocking, so we can read big buffers
                    // in the callback without having to use FIONREAD
                    // to make sure the callback doesn't block.

                inp1 = new VDKInput(this,outpipe[0]);
                io = 1;
                text->Clear();
                break;
        }
    }
    return true;
}

bool MyForm:: Quit(VDKObject*o)
{
    if(pid)
    {
        kill(pid, SIGTERM);
    }
    Application()->Terminate();
    return true;
}

void MyForm::Setup(void)
{
    VDKBox* vbox1 = new VDKBox(this, v_box);
    VDKFrame* frame1 = new VDKFrame(this, "You type here");
    VDKFrame* frame2 = new VDKFrame(this, "Forked friend");

    // Just to show user signals ....
    SignalConnect("KillSignal",&MyForm::Mourn, false);

    VDKBox* hbox1 = new VDKBox(this, h_box);
    vbox1->Add(frame1);
    vbox1->Add(frame2);    

    vbox1->Add(hbox1,c_justify, 0, 0);
    hbox1->Add((quit = new VDKLabelButton(this,"Quit")),
               c_justify, 1, 0);
    entry = new VDKEntry(this, 0);
    entry->Font = new VDKFont(this,
             "-*-lucidatypewriter-medium-r-normal-*-*-120-*-*-*-*-iso8859-1");
    frame1->Add(entry);    
    text = new VDKText(this, 0);
    text->Font = new VDKFont(this,
             "-*-lucidatypewriter-medium-r-normal-*-*-120-*-*-*-*-iso8859-1");
    frame2->Add(text);    
    Add(vbox1);    
    SetSize(500,-1);
    entry->GrabFocus();
}

bool MyForm::Mourn(VDKObject *)
{
  
    if(timer == 0)
    {
        timer = new VDKTimer(this, 30*1000);
    }

    if(io == 0)
        text->TextInsert("\nPress Quit to exit(or wait 30 seconds)\n");
    return true;
}


// This is a static member so we can access class members
// Somewhat contrived example
// In this case, we don't get SIGPIPE (at least not with glibc2.1, but 
// we should be ready to handle it

void MyApp::plumber(int)
{
    cerr << "Plumbed" << endl;
}

void MyApp::reaper(int)
{
    int sts;
    int pid;
    
    pid = waitpid(-1, &sts, WNOHANG);
    
    if(pid == ((MyForm*)MyApp::me->MainForm)->Pid())
    {
        cerr << "-- child dies, pid was " << pid << ", status " << sts << endl;
        ((MyForm*)MyApp::me->MainForm)->Pid(0);
    }
    else
    {
        cerr << "Unknown pid " << pid << " status = " << sts << endl;
    }
    
}

void MyApp::Setup()   
{
    if(*args == NULL)
    {
        cerr << "No command to run" << endl;
    }
    MainForm = new MyForm(this,"Running Command ...");
    MyApp::me = this;
    
    struct sigaction *sac = new struct sigaction;
    sigemptyset(&(sac->sa_mask));
    sac->sa_flags=0;
    sac->sa_handler = MyApp::reaper;
    sigaction(SIGCHLD, sac, NULL);  

    sigemptyset(&(sac->sa_mask));
    sac->sa_flags=0;
    sac->sa_handler = MyApp::plumber;
    sigaction(SIGPIPE, sac, NULL);  

    MainForm->Setup();       
    MainForm->Show();
}

MyApp *MyApp::me;
int main (int argc, char *argv[])   
{
    MyApp app(&argc, argv); 
    app.Run();      
    return 0;  
}

