/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_commandLine.h"

#include "visu_tools.h"

#include <stdlib.h>
#include <getopt.h>
#include <stdio.h>
#include <locale.h>
#include <string.h>

/**
 * SECTION:visu_commandLine
 * @short_description: All methods needed to parse options from the
 * command line.
 *
 * <para>V_Sim parses the command line at startup and store data in
 * private variables. All this values can be retrieve later by the
 * program through calls to commandLineGet_* methods.</para>
 */

static char *argFileName;

/* Forced resources file. */
static gchar *argResources;

/* Spin arguments. */
static char *argSpinFileName;
static int spinHidingMode;
static gboolean spinAndAtomic;

static char *argExportFileName;

/* Colourisation tool. */
static gchar *argColorizeFileName;
static int argColorizeColUsed[3];
static int argColorizePresetColor;
static gboolean argColorizeColUsed_isPresent;

/* Iso-surfaces tool. */
static gchar *argScalarFieldFileName;
static gchar *argIsoSurfacesFileName;
static float *argIsoValues;
static gchar **argIsoNames;
static float argNbIsoValues;
static gboolean argFitToBox;

/* Translations stuffs. */
static gboolean argTranslationsIsSet;
static float argTranslations[3];
/* Extension stuffs. */
static gboolean argExpandIsSet;
static float argExtension[3];

/* Coloured map. */
static int argMapPlaneId;
static gboolean argLogScale;
static int argNIsoLines;

/* Background image. */
static gchar *argBgImageFile;

/* Extended options. */
static OptionTable *argOptionTable;

static gchar* argPlanesFileName;

static int withGtk;

static int xWindowWidth, xWindowHeight;

#define OUT stdout
#define P   fprintf
void printInfoMessage()
{
  /* Remaining letters...
     j, k, l, q, w, y, z */
  P(OUT, _("V_Sim is a software to visualize atomic structures with"
		    " OpenGl rendering.\n\n"));
  P(OUT, _("usage: %s [-h|-e file|-g size|-c file|-s file|-m id|-t x:y:z|\n"
	   "              -x x:y:z|-u l:m:n|-p file|-d id|-a|-f file|-v v[:v]|\n"
	   "              -i file|-r file|-o id=value|-b id|-n value]\n"
	   "              [fileToRender]\n\n"), PACKAGE_TARNAME);
  P(OUT, _("-h, --help              show this little help.\n"));
  P(OUT, _("-r, --resources file    load the given resources file on startup\n"
	   "                        instead of looking for a valid resources\n"
	   "                        file in the standard locations.\n"));
  P(OUT, _("-e, --export file       made an image from the fileToRender\n"
	   "                        argument. The format is specified through\n"
	   "                        the extension of the file parameter.\n"));
  P(OUT, _("-g, --geometry size     specify the size of the rendering window,\n"
	   "                        the size parameter must have the following\n"
	   "                        format: <width>x<height> with positive non\n"
	   "                        null values.\n"));
  P(OUT, _("-c, --colorize file     the argument fileToRender must be called,\n"
	   "                        then the given file of the option is used to\n"
	   "                        colorize the elements.\n"));
  P(OUT, _("-u, --use-column l:m:n  must be used with the '--colorize' option,\n"
	   "                        it specifies the columns to use from the data\n"
	   "                        file for the three colour channels [l;m;n].\n"
	   "                        Columns are counted from 1 and set 0 to use\n"
	   "                        the constant 1.\n"));
  P(OUT, _("-d, --color-preset id   this option can be used with the '--colorize'\n"
	   "                        one or the '--build-map' one. It chooses a\n"
	   "                        preset color scheme. The id parameter is an\n"
	   "                        integer that corresponds to a defined color\n"
	   "                        shade (ranging from 0).\n"));
  P(OUT, _("-p, --planes file       the argument fileToRender must be called,\n"
	   "                        then the given file of the option is parsed\n"
	   "                        as a list of planes and they are rendered.\n"));
  P(OUT, _("-t, --translate x:y:z   a file must be loaded. It applies the given\n"
	   "                        translations to the loaded file. The units are\n"
	   "                        those of the file. This is available for\n"
	   "                        periodic file formats only.\n"));
  P(OUT, _("-x, --expand x:y:z      a file must be loaded. It applies the given\n"
	   "                        expansion to the loaded file. The values are\n"
	   "                        given in box coordinates. This is available\n"
	   "                        for periodic file formats only.\n"));
  P(OUT, _("-s, --spin-file file    use the given file as a spin indicator. If this\n"
	   "                        option is used, V_Sim switches automatically to\n"
	   "                        spin rendering whatever method is specified in\n"
	   "                        the parameter file.\n"));
  P(OUT, _("-m, --hiding-mode id    policy used to show or not null modulus spins\n"
	   "                        possible values are positives.\n"));
  P(OUT, _("-a, --spin-and-atomic   always draws atomic rendering on node position\n"
	   "                        in addition to spin rendering.\n"));
  P(OUT, _("-f, --scalar-field file the argument fileToRender must be called,\n"
	   "                        then the given file of the option is parsed\n"
	   "                        as a scalar field and loaded.\n"));
  P(OUT, _("-v, --iso-values v[:v]  must be used with the '--scalar-field' option,\n"
	   "                        then the given surfaces are built and rendered.\n"
	   "                        If a name is appended to a value using # as\n"
	   "                        separator, this name is used as the name for\n"
	   "                        the iso-surface (i.e. 0.25#Blue).\n"));
  P(OUT, _("-i, --iso-surfaces file the argument fileToRender must be given,\n"
	   "                        then the given file of the option is parsed\n"
	   "                        and surfaces are rendered.\n"));
  P(OUT, _("-b, --build-map id      the argument fileToRender must be given, as the\n"
	   "                        '--planes', '--color-preset' and\n"
	   "                        '--scalar-field' options used, then the given\n"
	   "                        plane 'id' is replaced by a coloured map using\n"
	   "                        given scalar field and shade. 'id' ranges from\n"
	   "                        0.\n"));
  P(OUT, _("    --log-scale         if given, plots are done with logarithm scale.\n"));
  P(OUT, _("-n, --n-iso-lines val   when positive, val isolines are plotted on\n"
	   "                        the coloured map.\n"));
  P(OUT, _("    --fit-to-box val    if val is not TRUE, the surfaces use their own\n"
	   "                        bounding box (default is TRUE).\n"));
  P(OUT, _("    --bg-image file     draw the given image on the background.\n"));
  P(OUT, _("-o, --option id=value   this is a generic way to give extended option.\n"
	   "                        to V_Sim. As much as -o can be used. Each one\n"
	   "                        store a key and its value (boolean, integer or.\n"
	   "                        float).\n"));
}

int parseCommandLine(int argc, char **argv)
{
  struct option long_options[] = {
    {"export"         , required_argument, 0, 'e'},
    {"resources"      , required_argument, 0, 'r'},
    {"help"           , no_argument,       0, 'h'},
    {"geometry"       , required_argument, 0, 'g'},
    {"spin-file"      , required_argument, 0, 's'},
    {"hiding-mode"    , required_argument, 0, 'm'},
    {"spin-and-atomic", no_argument,       0, 'a'},
    {"colorize"       , required_argument, 0, 'c'},
    {"use-column"     , required_argument, 0, 'u'},
    {"color-preset"   , required_argument, 0, 'd'},
    {"translate"      , required_argument, 0, 't'},
    {"expand"         , required_argument, 0, 'x'},
    {"planes"         , required_argument, 0, 'p'},
    {"scalar-field"   , required_argument, 0, 'f'},
    {"iso-values"     , required_argument, 0, 'v'},
    {"iso-surfaces"   , required_argument, 0, 'i'},
    {"build-map"      , required_argument, 0, 'b'},
    {"log-scale"      , no_argument,       0, 0}, 
    {"n-iso-lines"    , required_argument, 0, 'n'}, 
    {"fit-to-box"     , required_argument, 0, 0}, 
    {"bg-image"       , required_argument, 0, 0}, 
    {"option"         , required_argument, 0, 'o'},
    {0, 0, 0, 0}
  };
  int res, i, nb, valueInt;
  int option_index;
  gchar **tokens, **tokens2;
  Option *option;
  float valueFloat;

  /* We want to read . as floating point separator : in french it is , */
  setlocale(LC_NUMERIC, "C");

  argFileName = (char*)0;
  argSpinFileName = (char *)0;
  spinHidingMode = 1;
  spinAndAtomic = FALSE;
  argExportFileName = (char*)0;
  argColorizeFileName = (gchar*)0;
  for (i = 0; i < 2; i++)
    argColorizeColUsed[i] = 0;
  argColorizeColUsed_isPresent = FALSE;
  argColorizePresetColor = -1;
  argTranslationsIsSet = FALSE;
  argExpandIsSet       = FALSE;
  xWindowWidth = 600;
  xWindowHeight = 600;
  argPlanesFileName = (gchar*)0;
  argScalarFieldFileName = (gchar*)0;
  argIsoSurfacesFileName = (gchar*)0;
  argIsoValues = (float*)0;
  argFitToBox = TRUE;
  argOptionTable = (OptionTable*)0;
  argMapPlaneId = -1;
  argLogScale = FALSE;
  argNIsoLines = 0;
  argBgImageFile = (gchar*)0;

  withGtk = 1;

  option = (Option*)0;
  option_index = 0;
  opterr = 0;
  while (1)
    {
      res = getopt_long (argc, argv, "e:r:hg:s:m:ac:u:d:t:x:p:f:v:i:b:n:",
			 long_options, &option_index);

      /* test if there's no more option available. */
      if (res == -1)
	break;

      switch (res)
	{
	case 'e':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found with arg '%s'.\n",
		      long_options[option_index].name, optarg);
	  if (!optarg)
	    g_error("The option 'export' needs a parameter.\n");
	  else
	    argExportFileName = g_strdup(optarg);
	  /* This option does not need gtk. */
	  withGtk = 0;
	  break;
	case 'r':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found with arg '%s'.\n",
		      long_options[option_index].name, optarg);
	  if (!optarg)
	    g_error("The option 'resources' needs a parameter.\n");
	  else
	    argResources = g_strdup(optarg);
	  break;
	case 'h':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found.\n",
		      long_options[option_index].name);
	  printInfoMessage();
	  exit(0);
	  break;
	case 'g':
	  DBG_fprintf(stderr, "Visu Command: set the geometry of the X window (%s).\n", optarg);
	  res = sscanf(optarg, "%dx%d", &xWindowWidth, &xWindowHeight);
	  if (res != 2 || xWindowWidth <= 0 || xWindowHeight <=0)
	    {
	      g_warning("Wrong format for geometry"
			" option (<width>x<height> awaited).\n");
	      xWindowWidth = 600;
	      xWindowHeight = 600;
	      break;
	    }
	  break;
	case 's':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " spin rendering to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'spin-file' needs a parameter.\n");
	    }
	  argSpinFileName = g_strdup(optarg);
	  break;
	case 'm':
	  DBG_fprintf(stderr, "Visu Command: set hiding-mode to '%s' in spin rendering.\n", optarg);
	  res = sscanf(optarg, "%d", &spinHidingMode);
	  if (res != 1 || spinHidingMode < 0)
	    {
	      g_warning("Wrong format for hiding-mode"
			" option (integer awaited).\n");
	      spinHidingMode = 0;
	    }
	  break;
	case 'a':
	  DBG_fprintf(stderr, "Visu Command: set spin-and-atomic to TRUE in spin rendering.\n");
	  spinAndAtomic = TRUE;
	  break;
	case 'b':
	  res = sscanf(optarg, "%d", &argMapPlaneId);
	  if (res != 1 || argMapPlaneId < 0)
	    {
	      g_warning("Wrong format for build-map option (id >= 0 awaited).\n");
	      argMapPlaneId = 0;
	      break;
	    }
	  DBG_fprintf(stderr, "Visu Command: set the plane for coloured map '%d'.\n",
		      argMapPlaneId);
	  break;
	case 'n':
	  res = sscanf(optarg, "%d", &argNIsoLines);
	  if (res != 1 || argNIsoLines < 0)
	    {
	      g_warning("Wrong format for n-iso-lines option (id >= 0 awaited).\n");
	      argNIsoLines = 0;
	      break;
	    }
	  DBG_fprintf(stderr, "Visu Command: set the number of isolines"
		      " for coloured map '%d'.\n", argNIsoLines);
	  break;
	case 'c':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for colorization to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'colorize' needs a parameter.\n");
	    }
	  argColorizeFileName = g_strdup(optarg);
	  break;
	case 'u':
	  DBG_fprintf(stderr, "Visu Command: set the used columns for"
		      " colorize data (%s).\n", optarg);
	  res = sscanf(optarg, "%d:%d:%d", argColorizeColUsed,
		       argColorizeColUsed + 1, argColorizeColUsed + 2);
	  if (res != 3 || argColorizeColUsed[0] < 0 ||
	      argColorizeColUsed[1] < 0 || argColorizeColUsed[2] < 0)
	    {
	      g_warning("Wrong format for use-column"
			" option (<channel-1>:<channel-2>:<channel-3> awaited).\n");
	      for (i = 0; i < 2; i++)
		argColorizeColUsed[i] = 0;
	      break;
	    }
	  argColorizeColUsed_isPresent = TRUE;
	  break;
	case 'd':
	  DBG_fprintf(stderr, "Visu Command: set a previously defined color scheme (%s).\n", optarg);
	  res = sscanf(optarg, "%d", &argColorizePresetColor);
	  if (res != 1 || argColorizePresetColor < 0)
	    {
	      g_warning("Wrong format for color-preset"
			" option (positive integer awaited).\n");
	      argColorizePresetColor = -1;
	      break;
	    }
	  break;
	case 't':
	  DBG_fprintf(stderr, "Visu Command: set the translations (%s).\n", optarg);
	  res = sscanf(optarg, "%f:%f:%f", argTranslations,
		       argTranslations + 1, argTranslations + 2);
	  if (res != 3)
	    {
	      g_warning("Wrong format for translation"
			" option (<x>:<y>:<z> awaited).\n");
	      break;
	    }
	  else
	    argTranslationsIsSet = TRUE;
	  break;
	case 'x':
	  DBG_fprintf(stderr, "Visu Command: set the extension (%s).\n", optarg);
	  res = sscanf(optarg, "%f:%f:%f", argExtension,
		       argExtension + 1, argExtension + 2);
	  if (res != 3 || argExtension[0] < 0. ||
	      argExtension[1] < 0. || argExtension[2] < 0.)
	    {
	      g_warning("Wrong format for expand"
			" option (<x>:<y>:<z> awaited >= 0.).\n");
	      break;
	    }
	  else
	    argExpandIsSet = TRUE;
	  break;
	case 'p':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for planes to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'planes' needs a parameter.\n");
	    }
	  argPlanesFileName = g_strdup(optarg);
	  break;
	case 'f':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " a scalar field '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'scalar-field' needs a parameter.\n");
	    }
	  argScalarFieldFileName = g_strdup(optarg);
	  break;
	case 'v':
	  DBG_fprintf(stderr, "Visu Command: get the values for"
		      " a scalar field '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'iso-values' needs a parameter.\n");
	    }
	  tokens = g_strsplit(optarg, ":", -1);
	  /* Count number of tokens. */
	  for (nb = 0; tokens[nb]; nb++);
	  argIsoValues = g_malloc(sizeof(float) * (nb + 1));
	  argIsoNames = g_malloc(sizeof(gchar*) * (nb + 1));
	  nb = 0;
	  for (i = 0; tokens[i]; i++)
	    {
	      tokens2 = g_strsplit(tokens[i], "#", 2);
	      res = sscanf(tokens2[0], "%f", argIsoValues + nb);
	      if (res != 1)
		g_warning("Parse error reading values from option "
			  "'iso-values', unknown number '%s'.\n", tokens[i]);
	      else
		{
		  if (tokens2[1])
		    /* A name for the value is given. */
		    argIsoNames[nb] = g_strdup(tokens2[1]);
		  else
		    argIsoNames[nb] = (gchar*)0;
		  nb += 1;
		}
	      g_strfreev(tokens2);
	    }
	  argNbIsoValues = nb;
	  g_strfreev(tokens);
	  break;
	case 'i':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " an isosurfaces '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'iso-surfaces' needs a parameter.\n");
	    }
	  argIsoSurfacesFileName = g_strdup(optarg);
	  break;
	case 'o':
	  DBG_fprintf(stderr, "Visu Command: read an extended option.\n");
	  if (!optarg)
	    {
	      g_error("The option 'option' needs a parameter.\n");
	    }
	  tokens = g_strsplit(optarg, "=", 2);
	  if (!tokens[1])
	    {
	      g_error("The option 'option' needs a key and a value (key=value),"
		      " just the key is readable.\n");
	    }
	  if (!argOptionTable)
	    argOptionTable = toolOptionsNew_table();
	  if (tokens[1][0] == 'T')
	    {
	      option = toolOptionsNew_optionBoolean(tokens[0], tokens[0]);
	      toolOptionsSet_valueBoolean(option, TRUE);
	    }
	  else if (tokens[1][0] == 'F')
	    {
	      option = toolOptionsNew_optionBoolean(tokens[0], tokens[0]);
	      toolOptionsSet_valueBoolean(option, FALSE);
	    }
	  else if (sscanf(tokens[0], "%f", &valueFloat) == 1)
	    {
	      option = toolOptionsNew_optionFloat(tokens[0], tokens[0]);
	      toolOptionsSet_valueFloat(option, valueFloat);
	    }
	  else if (sscanf(tokens[0], "%d", &valueInt) == 1)
	    {
	      option = toolOptionsNew_optionInteger(tokens[0], tokens[0]);
	      toolOptionsSet_valueInteger(option, valueInt);
	    }
	  else
	    {
	      g_error("Unparsable value for option 'option'.\n");
	    }
	  toolOptionsAdd_optionToTable(argOptionTable, option);
	  g_strfreev(tokens);
	  break;
	case 0:
	  /* Long option only. */
	  if (!strcmp(long_options[option_index].name, "fit-to-box"))
	    argFitToBox = (!strcmp(optarg, "TRUE"));
	  else if (!strcmp(long_options[option_index].name, "log-scale"))
	    argLogScale = TRUE;
	  else if (!strcmp(long_options[option_index].name, "bg-image"))
	    argBgImageFile = g_strdup(optarg);
	  break;
	default:
	  g_warning("Unknown option '%s'.", argv[optind - 1]);
	  printInfoMessage();
	  exit(0);
	}
    }

  if (argc - optind == 1)
    {
      DBG_fprintf(stderr, "Visu Command: there is one argument '%s'.\n", argv[optind]);
      argFileName = g_strdup(argv[optind]);
    }
  else if (argc - optind == 0)
    return 0;
  else
    g_warning("This program allows only up to one argument.\n");

  /* Consistency check between options. */
  if (argScalarFieldFileName && argIsoSurfacesFileName)
    {
      g_error("The options --iso-surfaces and --scalar-field are exclusive.");
    }

  return 0;
}

char* commandLineGet_ArgFilename()
{
  return argFileName;
}

char* commandLineGet_ArgSpinFileName()
{
  return argSpinFileName;
}

int commandLineGet_WithGtk()
{
  return withGtk;
}

char* commandLineGet_ExportFileName()
{
  return argExportFileName;
}

void commandLineGet_XWindowGeometry(int *width, int *height)
{
  *width = xWindowWidth;
  *height = xWindowHeight;
}
gchar* commandLineGet_colorizeFileName()
{
  return argColorizeFileName;
}
int* commandLineGet_colorizeColUsed()
{
  if (argColorizeColUsed_isPresent)
    return argColorizeColUsed;
  else
    return (int*)0;
}
float* commandLineGet_translation()
{
  if (argTranslationsIsSet)
    return argTranslations;
  else
    return (float*)0;
}
float* commandLineGet_extension()
{
  if (argExpandIsSet)
    return argExtension;
  else
    return (float*)0;
}
int commandLineGet_presetColor()
{
  return argColorizePresetColor;
}
gchar* commandLineGet_planesFileName()
{
  return argPlanesFileName;
}
int commandLineGet_spinHidingMode()
{
  return spinHidingMode;
}
gboolean commandLineGet_spinAndAtomic()
{
  return spinAndAtomic;
}
float* commandLineGet_isoValues(int *nb)
{
  g_return_val_if_fail(nb, (float*)0);

  *nb = argNbIsoValues;
  return argIsoValues;
}
gchar** commandLineGet_isoNames(int *nb)
{
  g_return_val_if_fail(nb, (gchar**)0);

  *nb = argNbIsoValues;
  return argIsoNames;
}
gchar* commandLineGet_scalarFieldFileName()
{
  return argScalarFieldFileName;
}
gchar* commandLineGet_isoSurfacesFileName()
{
  return argIsoSurfacesFileName;
}
gboolean commandLineGet_fitToBox()
{
  return argFitToBox;
}
OptionTable* commandLineGet_options()
{
  return argOptionTable;
}
gchar* commandLineGet_resourcesFile()
{
  return argResources;
}
gboolean commandLineGet_coloredMap(int *planeId)
{
  *planeId = argMapPlaneId;
  return (argMapPlaneId >= 0);
}
gboolean commandLineGet_logScale()
{
  return argLogScale;
}
guint commandLineGet_nIsoLines()
{
  return (guint)argNIsoLines;
}
gchar* commandLineGet_bgImage()
{
  return argBgImageFile;
}
