
/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *	Jim Duchek <jim@linuxpimps.com>
 */

#include "sis6326glx.h"
#include "glx_symbols.h"

/*
 * sis6326
 * perform hardware accelerated clearing of the color and/or depth
 * buffer.  Software may still clear stencil buffers.
 * If all==GL_TRUE, clear whole buffer, else just clear region defined
 * by x,y,width,height
 */
GLbitfield sis6326Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
		     GLint x, GLint y, GLint width, GLint height )
{
	int		screenFormat;
	
	hwMsg( 10, "sis6326Clear( %i, %i, %i, %i, %i )\n", mask, x, y, width, height );

	if (all == GL_TRUE) {
		x = 0;
		y = 0;
		width = sis6326DB->width;
		height = sis6326DB->height;
	}

	if ( y + height > sis6326DB->height ) {
		height = sis6326DB->height - y;
	}
	if ( x + width > sis6326DB->width ) {
		width = sis6326DB->width - x;
	}
	if ( x < 0 ) {
		width += x;
		x = 0;
	}
	if ( y < 0 ) {
		height += y;
		y = 0;
	}
	if ( x >= sis6326DB->width || y >= sis6326DB->height || width < 1 || height < 1 ) {
		return 0;
	}

	/* bump the performance counter */
	sis6326glx.c_clears++;

	/* flip top to bottom */
	y = sis6326DB->height - y - height;

	/* color buffer */
	/* mesa doesn't let us just not clear the color buffer, so do it ourselves
	even if we are in software rendering */
	if ( (mask & GL_COLOR_BUFFER_BIT)) {
		int		r, g, b,a;
		int		color;

		r = 255 * ctx->Color.ClearColor[0];
		g = 255 * ctx->Color.ClearColor[1];
		b = 255 * ctx->Color.ClearColor[2];

		if ( sis6326DB->backBufferBlock ) {
			color = sis6326PackColor(r,g,b,255);
			WAITFIFOEMPTY(6);
			OUTREG(SIS6326_BITBLT_DST_ADDR, sis6326DB->backBufferBlock->ofs + ((y * sis6326DB->pitch + x) * sis6326glx.bytesPerPixel));
			OUTREG(SIS6326_BITBLT_DST_SRC_PITCH, ((sis6326DB->pitch * sis6326glx.bytesPerPixel) << 16));
			OUTREG(SIS6326_BITBLT_HEIGHT_WIDTH, (((height-1) << 16) | (width * sis6326glx.bytesPerPixel)));
			OUTREG(SIS6326_BITBLT_FG_COLOR, ((0xF0 << 24) | color));
			OUTREG(SIS6326_BITBLT_BG_COLOR, ((0xF0 << 24) | color));
			OUTREG(SIS6326_BITBLT_CMD, ((0x0 << 8) | (0x31)));
		} else {
			/* software clear */
			int		i, j;
			int		pixel;
			
			pixel = sis6326PackColor(r,g,b,a);
			if ( sis6326glx.bytesPerPixel == 4 ) {
				for ( i = 0 ; i < height ; i++ ) {
					unsigned *dest;
	
					dest = (unsigned *)sis6326DB->backBuffer + 
						( y + i ) * sis6326DB->pitch + x;
				
					for ( j = 0 ; j < width ; j++ ) {
						dest[j] = pixel;
					}
				}	
			} else {
				for ( i = 0 ; i < height ; i++ ) {
					unsigned short *dest;
	
					dest = (unsigned short *)sis6326DB->backBuffer + 
						( y + i ) * sis6326DB->pitch + x;
				
					for ( j = 0 ; j < width ; j++ ) {
						dest[j] = pixel;
					}
				}	
			}	
		}		
		mask &= ~GL_COLOR_BUFFER_BIT;
	}

	/* depth buffer */
	if ( mask & GL_DEPTH_BUFFER_BIT ) {
		if ( 1 /* !ctx->Depth.Mask FIXME: check this! */ ) {
			int zval = (ctx->Depth.Clear * 65535.0);

			if ( sis6326DB->depthBufferBlock ) {
				/* hardware accelerated clear */
				WAITFIFOEMPTY(6);
				OUTREG(SIS6326_BITBLT_DST_ADDR, sis6326DB->depthBufferBlock->ofs + ((y * sis6326DB->pitch + x) * 2));
				OUTREG(SIS6326_BITBLT_DST_SRC_PITCH, ((sis6326DB->pitch * 2) << 16));
				OUTREG(SIS6326_BITBLT_HEIGHT_WIDTH, (((height-1) << 16) | (width * 2)));
				OUTREG(SIS6326_BITBLT_FG_COLOR, ((0xF0 << 24) | zval));
				OUTREG(SIS6326_BITBLT_BG_COLOR, ((0xF0 << 24) | zval));
				OUTREG(SIS6326_BITBLT_CMD, ((0x0 << 8) | (0x31)));
			} else {
				/* software clear */
				int		i, j;

				for ( i = 0 ; i < height ; i++ ) {
					unsigned short *dest;

					dest = (unsigned short *)sis6326DB->depthBuffer +
						( y + i ) * sis6326DB->pitch + x;

					for ( j = 0 ; j < width ; j++ ) {
						dest[j] = zval;
					}
				}
			}
		}
		mask &= ~GL_DEPTH_BUFFER_BIT;
	}

	hwMsg(10, "Out of Clear.\n");
	return mask;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
